#             Asymptotics expansions for x > nu
#                `besseljy_large_argument`
#
#  This file contains the asymptotic asymptotic expansions for large arguments 
#  which is accurate in the fresnel regime |x| > nu following the derivations given by Heitman [1].
#  These routines can be used to calculate `besselj` and `bessely` using phase functions.
#  Convergence is much slower when x < 2*nu where it becomes more efficient to use other methods.
#
# [1] Heitman, Z., Bremer, J., Rokhlin, V., & Vioreanu, B. (2015). On the asymptotics of Bessel functions in the Fresnel regime. 
#     Applied and Computational Harmonic Analysis, 39(2), 347-356.

#####
##### Cutoff points where expansion is valid
#####

# Expansions have a minimum x value independent of nu where they won't provide good accuracy depending on input type
# Cutoffs were determined manually for each input type. 
# AbstractFloat cutoff gives relative error roughly for quadruple precision accuracy (1e-35)

besseljy_large_argument_min(x::Float32) = 15.0f0
besseljy_large_argument_min(x::Float64) = 20.0
#besseljy_large_argument_min(x) = 40.0

besseljy_large_argument_cutoff(v, x::Float32) = (x > 1.2f0*v && x > besseljy_large_argument_min(x))
besseljy_large_argument_cutoff(v, x::Float64) = (x > 1.65*v && x > besseljy_large_argument_min(x))
#besseljy_large_argument_cutoff(v, x) = (x > 4*v && x > besseljy_large_argument_min(x))

"""
    besseljy_large_argument(nu, x::T)

Asymptotic expansions for large arguments valid when x > 1.6*nu and x > 20.0.
Returns both (besselj(nu, x), bessely(nu, x)).
"""
function besseljy_large_argument(v, x::T) where T
    α, αp = _α_αp_asymptotic(v, x)
    S = promote_type(T, Float64)
    v, x = S(v), S(x)
    b = SQ2OPI(S) / sqrt(αp * x)

    # we need to calculate sin(x - v*pi/2 - pi/4) and cos(x - v*pi/2 - pi/4)
    # For improved accuracy this is expanded using the formula for sin(x+y+z)
    s, c = sincos(PIO2(S) * v)
    sα, cα = sincos(α)

    CMS = c - s
    CPS = c + s

    s1 = CMS * cα
    s2 = CPS * sα

    s3 = CMS * sα
    s4 = CPS * cα

    return SQ2O2(S) * (s1 + s2) * b, SQ2O2(S) * (s3 - s4) * b
end

# Float64
# can only use for x > 20.0
# 30 terms gives ~5e-16 relative error when x > 1.6*nu
# 20 terms gives ~2e-16 relative error when x > 2*nu
# 15 terms gives ~5e-18 relative error when x > 3*nu
# 10 terms gives ~3e-18 relative error when x > 5*nu
# 5 terms give  ~5e-15 relative erros when x > 20*nu
# 3 terms give  ~5e-15 relative erros when x > 50*nu
# 2 terms give  ~5e-15 relative erros when x > 500*nu

# Float32
# can only use for x > 15.0
# 30 terms gives ~2e-8 relative error when x > 1.2*nu
# 20 terms gives ~8e-8 relative error when x > 1.3*nu
# 15 terms gives ~8e-8 relative error when x > 1.4*nu
# 10 terms gives ~3e-8 relative error when x > 1.8*nu
# 5 terms give  ~8e-8 relative erros when x > 4*nu
# 3 terms give  ~2e-8 relative erros when x > 8*nu

# Float128
# can only use for x > 40.0
# 30 terms gives ~5e-34 relative error when x > 4*nu
# 20 terms gives ~6e-34 relative error when x > 5.5*nu
# 15 terms gives ~6e-34 relative error when x > 8.5*nu
# 10 terms gives ~3e-34 relative error when x > 30*nu

# can fit a rational function to trend (numer of terms on y axis; x/nu on x axis)
# to get the number of terms needed for desired precision as function of x/nu
# terms = a / (x/nu) + b / (x/nu)^2 + c
#
# to get number of terms required for Float64 precision
# a = 27.7479; b = 22.3588; c = 3.74567
# this method requires significantly more terms when x gets closer to nu
# so it becomes more efficient to use recurrence (or another algorithm) in this region

function _α_αp_asymptotic(v, x::Float64)
    if x > 5*v
        return _α_αp_poly_10(v, x)
    elseif x > 2*v
        return _α_αp_poly_20(v, x)
    else
        return _α_αp_poly_30(v, x)
    end
end
function _α_αp_asymptotic(v, x::Float32)
    v, x = Float64(v), Float64(x)
    if x > 4*v
        return _α_αp_poly_5(v, x)
    elseif x > 1.8*v
        return _α_αp_poly_10(v, x)
    else
        return _α_αp_poly_30(v, x)
    end
end

function _α_αp_poly_5(v, x::T) where T
    xinv = inv(x)^2
    μ = 4 * v^2

    s5 = evalpoly(μ, (211.27614974975586, -246.8455924987793, 37.067405700683594, -1.5151596069335938, 0.017223358154296875, -2.6702880859375e-5))
    s4 = evalpoly(μ, (-11.466461181640625, 13.1358642578125, -1.71624755859375, 0.0469970703125, -0.000152587890625))
    s3 = evalpoly(μ, (1.0478515625, -1.1591796875, 0.1123046875, -0.0009765625))
    s2 = evalpoly(μ, (-0.1953125, 0.203125, -0.0078125))
    s1 = evalpoly(μ, (0.125, -0.125))
    
    αp = evalpoly(xinv, (1.0, s1, s2, s3, s4, s5))
    α = x * evalpoly(xinv, (1.0, -s1, -s2/3, -s3/5, -s4/7, -s5/9))
    return α, αp
end


function _α_αp_poly_10(v, x::T) where T
    xinv = inv(x)^2
    μ = 4 * v^2

    s10 = evalpoly(μ, (-7.181611223268327e10, 8.659091218989731e10, -1.576853447634526e10, 1.0243675365394899e9, -3.1127612822052997e7, 499001.19046756154, -4427.281508191307, 21.557856538012857, -0.053067848184582544, 5.209072696743533e-5, -8.843926480039954e-9))
    s9 = evalpoly(μ, (8.369564497799761e8, -1.0061560971482614e9, 1.80175839935865e8, -1.1293844119552301e7, 322272.3393258663, -4655.44006793818, 34.77845393284224, -0.12591543863527477, 0.0001761710154823959, -4.1618477553129196e-8))
    s8 = evalpoly(μ, (-1.225106763286615e7, 1.4670878180751745e7, -2.5692206274838205e6, 153362.43350451812, -4001.4928083932027, 49.41718461737037, -0.2788656409829855, 0.0005833245813846588, -1.9976869225502014e-7))
    s7 = evalpoly(μ, (231884.6359563172, -276225.9131319225, 46889.68168082833, -2607.1347218453884, 59.29277476668358, -0.5644365847110748, 0.0018794238567352295, -9.834766387939453e-7))
    s6 = evalpoly(μ, (-5892.0457146167755, 6965.548173427582, -1128.6143367290497, 56.11658954620361, -1.0105445384979248, 0.005837917327880859, -5.0067901611328125e-6))
    s5 = evalpoly(μ, (211.27614974975586, -246.8455924987793, 37.067405700683594, -1.5151596069335938, 0.017223358154296875, -2.6702880859375e-5))
    s4 = evalpoly(μ, (-11.466461181640625, 13.1358642578125, -1.71624755859375, 0.0469970703125, -0.000152587890625))
    s3 = evalpoly(μ, (1.0478515625, -1.1591796875, 0.1123046875, -0.0009765625))
    s2 = evalpoly(μ, (-0.1953125, 0.203125, -0.0078125))
    s1 = evalpoly(μ, (0.125, -0.125))

    P = (1.0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10)
    α, αp = split_phase_evalpoly(xinv, P)
    return α*x, αp
end

function _α_αp_poly_20(v, x::T) where T
    xinv = inv(x)^2
    μ = 4 * v^2
    s20 = evalpoly(μ, (-1.1655823449858069e34, 1.4226232333293694e34, -2.7683048917097874e33, 2.0549175959945544e32, -7.766193794571668e30, 1.7289192571208397e29, -2.473355129610242e27, 2.406335820680262e25, -1.6548511836241395e23, 8.261988206301513e20, -3.048956971903377e18, 8.409146358453666e15, -1.7416884938569994e13, 2.7048317602344254e10, -3.1216507081066556e7, 26272.21159305084, -15.578788452840763, 0.0061163849392160095, -1.4107081622121022e-6, 1.4671084930995856e-10, -2.9236916960932355e-15))
    s19 = evalpoly(μ, (3.1436617397720927e31, -3.8346121964186915e31, 7.438172213027661e30, -5.487658176284203e29, 2.0543110479331377e28, -4.512190593153004e26, 6.33949148154985e24, -6.024489803885022e22, 4.020727203383251e20, -1.9328978639294188e18, 6.802550072768408e15, -1.767751414955178e13, 3.3964900375451138e10, -4.792703592867111e7, 48822.71795456563, -34.73905348751307, 0.01624286720956346, -4.43222001918643e-6, 5.421693881235296e-10, -1.2642991118241018e-14))
    s18 = evalpoly(μ, (-9.432675683095812e28, 1.1498154949132059e29, -2.2224123014148547e28, 1.6283472309548362e27, -6.03034763221407e25, 1.3043849059659297e24, -1.7950710383376238e22, 1.6602291470434327e20, -1.0700309670904285e18, 4.92026735148727e15, -1.6365119263912514e13, 3.957355516860315e10, -6.930551343828379e7, 86599.59532013819, -74.82883257079231, 0.042123857348712865, -1.3735762526818516e-5, 1.994760566671595e-9, -5.490670428493242e-14))
    s17 = evalpoly(μ, (3.167717078292175e26, -3.8584266638650446e26, 7.4277307068246446e25, -5.399729057323595e24, 1.9752248974202056e23, -4.197842582814349e21, 5.640147455618286e19, -5.0538188484912346e17, 3.1258357574848365e15, -1.3629748027461303e13, 4.233000035928751e10, -9.36296364559977e7, 145724.23973885347, -155.01792293897955, 0.10637041031529318, -4.191933776256961e-5, 7.303030923530292e-9, -2.3959289142515966e-13))
    s16 = evalpoly(μ, (-1.1986350446061028e24, 1.458736721654471e24, -2.795293505867967e23, 2.0139038975089457e22, -7.262922096794037e20, 1.5122633519520676e19, -1.975555191650512e17, 1.7050149800082885e15, -1.0037673890747877e13, 4.1025284625807785e10, -1.1700278886842367e8, 230919.39384718135, -307.22947128726474, 0.260626935759175, -0.0001257334987819726, 2.658906071879852e-8, -1.051117201091023e-12))
    s15 = evalpoly(μ, (5.149237018485509e21, -6.260437088008504e21, 1.1933270317945127e21, -8.508441473311652e19, 3.0180281284560543e18, -6.1344355915029736e16, 7.750474619361342e14, -6.393706192980096e12, 3.543452904857713e10, -1.3358182185805681e8, 341472.1144099866, -578.7806110577233, 0.6169921485050097, -0.00036976968916281605, 9.619824624385043e-8, -4.639413853091412e-12))
    s14 = evalpoly(μ, (-2.5330554576498356e19, 3.0761765051109855e19, -5.827690025164462e18, 4.1047854971841363e17, -1.4277576770556304e16, 2.819724813029272e14, -3.4214400566025547e12, 2.670056925037507e10, -1.3717353508462998e8, 465890.6725849346, -1028.1881576107903, 1.4039031226739573, -0.0010631539871280848, 3.4554354377824836e-7, -2.061961712485072e-11))
    s13 = evalpoly(μ, (1.4409993748135704e17, -1.747635582560889e17, 3.286944341694018e16, -2.2818860422499875e15, 7.75248528894739e13, -1.4783938404512664e12, 1.7064572063331423e10, -1.2415577730993801e8, 578046.6774168271, -1705.2186748496474, 3.0504712266430722, -0.0029777336087200013, 1.2309086638850886e-6, -9.237588471933122e-11))
    s12 = evalpoly(μ, (-9.588236853265692e14, 1.161016591203577e15, -2.164822377227822e14, 1.4768015151140877e13, -4.8748767898914606e11, 8.900147704533855e9, -9.641151011038888e7, 640028.5410983711, -2606.385824929625, 6.277422845836725, -0.008087900592073538, 4.342405588886322e-6, -4.176996526439325e-10))
    s11 = evalpoly(μ, (7.562132599676877e12, -9.139342212003133e12, 1.686263984004851e12, -1.1257747938365613e11, 3.584384568770644e9, -6.188986771037158e7, 616601.7257690447, -3609.808395761912, 12.103919447801218, -0.021179858537379914, 1.5144118606258417e-5, -1.909484126372263e-9))
    s10 = evalpoly(μ, (-7.181611223268327e10, 8.659091218989731e10, -1.576853447634526e10, 1.0243675365394899e9, -3.1127612822052997e7, 499001.19046756154, -4427.281508191307, 21.557856538012857, -0.053067848184582544, 5.209072696743533e-5, -8.843926480039954e-9))
    s9 = evalpoly(μ, (8.369564497799761e8, -1.0061560971482614e9, 1.80175839935865e8, -1.1293844119552301e7, 322272.3393258663, -4655.44006793818, 34.77845393284224, -0.12591543863527477, 0.0001761710154823959, -4.1618477553129196e-8))
    s8 = evalpoly(μ, (-1.225106763286615e7, 1.4670878180751745e7, -2.5692206274838205e6, 153362.43350451812, -4001.4928083932027, 49.41718461737037, -0.2788656409829855, 0.0005833245813846588, -1.9976869225502014e-7))
    s7 = evalpoly(μ, (231884.6359563172, -276225.9131319225, 46889.68168082833, -2607.1347218453884, 59.29277476668358, -0.5644365847110748, 0.0018794238567352295, -9.834766387939453e-7))
    s6 = evalpoly(μ, (-5892.0457146167755, 6965.548173427582, -1128.6143367290497, 56.11658954620361, -1.0105445384979248, 0.005837917327880859, -5.0067901611328125e-6))
    s5 = evalpoly(μ, (211.27614974975586, -246.8455924987793, 37.067405700683594, -1.5151596069335938, 0.017223358154296875, -2.6702880859375e-5))
    s4 = evalpoly(μ, (-11.466461181640625, 13.1358642578125, -1.71624755859375, 0.0469970703125, -0.000152587890625))
    s3 = evalpoly(μ, (1.0478515625, -1.1591796875, 0.1123046875, -0.0009765625))
    s2 = evalpoly(μ, (-0.1953125, 0.203125, -0.0078125))
    s1 = evalpoly(μ, (0.125, -0.125))

    P = (1.0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, s16, s17, s18, s19, s20)
    α, αp = split_phase_evalpoly(xinv, P)
    return α*x, αp
end

function _α_αp_poly_30(v, x::T) where T
    xinv = inv(x)^2
    μ = 4 * v^2
    s30 = even_odd_poly_add(μ, (-7.592029625744128e61, 9.300807898884993e61, -1.8453653700408117e61, 1.4213662513968141e60, -5.684267643368571e58, 1.3688721521734006e57, -2.1717210753568686e55, 2.410362055956752e53, -1.953160895382593e51, 1.1926329508677187e49, -5.621526725185766e46, 2.084010993835956e44, -6.16598654084696e41, 1.472798231204338e39, -2.8653790379578294e36, 4.5711759648399396e33, -6.008229949414824e30, 6.525582948238488e27, -5.863525845645105e24, 4.3559252317865593e21, -2.6682229171870643e18, 1.3409000748848645e15, -5.4845348768282446e11, 1.8042426705573353e8, -46913.0207873715, 9.39407807634952, -0.0013913710768178739, 1.4251320396079487e-7, -8.917800204478063e-12, 2.6068908049173686e-16, -1.5080064817014915e-21))
    s29 = even_odd_poly_add(μ, (8.871640240590993e58, -1.0865691207568645e59, 2.153044379065757e58, -1.654292598675498e57, 6.591312178536168e55, -1.5792626981209235e54, 2.4890507915218916e52, -2.7398535977577976e50, 2.1978557435213864e48, -1.3258581553903244e46, 6.160064953128667e43, -2.24525395106052e41, 6.512651083776791e38, -1.520126754804397e36, 2.879357904563898e33, -4.453253892004961e30, 5.646779978807957e27, -5.882834513685185e24, 5.036061410449725e21, -3.535410901341911e18, 2.0262400502486065e15, -9.410029770358712e11, 3.500781492364049e8, -102565.9203518178, 23.065519747567176, -0.003825019881917803, 4.374396753566091e-7, -3.048458029283708e-11, 9.900967356259312e-16, -6.3495009755852275e-21))
    s28 = even_odd_poly_add(μ, (-1.1113660587118494e56, 1.360796163396552e56, -2.692648562464633e55, 2.0634410442908336e54, -8.188692555802185e52, 1.9512412867297912e51, -3.0534326199740573e49, 3.3310997303957132e47, -2.6429272553053182e45, 1.573358131379098e43, -7.195503085038196e40, 2.5742410447166102e38, -7.305470671594484e35, 1.6621868613473995e33, -3.056128428896728e30, 4.5656808004766417e27, -5.560280477035352e24, 5.525982611412455e21, -4.4762143769251487e18, 2.944057191090032e15, -1.561376913130651e12, 6.60436019199837e8, -219122.78889988337, 55.602561558610454, -0.010370022987617815, 1.3297453832052619e-6, -1.0361746982713131e-10, 3.753418608703549e-15, -2.6783349569741323e-20))
    s27 = even_odd_poly_add(μ, (1.4962552090981924e53, -1.8315342344152888e53, 3.6186332587139524e52, -2.7651411770117913e51, 1.092593756883467e50, -2.5880088398643682e48, 4.0185482602383215e46, -4.341325948517365e44, 3.40330197791503e42, -1.996788871925338e40, 8.974814912978575e37, -3.1454308144632116e35, 8.712781142577153e32, -1.926817207371682e30, 3.4266421642703216e27, -4.923376108039163e24, 5.727723667496557e21, -5.393876507549732e18, 4.099268265682976e15, -2.498492707776039e12, 1.2086394541382809e9, -456609.83322422515, 131.40817769348385, -0.027695807224792045, 4.000233368100494e-6, -3.500503095889607e-10, 1.4200746208673407e-14, -1.1319755289852937e-19))
    s26 = even_odd_poly_add(μ, (-2.170796074547977e50, 2.6563886780963143e50, -5.239744551853566e49, 3.9915243203539826e48, -1.5697710844957169e47, 3.694226454576932e45, -5.687790414754595e43, 6.07923257088654e41, -4.703231744077508e39, 2.715669597352839e37, -1.1973880447446774e35, 4.101798034247621e32, -1.1059082744278117e30, 2.3689970250205793e27, -4.057770223206878e24, 5.577674944614657e21, -6.157880219725851e18, 5.448993990587677e15, -3.8435237604702446e12, 2.1401262084993436e9, -926134.0995426074, 303.9782133922326, -0.07278018905169194, 1.1898982367956518e-5, -1.1748038677566303e-9, 5.361217221358988e-14, -4.794249299231832e-19))
    s25 = even_odd_poly_add(μ, (3.4037481878842518e47, -4.163722422572948e47, 8.198385202264364e46, -6.22435990386475e45, 2.4353618503436353e44, -5.690731110001809e42, 8.680616408346572e40, -9.169497515863974e38, 6.991529767882026e36, -3.9660440566717967e34, 1.7117897755074633e32, -5.716342746132164e29, 1.495186253411991e27, -3.089698915505606e24, 5.071094031947313e21, -6.62564031827467e18, 6.884682491993269e15, -5.663687113496129e12, 3.6558083666029415e9, -1.8241160174217927e6, 687.0067933018984, -0.1879264018835715, 3.496541491784091e-5, -3.914759734732158e-9, 2.0193378048364477e-13, -2.035109906612696e-18))
    s24 = even_odd_poly_add(μ, (-5.786059033524315e44, 7.075318200534661e44, -1.3904375539321083e44, 1.051770598467428e43, -4.0921342782576984e41, 9.487906954286141e39, -1.4325415965940442e38, 1.493677625028016e36, -1.1206631946323043e34, 6.2329465039514265e31, -2.626782977025989e29, 8.52403241555426e26, -2.1544180262833415e24, 4.273241060880315e21, -6.678193343397941e18, 8.226793217207085e15, -7.9615210223039795e12, 6.004451011853662e9, -3.4795218728555855e6, 1513.831326319914, -0.4760701870709445, 0.00010139500470517733, -1.2944568154864108e-8, 7.586889731852131e-13, -8.660042155798706e-18))
    s23 = even_odd_poly_add(μ, (1.0699708275991454e42, -1.3078533633276038e42, 2.5647480608865752e41, -1.9322501955238208e40, 7.471528599453178e38, -1.7175047656287374e37, 2.56398527252613e35, -2.6350896428114214e33, 1.9417708082703794e31, -1.0563755426904998e29, 4.333903992576568e26, -1.3614388053852227e24, 3.3089693719445074e21, -6.261128573879847e18, 9.243346554045246e15, -1.0625592945166727e13, 9.4457596395905e9, -6.4083469576330995e6, 3244.624388704194, -1.1811472686414501, 0.0002898188382446864, -4.2443950630387665e-8, 2.84271079789386e-12, -3.694951319807448e-17))
    s22 = even_odd_poly_add(μ, (-2.1603839510459968e39, 2.639521237493071e39, -5.164169654060501e38, 3.873385769490781e37, -1.4875551809791352e36, 3.3870740530372218e34, -4.993086633044377e32, 5.049452384590897e30, -3.646484825050138e28, 1.934929035289944e26, -7.699788293506606e23, 2.3306500988206443e21, -5.414879939834449e18, 9.69892178268169e15, -1.3389590376401615e13, 1.4169012254830626e10, -1.1355350548005087e7, 6745.941285011119, -2.8643019266046954, 0.0008154154591207295, -1.3789599324701242e-7, 1.061958568923219e-11, -1.5810954484757452e-16))
    s21 = even_odd_poly_add(μ, (4.7820456670665866e36, -5.839767806978455e36, 1.139609569657776e36, -8.505736729879162e34, 3.241923388548771e33, -7.303696541971675e31, 1.061598557628623e30, -1.0542835823932327e28, 7.4416439961740466e25, -3.8383112218012e23, 1.4749530840065974e21, -4.277160387520148e18, 9.428034079487212e15, -1.582775791680076e13, 2.0161618573658318e10, -1.9279964972155876e7, 13562.78784549061, -6.773533616718505, 0.0022547380647248, -4.4351435924730097e-7, 3.9542564728199e-11, -6.787141437359297e-16))
    s20 = evalpoly(μ, (-1.1655823449858069e34, 1.4226232333293694e34, -2.7683048917097874e33, 2.0549175959945544e32, -7.766193794571668e30, 1.7289192571208397e29, -2.473355129610242e27, 2.406335820680262e25, -1.6548511836241395e23, 8.261988206301513e20, -3.048956971903377e18, 8.409146358453666e15, -1.7416884938569994e13, 2.7048317602344254e10, -3.1216507081066556e7, 26272.21159305084, -15.578788452840763, 0.0061163849392160095, -1.4107081622121022e-6, 1.4671084930995856e-10, -2.9236916960932355e-15))
    s19 = evalpoly(μ, (3.1436617397720927e31, -3.8346121964186915e31, 7.438172213027661e30, -5.487658176284203e29, 2.0543110479331377e28, -4.512190593153004e26, 6.33949148154985e24, -6.024489803885022e22, 4.020727203383251e20, -1.9328978639294188e18, 6.802550072768408e15, -1.767751414955178e13, 3.3964900375451138e10, -4.792703592867111e7, 48822.71795456563, -34.73905348751307, 0.01624286720956346, -4.43222001918643e-6, 5.421693881235296e-10, -1.2642991118241018e-14))
    s18 = evalpoly(μ, (-9.432675683095812e28, 1.1498154949132059e29, -2.2224123014148547e28, 1.6283472309548362e27, -6.03034763221407e25, 1.3043849059659297e24, -1.7950710383376238e22, 1.6602291470434327e20, -1.0700309670904285e18, 4.92026735148727e15, -1.6365119263912514e13, 3.957355516860315e10, -6.930551343828379e7, 86599.59532013819, -74.82883257079231, 0.042123857348712865, -1.3735762526818516e-5, 1.994760566671595e-9, -5.490670428493242e-14))
    s17 = evalpoly(μ, (3.167717078292175e26, -3.8584266638650446e26, 7.4277307068246446e25, -5.399729057323595e24, 1.9752248974202056e23, -4.197842582814349e21, 5.640147455618286e19, -5.0538188484912346e17, 3.1258357574848365e15, -1.3629748027461303e13, 4.233000035928751e10, -9.36296364559977e7, 145724.23973885347, -155.01792293897955, 0.10637041031529318, -4.191933776256961e-5, 7.303030923530292e-9, -2.3959289142515966e-13))
    s16 = evalpoly(μ, (-1.1986350446061028e24, 1.458736721654471e24, -2.795293505867967e23, 2.0139038975089457e22, -7.262922096794037e20, 1.5122633519520676e19, -1.975555191650512e17, 1.7050149800082885e15, -1.0037673890747877e13, 4.1025284625807785e10, -1.1700278886842367e8, 230919.39384718135, -307.22947128726474, 0.260626935759175, -0.0001257334987819726, 2.658906071879852e-8, -1.051117201091023e-12))
    s15 = evalpoly(μ, (5.149237018485509e21, -6.260437088008504e21, 1.1933270317945127e21, -8.508441473311652e19, 3.0180281284560543e18, -6.1344355915029736e16, 7.750474619361342e14, -6.393706192980096e12, 3.543452904857713e10, -1.3358182185805681e8, 341472.1144099866, -578.7806110577233, 0.6169921485050097, -0.00036976968916281605, 9.619824624385043e-8, -4.639413853091412e-12))
    s14 = evalpoly(μ, (-2.5330554576498356e19, 3.0761765051109855e19, -5.827690025164462e18, 4.1047854971841363e17, -1.4277576770556304e16, 2.819724813029272e14, -3.4214400566025547e12, 2.670056925037507e10, -1.3717353508462998e8, 465890.6725849346, -1028.1881576107903, 1.4039031226739573, -0.0010631539871280848, 3.4554354377824836e-7, -2.061961712485072e-11))
    s13 = evalpoly(μ, (1.4409993748135704e17, -1.747635582560889e17, 3.286944341694018e16, -2.2818860422499875e15, 7.75248528894739e13, -1.4783938404512664e12, 1.7064572063331423e10, -1.2415577730993801e8, 578046.6774168271, -1705.2186748496474, 3.0504712266430722, -0.0029777336087200013, 1.2309086638850886e-6, -9.237588471933122e-11))
    s12 = evalpoly(μ, (-9.588236853265692e14, 1.161016591203577e15, -2.164822377227822e14, 1.4768015151140877e13, -4.8748767898914606e11, 8.900147704533855e9, -9.641151011038888e7, 640028.5410983711, -2606.385824929625, 6.277422845836725, -0.008087900592073538, 4.342405588886322e-6, -4.176996526439325e-10))
    s11 = evalpoly(μ, (7.562132599676877e12, -9.139342212003133e12, 1.686263984004851e12, -1.1257747938365613e11, 3.584384568770644e9, -6.188986771037158e7, 616601.7257690447, -3609.808395761912, 12.103919447801218, -0.021179858537379914, 1.5144118606258417e-5, -1.909484126372263e-9))
    s10 = evalpoly(μ, (-7.181611223268327e10, 8.659091218989731e10, -1.576853447634526e10, 1.0243675365394899e9, -3.1127612822052997e7, 499001.19046756154, -4427.281508191307, 21.557856538012857, -0.053067848184582544, 5.209072696743533e-5, -8.843926480039954e-9))
    s9 = evalpoly(μ, (8.369564497799761e8, -1.0061560971482614e9, 1.80175839935865e8, -1.1293844119552301e7, 322272.3393258663, -4655.44006793818, 34.77845393284224, -0.12591543863527477, 0.0001761710154823959, -4.1618477553129196e-8))
    s8 = evalpoly(μ, (-1.225106763286615e7, 1.4670878180751745e7, -2.5692206274838205e6, 153362.43350451812, -4001.4928083932027, 49.41718461737037, -0.2788656409829855, 0.0005833245813846588, -1.9976869225502014e-7))
    s7 = evalpoly(μ, (231884.6359563172, -276225.9131319225, 46889.68168082833, -2607.1347218453884, 59.29277476668358, -0.5644365847110748, 0.0018794238567352295, -9.834766387939453e-7))
    s6 = evalpoly(μ, (-5892.0457146167755, 6965.548173427582, -1128.6143367290497, 56.11658954620361, -1.0105445384979248, 0.005837917327880859, -5.0067901611328125e-6))
    s5 = evalpoly(μ, (211.27614974975586, -246.8455924987793, 37.067405700683594, -1.5151596069335938, 0.017223358154296875, -2.6702880859375e-5))
    s4 = evalpoly(μ, (-11.466461181640625, 13.1358642578125, -1.71624755859375, 0.0469970703125, -0.000152587890625))
    s3 = evalpoly(μ, (1.0478515625, -1.1591796875, 0.1123046875, -0.0009765625))
    s2 = evalpoly(μ, (-0.1953125, 0.203125, -0.0078125))
    s1 = evalpoly(μ, (0.125, -0.125))

    P = (1.0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30)
    α, αp = split_phase_evalpoly(xinv, P)
    return α*x, αp
end
# this function evaluates the phase and integral at the same time (more efficiently) for example given some polynomial P with coeffficients s0, s1, ...
# we use this function when the size of P is large (length(P) > 20) as it is more efficient
# P = (s0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30)
# this functions solves
# αp = evalpoly(xinv, (s0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29))
# α = x * (evalpoly(xinv, (s0, -s1, -s2/3, -s3/5, -s4/7, -s5/9, -s6/11, -s7/13, -s8/15, -s9/17, -s10/19, -s11/21, -s12/23, -s13/25, -s14/27, -s15/29, -s16/31, -s17/33, -s18/35, -s19/37, -s20/39, -s21/41, -s22/43, -s23/45, -s24/47, -s25/49, -s26/51, -s27/53, -s28/55, -s29/57)))
function split_phase_evalpoly(x, P)
    N = length(P)
    xx = x*x

    αp1 = P[end]
    αp2 = P[end-1]

    d = -muladd(2, N, -3)
    α1 = αp1 / d
    α2 = -αp2 / (d + 2)
    d += 4

    for i in N-2:-2:2
        αp1 = muladd(xx, αp1, P[i])
        αp2 = muladd(xx, αp2, P[i-1])

        α1 = muladd(xx, α1, P[i] / d)
        α2 = muladd(xx, α2, P[i-1] / (d + 2))
        d += 4
    end

    if iszero(rem(N, 2)) 
        return muladd(x, α1, α2), muladd(x, αp1, αp2) 
    else
        αp1 = muladd(xx, αp1, P[1])
        α1 = muladd(xx, α1, P[1])
        return  muladd(x, α2, α1), muladd(x, αp2, αp1)
    end
end

# performs a second order horner scheme for polynomial evaluation
# computes the even and odd coefficients of the polynomial independently within a loop to reduce latency
# works for both even and odd degree polynomials
@inline function even_odd_poly_add(x, P)
    N = length(P)
    xx = x*x

    out = P[end]
    out2 = P[end-1]

    for i in N-2:-2:2
        out = muladd(xx, out, P[i])
        out2 = muladd(xx, out2, P[i-1])
    end
    if iszero(rem(N, 2)) 
        return muladd(out, x, out2) 
    else 
        out = muladd(xx, out, P[1]) 
        return muladd(x, out2, out) 
    end
end

# the following are the coefficients in exact fractions which can be used for arbitrary precision calculations
function _α_αp_asymptotic(v, x::BigFloat)
    T = BigFloat
    xinv = inv(x)^2
    μ = 4 * T(v)^2
    s0 = one(T)
    s1 = (1 - μ) / 8
    s2 = evalpoly(μ, (-25, 26, -1)) / 128
    s3 = evalpoly(μ, (1073, -1187, 115, -1)) / 1024
    s4 = evalpoly(μ, (-375733, 430436, -56238, 1540, -5)) / 32768
    s5 = evalpoly(μ, (55384775, -64709091, 9716998, -397190, 4515, -7)) / 262144
    s6 = evalpoly(μ, (-24713030909, 29215626566, -4733751627, 235370036, -4238531, 24486, -21)) / 4194304
    s7 = evalpoly(μ, (7780757249041, -9268603618823, 1573356635461, -87480924739, 1989535379, -18939349, 63063, -33)) / 33554432
    s8 = evalpoly(μ, (-26308967412122125, 31505470994964360, -5517359285625804, 329343318168440, -8593140373614, 106122595896, -598859404, 1252680, -429)) / 2147483648
    s9 = evalpoly(μ, (14378802319925055947, -17285630127691126691, 3095397360215483916, -194026764558396188, 5536596631240042, -79979851361130, 597489288988, -2163210764, 3026595, -715)) / 17179869184
    s10 = evalpoly(μ, (-19740662615375374580231, 23801928703130666089534, -4334421752432088249971, 281576004385356401192, -8556293060689145118, 137164402798287604, -1216961874423502, 5925778483368, -14587179035, 14318590, -2431)) / 274877906944
    s11 = evalpoly(μ, (16629305448257355302267575, -20097626064642945009122253, 3708133715826433118900337, -247560495216085669748963, 7882145023568373727078, -136097258378143928354, 1355921534379626242, -7938052610367302, 26616800349043, -46575001473, 33302269, -4199)) / 2199023255552
    s12 = evalpoly(μ, (-67471218624230362526181277601, 81699279492428006742420030332, -15233583205322702897380357650, 1039206680182498322210164588, -34303895772550410316410943, 626292216963766324300664, -6784356890740220944060, 45038004674954790968, -183408097342762543, 441734362333708, -569135407698, 305569628, -29393)) / 70368744177664
    s13 = evalpoly(μ, (81121053093143918050102987600099,  -98383136980007985475571939027975, 18503851640550924693860888790250, -1284587641197372537980918233562, 43642612323123398148213493825, -832261743620394930038568309, 9606500048207046619311900, -69893489053616393418780, 325411350127146969525, -959952773579760385, 1717262634951770, -1676314996330, 692939975, -52003)) / 562949953421312
    s14 = evalpoly(μ, (-228157352303612028397376534489640325, 277077347242874183788397478754999038, -52491165251522858022045953021809887, 3697262087110858234228068530237484, -128600978847262037677344810847005, 2539782323449194167276463807490, -30817592327971506033132319983, 240497347453138555094414952, -1235549362984466458531719, 4196370118897802443810, -9261095606965424565, 12645235160277420, -9576039800535, 3112379550, -185725)) / 9007199254740992
    s15 = evalpoly(μ, (371041630683099249807974389133866983377, -451112034187704470803165906576733421647, 85988274811534523057441254130668147353, -6130978215793605437752467509125332519, 217471845675333794186294440183867941, -4420326695043375922589356428203067, 55848055372320369378877454341437, -460715085251545483652077162563, 2553326909107535727375084819, -9625584690294666238157261, 24605658995427696249035, -41705538308621290485, 44458969761562935, -26644714149225, 6931814175, -334305)) / 72057594037927936
    s16 = evalpoly(μ, (-11055456952814105837238472522288088536664893, 13454471487641055458960459954964458339717648, -2578203195682435666976648565353960682213208, 185749848931968563987894480305354570867632, -6698863257342477965063978897873174851692, 139481675127549715032105727774178109456, -1822128051193260958105844837015204328, 15725987489026953079045248214882608, -92581200678991239243237978998382, 378391463021683664380997574832, -1079160251083042147293256808, 2129855479977547288466320, -2833691714368635092460, 2403859191332320560, -1159686836761560, 245240799120, -9694845)) / 9223372036854775808
    s17 = evalpoly(μ, (23373626496469884942898309902021582362191922675, -28470163678198171216731489192530581706160067107, 5480697895889118815544614489283816937852790808, -398429679951285151272097987810023499140966008, 14574587268291892957048020704423735025526308, -309746111147584727757459007548519123051204, 4161694266070984983741813756535503455880, -37290601157202922291181203252458959208, 230645968939091255561083278221370498, -1005697894606904987899324994096258, 3123402733071240567892677464424, -6908647765673021748575217288, 10752551023194100444762820, -11438303805173540795940, 7848750944406750840, -3093101181782040, 538868569635, -17678835)) / 73786976294838206464
    s18 = evalpoly(μ, (-111361378724077990746416029998046616050509978921539, 135746253866557423310337353662450724289897542908910, -26237613408296752490352020229134990522792887791555, 1922413096483679043237547734628813570357439558480, -71193778846050128718754631795120857490490262940, 1539945890173645033009867772439624727099098888, -21192458264539017819644078826589342129355820, 196005261947029152728400647710967544425520, -1263269593655108070478649445999468867210, 5808826406855320404193987142684204180, -19320522675016202958543734862586394, 46720207634051084004638119118640, -81821508434755789651511496140, 102238756592473887441057800, -88342292721143516508540, 49731073018185955920, -16216326143326155, 2354997610350, -64822395)) / 1180591620717411303424
    s19 = evalpoly(μ, (296910456667588147659333126541181625529873470176387611, -362168882223415616126501055899626700448641997962436017, 70251550305228319355272475757988113288672713290068649, -5182946608226016261211397666049774183662333597959843, 194024192762965317463261535095424641741096903713740, -4261643524285090368342561247478476304909756975140, 59874804181817284044814186630099605178719724308, -568996974525130964287568610366892787350713372, 3797469476403853060187807038916907856444554, -18255704174861238533251400128344883661470, 64248268923367993194175398884766099134, -166959400640594454410097385012706986, 320789414254074176405482585976668, -452658056185691378860864973396, 461117533742477601983387620, -328101083672095141216140, 153409543392289776435, -41861134526619705, 5120645093025, -119409675)) / 9444732965739290427392
    s20 = evalpoly(μ, (-3522756479350769975281617581321545613436218503050695678075, 4299614895888804153638859172375174476343649375612057689548, -836668815038360513614015909500916883814666806927189456922, 62106073474456005680385514002110287639480560824420522572, -2347188049597150004773910816777966916365233392848818383, 52253378249058169018788729968582230155091725819403248, -747525719315527247430397518232787428614442113610808, 7272703760709817711041445391295675625095681479472, -50014808087576306896550631589875112629326379366, 249703271598736427498837466608381894119961384, -921490701557006964516502956379076579410908, 2541508538413242992463438567320581532520, -5263930474873605158754566071747209670, 8174852381652706781000230751680944, -9434610352121065739568921445176, 7940288733304487926761165680, -4708399899738360283359135, 1848563918930071955820, -426360380309828250, 44340633437100, -883631595)) / 302231454903657293676544
    s21 = evalpoly(μ, (11562276954986118012905102199683696291379295134175971428710681, -14119692164821110293665160297543402876375196726712887276312589, 2755406866078403474813432478656763881170404858419576582318658, -205656094951908425907179975052824686393072466342278961851682, 7838489779258317368644351328312885893374241287287353084601, -176592546564392815596485971618573980366372738228899109621, 2566787812765782386429803068290417713174219989532914232, -25491012879019728230635774403534977262786325300355192, 179927911347099874189887493701009277459349916599058, -928046707950409032184149251437603291238784411418, 3566217731951601523416356645128571391393644908, -10341539254212039773512248413902653009876588, 22795587653797465357095136224432136887194, -38269170424460587731145121489014195346, 48747802517834826879462968473185912, -46616094912209768370505289864248, 32792808824738132450074479925, -16377399358557323704388985, 5451622125827462423970, -1072351920527800770, 95608054947405, -1641030105)) / 2417851639229258349412352 
    s22 = evalpoly(μ, (-83575806038418315461424176577671655789355064557837328722810810363, 102111532013648986390182274849560087911661660674277144494797501754, -19977913701325081051291504480331200861144453506726454660745981317, 1498443541140890887172300157818040316416714142527798061250817212, -57547003724390053887422306176263229331882560978641824071987057, 1310310808212309513721016150148686278236604804419891613787474, -19316068320832072657353419945987646188788279627258096785919, 195341227604691010556855787014189633450322695864689236944, -1410665489867534572898459538217020115453133972348502334, 7485394143628926776948795385198929132335854093895444, -29787113195477122503729199863502980387883252587074, 90162658590453244373324847901139399254001605096, -209478021420133615969565305958096327404239122, 375208862896212544106945468927927982466132, -517984688643055190232315977490791270222, 548137112105626862660002531852535248, -439288846984271326190410593921911, 260970963105735443862130025650, -110807313735818965400395785, 31544857671806945481660, -5334592853046971085, 410825322666330, -6116566755)) / 38685626227668133590597632
    s23 = evalpoly(μ, (331139932088074229583414385100484655583680323667219743432336893156375, -404761010994322659981740621417860548337506367222213959945034952405505, 79375107881275741011145002562545397482099096278323149191979217848795, -5980024707389856192480549544360322543977054376671317889548708990765, 231232610198222735170897082923113351106311448954569056231279745105, -5315419792588167914145829820169688781060054431572005135138461431, 79351500724953350171887383695707759766649852034188528391425045, -815520743985617485488470918603434536997805249424041983026435, 6009489576683595225413677597538524950365012612475993703590, -32693239520459805541072665529457785351317642575409616330, 134127831970732580141105863110451473403320285914221422, -421344902055805903571049169837400505463026456600930, 1024076418574775828492136039568457134302239053890, -1937725438179908789638378729405097765038430670, 2860677199060789249610678847660962264016810, -3288461736992539205784287844732495870726, 2923321014828730506772945071926726355, -1983287321121666482640542827954725, 1004162610804693250453533074375, -365547374035954079474050785, 89694586000567590763485, -13135766477702211915, 879776379205425, -11435320455)) / 309485009821345068724781056
    s24 = evalpoly(μ, (-11460470635629791545293372484006750122731817670656977258332556688633398921, 14014111505802311026665654556958455368190143118279169566775230812828711608, -2754045312221731365346800318745138513760434182644947440110635145128118868, 208324629757756119539887071410085261846231054839500347331614562331870856, -8105306990699817776465002381403873045496081436849592851326212688042930, 187927358523600279487081628876937144137383565949179064175790471702376, -2837440960584915871183716261829833363096514551129018419965987130436, 29585333404910010295435037804846483274351006167295070952283026520, -221970214270207127150632290469188405104591920361499699260389575, 1234562246394448342011056423641453973409852545995108093923760, -5202879714837953377224769143131813528966172834588809583784, 16883585637409754238127214329328823438340567417621760208, -42672645377509299293082661872272090002235553201735996, 84640259308513240076102188894192666414468595521872, -132275246878102596227254163245303375859517889320, 162948427496032610672483414621797748943261744, -157694170353957908988423048181218400323495, 118930405144020217084786109082007952664, -68919031103634934082636426953210628, 29984518585214620810437808556840, -9429541537363253968235837490, 2008334977729096678954440, -256393587362636099220, 15027383316324600, -171529806825)) / 19807040628566084398385987584
    s25 = evalpoly(μ, (53934542917465252257872000137728594960259370390866516239913881025916526618079, -65976815351979191514013436582276621696037856185100381323811701400123478350151, 12990859903190819767771277173922248418040797693438193819626123246996727763924, -986289196021334429598185221309997436594513963183161881910145274005112055556, 38589848892013007447965728844309701385952705788909865709295830738464656854, -901732338416406447829099270632322366537973763750405411674611994211884982, 13754985750489437020150084701211239326753483840283744285085048168887348, -145296487872202817071834551731992888383906147134262530708479742919492, 1107852113346147934451530333019409136722319574358275961326610338745, -6284447661214506168310202232419921778046818764894109430624191425, 27124391704832276375378035229827447472267671305885516783908200, -90579066415559029033446553653965206980661508469619065122888, 236921718948838484290563377929843042878945006979548048852, -489582335595648899206286073510649002814836556024449492, 803546924176475361367194165347536643073090978134088, -1049874615794655300129423477494049697668965354856, 1090921486669506169688142948646361250033103041, -897447046116032022931540194969487137324729, 579285958780450202216840934739373500228, -289042720546332850118715441421680692, 108860571736252830951542456446390, -29778127018305105531623841750, 5540491150978764772764420, -620318440935315513300, 31997684754555975, -322476036831)) / 158456325028528675187087900672
    s26 = evalpoly(μ, (-550362189374765423500020131694326871682947579295544126673454219279357840448787475, 673474540445653165097469530811639715030184284678513241901879111803771300974560774, -132843306523995345830830790301920189208579760537806586066532903781971404919028471, 10119716401044602598547436013299934060816525295000642596922239980740289514077464, -397984251496382810057558626001455259349582556891230525089025890178624740805426, 9365976765046902955936322300984376349463066112864638170465082090070459810828, -144202618664720643539154196259069508936470334045124013174312118132635730258, 1541268563482264821733153244434252845039676711071618846131329709235798072, -11924109086784630671902444007218893865417530446493249726209690870741433, 68850403902117609730665474296934648018000334821883146969897474938330, -303573934725339276628453846224414840422909218274232063761029728957, 1039929348025793684367471554854627430067811771656990473105571888, -2803810575751562150406590598593666192836314853381860916413356, 6006121001412454529935820912360762597858661908957386258344, -10287669718072870028476255463139544436162416218994465932, 14141085982837452113877229626968207248935797973216048, -15612081113338031406691783944922755502249136499861, 13814841005616967281507601209269830356947816730, -9744490403903133848524108305681120523846465, 5425864545536715503374357393342487043768, -2348028894354034128703204634739556482, 770676329325936928765925025265420, -184519700672202596543835960450, 30167504281690397397158040, -2978481656224275543615, 135923004572192838, -1215486600363)) / 2535301200456458802993406410752
    s27 = evalpoly(μ, (3034766102252701518157589464572634173727080006716405186421561822693903178222865453267, -3714792754552146381143399215357421132324512594933927037160786389545967909052342612209, 733946019586332057694627344519609691505114169934498357977357346958058975757065505005, -56083725964076635895744531425138465919432829322712289673161638652063716379066462215, 2216043410750309023932982985070868698985478186407806393141360294004890025943198090, -52491055348000482104122152189863250131359278473985994635664946071742204717323854, 815058418261954448107293230750826473065550287087398221401112444183687821279806, -8805255111079079336374463414380115404435700759914469371598474477417223017370, 69027164720990526029240636658930131205677059730900234667535242433097074161, -404996897924032627265358852167699489700618554104702707868970603595198491, 1820308721819928686570797781124614778454010567761204382775217813876903, -6379691615889053454414290791655251340101395503722656901263667198869, 17671619592072200881064220041665643126872351595003533197000304732, -39080495831276692607567240277191495033180367037410340189342132, 69500559940874112365836960661406024121439848953110460077172, -99857930856082720326359361513785333411820343692595376412, 116172037520695141323879606006603233207541346449393285, -109400812677637805606914204549170843912544659089943, 83143038039832916637956789152649648497314155371, -50675452490850399934703562637702549877087681, 24514120471966581387575303368968009290106, -9261147666508811758073848019756164382, 2665274485248682075861896295849070, -561737706437007489700084464810, 81134371682009320652590935, -7099863760968377393085, 288025351281817569, -2295919134019)) / 20282409603651670423947251286016
    s28 = evalpoly(μ, (-72131781191647102180135979166018125213328009812988639643980373344199924546055031219841657, 88320720553877044898016539857591409048541934216984061246144406353930481760466659135310060, -17476288340029290013244384062063718700017020569598272422382306529077447996191083120944250, 1339249806505389936360679880350661005143397811790702381254671942509756480768691050430780, -53147653291249013517567215476928286085680620344084765055885408763582768328625269986515, 1266428000416318763460583208865588275947660408675059698889268273293866046141098646088, -19817910750548832205286511761977528667151953139167471097483757983625632824415854052, 216200733319998411897616656521528132036058294484119226909293000883829666320165320, -1715357860632234663803938553870596959687369271423120785238819271587494683750241, 10211678103637396902724570320414980517346978304968771663479533793948404676100, -46701485080026857458041597651829261081340501743039967654292046914103115238, 167077796119918348572545316722339899638809234995433324513630413185024980, -474152155227979464991161392169467129570306842144689118967678169058355, 1078820952308996014951396457847753147289281214958590369482040062640, -1983540755079932526194570933278727505224710158375196934944380920, 2963296259673455351858339135295437559242914546967478677045808, -3608828356685398684317572170795607088256593908628099432291, 3586567769194371717203744378289395696784836160566682420, -2905229230929609915630859596422130790818280882266278, 1910802363080455301530232591650967617033975786020, -1013391555129635701107834029596318350575760529, 428647483469320100810247742835039527501960, -142218821054781826089282598333880233700, 36088125713407805490928679853329800, -6730529722692741129622542935235, 863054096983483927763931228, -67251582884092645505082, 2436107956342200140, -17383387729001)) / 649037107316853453566312041152512
    s29 = evalpoly(μ, (460641897512717772444684472005371318928375783673161434624040792742355634436184850536871203675, -564178943228681784408128036377843627979833026582101974626289086576214861275629425222702448847, 111792455657091985902006439173302723560601542405476379242249078132812659086463986818301116178, -8589578263200204701470353360882571455715307585778549660943403684958602296026835769312573986, 342240495182356923410762074560927370205390456163884415965761227013197038811878947336320765, -8200000746254507295304096959671163015672807825961507454633484615000277725706606017152241, 129238906055527429629281249867910788422871402842227214411781858297630187356971956903236, -1422613322848315696593640888405216513575461590729332023127907398193731352234855477764, 11411919472598822437353439396939566156317984524435454538058672839354271255407920619, -68842491350959631082153901358155985737647859495682120491706731168043077431302735, 319848859045004698475705125867452815773463133628860238924577037584188264819934, -1165802503670444737174000655118383418045988242850127477428261836544612848430, 3381561776302767309753579289493706303221455529588479829370843821075902949, -7892949373545612140613272341863738840408667132927648754066972556970953, 14950481002464553128848507962612606727458261014192936783633863754360, -23122616193715352907955207763774332462501115298399429292264871032, 29319757944801916188347969656769373510907436077849765422140169, -30545423164687847712975609948618551041034999948860404860325, 26148725840866580994656722166781495459996912551090067118, -18356902916667388449505459868039364581979090609382750, 10520839847543290180652936889961450086351879969551, -4885966801535274414624198426720325555016669227, 1817709674521871694893131527594007075961604, -532552706035476905379118498186676823108, 119763025725766078479808600789766705, -19860638716715065331302542747837, 2271316652152616314798490274, -158284990488250672349842, 5140876170036110735, -32968493968795)) / 5192296858534827628530496329220096
    s30 = evalpoly(μ, (-6307211452104732174943448902500511410516752502135177977516479793936266104375354563403736592992661, 7726808901634633472764423487483987147912793024327355069790059175019923981633065149186691588112366, -1533069570193898638776969946027497754413700119418200914994976211124695059017720280563720165116679, 118082488351281628256212364427051509440951666348540358507793500087772127619198529380156671977052, -4722304804437407702774315126971884792839130791614396352799127518832880404913753883259609681633, 113721449207452105340760342113513866981469050815323620717113645049876718668309441770210232122, -1804195282750295484432387280834799906106198812198732749587879723377689138780815596231991491, 20024504529721268400575321293109749034087666595823477974729050069147156000605788822571736, -162262258900929736852882667680735927088036383095118777414049084086989027288294045992673, 990800691868092463873685007902517162497606862693782162842631470654113482500613909342, -4670181688856261225457142767754524003917312775851464792868017769459561728687864187, 17313285978314366363824489788003621278946387177174138648380949803667760165547492, -51225012073292313036848862087206475698191501546208104766878180668936339463693, 122355290066206966065274608457537774339544059725077257041985190390602745786, -238046377236799748837585319989470526266751308352706861653647596375755319, 379758441632773228328580291161418519539456909127730054048670703056080, -499144215867223345245911370032323062802545746779345685104627948303, 542124221475954160835505089973782190763373791511623456852516698, -487122669252493559223512526400713858958886216235592498481701, 361876110352287076667069743255190067307582588617193373284, -221667287532496449333210348405388543867822016580450643, 111397619942940758738781350025908679042794150454014, -45563733138367989946335361691477264600710356601, 14989061680590944114321512213343142442045656, -3897381287338369243170639191805209007419, 780429473354568256573331148798859610, -115590586739482666082002003622553, 11839533779605892925181369564, -740861855788043106212095, 21657201499065248270, -125280277081421)) / 83076749736557242056487941267521536
    s31 = evalpoly(μ, (46182079411090538580095089404942255695320530836210437733165170316163624612655848461364888827345905489, -56589815938738489648863263832934051923377354045914412489016056250146948672027944802988658352618891999, 11241583195200442692834110241769427027884989610098224891681009954933318276683113568160633339693595649, -867843403785258280181868308380383076404643209612762846277005473135727721639070438841777105882928751, 34826067703857573462990143002859739905376488212086953273803145530518509255080039061262282200125561, -842632440660966679941827102290359515945138298364592425230657557998803436135121413134292330107575, 13450170654960047302052941916727450371659908826619113751880711216129789590258433346038732332425, -150423827466671554817403807085438951680192267112529521939086066646687932474851014386490102247, 1230303131217372600115912102438485397226173492790596780852223405648551065572686437981930085, -7596722836105864768528712697484067335883422776696888240530364001145803974202857888950923, 36283778218425900991948662450401425089152917764886969580527341133988577931147563015701, -136612913412746323823099395629801246971887588829206879963137829532692846755233552411, 411567070065282272951768798427985366408477463155756408128666571656469786690052445, -1003873047146554950160406507427005345385217243007283021792162491351924476578195, 2000921895269749382020198707732648329333292789542094982041611431151186749933, -3282459656413060227257926591956836941430986016122199166050314327357866563, 4455424759035844137149173708152328271622727252144185877924585628290579, -5021931687769280215531437107865192817259990605542464970848282769533, 4709965800840740080632684125405464125918977887569860130776024483, -3677067638747848313827963187329821467457994503962714399236525, 2386436645637972767431786780223292421918297198556726971627, -1283380965474037387777195267253649845095676641967230629, 568749034477374070950270653906653019830271331153819, -205967886701098007300310452992576713489985085621, 60211048305050119638506114985861585545790583, -13958644821679115348256350410386760028633, 2499753933136077047949624164807916679, -332043855649674625203038036426633, 30579938614455692722089247999, -1724653213420908752418129, 45539255438819452079, -238436656380769)) / 664613997892457936451903530140172288
    
    αp = evalpoly(xinv, (s0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10, s11, s12, s13, s14, s15, s16, s17, s18, s19, s20, s21, s22, s23, s24, s25, s26, s27, s28, s29, s30, s31))
    α = x * (evalpoly(xinv, (s0, -s1, -s2/3, -s3/5, -s4/7, -s5/9, -s6/11, -s7/13, -s8/15, -s9/17, -s10/19, -s11/21, -s12/23, -s13/25, -s14/27, -s15/29, -s16/31, -s17/33, -s18/35, -s19/37, -s20/39, -s21/41, -s22/43, -s23/45, -s24/47, -s25/49, -s26/51, -s27/53, -s28/55, -s29/57, -s30/59, -s31/61)))
    return α, αp
end
