const GL_NW = permutedims(
    [
        -0.999953919435642 0.00011825670068171721
        -0.9997572122115891 0.00027526103865831744
        -0.9994033532930161 0.00043245460060038196
        -0.9988923197258543 0.0005896003421932499
        -0.998224182256913 0.0007466574055662421
        -0.9973990436722557 0.0009035982480626941
        -0.9964170329848294 0.0010603974326420873
        -0.9952783043339365 0.0012170300415936897
        -0.9939830366739069 0.0013734713364273322
        -0.992531433650476 0.0015296966649272486
        -0.9909237235316215 0.0016856814323170032
        -0.9891601591554368 0.0018414010924821323
        -0.9872410178826077 0.0019968311464057843
        -0.9851666015488034 0.002151947143493035
        -0.9829372364150317 0.002306724684154767
        -0.9805532731150799 0.0024611394229790977
        -0.9780150865996246 0.002615167072191434
        -0.9753230760767991 0.0027687834052614944
        -0.9724776649491127 0.002921964260586262
        -0.9694793007466641 0.00307468554521151
        -0.9663284550566241 0.0032269232385712526
        -0.963025623448975 0.0033786533962332863
        -0.959571325398504 0.0035298521536436056
        -0.9559661042030546 0.0036804957298652166
        -0.9522105268980449 0.003830560431308329
        -0.9483051841672582 0.003980022655449804
        -0.944250690249923 0.0041288588945403645
        -0.94004768284409 0.004277045739298241
        -0.9356968230063247 0.004424559882588365
        -0.931198795047727 0.004571378123086171
        -0.9265543064262952 0.004717477368925247
        -0.9217640876356519 0.0048628346413281485
        -0.9168288920901461 0.005007427078219678
        -0.911749496006352 0.0051512319378220145
        -0.9065266982809823 0.005294226602231066
        -0.901161320365234 0.005436388580973459
        -0.8956542061355872 0.0055776955145435516
        -0.890006221761078 0.0057181251779199384
        -0.8842182555670642 0.00585765548406083
        -0.8782912178955077 0.005996264487377793
        -0.8722260409617927 0.006133930387187249
        -0.866023678708105 0.006270631531139229
        -0.8596851066533939 0.006406346418622805
        -0.85321132173994 0.00654105370414767
        -0.8466033421765535 0.006674732200701343
        -0.8398622072784298 0.006807360883081451
        -0.8329889773036814 0.006938918891202568
        -0.8259847332865807 0.007069385533377089
        -0.818850576867531 0.007198740289569644
        -0.8115876301197994 0.007326962814624488
        -0.804197035373034 0.007454032941465406
        -0.796679955033597 0.007579930684267617
        -0.7890375714017384 0.007704636241601166
        -0.7812710864856423 0.007828129999545307
        -0.7733817218123715 0.007950392534773417
        -0.7653707182357452 0.00807140461760791
        -0.7572393357411735 0.008191147215044737
        -0.7489888532474854 0.008309601493746885
        -0.740620568405778 0.008426748823006541
        -0.732135797395319 0.008542570777675352
        -0.7235358747165359 0.008657049141062358
        -0.7148221529811238 0.008770165907799143
        -0.705996002699303 0.008881903286671764
        -0.6970588120642633 0.008992243703418954
        -0.6880119867338267 0.009101169803496257
        -0.6788569496093618 0.00920866445480557
        -0.6695951406119885 0.00931471075038971
        -0.6602280164561035 0.009419292011091571
        -0.6507570504202663 0.009522391788177445
        -0.641183732115479 0.00962399386592412
        -0.6315095672508971 0.0097240822641693
        -0.6217360773970098 0.009822641240825014
        -0.6118647997463229 0.009919655294353544
        -0.6018972868715882 0.010015109166205537
        -0.5918351064816105 0.010108987843219902
        -0.5816798411746766 0.010201276559985107
        -0.5714330881896406 0.010291960801161504
        -0.5610964591547072 0.010381026303764331
        -0.550671579833952 0.010468459059407015
        -0.5401600898716189 0.010554245316504453
        -0.529563642534233 0.010638371582435857
        -0.5188839044505712 0.010720824625666934
        -0.5081225553495348 0.01080159147783093
        -0.49728128779595404 0.010880659435768353
        -0.48636180692438263 0.01095801606352491
        -0.4753658301709075 0.011033649194307498
        -0.4642950870030309 0.011107546932397785
        -0.4531513186476536 0.011179697655023212
        -0.4419362778172123 0.011250090014185036
        -0.43065172843401034 0.011318712938443162
        -0.4192994453527847 0.011385555634657481
        -0.4078812140815536 0.011450607589685441
        -0.39639883050078745 0.011513858572035563
        -0.38485410058095026 0.011575298633476675
        -0.37324884009845394 0.011634918110602589
        -0.3615848743500686 0.01169270762635197
        -0.34986403786583664 0.011748658091483198
        -0.33808817412053493 0.011802760706003907
        -0.32625913524372974 0.011855006960555099
        -0.3143787817284691 0.011905388637749498
        -0.30244898213866317 0.011953897813463982
        -0.29047161281519085 0.012000526858085933
        -0.27844855758078807 0.012045268437713197
        -0.2663817074437531 0.012088115515307618
        -0.2542729603005287 0.012129061351801793
        -0.24212422063719571 0.012168099507159044
        -0.2299373992299318 0.012205223841386304
        -0.21771441284448234 0.012240428515499802
        -0.20545718393468665 0.012273707992443457
        -0.19316764034011163 0.012305057037959772
        -0.1808477149828374 0.012334470721413042
        -0.16849934556344323 0.012361944416564878
        -0.15612447425624337 0.012387473802301837
        -0.14372504740381872 0.012411054863315053
        -0.1313030152108904 0.01243268389073175
        -0.11886033143758944 0.01245235748269861
        -0.10639895309216538 0.012470072544916794
        -0.09392084012318516 0.01248582629112865
        -0.08142795511126816 0.01249961624355592
        -0.0689222629604074 0.012511440233289444
        -0.05640573058892591 0.012521296400630325
        -0.043880326620117996 0.01252918319538237
        -0.031348021072617915 0.01253509937709596
        -0.01881078505055355 0.012539044015263127
        -0.0062705904335270835 0.012541016489463895
        0.0062705904335270835 0.012541016489463895
        0.01881078505055355 0.012539044015263127
        0.031348021072617915 0.01253509937709596
        0.043880326620117996 0.01252918319538237
        0.05640573058892591 0.012521296400630325
        0.0689222629604074 0.012511440233289444
        0.08142795511126816 0.01249961624355592
        0.09392084012318516 0.01248582629112865
        0.10639895309216538 0.012470072544916794
        0.11886033143758944 0.01245235748269861
        0.1313030152108904 0.01243268389073175
        0.14372504740381872 0.012411054863315053
        0.15612447425624337 0.012387473802301837
        0.16849934556344323 0.012361944416564878
        0.1808477149828374 0.012334470721413042
        0.19316764034011163 0.012305057037959772
        0.20545718393468665 0.012273707992443457
        0.21771441284448234 0.012240428515499802
        0.2299373992299318 0.012205223841386304
        0.24212422063719571 0.012168099507159044
        0.2542729603005287 0.012129061351801793
        0.2663817074437531 0.012088115515307618
        0.27844855758078807 0.012045268437713197
        0.29047161281519085 0.012000526858085933
        0.30244898213866317 0.011953897813463982
        0.3143787817284691 0.011905388637749498
        0.32625913524372974 0.011855006960555099
        0.33808817412053493 0.011802760706003907
        0.34986403786583664 0.011748658091483198
        0.3615848743500686 0.01169270762635197
        0.37324884009845394 0.011634918110602589
        0.38485410058095026 0.011575298633476675
        0.39639883050078745 0.011513858572035563
        0.4078812140815536 0.011450607589685441
        0.4192994453527847 0.011385555634657481
        0.43065172843401034 0.011318712938443162
        0.4419362778172123 0.011250090014185036
        0.4531513186476536 0.011179697655023212
        0.4642950870030309 0.011107546932397785
        0.4753658301709075 0.011033649194307498
        0.48636180692438263 0.01095801606352491
        0.49728128779595404 0.010880659435768353
        0.5081225553495348 0.01080159147783093
        0.5188839044505712 0.010720824625666934
        0.529563642534233 0.010638371582435857
        0.5401600898716189 0.010554245316504453
        0.550671579833952 0.010468459059407015
        0.5610964591547072 0.010381026303764331
        0.5714330881896406 0.010291960801161504
        0.5816798411746766 0.010201276559985107
        0.5918351064816105 0.010108987843219902
        0.6018972868715882 0.010015109166205537
        0.6118647997463229 0.009919655294353544
        0.6217360773970098 0.009822641240825014
        0.6315095672508971 0.0097240822641693
        0.641183732115479 0.00962399386592412
        0.6507570504202663 0.009522391788177445
        0.6602280164561035 0.009419292011091571
        0.6695951406119885 0.00931471075038971
        0.6788569496093618 0.00920866445480557
        0.6880119867338267 0.009101169803496257
        0.6970588120642633 0.008992243703418954
        0.705996002699303 0.008881903286671764
        0.7148221529811238 0.008770165907799143
        0.7235358747165359 0.008657049141062358
        0.732135797395319 0.008542570777675352
        0.740620568405778 0.008426748823006541
        0.7489888532474854 0.008309601493746885
        0.7572393357411735 0.008191147215044737
        0.7653707182357452 0.00807140461760791
        0.7733817218123715 0.007950392534773417
        0.7812710864856423 0.007828129999545307
        0.7890375714017384 0.007704636241601166
        0.796679955033597 0.007579930684267617
        0.804197035373034 0.007454032941465406
        0.8115876301197994 0.007326962814624488
        0.818850576867531 0.007198740289569644
        0.8259847332865807 0.007069385533377089
        0.8329889773036814 0.006938918891202568
        0.8398622072784298 0.006807360883081451
        0.8466033421765535 0.006674732200701343
        0.85321132173994 0.00654105370414767
        0.8596851066533939 0.006406346418622805
        0.866023678708105 0.006270631531139229
        0.8722260409617927 0.006133930387187249
        0.8782912178955077 0.005996264487377793
        0.8842182555670642 0.00585765548406083
        0.890006221761078 0.0057181251779199384
        0.8956542061355872 0.0055776955145435516
        0.901161320365234 0.005436388580973459
        0.9065266982809823 0.005294226602231066
        0.911749496006352 0.0051512319378220145
        0.9168288920901461 0.005007427078219678
        0.9217640876356519 0.0048628346413281485
        0.9265543064262952 0.004717477368925247
        0.931198795047727 0.004571378123086171
        0.9356968230063247 0.004424559882588365
        0.94004768284409 0.004277045739298241
        0.944250690249923 0.0041288588945403645
        0.9483051841672582 0.003980022655449804
        0.9522105268980449 0.003830560431308329
        0.9559661042030546 0.0036804957298652166
        0.959571325398504 0.0035298521536436056
        0.963025623448975 0.0033786533962332863
        0.9663284550566241 0.0032269232385712526
        0.9694793007466641 0.00307468554521151
        0.9724776649491127 0.002921964260586262
        0.9753230760767991 0.0027687834052614944
        0.9780150865996246 0.002615167072191434
        0.9805532731150799 0.0024611394229790977
        0.9829372364150317 0.002306724684154767
        0.9851666015488034 0.002151947143493035
        0.9872410178826077 0.0019968311464057843
        0.9891601591554368 0.0018414010924821323
        0.9909237235316215 0.0016856814323170032
        0.992531433650476 0.0015296966649272486
        0.9939830366739069 0.0013734713364273322
        0.9952783043339365 0.0012170300415936897
        0.9964170329848294 0.0010603974326420873
        0.9973990436722557 0.0009035982480626941
        0.998224182256913 0.0007466574055662421
        0.9988923197258543 0.0005896003421932499
        0.9994033532930161 0.00043245460060038196
        0.9997572122115891 0.00027526103865831744
        0.999953919435642 0.00011825670068171721
    ],
)

"""
    abstract type AbstractParametricCurve <: Function end 

Abstract type for representing a parametric curve parametrised over `0 ≤ t ≤ 1`. The curves represented by this 
abstract type should not be self-intersecting, with the exception of allowing for closed curves.

The structs that subtype this abstract type must implement are:
- [`differentiate`](@ref).
- [`twice_differentiate`](@ref).
- [`thrice_differentiate`](@ref) (only if you have not manually defined [`total_variation`](@ref)).
- The struct must be callable so that `c(t)`, where `c` an instance of the struct, returns the associated value of the curve at `t`.
- If the struct does not implement [`point_position_relative_to_curve`](@ref), then the struct must implement [`get_closest_point`](@ref). Alternatively,
   rather than implementing [`get_closest_point`](@ref), the struct should have a `lookup_table` field as a `Vector{NTuple{2,Float64}}`, which returns values on the curve at a set of points, 
   where `lookup_table[i]` is the value of the curve at `t = (i - 1) / (length(lookup_table) - 1)`.

Functions that are defined for all [`AbstractParametricCurve`](@ref) subtypes are:
- [`arc_length`](@ref)
- [`curvature`](@ref)
- [`total_variation`](@ref)

!!! note "Efficiently computing the total variation"

    The curves in this package evaluate the total variation not by evaluating the integral itself, but by taking care of the changes in orientation in the curve
    to efficiently compute it. This is done by using the orientation markers of the curves, obtained using [`orientation_markers`](@ref), that stored in the field 
    `orientation_markers` of these curves. The function [`marked_total_variation`](@ref) is then used to evaluate it. You may like to consider using these functions for 
    any curve you wish to implement yourself, using e.g. the [`BezierCurve`](@ref) struct's implementation as a reference.
"""
abstract type AbstractParametricCurve <: Function end # defined in t ∈ [0, 1]

Base.show(io::IO, c::C) where {C<:AbstractParametricCurve} = print(io, string(C))

"""
    is_curve_bounded(c::AbstractParametricCurve) -> Bool

Returns `true` if `c` is not a [`PiecewiseLinear`](@ref) curve. This is equivalent to `!is_piecewise_linear(c)`.
"""
is_curve_bounded(c::AbstractParametricCurve) = !is_piecewise_linear(c)

"""
    is_piecewise_linear(c::AbstractParametricCurve) -> Bool

Returns `true` if `c` is [`PiecewiseLinear`](@ref), and `false` otherwise.
"""
is_piecewise_linear(::AbstractParametricCurve) = false

"""
    is_linear(c::AbstractParametricCurve) -> Bool

Returns `true` if `c` is [`LineSegment`](@ref), and `false` otherwise.
"""
is_linear(::AbstractParametricCurve) = false

"""
    is_interpolating(c::AbstractParametricCurve) -> Bool

Returns `true` if `c` goes through all its control points, and `false` otherwise.
"""
is_interpolating(::AbstractParametricCurve) = false

"""
    has_lookup_table(c::AbstractParametricCurve) -> Bool

Returns `true` if `c` has a lookup table, and `false` otherwise.
"""
has_lookup_table(::AbstractParametricCurve) = false

@doc """
    arc_length(c::AbstractParametricCurve) -> Float64
    arc_length(c::AbstractParametricCurve, t₁, t₂) -> Float64

Returns the arc length of the [`AbstractParametricCurve`] `c`. The second method returns the arc length in the interval `[t₁, t₂]`, where `0 ≤ t₁ ≤ t₂ ≤ 1`.
"""
arc_length

arc_length(c::AbstractParametricCurve) = arc_length(c, 0.0, 1.0)
function arc_length(c::AbstractParametricCurve, t₁, t₂)
    # The integral to evaluate is ∫ √(x′(t)² + y′(t)²) dt
    scale = (t₂ - t₁) / 2
    shift = (t₂ + t₁) / 2
    s = 0.0
    for (x, w) in eachcol(GL_NW)
        t = scale * x + shift
        c′ = differentiate(c, t)
        s += w * norm(c′)
    end
    return scale * s
end

@doc """
    differentiate(c::AbstractParametricCurve, t) -> NTuple{2, Float64}

Evaluates the derivative of `c` at `t`.
"""
differentiate

@doc """
    twice_differentiate(c::AbstractParametricCurve, t) -> NTuple{2, Float64}

Evaluates the second derivative of `c` at `t`.
"""
twice_differentiate

@doc """
    thrice_differentiate(c::AbstractParametricCurve, t) -> NTuple{2, Float64}

Evaluates the third derivative of `c` at `t`.
"""
thrice_differentiate

"""
    curvature(c::AbstractParametricCurve, t) -> Float64

Returns the curvature of the [`AbstractParametricCurve`] `c` at `t`.
"""
function curvature(c::AbstractParametricCurve, t)
    x′, y′ = getxy(differentiate(c, t))
    x′′, y′′ = getxy(twice_differentiate(c, t))
    return (x′ * y′′ - y′ * x′′) / (x′^2 + y′^2)^(3 / 2)
end

@doc """
    total_variation(c::AbstractParametricCurve) -> Float64
    total_variation(c::AbstractParametricCurve, t₁, t₂) -> Float64

Returns the total variation of a curve `c`, or the subcurve over `[t₁, t₂]` with `0 ≤ t₁ ≤ t₂ ≤ 1`,
defined as the integral of the absolute curvature over this interval. (This is also known as the total absolute curvature.)
"""
total_variation

total_variation(c::AbstractParametricCurve) = total_variation(c, 0.0, 1.0)
function total_variation(c::AbstractParametricCurve, t₁, t₂)
    scale = (t₂ - t₁) / 2
    shift = (t₂ + t₁) / 2
    s = 0.0
    for (x, w) in eachcol(GL_NW)
        t = scale * x + shift
        κ = abs(curvature(c, t))
        c′ = getxy(differentiate(c, t))
        ds = norm(c′)
        s += w * κ * ds
    end
    return scale * s
end

"""
    marked_total_variation(b::AbstractParametricCurve, t₁, t₂)

Returns the total variation of the curve `b` over the interval `[t₁, t₂]` using the orientation markers of `b`.
"""
function marked_total_variation(b::AbstractParametricCurve, t₁, t₂)
    i₁ = min(lastindex(b.orientation_markers) - 1, searchsortedlast(b.orientation_markers, t₁)) # avoid issues at t = 1
    i₂ = min(lastindex(b.orientation_markers) - 1, searchsortedlast(b.orientation_markers, t₂))
    if i₁ == i₂
        T₁ = differentiate(b, t₁)
        T₂ = differentiate(b, t₂)
        θ = angle_between(T₁, T₂)
        θ > π && (θ = 2π - θ)
        return θ
    elseif i₁ == i₂ - 1
        T₁ = differentiate(b, t₁)
        T₂ = differentiate(b, b.orientation_markers[i₂])
        T₃ = differentiate(b, t₂)
        θ₁ = angle_between(T₁, T₂)
        θ₁ > π && (θ₁ = 2π - θ₁)
        θ₂ = angle_between(T₂, T₃)
        θ₂ > π && (θ₂ = 2π - θ₂)
        return θ₁ + θ₂
    else
        T₁ = differentiate(b, t₁)
        T₂ = differentiate(b, b.orientation_markers[i₁+1])
        θ = angle_between(T₁, T₂)
        θ > π && (θ = 2π - θ)
        Δθ = θ
        for i in (i₁+1):(i₂-1)
            T₁ = T₂
            T₂ = differentiate(b, b.orientation_markers[i+1])
            θ = angle_between(T₁, T₂)
            θ > π && (θ = 2π - θ)
            Δθ += θ
        end
        T₁ = T₂
        T₂ = differentiate(b, t₂)
        θ = angle_between(T₁, T₂)
        θ > π && (θ = 2π - θ)
        Δθ += θ
        return Δθ
    end
end

"""
    point_position_relative_to_curve([kernel::AbstractPredicateKernel=AdaptiveKernel(),] e::AbstractParametricCurve, p) -> Certificate 

Returns the position of the point `p` relative to the curve `c`. This function returns a [`Certificate`]:

- `Left`: `p` is to the left of `c`.
- `Right`: `p` is to the right of `c`.
- `On`: `p` is on `c`.

The `kernel` argument determines how this result is computed, and should be 
one of [`ExactKernel`](@ref), [`FastKernel`](@ref), and [`AdaptiveKernel`](@ref) (the default).
See the documentation for more information about these choices.
"""
function point_position_relative_to_curve(kernel::AbstractPredicateKernel, b::AbstractParametricCurve, p)
    t, q = get_closest_point(b, p)
    qx, qy = getxy(q)
    q′ = differentiate(b, t)
    q′x, q′y = getxy(q′)
    τx, τy = qx + q′x, qy + q′y
    return point_position_relative_to_curve(kernel, LineSegment(q, (τx, τy)), p)
end
point_position_relative_to_curve(b::AbstractParametricCurve, p) = point_position_relative_to_curve(AdaptiveKernel(), b, p)

"""
    convert_lookup_idx(b::AbstractParametricCurve, i) -> Float64

Converts the index `i` of the lookup table of the curve `b` to the corresponding `t`-value.
"""
function convert_lookup_idx(b::AbstractParametricCurve, i)
    n = length(b.lookup_table)
    return (i - 1) / (n - 1)
end

const PROJECTION_INTERVAL_TOL = 1.0e-12
"""
    get_closest_point(b::AbstractParametricCurve p) -> (Float64, NTuple{2,Float64})

Returns the `t`-value and the associated point `q` on the curve `b` that is nearest to `p` using a binary search. The search is done until the 
binary search interval is smaller than `1e-12`. This function will only work if the curve `b` has a lookup table.

!!! danger "Loops"

    This function is only tested on loop-free curves. It is not guaranteed to work on curves with loops. Moreover, for this function to be accurate, 
    you want the lookup table in `b` to be sufficiently dense.
"""
function get_closest_point(b::AbstractParametricCurve, p)
    has_ctrl_points = hasfield(typeof(b), :control_points)
    left_flag = has_ctrl_points ? (p == b.control_points[begin]) : (b(0.0) == p)
    right_flag = has_ctrl_points ? (p == b.control_points[end]) : (b(1.0) == p)
    if left_flag
        return 0.0, p
    elseif right_flag
        return 1.0, p
    end
    i, δ = _get_closest_point_lookup_table(b, p)
    if i == 1
        t, q = _get_closest_point_left_search(b, p, δ)
    elseif i == length(b.lookup_table)
        t, q = _get_closest_point_right_search(b, p, δ)
    else
        tmid, pmid = convert_lookup_idx(b, i), b.lookup_table[i]
        tleft, pleft = convert_lookup_idx(b, i - 1), b.lookup_table[i-1]
        tright, pright = convert_lookup_idx(b, i + 1), b.lookup_table[i+1]
        t, q = _get_closest_point_interior_search(b, p, tmid, pmid, tleft, pleft, tright, pright, δ)
    end
    return t, q
end

function _get_closest_point_lookup_table(b::AbstractParametricCurve, p)
    δ = Inf
    i = 0
    for (j, q) in enumerate(b.lookup_table)
        δ′ = dist_sqr(p, q)
        if δ′ < δ
            δ = δ′
            i = j
        end
    end
    return i, δ
end
function _get_closest_point_interior_search(b::AbstractParametricCurve, p, tmid, pmid, tleft, pleft, tright, pright, δ)
    δleft, δright, δmid = dist_sqr(p, pleft), dist_sqr(p, pright), δ
    w = tright - tleft
    while w > PROJECTION_INTERVAL_TOL # Keep middle as closest
        tleftmid, trightmid = midpoint(tleft, tmid), midpoint(tmid, tright)
        pleftmid, prightmid = b(tleftmid), b(trightmid)
        δleftmid, δrightmid = dist_sqr(p, pleftmid), dist_sqr(p, prightmid)
        if δleftmid < δrightmid
            # Choose the left-middle as the new center 
            tleft, tright, tmid = tleft, tmid, tleftmid
            pleft, pright, pmid = pleft, pmid, pleftmid
            δleft, δright, δmid = δleft, δmid, δleftmid
        else
            # Choose the right-middle as the new center 
            tleft, tright, tmid = tmid, tright, trightmid
            pleft, pright, pmid = pmid, pright, prightmid
            δleft, δright, δmid = δmid, δright, δrightmid
        end
        w = tright - tleft
    end
    return tmid, pmid
end
function _get_closest_point_left_search(b::AbstractParametricCurve, p, δ)
    tleft, pleft = 0.0, b.lookup_table[begin]
    tright, pright = convert_lookup_idx(b, 2), b.lookup_table[2]
    δleft, δright = δ, dist_sqr(p, pright)
    tmid = midpoint(tleft, tright)
    pmid = b(tmid)
    δmid = dist_sqr(p, pmid)
    w = tmid - tleft
    while δmid > δleft && w > PROJECTION_INTERVAL_TOL
        tright, pright, δright = tmid, pmid, δmid
        tmid = midpoint(tleft, tmid)
        pmid = b(tmid)
        δmid = dist_sqr(p, pmid)
    end
    if δmid < δleft
        return _get_closest_point_interior_search(b, p, tmid, pmid, tleft, pleft, tright, pright, δmid)
    else
        return tleft, pleft
    end
end
function _get_closest_point_right_search(b::AbstractParametricCurve, p, δ)
    tleft, pleft = convert_lookup_idx(b, length(b.lookup_table) - 1), b.lookup_table[end-1]
    tright, pright = 1.0, b.lookup_table[end]
    δleft, δright = dist_sqr(p, pleft), δ
    tmid = midpoint(tleft, tright)
    pmid = b(tmid)
    δmid = dist_sqr(p, pmid)
    w = tright - tmid
    while δmid > δright && w > PROJECTION_INTERVAL_TOL
        tleft, pleft, δleft = tmid, pmid, δmid
        tmid = midpoint(tmid, tright)
        pmid = b(tmid)
        δmid = dist_sqr(p, pmid)
    end
    if δmid < δright
        return _get_closest_point_interior_search(b, p, tmid, pmid, tleft, pleft, tright, pright, δmid)
    else
        return tright, pright
    end
end

"""
    process_roots_and_residuals!(roots, residuals, tol) -> Vector{Float64}

Processes the roots and residuals of a root-finding algorithm. This function removes all `NaN` values from `roots` and `residuals`, sorts the roots in ascending order, and removes all roots with residuals greater than `tol`. The 
returned vector is the vector of roots with duplicates (i.e. roots that are within `tol` of each other) removed.
"""
function process_roots_and_residuals!(roots, residuals, tol)
    nan_idx = findall(isnan, roots)
    deleteat!(roots, nan_idx)
    deleteat!(residuals, nan_idx)
    sort_idx = sortperm(roots)
    permute!(roots, sort_idx)
    permute!(residuals, sort_idx)
    bad_idx = Int[]
    for (i, resid) in enumerate(residuals)
        if resid > tol
            push!(bad_idx, i)
        end
    end
    deleteat!(roots, bad_idx)
    return uniquetol(roots; tol)
end

"""
    protect_against_bad_division!(roots, residuals, val, i) -> Bool

Protects against bad division in root-finding algorithms. This function checks if `val` is close to `0` or if `roots[i]` is outside of `[0, 1]`. If either of these conditions are true, then `roots[i]` and `residuals[i]` are set to `NaN` and `true` is returned. Otherwise, `false` is returned.
"""
function protect_against_bad_division!(roots, residuals, val, i)
    if abs(val) < 1.0e-8 || roots[i] < 0 || roots[i] > 1
        roots[i] = NaN
        residuals[i] = NaN
        return true
    end
    return false
end

"""
    horizontal_turning_points(c::AbstractParametricCurve; steps=200, iters = 50, tol = 1e-5) -> Vector{Float64}

Returns points `t` such that `x'(t) = 0` and `0 ≤ t ≤ 1`, where `x'` is the derivative of the `x`-coordinate of `c`. This function uses Newton's method to find the roots of `x'`.

!!! danger "High-degree curves"

    For curves of very high degree, such as Bezier curves with `steps` control points or greater, this function might fail to return all 
    turning points.

# Arguments 
- `c::AbstractParametricCurve`: The curve to find the horizontal turning points of.

# Keyword Arguments
- `steps=200`: The number of `t`-values to use for seeding Newton's method. In particular, Newton's method is run for each initial value in `LinRange(0, 1, steps)`.
- `iters=50`: The number of iterations to run Newton's method for.
- `tol=1e-5`: The tolerance to use for [`uniquetol`](@ref). Also used for deciding whether a root is a valid root, i.e. if `abs(x'(t)) > tol` for a found root `t`, then `t` is not a valid root and is rejected.

# Output 
- `t`: All turning points, given in sorted order.
"""
function horizontal_turning_points(c::AbstractParametricCurve; steps=200, iters=50, tol=1.0e-5)
    roots = collect(LinRange(0, 1, steps))
    residuals = fill(Inf, steps)
    for i in eachindex(roots)
        x′ = getx(differentiate(c, roots[i]))
        x′′ = getx(twice_differentiate(c, roots[i]))
        roots[i] -= x′ / x′′
        residuals[i] = abs(x′)
        for _ in 1:iters
            protect_against_bad_division!(roots, residuals, x′′, i) && break
            x′ = getx(differentiate(c, roots[i]))
            x′′ = getx(twice_differentiate(c, roots[i]))
            roots[i] -= x′ / x′′
            residuals[i] = abs(x′)
        end
    end
    return process_roots_and_residuals!(roots, residuals, tol)
end

"""
    vertical_turning_points(c::AbstractParametricCurve; steps=200, iters = 50, tol = 1e-5) -> Vector{Float64}

Returns points `t` such that `y'(t) = 0` and `0 ≤ t ≤ 1`, where `y'` is the derivative of the `y`-coordinate of `c`. This function uses Newton's method to find the roots of `y'`.

!!! danger "High-degree curves"

    For curves of very high degree, such as Bezier curves with `steps` control points or greater, this function might fail to return all 
    turning points.

# Arguments 
- `c::AbstractParametricCurve`: The curve to find the vertical turning points of.

# Keyword Arguments
- `steps=200`: The number of `t`-values to use for seeding Newton's method. In particular, Newton's method is run for each initial value in `LinRange(0, 1, steps)`.
- `iters=50`: The number of iterations to run Newton's method for.
- `tol=1e-5`: The tolerance to use for [`uniquetol`](@ref). Also used for deciding whether a root is a valid root, i.e. if `abs(y'(t)) > tol` for a found root `t`, then `t` is not a valid root and is rejected.

# Output 
- `t`: All turning points, given in sorted order.
"""
function vertical_turning_points(c::AbstractParametricCurve; steps=200, iters=50, tol=1.0e-5)
    roots = collect(LinRange(0, 1, steps))
    residuals = fill(Inf, steps)
    for i in eachindex(roots)
        y′ = gety(differentiate(c, roots[i]))
        y′′ = gety(twice_differentiate(c, roots[i]))
        roots[i] -= y′ / y′′
        residuals[i] = abs(y′)
        for _ in 1:iters
            protect_against_bad_division!(roots, residuals, y′′, i) && break
            y′ = gety(differentiate(c, roots[i]))
            y′′ = gety(twice_differentiate(c, roots[i]))
            roots[i] -= y′ / y′′
            residuals[i] = abs(y′)
        end
    end
    return process_roots_and_residuals!(roots, residuals, tol)
end

"""
    horizontal_inflection_points(c::AbstractParametricCurve; steps=200, iters = 50, tol = 1e-5) -> Vector{Float64}

Returns points `t` such that `x''(t) = 0` and `0 ≤ t ≤ 1`, where `x''` is the second derivative of the `x`-coordinate of `c`. This function uses Newton's method to find the roots of `x''`.
Note that these are only technically inflection points if `x'''(t) ≠ 0` at these points, but this is not checked.

!!! danger "High-degree curves"

    For curves of very high degree, such as Bezier curves with `steps` control points or greater, this function might fail to return all 
    inflection points.

# Arguments
- `c::AbstractParametricCurve`: The curve to find the horizontal inflection points of.

# Keyword Arguments
- `steps=200`: The number of `t`-values to use for seeding Newton's method. In particular, Newton's method is run for each initial value in `LinRange(0, 1, steps)`.
- `iters=50`: The number of iterations to run Newton's method for.
- `tol=1e-5`: The tolerance to use for [`uniquetol`](@ref). Also used for deciding whether a root is a valid root, i.e. if `abs(x''(t)) > tol` for a found root `t`, then `t` is not a valid root and is rejected.

# Output
- `t`: All inflection points, given in sorted order.
"""
function horizontal_inflection_points(c::AbstractParametricCurve; steps=200, iters=50, tol=1.0e-5)
    roots = collect(LinRange(0, 1, steps))
    residuals = fill(Inf, steps)
    for i in eachindex(roots)
        x′′ = getx(twice_differentiate(c, roots[i]))
        x′′′ = getx(thrice_differentiate(c, roots[i]))
        roots[i] -= x′′ / x′′′
        residuals[i] = abs(x′′)
        for _ in 1:iters
            protect_against_bad_division!(roots, residuals, x′′′, i) && break
            x′′ = getx(twice_differentiate(c, roots[i]))
            x′′′ = getx(thrice_differentiate(c, roots[i]))
            roots[i] -= x′′ / x′′′
            residuals[i] = abs(x′′)
        end
    end
    return process_roots_and_residuals!(roots, residuals, tol)
end

"""
    vertical_inflection_points(c::AbstractParametricCurve; steps=200, iters = 50, tol = 1e-5) -> Vector{Float64}

Returns points `t` such that `y''(t) = 0` and `0 ≤ t ≤ 1`, where `y''` is the second derivative of the `y`-coordinate of `c`. This function uses Newton's method to find the roots of `y''`.
Note that these are only technically inflection points if `y'''(t) ≠ 0` at these points, but this is not checked.

!!! danger "High-degree curves"

    For curves of very high degree, such as Bezier curves with `steps` control points or greater, this function might fail to return all 
    inflection points.

# Arguments
- `c::AbstractParametricCurve`: The curve to find the vertical inflection points of.

# Keyword Arguments
- `steps=200`: The number of `t`-values to use for seeding Newton's method. In particular, Newton's method is run for each initial value in `LinRange(0, 1, steps)`.
- `iters=50`: The number of iterations to run Newton's method for.
- `tol=1e-5`: The tolerance to use for [`uniquetol`](@ref). Also used for deciding whether a root is a valid root, i.e. if `abs(y''(t)) > tol` for a found root `t`, then `t` is not a valid root and is rejected.

# Output
- `t`: All inflection points, given in sorted order.
"""
function vertical_inflection_points(c::AbstractParametricCurve; steps=200, iters=50, tol=1.0e-5)
    roots = collect(LinRange(0, 1, steps))
    residuals = fill(Inf, steps)
    for i in eachindex(roots)
        y′′ = gety(twice_differentiate(c, roots[i]))
        y′′′ = gety(thrice_differentiate(c, roots[i]))
        roots[i] -= y′′ / y′′′
        residuals[i] = abs(y′′)
        for _ in 1:iters
            protect_against_bad_division!(roots, residuals, y′′′, i) && break
            y′′ = gety(twice_differentiate(c, roots[i]))
            y′′′ = gety(thrice_differentiate(c, roots[i]))
            roots[i] -= y′′ / y′′′
            residuals[i] = abs(y′′)
        end
    end
    return process_roots_and_residuals!(roots, residuals, tol)
end

"""
    inflection_points(c::AbstractParametricCurve; steps=200, iters = 50, tol = 1e-5) -> Vector{Float64}

Returns points `t` such that `κ(t) = 0` and `0 ≤ t ≤ 1`, where `κ` is the curvature of `c`. This function uses Newton's method to find the roots of `κ`.

!!! danger "High-degree curves"

    For curves of very high degree, such as Bezier curves with `steps` control points or greater, this function might fail to return all 
    inflection points.

# Arguments
- `c::AbstractParametricCurve`: The curve to find the inflection points of.

# Keyword Arguments
- `steps=200`: The number of `t`-values to use for seeding Newton's method. In particular, Newton's method is run for each initial value in `LinRange(0, 1, steps)`.
- `iters=50`: The number of iterations to run Newton's method for.
- `tol=1e-5`: The tolerance to use for [`uniquetol`](@ref). Also used for deciding whether a root is a valid root, i.e. if `abs(κ(t)) > tol` for a found root `t`, then `t` is not a valid root and is rejected.
"""
function inflection_points(c::AbstractParametricCurve; steps=200, iters=50, tol=1.0e-5)
    roots = collect(LinRange(0, 1, steps))
    residuals = fill(Inf, steps)
    for i in eachindex(roots)
        x′, y′ = getxy(differentiate(c, roots[i]))
        x′′, y′′ = getxy(twice_differentiate(c, roots[i]))
        x′′′, y′′′ = getxy(thrice_differentiate(c, roots[i]))
        κ = x′ * y′′ - y′ * x′′ # don't need to divide by (x′^2 + y′^2)^(3 / 2) since we're only checking if it's zero
        κ′ = x′ * y′′′ - y′ * x′′′
        roots[i] -= κ / κ′
        residuals[i] = abs(κ)
        for _ in 1:iters
            protect_against_bad_division!(roots, residuals, κ′, i) && break
            x′, y′ = getxy(differentiate(c, roots[i]))
            x′′, y′′ = getxy(twice_differentiate(c, roots[i]))
            x′′′, y′′′ = getxy(thrice_differentiate(c, roots[i]))
            κ = x′ * y′′ - y′ * x′′
            κ′ = x′ * y′′′ - y′ * x′′′
            roots[i] -= κ / κ′
            residuals[i] = abs(κ)
        end
    end
    return process_roots_and_residuals!(roots, residuals, tol)
end

"""
    orientation_markers(c::AbstractParametricCurve; steps=200, iters=50, tol=1e-5) -> Vector{Float64}

Finds all orientation markers of the [`AbstractParametricCurve`](@ref) `c`. These are points `t` where any of the following 
conditions hold (not necessarily simultaneously), letting `c(t) = (x(t), y(t))`:

- `x'(t) = 0`
- `y'(t) = 0`
- `κ(t; x) = 0`, where `κ(t; x)` is the curvature of the component function `x(t)`
- `κ(t; y) = 0`, where `κ(t; y)` is the curvature of the component function `y(t)`
- `κ(t) = 0`, where `κ` is the curvature of `c(t)`

Note that the third and fourth conditions give all the inflection points of the component functions, and similarly for the fifth condition.

See also [`horizontal_turning_points`](@ref), [`vertical_turning_points`](@ref), [`horizontal_inflection_points`](@ref), [`vertical_inflection_points`](@ref), and [`inflection_points`](@ref).

!!! danger "High-degree curves"

    For curves of very high degree, such as Bezier curves with `steps` control points or greater, this function might fail to return all 
    inflection points.

# Arguments 
- `c::AbstractParametricCurve`: The [`AbstractParametricCurve`](@ref).

# Keyword Arguments 
- `steps=200`: The number of equally spaced points to use for initialising Newton's method.
- `iters=50`: How many iterations to use for Newton's method.
- `tol=1e-5`: The tolerance used for determining if two `t`-values are the same.

# Output 
- `markers::Vector{Float64}`: The `t`-values of the orientation markers of `b`. The returned vector is sorted, and also includes the 
   endpoints `0` and `1`; any `t`-values outside of `[0, 1]` are discarded, and multiplicity 
   of any `t` is not considered (so the `t`-values in the returned vector are unique). These values can be used to split the curve into monotone pieces, meaning 
   the orientation is monotone. These markers also guarantee that, over any monotone piece, the orientation changes by an angle of at most `π/2`.
"""
function orientation_markers(c::AbstractParametricCurve; steps=200, iters=50, tol=1.0e-5)
    t₁ = horizontal_turning_points(c, steps=steps, iters=iters, tol=tol)
    t₂ = vertical_turning_points(c, steps=steps, iters=iters, tol=tol)
    t₃ = horizontal_inflection_points(c, steps=steps, iters=iters, tol=tol)
    t₄ = vertical_inflection_points(c, steps=steps, iters=iters, tol=tol)
    t₅ = inflection_points(c, steps=steps, iters=iters, tol=tol)
    all_t = vcat(t₁, t₂, t₃, t₄, t₅)
    isempty(all_t) && return [0.0, 1.0]
    sort!(all_t)
    all_t[1] ≠ 0.0 && pushfirst!(all_t, 0.0)
    all_t[end] ≠ 1.0 && push!(all_t, 1.0)
    return uniquetol(all_t; tol)
end

"""
    get_equidistant_split(c::AbstractParametricCurve, t₁, t₂) -> Float64 

Returns a value of `t` such that the arc length along `c` from `t₁` to `t` is equal to the arc length along `c` from `t` to `t₂`. 
Uses the bisection method to compute the `t`-value.
"""
function get_equidistant_split(c::AbstractParametricCurve, t₁, t₂)
    a = t₁
    s₁₂ = arc_length(c, a, t₂)
    s = s₁₂ / 2
    t = midpoint(a, t₂)
    for _ in 1:100 # limit iterations to 100 
        s₁t = arc_length(c, a, t)
        if abs(s₁t - s) < 1.0e-3 || abs(t₁ - t₂) < 2.0e-8
            return t
        end
        s₁t > s ? (t₂ = t) : (t₁ = t)
        t = midpoint(t₁, t₂)
    end
    return t
end

"""
    get_equivariation_split(c::AbstractParametricCurve, t₁, t₂) -> Float64, Float64

Returns a value of `t` such that the total variation of `c` from `t₁` to `t` is equal to the total variation of `c` from `t` to `t₂`.
Uses the bisection method to compute the `t`-value. Also returns the new total variation of the two pieces.
"""
function get_equivariation_split(c::AbstractParametricCurve, t₁, t₂)
    a = t₁
    Δθ₁₂ = total_variation(c, a, t₂)
    Δθ = Δθ₁₂ / 2
    Δθ₁t = Δθ
    t = get_equidistant_split(c, a, t₂)
    for _ in 1:100 # limit iterations to 100 
        Δθ₁t = total_variation(c, a, t)
        if abs(Δθ₁t - Δθ) < 1.0e-4 || abs(t₁ - t₂) < 2.0e-8
            return t, Δθ₁t
        end
        Δθ₁t > Δθ ? (t₂ = t) : (t₁ = t)
        t = midpoint(t₁, t₂)
    end
    return t, Δθ₁t
end

"""
    get_inverse(c::AbstractParametricCurve, p) -> Float64

Given a point `p` on `c`, returns the `t`-value such that `c(t) ≈ p`.
"""
function get_inverse(c::AbstractParametricCurve, p)
    t, _ = get_closest_point(c, p)
    return t
end

"""
    angle_between(c₁::AbstractParametricCurve, c₂::AbstractParametricCurve) -> Float64

Given two curves `c₁` and `c₂` such that `c₁(1) == c₂(0)`, returns the angle between the two curves, treating the interior of the 
curves as being left of both.
"""
function angle_between(c₁::AbstractParametricCurve, c₂::AbstractParametricCurve)
    T₁x, T₁y = getxy(differentiate(c₁, 1.0))
    T₂x, T₂y = getxy(differentiate(c₂, 0.0))
    qx, qy = getxy(c₁(1.0))
    px, py = qx - T₁x, qy - T₁y
    rx, ry = qx + T₂x, qy + T₂y
    L₁ = LineSegment((px, py), (qx, qy), 0.0)
    L₂ = LineSegment((qx, qy), (rx, ry), 0.0)
    return angle_between(L₁, L₂)
end

"""
    get_circle_intersection(c::AbstractParametricCurve, t₁, t₂, r) -> (Float64, NTuple{2,Float64}) 

Given a circle centered at `c(t₁)` with radius `r`, finds the first intersection of the circle with 
the curve after `t₁` and less than `t₂`. It is assumed that such an intersection exists. The returned value 
is `(t, q)`, where `t` is the parameter value of the intersection and `q` is the point of intersection.
"""
function get_circle_intersection(c::AbstractParametricCurve, t₁, t₂, r)
    tᵢ, tⱼ, p = _get_interval_for_get_circle_intersection(c, t₁, t₂, r)
    t = midpoint(tᵢ, tⱼ)
    for _ in 1:100
        q = c(t)
        δ = dist(p, q)
        if abs(δ - r) < 1.0e-3 || abs(tᵢ - tⱼ) < 2.0e-8
            return t, q
        end
        (δ > r) ? (tⱼ = t) : (tᵢ = t)
        t = midpoint(tᵢ, tⱼ)
    end
    return t, c(t)
end

"""
    _get_interval_for_get_circle_intersection(c::AbstractParametricCurve, t₁, t₂, r) -> (Float64, Float64, NTuple{2, Float64})

Given a circle centered at `c(t₁)` with radius `r`, finds an initial interval for [`get_circle_intersection`](@ref)
to perform bisection on to find a point of intersection. The returned interval is `(tᵢ, tⱼ)`, 
where `tᵢ` is the parameter value of the first point in the interval and `tⱼ` 
is the parameter value of the last point in the interval. (The interval does not have to be sorted.) The third returned value is `p = c(t₁)`.
"""
function _get_interval_for_get_circle_intersection(c::AbstractParametricCurve, t₁, t₂, r)
    if has_lookup_table(c)
        return _get_interval_for_get_circle_intersection_lookup(c, t₁, t₂, r)
    else
        return _get_interval_for_get_circle_intersection_direct(c, t₁, t₂, r)
    end
end
function _get_interval_for_get_circle_intersection_direct(c::AbstractParametricCurve, t₁, t₂, r)
    t = LinRange(t₁, t₂, 500)
    tᵢ, tⱼ = t₁, t₂
    p = c(t₁)
    for τ in t
        q = c(τ)
        δ = dist(p, q)
        if δ > r
            tⱼ = τ
            break
        else
            tᵢ = τ
        end
    end
    return tᵢ, tⱼ, p
end
function _get_interval_for_get_circle_intersection_lookup(c::AbstractParametricCurve, t₁, t₂, r)
    n = length(c.lookup_table)
    p = c(t₁) # the center
    i₁ = floor(Int, t₁ * (n - 1)) + 1
    i₂ = ceil(Int, t₂ * (n - 1)) + 1
    i = i₁
    itr = t₁ < t₂ ? (i₁:1:i₂) : (i₁:-1:i₂) # explicit :1 step so that bothr anges are a StepRange
    for outer i in itr
        t = convert_lookup_idx(c, i)
        q = c(t)
        δ = dist(p, q)
        δ > r && break
    end
    i, j = i - 1, i # δ = r somewhere inside here 
    tᵢ, tⱼ = convert_lookup_idx(c, i), convert_lookup_idx(c, j)
    return tᵢ, tⱼ, p
end

"""
    reverse(curve::AbstractParametricCurve) -> AbstractParametricCurve

Returns an [`AbstractParametricCurve`](@ref) that reverses the orientation of `curve`. In particular, 
`c(t) = c̄(1-t)` for all `t` in `[0, 1]`, where `c` is the original curve and `c̄` is the reversed curve.
"""
function Base.reverse(c::AbstractParametricCurve)
    return _reverse(c)
end

_plcopy(c::AbstractParametricCurve; kwargs...) = copy(c) # for handling PiecewiseLinear