function test_adapt_methods(AT, synchronize)
    """Test new Adapt methods for all quantum optics types."""
    cache = AllocCache()
    
    for n in test_sizes
        for r in 1:round_count
            @cached cache begin
                
                @testset "Adapt Methods - Size $n" begin
                    # Create test bases
                    b1 = GenericBasis(n)
                    b2 = GenericBasis(n)
                    cb = b1 ⊗ b2
                    
                    # Test Ket adaptation 
                    ket_data = rand(ComplexF64, n)
                    normalize!(ket_data)
                    cpu_ket = Ket(b1, ket_data)
                    gpu_ket = Adapt.adapt(AT, cpu_ket)
                    
                    @test typeof(gpu_ket.data) <: AT
                    @test gpu_ket.basis == cpu_ket.basis
                    @test verify_gpu_result(cpu_ket, gpu_ket)
                    
                    # Test Bra adaptation
                    bra_data = rand(ComplexF64, n)  
                    normalize!(bra_data)
                    cpu_bra = Bra(b1, bra_data)
                    gpu_bra = Adapt.adapt(AT, cpu_bra)
                    
                    @test typeof(gpu_bra.data) <: AT
                    @test gpu_bra.basis == cpu_bra.basis
                    @test verify_gpu_result(cpu_bra, gpu_bra)
                    
                    # Test SuperOperator adaptation
                    super_data = rand(ComplexF64, n*n, n*n)
                    cpu_super = SuperOperator([b1,b1], [b1,b1], super_data)
                    gpu_super = Adapt.adapt(AT, cpu_super)
                    
                    @test typeof(gpu_super.data) <: AT
                    @test gpu_super.basis_l == cpu_super.basis_l
                    @test gpu_super.basis_r == cpu_super.basis_r
                    @test verify_gpu_result(cpu_super, gpu_super)
                    
                    # Test LazyKet adaptation 
                    ket1 = Ket(b1, rand(ComplexF64, n))
                    ket2 = Ket(b2, rand(ComplexF64, n))
                    cpu_lazy_ket = LazyKet(cb, (ket1, ket2))
                    gpu_lazy_ket = Adapt.adapt(AT, cpu_lazy_ket)
                    
                    @test typeof(gpu_lazy_ket.kets[1].data) <: AT
                    @test typeof(gpu_lazy_ket.kets[2].data) <: AT
                    @test gpu_lazy_ket.basis == cpu_lazy_ket.basis
                    
                    # Test LazySum adaptation
                    op1 = Operator(b1, b1, rand(ComplexF64, n, n))
                    op2 = Operator(b1, b1, rand(ComplexF64, n, n))
                    cpu_lazy_sum = LazySum([1.0, 2.0], [op1, op2])
                    gpu_lazy_sum = Adapt.adapt(AT, cpu_lazy_sum)
                    
                    @test typeof(gpu_lazy_sum.operators[1].data) <: AT
                    @test typeof(gpu_lazy_sum.operators[2].data) <: AT
                    @test gpu_lazy_sum.basis_l == cpu_lazy_sum.basis_l
                    @test gpu_lazy_sum.basis_r == cpu_lazy_sum.basis_r
                    @test gpu_lazy_sum.factors == cpu_lazy_sum.factors
                    
                    # Test LazyProduct adaptation
                    cpu_lazy_prod = LazyProduct([op1, op2])
                    gpu_lazy_prod = Adapt.adapt(AT, cpu_lazy_prod)
                    
                    @test typeof(gpu_lazy_prod.operators[1].data) <: AT
                    @test typeof(gpu_lazy_prod.operators[2].data) <: AT
                    @test gpu_lazy_prod.basis_l == cpu_lazy_prod.basis_l
                    @test gpu_lazy_prod.basis_r == cpu_lazy_prod.basis_r
                    
                    # Test LazyTensor adaptation (for composite systems)
                    if n >= 2
                        indices = [1]
                        sub_op = Operator(b1, b1, rand(ComplexF64, n, n))
                        cpu_lazy_tensor = LazyTensor(cb, cb, indices, (sub_op,))
                        gpu_lazy_tensor = Adapt.adapt(AT, cpu_lazy_tensor)
                        
                        @test typeof(gpu_lazy_tensor.operators[1].data) <: AT
                        @test gpu_lazy_tensor.basis_l == cpu_lazy_tensor.basis_l
                        @test gpu_lazy_tensor.basis_r == cpu_lazy_tensor.basis_r
                        @test gpu_lazy_tensor.indices == cpu_lazy_tensor.indices
                    end
                    
                    # Test LazyDirectSum adaptation 
                    op_ds1 = Operator(b1, b1, rand(ComplexF64, n, n))
                    op_ds2 = Operator(b2, b2, rand(ComplexF64, n, n)) 
                    cpu_lazy_ds = LazyDirectSum(op_ds1, op_ds2)
                    gpu_lazy_ds = Adapt.adapt(AT, cpu_lazy_ds)
                    
                    @test typeof(gpu_lazy_ds.operators[1].data) <: AT
                    @test typeof(gpu_lazy_ds.operators[2].data) <: AT
                    @test gpu_lazy_ds.basis_l == cpu_lazy_ds.basis_l
                    @test gpu_lazy_ds.basis_r == cpu_lazy_ds.basis_r
                    
                    synchronize()
                end
                
            end
        end
    end
end