# Entanglement Swap with Emissive Memories
## Overview
This folder contains the functions required to generate the spin-spin density matrix generated by linear photonic entanglement swap with emissive memories. The functions are written in MATLAB. The functions are:
- midswap_dual_rail.m: Generates the spin-spin density matrix for linear photonic entanglement swap with emissive memories emitting dual rail photonic qubits from the papers Ref. 1.
- midswap_single_rail.m: Generates the spin-spin density matrix for linear photonic entanglement swap with emissive memories emitting single rail photonic qubits from the papers Ref. 1.
  
## Detailed Descriptions
### midswap_dual_rail.m
This function generates the spin-spin density matrix for linear photonic entanglement swap with emissive memories emitting dual rail photonic qubits 

**Inputs:**
- eA, eB: Link efficiencies for memories A and B upto the swap (include link loss, detector efficiency, etc.)
    - Range: $[0,1]$ 
    - Typical value: $1\leftrightarrow10^{-4}$
- gA, gB: Memory initialization parameter for memories A and B 
  -  Range: $[0,1]$
  -  Typical value: $0.5$
  - Memory emission model: $\sqrt{1-g_k} \ket{0} _M\otimes \ket{0,1}_P + \sqrt{g_k} \ket{1}_M\otimes \ket{1,0}_P$
- Pd: Detector dark count probability per photonic mode (assumed to be the same for both detectors)
    -  Range: $[0,1)$
    -  Typical value: $10^{-8}$ 
- Vis: Interferometer visibility for the midpoint swap ()
    -  Range: $[0,1]$
    -  Typical value: $0.99$ 
   
**Output:**
- Mv: Spin-spin density matrix for the two memories after the midpoint swap
  - Basis order: $\ket{0,0}, \ket{0,1}, \ket{1,0}, \ket{1,1}$

### midswap_single_rail.m
This function generates the spin-spin density matrix for linear photonic entanglement swap with emissive memories emitting single rail photonic qubits

**Inputs:**
- eA, eB: Link efficiencies for memories A and B upto the swap (include link loss, detector efficiency, etc.)
    - Range: $[0,1]$ 
    - Typical value: $1\leftrightarrow10^{-4}$
- gA, gB: Memory initialization parameter for memories A and B 
  -  Range: $[0,1]$
  -  Typical value: $0.5$
  - Memory emission model: $\sqrt{1-g_k} \ket{0} _M\otimes \ket{1}_P + \sqrt{g_k} \ket{1}_M\otimes \ket{0}_P$
- Pd: Detector dark count probability per photonic mode (assumed to be the same for both detectors)
    -  Range: $[0,1)$
    -  Typical value: $10^{-8}$ 
- Vis: Interferometer visibility for the midpoint swap' can be complex to account for phase instability
    -  Range (absolute value): $[0,1]$
    -  Typical value: $0.99$ 
   
**Output:**
- Mv: Spin-spin density matrix for the two memories after the midpoint swap
  - Basis order: $\ket{0,0}, \ket{0,1}, \ket{1,0}, \ket{1,1}$

## Usage
The functions can be used as follows:
```matlab
% Parameters
eA = 0.9; % Link efficiency for memory A
eB = 0.9; % Link efficiency for memory B
gA = 0.5; % Initialization parameter for memory A
gB = 0.5; % Initialization parameter for memory B
Pd = 1e-8; % Dark count probability per photonic mode
Vis = 0.99; % Interferometer visibility
%% Generate the spin-spin density matrix
M_dual= midswap_dual_rail(eA,eB,gA,gB,Pd,Vis); % For dual rail photonic qubits
M_single = midswap_single_rail(eA,eB,gA,gB,Pd,Vis); % For single rail photonic qubits

%% Calculate the probability of success
P_succ_dual = trace(M_dual)*4; % Multiply by 4 to account for all click patterns
P_succ_single = trace(M_single)*2; % Multiply by 2 to account for all click patterns

%% Calculate the fidelity
F_dual = 0.5*(M_dual(2,2)+M_dual(3,3)+M_dual(2,3)+M_dual(3,2));
F_single = 0.5*(M_single(2,2)+M_single(3,3)+M_single(2,3)+M_single(3,2));

```


## Reference
1. [P. Dhara, D. Englund, S. Guha, Entangling quantum memories via heralded photonic Bell measurement. Phys. Rev. Res. 5, 033149 (2023).](https://journals.aps.org/prresearch/abstract/10.1103/PhysRevResearch.5.033149)