# Cascaded Source Swapped With Emissive Memories

## Overview
This folder contains the functions required to generate the spin-spin density matrix for a cascaded source swapped with emissive memories. The functions are written in MATLAB. The functions are:
- cascaded_source_photonic.m : Generates the photonic density matrix for a cascaded source from the papers Refs. 1 and 2.
- cascaded_source_spin.m : Generates the spin-spin density matrix for a cascaded source swapped with emissive spin memories from the papers Ref. 3.
- cascaded_source_basis.m : Generates the photonic Fock basis ordering for the cascaded source density matrix generated by cascaded_source_photonic.m.
- ret_cXY.m : Functions called by cascaded_source_spin.m to generate the spin-spin density matrix for a cascaded source swapped with emissive spin memories.
  
## Detailed Descriptions
### cascaded_source_photonic.m
This function generates the photonic density matrix for a cascaded source from the papers Refs. 1 and 2. 

**Inputs:**
- Ns: mean photon number per mode of the cascaded source model
  - Allowed range: $(0,0.2)]$
  - Typical value: $10^{-2}\leftrightarrow 10^{-4}$
- eAs: out-coupling efficiency for Alice's side 
  - Allowed range: $[0,1]$
  - Typical value: $1\leftrightarrow 10^{-4}$
- eBs: out-coupling efficiency for Bob's side 
  - Allowed range: $[0,1]$
  -   Typical value: $1\leftrightarrow 10^{-4}$
- eD: detector efficiency 
  -   Allowed range: $[0,1]$
  -   Typical value: $0.9$
- Pd: dark click probability per photonic mode 
    -   Allowed range: $[0,1)$
    -   Typical value: $10^{-8}$
- VisF: visibility of the Franson interferometer 
  -   Allowed range: $[0,1]$
  -   Typical value: $0.99$   

**Outputs:**
  - M: Output state density matrix of the cascaded source(in the Fock basis)

__Use 'cascaded_source_basis.m' to obtain the basis ordering__

### cascaded_source_spin.m
This function generates the spin-spin density matrix for a cascaded source swapped with emissive spin memories.

**Inputs:**
- Ns: mean photon number per mode of the cascaded source model
  - Allowed range: $(0,0.2)]$
  - Typical value: $10^{-2}\leftrightarrow 10^{-4}$
- gA: qubit initialization parameter on Alice's side 
  -  Allowed range: $[0,1]$
  -  Typical value: $0.5$
- gB: qubit initialization parameter on Bob's side 
  -  Allowed range: $[0,1]$
  -  Typical value: $0.5$
- eAm: memory out-coupling efficiency for Alice's side (Allowed range: [0,1])
  -  Allowed range: $[0,1]$
  -  Typical value: $1\leftrightarrow 10^{-4}$
- eBm: memory out-coupling efficiency for Bob's side (Allowed range: [0,1])
  -  Allowed range: $[0,1]$
  -  Typical value: $1\leftrightarrow 10^{-4}$
- eAs: source out-coupling efficiency for Alice's side (Allowed range: [0,1])
  -  Allowed range: $[0,1]$
  -  Typical value: $1\leftrightarrow 10^{-4}$
- eBs: source out-coupling efficiency for Bob's side (Allowed range: [0,1])
  -  Allowed range: $[0,1]$
  -  Typical value: $1\leftrightarrow 10^{-4}$
- eD: detector efficiency (Allowed range: [0,1])
  -   Allowed range: $[0,1]$
  -   Typical value: $0.9$
- Pd: dark click probability per photonic mode on source's swap 
  -   Allowed range: $[0,1)$
  -   Typical value: $10^{-8}$
- Pdo1: dark click probability per photonic mode on Alice side swap 
  -   Allowed range: $[0,1)$
  -   Typical value: $10^{-8}$
- Pdo2: dark click probability per photonic mode on Bob side swap
  -   Allowed range: $[0,1)$
  -   Typical value: $10^{-8}$
- VisF: product of visibilities of all three  interferometers (Allowed range: [0,1])
  -   Allowed range: $[0,1]$
  -   Typical value: $0.9$
  
**Outputs:**
- M: Output spin state density matrix of the cascaded source (in the logical spin basis)
  - Basis order:  $\ket{0,0}, \ket{0,1}, \ket{1,0}, \ket{1,1}$

### cascaded_source_basis.m
This function generates the photonic Fock basis ordering for the cascaded source density matrix generated by cascaded_source_photonic.m.
**Inputs:**
None
**Outputs:**
- vec: Fock basis ordering for the cascaded source density matrix generated by cascaded_source_photonic.m a vec is a 36x4 array of the basis vectors (we limit ourselves  to the Fock space of upto 2 photons in a pair of modes)

## Usage
### Example 1: Cascaded source density matrix
```matlab
% Cascaded source density matrix
Ns = 1e-3; % mean photon number per mode of the cascaded source model
eAs = 1; % out-coupling efficiency for Alice's side
eBs = 1; % out-coupling efficiency for Bob's side
eD = 0.9; % detector efficiency
Pd = 1e-8; % dark click probability per photonic mode
VisF = 0.99; % visibility of the Franson interferometer
M = cascaded_source_photonic(Ns,eAs,eBs,eD,Pd,VisF);
%% To obtain the basis ordering
vec = cascaded_source_basis();
%% To obtain the probability of a successful state generation (factor of 4 for the different click patterns)
prob_success=trace(M)*4;

%Fidelity with \ket{\Psi^\pm} photonic Bell states
fidelity=0.5*(M(10,10)+M(20,20)+M(10,20)+M(20,10));
```
### Example 2: Cascaded source swapped with emissive memories density matrix
```matlab
% Cascaded source swapped with emissive memories density matrix
Ns = 1e-3; % mean photon number per mode of the cascaded source model
gA = 0.5; % qubit initialization parameter on Alice's side
gB = 0.5; % qubit initialization parameter on Bob's side
eAm = 1; % memory out-coupling efficiency for Alice's side
eBm = 1; % memory out-coupling efficiency for Bob's side
eAs = 1; % source out-coupling efficiency for Alice's side
eBs = 1; % source out-coupling efficiency for Bob's side
eD = 0.9; % detector efficiency
Pd = 1e-8; % dark click probability per photonic mode on source's swap
Pdo1 = 1e-8; % dark click probability per photonic mode on Alice side swap
Pdo2 = 1e-8; % dark click probability per photonic mode on Bob side swap
VisF = 0.99; % product of visibilities of all three  interferometers
M = cascaded_source_spin(Ns,gA,gB,eAm,eBm,eAs,eBs,eD,Pd,Pdo1,Pdo2,VisF);
%% To obtain the probability of a successful state generation (factor of 4^3 for the different click patterns on 3 swaps)
prob_success=trace(M)*64;
%Fidelity with \ket{\Psi^\pm} spin Bell states
fidelity=0.5*(M(2,2)+M(3,3)+M(2,3)+M(3,2));
```

## Code Dependencies
- ret_cXY.m's are required for cascaded_source_spin.m


## Reference
1. [P. Dhara, S. J. Johnson, C. N. Gagatsos, P. G. Kwiat, S. Guha, Heralded Multiplexed High-Efficiency Cascaded Source of Dual-Rail Entangled Photon Pairs Using Spontaneous Parametric Down-Conversion. Phys. Rev. Appl. 17, 034071 (2022).](https://arxiv.org/abs/2107.14360)
2. [K. C. Chen, et al., Zero-Added-Loss Entangled-Photon Multiplexing for Ground- and Space-Based Quantum Networks. Phys. Rev. Appl. 19, 054029 (2023).](https://arxiv.org/abs/2206.03670)
3. [P. Dhara, D. Englund, S. Guha, Entangling quantum memories via heralded photonic Bell measurement. Phys. Rev. Res. 5, 033149 (2023).](https://journals.aps.org/prresearch/abstract/10.1103/PhysRevResearch.5.033149)
