import { Component, createComponentVNode, linkEvent, createVNode } from 'inferno';
import { parsePath, createBrowserHistory, createHashHistory, createMemoryHistory } from 'history';
import pathToRegexp from 'path-to-regexp-es6';
import hoistNonReactStatics from 'hoist-non-inferno-statics';

const isArray = Array.isArray;
function isNullOrUndef(o) {
  return o === void 0 || o === null;
}
function isInvalid(o) {
  return o === null || o === false || o === true || o === void 0;
}
function isString(o) {
  return typeof o === 'string';
}
function isUndefined(o) {
  return o === void 0;
}
function combineFrom(first, second) {
  const out = {};
  if (first) {
    for (const key in first) {
      out[key] = first[key];
    }
  }
  if (second) {
    for (const key in second) {
      out[key] = second[key];
    }
  }
  return out;
}

function combinePath({
  pathname = '/',
  search = '',
  hash = ''
}) {
  return pathname + search + hash;
}
function invariant(condition, format, a, b, c, d, e, f) {
  if (!condition) {
    let error;
    if (format === undefined) {
      error = new Error('Minified exception occurred; use the non-minified dev environment ' + 'for the full error message and additional helpful warnings.');
    } else {
      const args = [a, b, c, d, e, f];
      let argIndex = 0;
      error = new Error(format.replace(/%s/g, function () {
        return args[argIndex++];
      }));
      error.name = 'Invariant Violation';
    }
    error.framesToPop = 1; // we don't care about invariant's own frame
    throw error;
  }
}

const patternCache = {};
const cacheLimit = 10000;
let cacheCount = 0;
const compilePath = (pattern, options) => {
  const cacheKey = `${options.end}${options.strict}${options.sensitive}`;
  const cache = patternCache[cacheKey] || (patternCache[cacheKey] = {});
  if (cache[pattern]) {
    return cache[pattern];
  }
  const keys = [];
  const re = pathToRegexp(pattern, keys, options);
  const compiledPattern = {
    re,
    keys
  };
  if (cacheCount < cacheLimit) {
    cache[pattern] = compiledPattern;
    cacheCount++;
  }
  return compiledPattern;
};
/**
 * Public API for matching a URL pathname to a path pattern.
 */
function matchPath(pathname, options) {
  if (typeof options === 'string') {
    options = {
      path: options
    };
  }
  const {
    path = '/',
    exact = false,
    strict = false,
    sensitive = false,
    loader,
    initialData = {}
  } = options;
  const {
    re,
    keys
  } = compilePath(path, {
    end: exact,
    strict,
    sensitive
  });
  const match = re.exec(pathname);
  if (!match) {
    return null;
  }
  const loaderData = initialData[path];
  const [url, ...values] = match;
  const isExact = pathname === url;
  if (exact && !isExact) {
    return null;
  }
  return {
    isExact,
    loader,
    loaderData,
    params: keys.reduce((memo, key, index) => {
      memo[key.name] = values[index];
      return memo;
    }, {}),
    path,
    url: path === '/' && url === '' ? '/' : url // the matched portion of the URL
  };
}

function getMatch(pathname, {
  path,
  exact,
  strict,
  sensitive,
  loader,
  from
}, router) {
  path ??= from;
  const {
    initialData,
    route
  } = router; // This is the parent route
  return path ? matchPath(pathname, {
    path,
    exact,
    strict,
    sensitive,
    loader,
    initialData
  }) : route.match;
}
function extractFirstMatchFromChildren(pathname, children, router) {
  if (isArray(children)) {
    for (let i = 0; i < children.length; ++i) {
      const nestedMatch = extractFirstMatchFromChildren(pathname, children[i], router);
      if (nestedMatch.match) return nestedMatch;
    }
    return {};
  }
  return {
    _child: children,
    match: getMatch(pathname, children.props, router)
  };
}
class Switch extends Component {
  constructor(props, context) {
    super(props, context);
    const {
      router
    } = context;
    const {
      location,
      children
    } = props;
    const pathname = (location || router.route.location).pathname;
    const {
      match,
      _child
    } = extractFirstMatchFromChildren(pathname, children, router);
    this.state = {
      _child,
      match
    };
  }
  componentWillReceiveProps(nextProps, nextContext) {
    const {
      router
    } = nextContext;
    const {
      location,
      children
    } = nextProps;
    const pathname = (location || router.route.location).pathname;
    const {
      match,
      _child
    } = extractFirstMatchFromChildren(pathname, children, router);
    this.setState({
      match,
      _child
    });
  }
  render({
    children,
    location
  }, {
    match,
    _child
  }, context) {
    if (isInvalid(children)) {
      return null;
    }
    if (match) {
      location ??= context.router.route.location;
      return createComponentVNode(_child.flags, _child.type, combineFrom(_child.props, {
        location,
        computedMatch: match
      }));
    }
    return null;
  }
}

class Route extends Component {
  constructor(props, context) {
    super(props, context);
    const match = this.computeMatch(props, context.router);
    this.state = {
      __loaderData__: match?.loaderData,
      match
    };
  }
  getChildContext() {
    const parentRouter = this.context.router;
    const router = combineFrom(parentRouter, null);
    router.route = {
      location: this.props.location || parentRouter.route.location,
      match: this.state.match
    };
    return {
      router
    };
  }
  computeMatch({
    computedMatch,
    ...props
  }, router) {
    if (!isNullOrUndef(computedMatch)) {
      // <Switch> already computed the match for us
      return computedMatch;
    }
    const {
      path,
      strict,
      exact,
      sensitive,
      loader
    } = props;
    const {
      route,
      initialData
    } = router; // This is the parent route
    const pathname = (props.location || route.location).pathname;
    return path ? matchPath(pathname, {
      path,
      strict,
      exact,
      sensitive,
      loader,
      initialData
    }) : route.match;
  }
  componentWillReceiveProps(nextProps, nextContext) {
    const match = this.computeMatch(nextProps, nextContext.router);
    this.setState({
      __loaderData__: match?.loaderData,
      match
    });
  }
  render(props, state, context) {
    const {
      match,
      __loaderData__
    } = state;
    const {
      children,
      component,
      render,
      loader
    } = props;
    const {
      history,
      route,
      staticContext
    } = context.router;
    const location = props.location || route.location;
    const renderProps = {
      match,
      location,
      history,
      staticContext,
      component,
      render,
      loader,
      __loaderData__
    };
    // If we have a loader we don't render until it has been resolved
    if (!isUndefined(loader) && isUndefined(__loaderData__)) {
      return null;
    }
    if (component) {
      return match ? createComponentVNode(2 /* VNodeFlags.ComponentUnknown */, component, renderProps) : null;
    }
    if (render) {
      // @ts-ignore
      return match ? render(renderProps, this.context) : null;
    }
    if (typeof children === 'function') {
      return children(renderProps);
    }
    return children;
  }
}

function resolveLoaders(loaderEntries) {
  const promises = loaderEntries.map(({
    path,
    params,
    request,
    loader
  }) => {
    return resolveEntry(path, params, request, loader);
  });
  return Promise.all(promises).then(result => {
    return Object.fromEntries(result);
  });
}
function traverseLoaders(location, tree, base) {
  return _traverseLoaders(location, tree, base, false);
}
function _isSwitch(node) {
  // Using the same patterns as for _isRoute, but I don't have a test where
  // I pass a Switch via an array, but it is better to be consistent.
  return node?.type?.prototype instanceof Switch || node?.type === Switch;
}
function _isRoute(node) {
  // So the === check is needed if routes are passed in an array,
  // the instanceof test if routes are passed as children to a Component
  // This feels inconsistent, but at least it works.
  return node?.type?.prototype instanceof Route || node?.type === Route;
}
// Optionally pass base param during SSR to get fully qualified request URI passed to loader in request param
function _traverseLoaders(location, tree, base, parentIsSwitch = false) {
  // Make sure tree isn't null
  if (isNullOrUndef(tree)) return [];
  if (Array.isArray(tree)) {
    let hasMatch = false;
    const entriesOfArr = tree.reduce((res, node) => {
      if (parentIsSwitch && hasMatch) return res;
      const outpArr = _traverseLoaders(location, node, base, _isSwitch(node));
      if (parentIsSwitch && outpArr.length > 0) {
        hasMatch = true;
      }
      return [...res, ...outpArr];
    }, []);
    return entriesOfArr;
  }
  const outp = [];
  if (_isRoute(tree) && tree.props) {
    // TODO: Should we check if we are in Router? It is defensive and could save a bit of time, but is it worth it?
    const {
      path,
      exact = false,
      strict = false,
      sensitive = false
    } = tree.props;
    const match = matchPath(location, {
      exact,
      path,
      sensitive,
      strict
    });
    // So we can bail out of recursion it this was a Route which didn't match
    if (!match) {
      return outp;
    } else if (!tree.context && tree.props?.loader && tree.props?.path) {
      // Add any loader on this node (but only on the VNode)
      const {
        params
      } = match;
      const controller = new AbortController();
      const request = createClientSideRequest(location, controller.signal, base);
      outp.push({
        controller,
        loader: tree.props.loader,
        params,
        path,
        request
      });
    }
  }
  // Traverse children
  const children = tree.children ?? tree.props?.children;
  if (isNullOrUndef(children)) return outp;
  const entries = _traverseLoaders(location, children, base, _isSwitch(tree));
  return [...outp, ...entries];
}
function resolveEntry(path, params, request, loader) {
  return loader({
    params,
    request
  }).then(res => {
    // This implementation is based on:
    // https://github.com/remix-run/react-router/blob/4f3ad7b96e6e0228cc952cd7eafe2c265c7393c7/packages/router/router.ts#L2787-L2879
    // Check if regular data object (from tests or initialData)
    if (typeof res.json !== 'function') {
      return [path, {
        res
      }];
    }
    const contentType = res.headers.get('Content-Type');
    let dataPromise;
    // Check between word boundaries instead of startsWith() due to the last
    // paragraph of https://httpwg.org/specs/rfc9110.html#field.content-type
    if (contentType && /\bapplication\/json\b/.test(contentType)) {
      dataPromise = res.json();
    } else {
      dataPromise = res.text();
    }
    return dataPromise.then(body => {
      // We got a JSON error
      if (!res.ok) {
        return [path, {
          err: body
        }];
      }
      // We got JSON response
      return [path, {
        res: body
      }];
    })
    // Could not parse JSON
    .catch(err => [path, {
      err
    }]);
  })
  // Could not fetch data
  .catch(err => [path, {
    err
  }]);
}
const inBrowser = typeof window === 'undefined';
function createClientSideRequest(location, signal,
// submission?: Submission
base) {
  const url = inBrowser || !isUndefined(base) ? createClientSideURL(location, base) : location.toString();
  const init = {
    signal
  };
  // Content-Type is inferred (https://fetch.spec.whatwg.org/#dom-request)
  return new Request(url, init);
}
/**
 * Parses a string URL path into its separate pathname, search, and hash components.
 */
function createClientSideURL(location, base) {
  if (base === undefined && typeof window !== 'undefined') {
    // window.location.origin is "null" (the literal string value) in Firefox
    // under certain conditions, notably when serving from a local HTML file
    // See https://bugzilla.mozilla.org/show_bug.cgi?id=878297
    base = window?.location?.origin !== 'null' ? window.location.origin : window.location.href;
  }
  const url = new URL(location.toString(), base);
  url.hash = '';
  return url;
}
// TODO: react-router supports submitting forms with loaders, this is related to that
// function isMutationMethod(method?: string): method is MutationFormMethod {
//   return validMutationMethods.has(method as MutationFormMethod);
// }
// function convertFormDataToSearchParams(formData: FormData): URLSearchParams {
//   let searchParams = new URLSearchParams();
//   for (let [key, value] of formData.entries()) {
//     // invariant(
//     //   typeof value === "string",
//     //   'File inputs are not supported with encType "application/x-www-form-urlencoded", ' +
//     //     'please use "multipart/form-data" instead.'
//     // );
//     if (typeof value === "string") {
//       searchParams.append(key, value);
//     }
//   }
//   return searchParams;
// }

/**
 * The public API for putting history on context.
 */
class Router extends Component {
  constructor(props, context) {
    super(props, context);
    this.unlisten = void 0;
    this._loaderFetchControllers = [];
    this._loaderIteration = 0;
    const match = this.computeMatch(props.history.location.pathname);
    this.state = {
      initialData: this.props.initialData,
      match
    };
  }
  getChildContext() {
    const parentRouter = this.context.router;
    const router = combineFrom(parentRouter, null);
    router.history = this.props.history;
    router.route = {
      location: router.history.location,
      match: this.state?.match // Why are we sending this? it appears useless.
    };

    router.initialData = this.state?.initialData; // this is a dictionary of all data available
    return {
      router
    };
  }
  computeMatch(pathname) {
    return {
      isExact: pathname === '/',
      loader: undefined,
      params: {},
      path: '/',
      url: '/'
    };
  }
  componentWillMount() {
    const {
      history
    } = this.props;
    // Do this here so we can setState when a <Redirect> changes the
    // location in componentWillMount. This happens e.g. when doing
    // server rendering using a <StaticRouter>.
    this.unlisten = history.listen(() => {
      const match = this.computeMatch(history.location.pathname);
      this._matchAndResolveLoaders(match);
    });
    // First execution of loaders
    if (isUndefined(this.props.initialData)) {
      this._matchAndResolveLoaders(this.state?.match);
    }
  }
  _matchAndResolveLoaders(match) {
    // Keep track of invokation order
    // Bumping the counter needs to be done first because calling abort
    // triggers promise to resolve with "aborted"
    this._loaderIteration = (this._loaderIteration + 1) % 10000;
    const currentIteration = this._loaderIteration;
    for (const controller of this._loaderFetchControllers) {
      controller.abort();
    }
    this._loaderFetchControllers = [];
    const {
      history,
      children
    } = this.props;
    const loaderEntries = traverseLoaders(history.location.pathname, children);
    if (loaderEntries.length === 0) {
      this.setState({
        match
      });
      return;
    }
    // Store AbortController instances for each matched loader
    this._loaderFetchControllers = loaderEntries.map(e => e.controller);
    resolveLoaders(loaderEntries).then(initialData => {
      // On multiple pending navigations, only update interface with last
      // in case they resolve out of order
      if (currentIteration === this._loaderIteration) {
        this.setState({
          initialData,
          match
        });
      }
    });
  }
  componentWillUnmount() {
    this.unlisten();
  }
  render(props) {
    return props.children;
  }
}

function addLeadingSlash(path) {
  return path.charAt(0) === '/' ? path : '/' + path;
}
// tslint:disable-next-line:no-empty
const noop = () => {};
class StaticRouter extends Component {
  constructor(...args) {
    super(...args);
    this.createHref = path => addLeadingSlash((this.props.basename || '') + createURL(path));
    this.handlePush = location => {
      const {
        basename,
        context
      } = this.props;
      context.action = 'PUSH';
      context.location = addBasename(basename, isString(location) ? parsePath(location) : location);
      context.url = createURL(context.location);
    };
    this.handleReplace = location => {
      const {
        basename,
        context
      } = this.props;
      context.action = 'REPLACE';
      context.location = addBasename(basename, isString(location) ? parsePath(location) : location);
      context.url = createURL(context.location);
    };
    // tslint:disable-next-line:no-empty
    this.handleListen = () => noop;
    // tslint:disable-next-line:no-empty
    this.handleBlock = () => noop;
  }
  getChildContext() {
    return {
      router: {
        initialData: this.props.initialData,
        staticContext: this.props.context
      }
    };
  }
  render({
    basename,
    context,
    location,
    ...props
  }) {
    return createComponentVNode(4 /* VNodeFlags.ComponentClass */, Router, combineFrom(props, {
      history: {
        action: 'POP',
        block: this.handleBlock,
        createHref: this.createHref,
        go: staticHandler('go'),
        goBack: staticHandler('goBack'),
        goForward: staticHandler('goForward'),
        listen: this.handleListen,
        location: stripBasename(basename, createLocation(location)),
        push: this.handlePush,
        replace: this.handleReplace
      }
    }));
  }
}
StaticRouter.defaultProps = {
  basename: '',
  location: '/'
};
function normalizeLocation({
  pathname = '/',
  search,
  hash
}) {
  return {
    hash: (hash || '') === '#' ? '' : hash,
    pathname,
    search: (search || '') === '?' ? '' : search
  };
}
function addBasename(basename, location) {
  if (!basename) {
    return location;
  }
  return combineFrom(location, {
    pathname: addLeadingSlash(basename) + location.pathname
  });
}
function stripBasename(basename, location) {
  if (!basename) {
    return location;
  }
  const base = addLeadingSlash(basename);
  if (location.pathname.indexOf(base) !== 0) {
    return location;
  }
  return combineFrom(location, {
    pathname: location.pathname.substring(base.length)
  });
}
function createLocation(location) {
  return typeof location === 'string' ? parsePath(location) : normalizeLocation(location);
}
function createURL(location) {
  return typeof location === 'string' ? location : combinePath(location);
}
function staticHandler(methodName) {
  return () => {
    invariant(false, 'You cannot %s with <StaticRouter>', methodName);
  };
}

class BrowserRouter extends Component {
  constructor(props, context) {
    super(props, context);
    this.history = void 0;
    this.history = createBrowserHistory();
  }
  render() {
    return createComponentVNode(4 /* VNodeFlags.ComponentClass */, Router, {
      children: this.props.children,
      history: this.history,
      initialData: this.props.initialData
    });
  }
}

class HashRouter extends Component {
  constructor(props, context) {
    super(props, context);
    this.history = void 0;
    this.history = createHashHistory();
  }
  render() {
    return createComponentVNode(4 /* VNodeFlags.ComponentClass */, Router, {
      children: this.props.children,
      history: this.history
    });
  }
}

class MemoryRouter extends Component {
  constructor(props, context) {
    super(props, context);
    this.history = void 0;
    this.history = createMemoryHistory(props);
  }
  render() {
    return createComponentVNode(4 /* VNodeFlags.ComponentClass */, Router, {
      children: this.props.children,
      history: this.history,
      initialData: this.props.initialData
    });
  }
}

const normalizeToLocation = to => {
  return isString(to) ? parsePath(to) : to;
};
const splitLocation = location => {
  const {
    key = '',
    state,
    ...to
  } = location;
  return {
    to,
    state
  };
};

const isModifiedEvent = event => Boolean(event.metaKey || event.altKey || event.ctrlKey || event.shiftKey);
function handleClick({
  props,
  context
}, event) {
  if (props.onClick) {
    props.onClick(event);
  }
  if (!event.defaultPrevented &&
  // onClick prevented default
  event.button === 0 &&
  // ignore everything but left clicks
  !props.target &&
  // let browser handle "target=_blank" etc.
  !isModifiedEvent(event) // ignore clicks with modifier keys
  ) {
    event.preventDefault();
    const {
      history
    } = context.router;
    const {
      replace = false,
      to: toPropIn
    } = props;
    const {
      to,
      state
    } = splitLocation(normalizeToLocation(toPropIn));
    if (replace) {
      history.replace(to, state);
    } else {
      history.push(to, state);
    }
  }
}
/**
 * The public API for rendering a history-aware <a>.
 */
function Link(props, context) {
  const {
    replace,
    children,
    className,
    to = '',
    innerRef,
    ...rest
  } = props;
  invariant(context.router, 'You should not use <Link> outside a <Router>');
  const href = context.router.history.createHref(isString(to) ? parsePath(to) : to);
  const newProps = combineFrom(rest, null);
  newProps.href = href;
  newProps.onClick = linkEvent({
    context,
    props
  }, handleClick);
  return createVNode(1 /* VNodeFlags.HtmlElement */, 'a', className, children, 0 /* ChildFlags.UnknownChildren */, newProps, null, innerRef);
}

function filter(i) {
  return i;
}
/**
 * A <Link> wrapper that knows if it's "active" or not.
 */
function NavLink({
  to,
  exact,
  strict,
  onClick,
  location: linkLocation,
  activeClassName = 'active',
  className: classNameProp,
  activeStyle,
  style: styleProp,
  isActive: getIsActive,
  ariaCurrent = 'true',
  ...rest
}) {
  function linkComponent({
    location,
    match
  }) {
    const isActive = Boolean(getIsActive ? getIsActive(match, location) : match);
    const className = typeof classNameProp === 'function' ? classNameProp(isActive) : classNameProp;
    const style = typeof styleProp === 'function' ? styleProp(isActive) : styleProp;
    return createComponentVNode(8 /* VNodeFlags.ComponentFunction */, Link, combineFrom({
      'aria-current': isActive && ariaCurrent || null,
      className: isActive ? [className, activeClassName].filter(filter).join(' ') : className,
      onClick,
      style: isActive ? combineFrom(style, activeStyle) : style,
      to
    }, rest));
  }
  return createComponentVNode(4 /* VNodeFlags.ComponentClass */, Route, {
    children: linkComponent,
    exact,
    location: linkLocation,
    path: typeof to === 'object' ? to.pathname : to,
    strict
  });
}

/**
 * The public API for matching a single path and rendering.
 */
class Prompt extends Component {
  constructor(...args) {
    super(...args);
    this.unblock = void 0;
  }
  enable(message) {
    if (this.unblock) {
      this.unblock();
    }
    this.unblock = this.context.router.history.block(tx => {
      if (message && window.confirm(message)) {
        this.unblock();
        tx.retry();
      }
    });
  }
  disable() {
    if (this.unblock) {
      this.unblock();
      this.unblock = null;
    }
  }
  componentWillMount() {
    invariant(this.context.router, 'You should not use <Prompt> outside a <Router>');
    if (this.props.when) {
      this.enable(this.props.message);
    }
  }
  componentWillReceiveProps(nextProps) {
    if (nextProps.when) {
      if (!this.props.when || this.props.message !== nextProps.message) {
        this.enable(nextProps.message);
      }
    } else {
      this.disable();
    }
  }
  componentWillUnmount() {
    this.disable();
  }
  render() {
    return null;
  }
}

function getLocationTarget(to) {
  if (!isString(to)) {
    to = combinePath(to);
  }
  return parsePath(to);
}
class Redirect extends Component {
  isStatic() {
    return this.context.router && this.context.router.staticContext;
  }
  componentWillMount() {
    invariant(this.context.router, 'You should not use <Redirect> outside a <Router>');
    if (this.isStatic()) {
      this.perform();
    }
  }
  componentDidMount() {
    if (!this.isStatic()) {
      this.perform();
    }
  }
  componentDidUpdate(prevProps) {
    const prevTo = getLocationTarget(prevProps.to);
    const nextTo = getLocationTarget(this.props.to);
    if (prevTo.pathname === nextTo.pathname && prevTo.search === nextTo.search) {
      // tslint:disable-next-line:no-console
      console.error(`You tried to redirect to the same route you're currently on: "${nextTo.pathname}${nextTo.search}"`);
      return;
    }
    this.perform();
  }
  perform() {
    const {
      history
    } = this.context.router;
    const {
      push = false,
      to
    } = this.props;
    if (push) {
      history.push(to);
    } else {
      history.replace(to);
    }
  }
  render() {
    return null;
  }
}

/**
 * A public higher-order component to access the imperative API
 */
function withRouter(Com) {
  const C = function (props) {
    const {
      wrappedComponentRef,
      ...remainingProps
    } = props;
    return createComponentVNode(4 /* VNodeFlags.ComponentClass */, Route, {
      render(routeComponentProps) {
        return createComponentVNode(2 /* VNodeFlags.ComponentUnknown */, Com, combineFrom(remainingProps, routeComponentProps), null, wrappedComponentRef);
      }
    });
  };
  C.displayName = `withRouter(${Com.displayName || Com.name})`;
  C.WrappedComponent = Com;
  return hoistNonReactStatics(C, Com);
}

function useLoaderData(props) {
  return props.__loaderData__?.res;
}
function useLoaderError(props) {
  return props.__loaderData__?.err;
}

export { BrowserRouter, HashRouter, Link, MemoryRouter, NavLink, Prompt, Redirect, Route, Router, StaticRouter, Switch, createClientSideURL, matchPath, resolveLoaders, traverseLoaders, useLoaderData, useLoaderError, withRouter };
