"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JSONSchemaRenderer = void 0;
const collection_utils_1 = require("collection-utils");
const Description_1 = require("../../attributes/Description");
const ConvenienceRenderer_1 = require("../../ConvenienceRenderer");
const Support_1 = require("../../support/Support");
const Type_1 = require("../../Type");
const TypeUtils_1 = require("../../Type/TypeUtils");
const utils_1 = require("./utils");
class JSONSchemaRenderer extends ConvenienceRenderer_1.ConvenienceRenderer {
    makeNamedTypeNamer() {
        return utils_1.namingFunction;
    }
    namerForObjectProperty() {
        return null;
    }
    makeUnionMemberNamer() {
        return null;
    }
    makeEnumCaseNamer() {
        return null;
    }
    nameForType(t) {
        return (0, Support_1.defined)(this.names.get(this.nameForNamedType(t)));
    }
    makeOneOf(types) {
        const first = (0, collection_utils_1.iterableFirst)(types);
        if (first === undefined) {
            return (0, Support_1.panic)("Must have at least one type for oneOf");
        }
        if (types.size === 1) {
            return this.schemaForType(first);
        }
        return {
            anyOf: Array.from(types).map((t) => this.schemaForType(t)),
        };
    }
    makeRef(t) {
        return { $ref: `#/definitions/${this.nameForType(t)}` };
    }
    addAttributesToSchema(t, schema) {
        const attributes = this.typeGraph.attributeStore.attributesForType(t);
        for (const [kind, attr] of attributes) {
            kind.addToSchema(schema, t, attr);
        }
    }
    schemaForType(t) {
        const schema = (0, TypeUtils_1.matchTypeExhaustive)(t, (_noneType) => {
            return (0, Support_1.panic)("none type should have been replaced");
        }, (_anyType) => ({}), (_nullType) => ({ type: "null" }), (_boolType) => ({ type: "boolean" }), (_integerType) => ({ type: "integer" }), (_doubleType) => ({ type: "number" }), (_stringType) => ({ type: "string" }), (arrayType) => ({
            type: "array",
            items: this.schemaForType(arrayType.items),
        }), (classType) => this.makeRef(classType), (mapType) => this.definitionForObject(mapType, undefined), (objectType) => this.makeRef(objectType), (enumType) => this.makeRef(enumType), (unionType) => {
            if (this.unionNeedsName(unionType)) {
                return this.makeRef(unionType);
            }
            return this.definitionForUnion(unionType);
        }, (transformedStringType) => {
            const target = Type_1.transformedStringTypeTargetTypeKindsMap.get(transformedStringType.kind);
            if (target === undefined) {
                return (0, Support_1.panic)(`Unknown transformed string type ${transformedStringType.kind}`);
            }
            return { type: "string", format: target.jsonSchema };
        });
        if (schema.$ref === undefined) {
            this.addAttributesToSchema(t, schema);
        }
        return schema;
    }
    definitionForObject(o, title) {
        let properties;
        let required;
        if (o.getProperties().size === 0) {
            properties = undefined;
            required = undefined;
        }
        else {
            const props = {};
            const req = [];
            for (const [name, p] of o.getProperties()) {
                const prop = this.schemaForType(p.type);
                if (prop.description === undefined) {
                    (0, Description_1.addDescriptionToSchema)(prop, this.descriptionForClassProperty(o, name));
                }
                props[name] = prop;
                if (!p.isOptional) {
                    req.push(name);
                }
            }
            properties = props;
            required = req.sort();
        }
        const additional = o.getAdditionalProperties();
        const additionalProperties = additional !== undefined ? this.schemaForType(additional) : false;
        const schema = {
            type: "object",
            additionalProperties,
            properties,
            required,
            title,
        };
        this.addAttributesToSchema(o, schema);
        return schema;
    }
    definitionForUnion(u, title) {
        const oneOf = this.makeOneOf(u.sortedMembers);
        if (title !== undefined) {
            oneOf.title = title;
        }
        this.addAttributesToSchema(u, oneOf);
        return oneOf;
    }
    definitionForEnum(e, title) {
        const schema = { type: "string", enum: Array.from(e.cases), title };
        this.addAttributesToSchema(e, schema);
        return schema;
    }
    emitSourceStructure() {
        // FIXME: Find a good way to do multiple top-levels.  Maybe multiple files?
        const topLevelType = this.topLevels.size === 1
            ? this.schemaForType((0, Support_1.defined)((0, collection_utils_1.mapFirst)(this.topLevels)))
            : {};
        const schema = Object.assign({ $schema: "http://json-schema.org/draft-06/schema#" }, topLevelType);
        const definitions = {};
        this.forEachObject("none", (o, name) => {
            const title = (0, Support_1.defined)(this.names.get(name));
            definitions[title] = this.definitionForObject(o, title);
        });
        this.forEachUnion("none", (u, name) => {
            if (!this.unionNeedsName(u))
                return;
            const title = (0, Support_1.defined)(this.names.get(name));
            definitions[title] = this.definitionForUnion(u, title);
        });
        this.forEachEnum("none", (e, name) => {
            const title = (0, Support_1.defined)(this.names.get(name));
            definitions[title] = this.definitionForEnum(e, title);
        });
        schema.definitions = definitions;
        this.emitMultiline(JSON.stringify(schema, undefined, "    "));
    }
}
exports.JSONSchemaRenderer = JSONSchemaRenderer;
