"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.flattenStrings = flattenStrings;
const collection_utils_1 = require("collection-utils");
const TypeAttributes_1 = require("../attributes/TypeAttributes");
const Support_1 = require("../support/Support");
const TypeUtils_1 = require("../Type/TypeUtils");
// A union needs replacing if it contains more than one string type, one of them being
// a basic string type.
function unionNeedsReplacing(u) {
    const stringMembers = u.stringTypeMembers;
    if (stringMembers.size <= 1)
        return undefined;
    const stringType = u.findMember("string");
    if (stringType === undefined)
        return undefined;
    (0, Support_1.assert)(!(0, TypeUtils_1.stringTypesForType)(stringType).isRestricted, "We must only flatten strings if we have no restriced strings");
    return stringMembers;
}
// Replaces all string types in an enum with the basic string type.
function replaceUnion(group, builder, forwardingRef) {
    (0, Support_1.assert)(group.size === 1);
    const u = (0, Support_1.defined)((0, collection_utils_1.iterableFirst)(group));
    const stringMembers = (0, Support_1.defined)(unionNeedsReplacing(u));
    const stringAttributes = (0, TypeUtils_1.combineTypeAttributesOfTypes)("union", stringMembers);
    const types = [];
    for (const t of u.members) {
        if (stringMembers.has(t))
            continue;
        types.push(builder.reconstituteType(t));
    }
    if (types.length === 0) {
        return builder.getStringType((0, TypeAttributes_1.combineTypeAttributes)("union", stringAttributes, u.getAttributes()), undefined, forwardingRef);
    }
    types.push(builder.getStringType(stringAttributes, undefined));
    return builder.getUnionType(u.getAttributes(), new Set(types), forwardingRef);
}
function flattenStrings(graph, stringTypeMapping, debugPrintReconstitution) {
    const allUnions = graph.allNamedTypesSeparated().unions;
    const unionsToReplace = Array.from(allUnions)
        .filter(unionNeedsReplacing)
        .map((t) => [t]);
    return graph.rewrite("flatten strings", stringTypeMapping, false, unionsToReplace, debugPrintReconstitution, replaceUnion);
}
