/**
 * The read-only interface of [[DefaultMap]].
 *
 * @typeParam K The key type.
 * @typeParam V The value type.
 */
export interface ReadonlyDefaultMap<K, V> extends ReadonlyMap<K, V> {
    get(k: K): V;
}
/**
 * A `Map` that returns default values for keys that are not present.
 *
 * To map strings to `Set`s of numbers, for example, `DefaultMap` can
 * be used like this:
 *
 * ```
 * const m = new DefaultMap<string, Set<number>>(() => new Set());
 * m.get("foo"); // returns an empty `Set`
 * m.get("bar").add(1);
 * m.get("bar"); // returns a `Set` containing `1`.
 * ```
 *
 * @typeParam K The key type.
 * @typeParam V The value type.
 */
export declare class DefaultMap<K, V> extends Map<K, V> implements ReadonlyDefaultMap<K, V> {
    private readonly _defaultFunc;
    /**
     *
     * @param _defaultFunc Must return the default value for key `k`.  Will only be called when necessary.
     * @param initKVPs An array of [key, value] arrays to initialize the map with.
     */
    constructor(_defaultFunc: (k: K) => V, initKVPs?: readonly (readonly [K, V])[]);
    /**
     * Returns the value for `k`.  If `k` is not present, creates
     * the default value via [[_defaultFunc]], sets that default
     * as the value for `k` and returns it.
     */
    get(k: K): V;
    /**
     * Sets the value for `k` to `f(v)` where `v` is the previous
     * value for `k`, or the default if not present.  Returns the new
     * value.
     *
     * These two lines are equivalent:
     *
     * ```
     * m.set("foo", m.get("foo") + 1);
     * m.update("foo", x => x + 1);
     * ```
     */
    update(k: K, f: (v: V) => V): V;
}
