# Braket.jl

**Braket.jl is not an officially supported AWS product.**

This package is a Julia implementation of the [Amazon Braket SDK](https://github.com/aws/amazon-braket-sdk-python) allowing customers to access [Quantum Hardware and Simulators](https://aws.amazon.com/braket/quantum-computers/). 

This is *experimental* software, and support may be discontinued in the future. For a fully supported SDK, please use
the [Python SDK](https://github.com/aws/amazon-braket-sdk-python). We may change, remove, or deprecate parts of the API when making new releases.
Please review the [CHANGELOG](CHANGELOG.md) for information about changes in each release. 

[![Stable docs](https://img.shields.io/badge/docs-stable-blue.svg)](https://amazon-braket.github.io/Braket.jl/stable)
[![Latest docs](https://img.shields.io/badge/docs-dev-blue.svg)](https://amazon-braket.github.io/Braket.jl/dev)
[![CI](https://github.com/amazon-braket/braket.jl/actions/workflows/CI.yml/badge.svg)](https://github.com/amazon-braket/braket.jl/actions/workflows/CI.yml)
[![codecov](https://codecov.io/gh/amazon-braket/braket.jl/graph/badge.svg?token=xocMQMNqAN)](https://codecov.io/gh/amazon-braket/braket.jl)

## Installation & Prerequisites

You do *not* need a Python installation or the Python Amazon Braket SDK installed to use this package.
However, to use the Amazon Braket Analog Hamiltonian Simulation [Local Simulator](https://docs.aws.amazon.com/braket/latest/developerguide/braket-send-to-local-simulator.html)
you'll need to install the sub-package `PyBraket.jl`,
included in this repository. See its [`README`](PyBraket/README.md) for more information.

For *gate based* local simulations, with or without noise, see [`BraketSimulator.jl`](https://github.com/amazon-braket/braketsimulator.jl).

All necessary Julia packages will be installed for you when you run `Pkg.add("Braket")`
or `] instantiate` (if you're doing a [`dev`](https://pkgdocs.julialang.org/v1/managing-packages/#developing) install).

If you want to run tasks on Amazon Braket's [managed simulators or QPUs](https://docs.aws.amazon.com/braket/latest/developerguide/braket-devices.html) or run
[managed jobs](https://docs.aws.amazon.com/braket/latest/developerguide/braket-jobs-works.html),
you will need an AWS account and to have [onboarded](https://docs.aws.amazon.com/braket/latest/developerguide/braket-enable-overview.html) with the Amazon Braket service.
`Braket.jl` can load your AWS credentials from your environment variables or your `~/.aws/config` file thanks to the [`AWS.jl`](https://github.com/JuliaCloud/AWS.jl) package.

## Usage Notes

Keep in mind that the first qubit has index `0`, **not** index `1`.
Some Amazon Braket Hybrid Jobs return results as pickled objects, which are currently not decompressible using `Braket.jl`.
In such cases you may need to download and extract the results using `PyBraket.jl` and the Amazon Braket SDK.

## Examples

Constructing a simple circuit:

```julia
using Braket

c = Circuit()
c = H(c, 0) # qubits are 0-indexed
c = CNot(c, 0, 1)
c = Probability(c)
```

Measuring expectation values on a QPU:

```julia
using Braket, Braket.Observables

c = Circuit()
c = H(c, 0)
c = CNot(c, 0, 1)
c = Expectation(c, Observables.X()) # measure X on all qubits

dev = AwsDevice("arn:aws:braket:us-east-1::device/qpu/ionq/Harmony")
res = result(dev(c, shots=10))
```

Simulating a noisy circuit locally:

```julia
using BraketSimulator, Braket

c = Circuit()
c = H(c, 0) # qubits are 0-indexed
c = CNot(c, 0, 1)
c = BitFlip(c, 0, 0.1)
c = PhaseFlip(c, 1, 0.2)
c = Probability(c)

dev = LocalSimulator("braket_dm_v2")
res = result(simulate(dev, c, shots=200))
``` 

## TODO and development roadmap

As Amazon Braket expands its featureset, we welcome contributions and development from the Julia community.  

What's currently implemented in *pure* Julia:

- All of the [`Amazon Braket schemas`](https://github.com/aws/amazon-braket-schemas-python).
- Submitting [`Amazon Braket Hybrid Jobs`](https://docs.aws.amazon.com/braket/latest/developerguide/braket-jobs.html)
- Abstract interface for `LocalSimulator`s
- Local jobs
- Building and submitting circuits to managed simulators and QPUs
- Reading results from managed simulators and QPUs
- Cancelling tasks and jobs
- Fetching quantum task and device information from AWS
- Searching device information and looking up availability windows
- Noise models
- Convenience `Circuit` features
- Pretty printing of `Circuit`s
- Cost tracking
- All gates and noise operations
- Analog Hamiltonian Simulation

**Features to add:**

- [Support for pickled jobs results](https://github.com/amazon-braket/Braket.jl/issues/18)
- More robust entry point verification for jobs
- [Pulse control](https://github.com/amazon-braket/Braket.jl/issues/15)

## Security

See [CONTRIBUTING](CONTRIBUTING.md#security-issue-notifications) for more information.

## License

This project is licensed under the Apache-2.0 License.
