
abstract type CMF end
abstract type CIE1931_CMF <: CMF end
abstract type CIE1964_CMF <: CMF end
abstract type CIE1931J_CMF <: CMF end
abstract type CIE1931JV_CMF <: CMF end
abstract type CIE2006_2_CMF <: CMF end
abstract type CIE2006_10_CMF <: CMF end

"""
    colormatch(wavelength)
    colormatch(matchingfunction, wavelength)

Evaluate the CIE standard observer color match function.

# Arguments

- `matchingfunction` (optional): a type used to specify the matching function.
  Choices include:
  - `CIE1931_CMF` (the default, the CIE 1931 2° matching function)
  - `CIE1964_CMF` (the CIE 1964 10° color matching function)
  - `CIE1931J_CMF` (Judd adjustment to `CIE1931_CMF`)
  - `CIE1931JV_CMF` (Judd-Vos adjustment to `CIE1931_CMF`)
  - `CIE2006_2_CMF` (transformed from the CIE 2006 2° LMS cone fundamentals)
  - `CIE2006_10_CMF` (transformed from the CIE 2006 10° LMS cone fundamentals)
- `wavelength`: Wavelength of stimulus in nanometers.

Returns the XYZ value of perceived color.

!!! note
    As of February 2020, only `CIE1931_CMF` and `CIE1964_CMF` have been
    standardized by the ISO/CIE. `CIE2006_2_CMF` and `CIE2006_10_CMF` are
    proposals which have yet to be ratified by the CIE, even though they are
    sometimes referred to as CIE 2012 CMFs.

"""
function colormatch(wavelen::Real)
    return colormatch(CIE1931_CMF, wavelen)
end


@deprecate cie_color_match colormatch


# Interpolate between values in a cmf table, where the table starts
# at `start` nm and has `step` nm between entries.
function interpolate_table(tbl, start, step, wavelen)
    n = size(tbl, 1)
    stop = start + step * (n - 1)
    i = (wavelen - start) / step

    a = floor(Integer, i) + 1
    @inbounds ac = 1 <= a <= n ? tbl[a,:] : [0.0, 0.0, 0.0]

    b = ceil(Integer, i) + 1
    @inbounds bc = 1 <= b <= n ? tbl[b,:] : [0.0, 0.0, 0.0]

    p = i % 1.0
    ac = p * bc + (1.0 - p) * ac

    return @inbounds XYZ(ac[1], ac[2], ac[3])
end


function colormatch(::Type{CIE1931_CMF}, wavelen::Real)
    return interpolate_table(cie1931_cmf_table, 360.0, 1.0, wavelen)
end

# CIE 1931 2° color matching function, 1nm increments starting at 360nm
const cie1931_cmf_table =
    [0.000129900000  0.000003917000  0.000606100000;
     0.000145847000  0.000004393581  0.000680879200;
     0.000163802100  0.000004929604  0.000765145600;
     0.000184003700  0.000005532136  0.000860012400;
     0.000206690200  0.000006208245  0.000966592800;
     0.000232100000  0.000006965000  0.001086000000;
     0.000260728000  0.000007813219  0.001220586000;
     0.000293075000  0.000008767336  0.001372729000;
     0.000329388000  0.000009839844  0.001543579000;
     0.000369914000  0.000011043230  0.001734286000;
     0.000414900000  0.000012390000  0.001946000000;
     0.000464158700  0.000013886410  0.002177777000;
     0.000518986000  0.000015557280  0.002435809000;
     0.000581854000  0.000017442960  0.002731953000;
     0.000655234700  0.000019583750  0.003078064000;
     0.000741600000  0.000022020000  0.003486000000;
     0.000845029600  0.000024839650  0.003975227000;
     0.000964526800  0.000028041260  0.004540880000;
     0.001094949000  0.000031531040  0.005158320000;
     0.001231154000  0.000035215210  0.005802907000;
     0.001368000000  0.000039000000  0.006450001000;
     0.001502050000  0.000042826400  0.007083216000;
     0.001642328000  0.000046914600  0.007745488000;
     0.001802382000  0.000051589600  0.008501152000;
     0.001995757000  0.000057176400  0.009414544000;
     0.002236000000  0.000064000000  0.010549990000;
     0.002535385000  0.000072344210  0.011965800000;
     0.002892603000  0.000082212240  0.013655870000;
     0.003300829000  0.000093508160  0.015588050000;
     0.003753236000  0.000106136100  0.017730150000;
     0.004243000000  0.000120000000  0.020050010000;
     0.004762389000  0.000134984000  0.022511360000;
     0.005330048000  0.000151492000  0.025202880000;
     0.005978712000  0.000170208000  0.028279720000;
     0.006741117000  0.000191816000  0.031897040000;
     0.007650000000  0.000217000000  0.036210000000;
     0.008751373000  0.000246906700  0.041437710000;
     0.010028880000  0.000281240000  0.047503720000;
     0.011421700000  0.000318520000  0.054119880000;
     0.012869010000  0.000357266700  0.060998030000;
     0.014310000000  0.000396000000  0.067850010000;
     0.015704430000  0.000433714700  0.074486320000;
     0.017147440000  0.000473024000  0.081361560000;
     0.018781220000  0.000517876000  0.089153640000;
     0.020748010000  0.000572218700  0.098540480000;
     0.023190000000  0.000640000000  0.110200000000;
     0.026207360000  0.000724560000  0.124613300000;
     0.029782480000  0.000825500000  0.141701700000;
     0.033880920000  0.000941160000  0.161303500000;
     0.038468240000  0.001069880000  0.183256800000;
     0.043510000000  0.001210000000  0.207400000000;
     0.048995600000  0.001362091000  0.233692100000;
     0.055022600000  0.001530752000  0.262611400000;
     0.061718800000  0.001720368000  0.294774600000;
     0.069212000000  0.001935323000  0.330798500000;
     0.077630000000  0.002180000000  0.371300000000;
     0.086958110000  0.002454800000  0.416209100000;
     0.097176720000  0.002764000000  0.465464200000;
     0.108406300000  0.003117800000  0.519694800000;
     0.120767200000  0.003526400000  0.579530300000;
     0.134380000000  0.004000000000  0.645600000000;
     0.149358200000  0.004546240000  0.718483800000;
     0.165395700000  0.005159320000  0.796713300000;
     0.181983100000  0.005829280000  0.877845900000;
     0.198611000000  0.006546160000  0.959439000000;
     0.214770000000  0.007300000000  1.039050100000;
     0.230186800000  0.008086507000  1.115367300000;
     0.244879700000  0.008908720000  1.188497100000;
     0.258777300000  0.009767680000  1.258123300000;
     0.271807900000  0.010664430000  1.323929600000;
     0.283900000000  0.011600000000  1.385600000000;
     0.294943800000  0.012573170000  1.442635200000;
     0.304896500000  0.013582720000  1.494803500000;
     0.313787300000  0.014629680000  1.542190300000;
     0.321645400000  0.015715090000  1.584880700000;
     0.328500000000  0.016840000000  1.622960000000;
     0.334351300000  0.018007360000  1.656404800000;
     0.339210100000  0.019214480000  1.685295900000;
     0.343121300000  0.020453920000  1.709874500000;
     0.346129600000  0.021718240000  1.730382100000;
     0.348280000000  0.023000000000  1.747060000000;
     0.349599900000  0.024294610000  1.760044600000;
     0.350147400000  0.025610240000  1.769623300000;
     0.350013000000  0.026958570000  1.776263700000;
     0.349287000000  0.028351250000  1.780433400000;
     0.348060000000  0.029800000000  1.782600000000;
     0.346373300000  0.031310830000  1.782968200000;
     0.344262400000  0.032883680000  1.781699800000;
     0.341808800000  0.034521120000  1.779198200000;
     0.339094100000  0.036225710000  1.775867100000;
     0.336200000000  0.038000000000  1.772110000000;
     0.333197700000  0.039846670000  1.768258900000;
     0.330041100000  0.041768000000  1.764039000000;
     0.326635700000  0.043766000000  1.758943800000;
     0.322886800000  0.045842670000  1.752466300000;
     0.318700000000  0.048000000000  1.744100000000;
     0.314025100000  0.050243680000  1.733559500000;
     0.308884000000  0.052573040000  1.720858100000;
     0.303290400000  0.054980560000  1.705936900000;
     0.297257900000  0.057458720000  1.688737200000;
     0.290800000000  0.060000000000  1.669200000000;
     0.283970100000  0.062601970000  1.647528700000;
     0.276721400000  0.065277520000  1.623412700000;
     0.268917800000  0.068042080000  1.596022300000;
     0.260422700000  0.070911090000  1.564528000000;
     0.251100000000  0.073900000000  1.528100000000;
     0.240847500000  0.077016000000  1.486111400000;
     0.229851200000  0.080266400000  1.439521500000;
     0.218407200000  0.083666800000  1.389879900000;
     0.206811500000  0.087232800000  1.338736200000;
     0.195360000000  0.090980000000  1.287640000000;
     0.184213600000  0.094917550000  1.237422300000;
     0.173327300000  0.099045840000  1.187824300000;
     0.162688100000  0.103367400000  1.138761100000;
     0.152283300000  0.107884600000  1.090148000000;
     0.142100000000  0.112600000000  1.041900000000;
     0.132178600000  0.117532000000  0.994197600000;
     0.122569600000  0.122674400000  0.947347300000;
     0.113275200000  0.127992800000  0.901453100000;
     0.104297900000  0.133452800000  0.856619300000;
     0.095640000000  0.139020000000  0.812950100000;
     0.087299550000  0.144676400000  0.770517300000;
     0.079308040000  0.150469300000  0.729444800000;
     0.071717760000  0.156461900000  0.689913600000;
     0.064580990000  0.162717700000  0.652104900000;
     0.057950010000  0.169300000000  0.616200000000;
     0.051862110000  0.176243100000  0.582328600000;
     0.046281520000  0.183558100000  0.550416200000;
     0.041150880000  0.191273500000  0.520337600000;
     0.036412830000  0.199418000000  0.491967300000;
     0.032010000000  0.208020000000  0.465180000000;
     0.027917200000  0.217119900000  0.439924600000;
     0.024144400000  0.226734500000  0.416183600000;
     0.020687000000  0.236857100000  0.393882200000;
     0.017540400000  0.247481200000  0.372945900000;
     0.014700000000  0.258600000000  0.353300000000;
     0.012161790000  0.270184900000  0.334857800000;
     0.009919960000  0.282293900000  0.317552100000;
     0.007967240000  0.295050500000  0.301337500000;
     0.006296346000  0.308578000000  0.286168600000;
     0.004900000000  0.323000000000  0.272000000000;
     0.003777173000  0.338402100000  0.258817100000;
     0.002945320000  0.354685800000  0.246483800000;
     0.002424880000  0.371698600000  0.234771800000;
     0.002236293000  0.389287500000  0.223453300000;
     0.002400000000  0.407300000000  0.212300000000;
     0.002925520000  0.425629900000  0.201169200000;
     0.003836560000  0.444309600000  0.190119600000;
     0.005174840000  0.463394400000  0.179225400000;
     0.006982080000  0.482939500000  0.168560800000;
     0.009300000000  0.503000000000  0.158200000000;
     0.012149490000  0.523569300000  0.148138300000;
     0.015535880000  0.544512000000  0.138375800000;
     0.019477520000  0.565690000000  0.128994200000;
     0.023992770000  0.586965300000  0.120075100000;
     0.029100000000  0.608200000000  0.111700000000;
     0.034814850000  0.629345600000  0.103904800000;
     0.041120160000  0.650306800000  0.096667480000;
     0.047985040000  0.670875200000  0.089982720000;
     0.055378610000  0.690842400000  0.083845310000;
     0.063270000000  0.710000000000  0.078249990000;
     0.071635010000  0.728185200000  0.073208990000;
     0.080462240000  0.745463600000  0.068678160000;
     0.089739960000  0.761969400000  0.064567840000;
     0.099456450000  0.777836800000  0.060788350000;
     0.109600000000  0.793200000000  0.057250010000;
     0.120167400000  0.808110400000  0.053904350000;
     0.131114500000  0.822496200000  0.050746640000;
     0.142367900000  0.836306800000  0.047752760000;
     0.153854200000  0.849491600000  0.044898590000;
     0.165500000000  0.862000000000  0.042160000000;
     0.177257100000  0.873810800000  0.039507280000;
     0.189140000000  0.884962400000  0.036935640000;
     0.201169400000  0.895493600000  0.034458360000;
     0.213365800000  0.905443200000  0.032088720000;
     0.225749900000  0.914850100000  0.029840000000;
     0.238320900000  0.923734800000  0.027711810000;
     0.251066800000  0.932092400000  0.025694440000;
     0.263992200000  0.939922600000  0.023787160000;
     0.277101700000  0.947225200000  0.021989250000;
     0.290400000000  0.954000000000  0.020300000000;
     0.303891200000  0.960256100000  0.018718050000;
     0.317572600000  0.966007400000  0.017240360000;
     0.331438400000  0.971260600000  0.015863640000;
     0.345482800000  0.976022500000  0.014584610000;
     0.359700000000  0.980300000000  0.013400000000;
     0.374083900000  0.984092400000  0.012307230000;
     0.388639600000  0.987418200000  0.011301880000;
     0.403378400000  0.990312800000  0.010377920000;
     0.418311500000  0.992811600000  0.009529306000;
     0.433449900000  0.994950100000  0.008749999000;
     0.448795300000  0.996710800000  0.008035200000;
     0.464336000000  0.998098300000  0.007381600000;
     0.480064000000  0.999112000000  0.006785400000;
     0.495971300000  0.999748200000  0.006242800000;
     0.512050100000  1.000000000000  0.005749999000;
     0.528295900000  0.999856700000  0.005303600000;
     0.544691600000  0.999304600000  0.004899800000;
     0.561209400000  0.998325500000  0.004534200000;
     0.577821500000  0.996898700000  0.004202400000;
     0.594500000000  0.995000000000  0.003900000000;
     0.611220900000  0.992600500000  0.003623200000;
     0.627975800000  0.989742600000  0.003370600000;
     0.644760200000  0.986444400000  0.003141400000;
     0.661569700000  0.982724100000  0.002934800000;
     0.678400000000  0.978600000000  0.002749999000;
     0.695239200000  0.974083700000  0.002585200000;
     0.712058600000  0.969171200000  0.002438600000;
     0.728828400000  0.963856800000  0.002309400000;
     0.745518800000  0.958134900000  0.002196800000;
     0.762100000000  0.952000000000  0.002100000000;
     0.778543200000  0.945450400000  0.002017733000;
     0.794825600000  0.938499200000  0.001948200000;
     0.810926400000  0.931162800000  0.001889800000;
     0.826824800000  0.923457600000  0.001840933000;
     0.842500000000  0.915400000000  0.001800000000;
     0.857932500000  0.907006400000  0.001766267000;
     0.873081600000  0.898277200000  0.001737800000;
     0.887894400000  0.889204800000  0.001711200000;
     0.902318100000  0.879781600000  0.001683067000;
     0.916300000000  0.870000000000  0.001650001000;
     0.929799500000  0.859861300000  0.001610133000;
     0.942798400000  0.849392000000  0.001564400000;
     0.955277600000  0.838622000000  0.001513600000;
     0.967217900000  0.827581300000  0.001458533000;
     0.978600000000  0.816300000000  0.001400000000;
     0.989385600000  0.804794700000  0.001336667000;
     0.999548800000  0.793082000000  0.001270000000;
     1.009089200000  0.781192000000  0.001205000000;
     1.018006400000  0.769154700000  0.001146667000;
     1.026300000000  0.757000000000  0.001100000000;
     1.033982700000  0.744754100000  0.001068800000;
     1.040986000000  0.732422400000  0.001049400000;
     1.047188000000  0.720003600000  0.001035600000;
     1.052466700000  0.707496500000  0.001021200000;
     1.056700000000  0.694900000000  0.001000000000;
     1.059794400000  0.682219200000  0.000968640000;
     1.061799200000  0.669471600000  0.000929920000;
     1.062806800000  0.656674400000  0.000886880000;
     1.062909600000  0.643844800000  0.000842560000;
     1.062200000000  0.631000000000  0.000800000000;
     1.060735200000  0.618155500000  0.000760960000;
     1.058443600000  0.605314400000  0.000723680000;
     1.055224400000  0.592475600000  0.000685920000;
     1.050976800000  0.579637900000  0.000645440000;
     1.045600000000  0.566800000000  0.000600000000;
     1.039036900000  0.553961100000  0.000547866700;
     1.031360800000  0.541137200000  0.000491600000;
     1.022666200000  0.528352800000  0.000435400000;
     1.013047700000  0.515632300000  0.000383466700;
     1.002600000000  0.503000000000  0.000340000000;
     0.991367500000  0.490468800000  0.000307253300;
     0.979331400000  0.478030400000  0.000283160000;
     0.966491600000  0.465677600000  0.000265440000;
     0.952847900000  0.453403200000  0.000251813300;
     0.938400000000  0.441200000000  0.000240000000;
     0.923194000000  0.429080000000  0.000229546700;
     0.907244000000  0.417036000000  0.000220640000;
     0.890502000000  0.405032000000  0.000211960000;
     0.872920000000  0.393032000000  0.000202186700;
     0.854449900000  0.381000000000  0.000190000000;
     0.835084000000  0.368918400000  0.000174213300;
     0.814946000000  0.356827200000  0.000155640000;
     0.794186000000  0.344776800000  0.000135960000;
     0.772954000000  0.332817600000  0.000116853300;
     0.751400000000  0.321000000000  0.000100000000;
     0.729583600000  0.309338100000  0.000086133330;
     0.707588800000  0.297850400000  0.000074600000;
     0.685602200000  0.286593600000  0.000065000000;
     0.663810400000  0.275624500000  0.000056933330;
     0.642400000000  0.265000000000  0.000049999990;
     0.621514900000  0.254763200000  0.000044160000;
     0.601113800000  0.244889600000  0.000039480000;
     0.581105200000  0.235334400000  0.000035720000;
     0.561397700000  0.226052800000  0.000032640000;
     0.541900000000  0.217000000000  0.000030000000;
     0.522599500000  0.208161600000  0.000027653330;
     0.503546400000  0.199548800000  0.000025560000;
     0.484743600000  0.191155200000  0.000023640000;
     0.466193900000  0.182974400000  0.000021813330;
     0.447900000000  0.175000000000  0.000020000000;
     0.429861300000  0.167223500000  0.000018133330;
     0.412098000000  0.159646400000  0.000016200000;
     0.394644000000  0.152277600000  0.000014200000;
     0.377533300000  0.145125900000  0.000012133330;
     0.360800000000  0.138200000000  0.000010000000;
     0.344456300000  0.131500300000  0.000007733333;
     0.328516800000  0.125024800000  0.000005400000;
     0.313019200000  0.118779200000  0.000003200000;
     0.298001100000  0.112769100000  0.000001333333;
     0.283500000000  0.107000000000  0.000000000000;
     0.269544800000  0.101476200000  0.000000000000;
     0.256118400000  0.096188640000  0.000000000000;
     0.243189600000  0.091122960000  0.000000000000;
     0.230727200000  0.086264850000  0.000000000000;
     0.218700000000  0.081600000000  0.000000000000;
     0.207097100000  0.077120640000  0.000000000000;
     0.195923200000  0.072825520000  0.000000000000;
     0.185170800000  0.068710080000  0.000000000000;
     0.174832300000  0.064769760000  0.000000000000;
     0.164900000000  0.061000000000  0.000000000000;
     0.155366700000  0.057396210000  0.000000000000;
     0.146230000000  0.053955040000  0.000000000000;
     0.137490000000  0.050673760000  0.000000000000;
     0.129146700000  0.047549650000  0.000000000000;
     0.121200000000  0.044580000000  0.000000000000;
     0.113639700000  0.041758720000  0.000000000000;
     0.106465000000  0.039084960000  0.000000000000;
     0.099690440000  0.036563840000  0.000000000000;
     0.093330610000  0.034200480000  0.000000000000;
     0.087400000000  0.032000000000  0.000000000000;
     0.081900960000  0.029962610000  0.000000000000;
     0.076804280000  0.028076640000  0.000000000000;
     0.072077120000  0.026329360000  0.000000000000;
     0.067686640000  0.024708050000  0.000000000000;
     0.063600000000  0.023200000000  0.000000000000;
     0.059806850000  0.021800770000  0.000000000000;
     0.056282160000  0.020501120000  0.000000000000;
     0.052971040000  0.019281080000  0.000000000000;
     0.049818610000  0.018120690000  0.000000000000;
     0.046770000000  0.017000000000  0.000000000000;
     0.043784050000  0.015903790000  0.000000000000;
     0.040875360000  0.014837180000  0.000000000000;
     0.038072640000  0.013810680000  0.000000000000;
     0.035404610000  0.012834780000  0.000000000000;
     0.032900000000  0.011920000000  0.000000000000;
     0.030564190000  0.011068310000  0.000000000000;
     0.028380560000  0.010273390000  0.000000000000;
     0.026344840000  0.009533311000  0.000000000000;
     0.024452750000  0.008846157000  0.000000000000;
     0.022700000000  0.008210000000  0.000000000000;
     0.021084290000  0.007623781000  0.000000000000;
     0.019599880000  0.007085424000  0.000000000000;
     0.018237320000  0.006591476000  0.000000000000;
     0.016987170000  0.006138485000  0.000000000000;
     0.015840000000  0.005723000000  0.000000000000;
     0.014790640000  0.005343059000  0.000000000000;
     0.013831320000  0.004995796000  0.000000000000;
     0.012948680000  0.004676404000  0.000000000000;
     0.012129200000  0.004380075000  0.000000000000;
     0.011359160000  0.004102000000  0.000000000000;
     0.010629350000  0.003838453000  0.000000000000;
     0.009938846000  0.003589099000  0.000000000000;
     0.009288422000  0.003354219000  0.000000000000;
     0.008678854000  0.003134093000  0.000000000000;
     0.008110916000  0.002929000000  0.000000000000;
     0.007582388000  0.002738139000  0.000000000000;
     0.007088746000  0.002559876000  0.000000000000;
     0.006627313000  0.002393244000  0.000000000000;
     0.006195408000  0.002237275000  0.000000000000;
     0.005790346000  0.002091000000  0.000000000000;
     0.005409826000  0.001953587000  0.000000000000;
     0.005052583000  0.001824580000  0.000000000000;
     0.004717512000  0.001703580000  0.000000000000;
     0.004403507000  0.001590187000  0.000000000000;
     0.004109457000  0.001484000000  0.000000000000;
     0.003833913000  0.001384496000  0.000000000000;
     0.003575748000  0.001291268000  0.000000000000;
     0.003334342000  0.001204092000  0.000000000000;
     0.003109075000  0.001122744000  0.000000000000;
     0.002899327000  0.001047000000  0.000000000000;
     0.002704348000  0.000976589600  0.000000000000;
     0.002523020000  0.000911108800  0.000000000000;
     0.002354168000  0.000850133200  0.000000000000;
     0.002196616000  0.000793238400  0.000000000000;
     0.002049190000  0.000740000000  0.000000000000;
     0.001910960000  0.000690082700  0.000000000000;
     0.001781438000  0.000643310000  0.000000000000;
     0.001660110000  0.000599496000  0.000000000000;
     0.001546459000  0.000558454700  0.000000000000;
     0.001439971000  0.000520000000  0.000000000000;
     0.001340042000  0.000483913600  0.000000000000;
     0.001246275000  0.000450052800  0.000000000000;
     0.001158471000  0.000418345200  0.000000000000;
     0.001076430000  0.000388718400  0.000000000000;
     0.000999949300  0.000361100000  0.000000000000;
     0.000928735800  0.000335383500  0.000000000000;
     0.000862433200  0.000311440400  0.000000000000;
     0.000800750300  0.000289165600  0.000000000000;
     0.000743396000  0.000268453900  0.000000000000;
     0.000690078600  0.000249200000  0.000000000000;
     0.000640515600  0.000231301900  0.000000000000;
     0.000594502100  0.000214685600  0.000000000000;
     0.000551864600  0.000199288400  0.000000000000;
     0.000512429000  0.000185047500  0.000000000000;
     0.000476021300  0.000171900000  0.000000000000;
     0.000442453600  0.000159778100  0.000000000000;
     0.000411511700  0.000148604400  0.000000000000;
     0.000382981400  0.000138301600  0.000000000000;
     0.000356649100  0.000128792500  0.000000000000;
     0.000332301100  0.000120000000  0.000000000000;
     0.000309758600  0.000111859500  0.000000000000;
     0.000288887100  0.000104322400  0.000000000000;
     0.000269539400  0.000097335600  0.000000000000;
     0.000251568200  0.000090845870  0.000000000000;
     0.000234826100  0.000084800000  0.000000000000;
     0.000219171000  0.000079146670  0.000000000000;
     0.000204525800  0.000073858000  0.000000000000;
     0.000190840500  0.000068916000  0.000000000000;
     0.000178065400  0.000064302670  0.000000000000;
     0.000166150500  0.000060000000  0.000000000000;
     0.000155023600  0.000055981870  0.000000000000;
     0.000144621900  0.000052225600  0.000000000000;
     0.000134909800  0.000048718400  0.000000000000;
     0.000125852000  0.000045447470  0.000000000000;
     0.000117413000  0.000042400000  0.000000000000;
     0.000109551500  0.000039561040  0.000000000000;
     0.000102224500  0.000036915120  0.000000000000;
     0.000095394450  0.000034448680  0.000000000000;
     0.000089023900  0.000032148160  0.000000000000;
     0.000083075270  0.000030000000  0.000000000000;
     0.000077512690  0.000027991250  0.000000000000;
     0.000072313040  0.000026113560  0.000000000000;
     0.000067457780  0.000024360240  0.000000000000;
     0.000062928440  0.000022724610  0.000000000000;
     0.000058706520  0.000021200000  0.000000000000;
     0.000054770280  0.000019778550  0.000000000000;
     0.000051099180  0.000018452850  0.000000000000;
     0.000047676540  0.000017216870  0.000000000000;
     0.000044485670  0.000016064590  0.000000000000;
     0.000041509940  0.000014990000  0.000000000000;
     0.000038733240  0.000013987280  0.000000000000;
     0.000036142030  0.000013051550  0.000000000000;
     0.000033723520  0.000012178180  0.000000000000;
     0.000031464870  0.000011362540  0.000000000000;
     0.000029353260  0.000010600000  0.000000000000;
     0.000027375730  0.000009885877  0.000000000000;
     0.000025524330  0.000009217304  0.000000000000;
     0.000023793760  0.000008592362  0.000000000000;
     0.000022178700  0.000008009133  0.000000000000;
     0.000020673830  0.000007465700  0.000000000000;
     0.000019272260  0.000006959567  0.000000000000;
     0.000017966400  0.000006487995  0.000000000000;
     0.000016749910  0.000006048699  0.000000000000;
     0.000015616480  0.000005639396  0.000000000000;
     0.000014559770  0.000005257800  0.000000000000;
     0.000013573870  0.000004901771  0.000000000000;
     0.000012654360  0.000004569720  0.000000000000;
     0.000011797230  0.000004260194  0.000000000000;
     0.000010998440  0.000003971739  0.000000000000;
     0.000010253980  0.000003702900  0.000000000000;
     0.000009559646  0.000003452163  0.000000000000;
     0.000008912044  0.000003218302  0.000000000000;
     0.000008308358  0.000003000300  0.000000000000;
     0.000007745769  0.000002797139  0.000000000000;
     0.000007221456  0.000002607800  0.000000000000;
     0.000006732475  0.000002431220  0.000000000000;
     0.000006276423  0.000002266531  0.000000000000;
     0.000005851304  0.000002113013  0.000000000000;
     0.000005455118  0.000001969943  0.000000000000;
     0.000005085868  0.000001836600  0.000000000000;
     0.000004741466  0.000001712230  0.000000000000;
     0.000004420236  0.000001596228  0.000000000000;
     0.000004120783  0.000001488090  0.000000000000;
     0.000003841716  0.000001387314  0.000000000000;
     0.000003581652  0.000001293400  0.000000000000;
     0.000003339127  0.000001205820  0.000000000000;
     0.000003112949  0.000001124143  0.000000000000;
     0.000002902121  0.000001048009  0.000000000000;
     0.000002705645  0.000000977058  0.000000000000;
     0.000002522525  0.000000910930  0.000000000000;
     0.000002351726  0.000000849251  0.000000000000;
     0.000002192415  0.000000791721  0.000000000000;
     0.000002043902  0.000000738090  0.000000000000;
     0.000001905497  0.000000688110  0.000000000000;
     0.000001776509  0.000000641530  0.000000000000;
     0.000001656215  0.000000598090  0.000000000000;
     0.000001544022  0.000000557575  0.000000000000;
     0.000001439440  0.000000519808  0.000000000000;
     0.000001341977  0.000000484612  0.000000000000;
     0.000001251141  0.000000451810  0.000000000000];


function colormatch(::Type{CIE1964_CMF}, wavelen::Real)
    return interpolate_table(cie1964_cmf_table, 360.0, 1.0, wavelen)
end

# CIE 1964 10° color matching function, 1nm increments starting at 360nm
const cie1964_cmf_table =
    [0.000000122200  0.000000013398  0.000000535027;
     0.000000185138  0.000000020294  0.000000810720;
     0.000000278830  0.000000030560  0.000001221200;
     0.000000417470  0.000000045740  0.000001828700;
     0.000000621330  0.000000068050  0.000002722200;
     0.000000919270  0.000000100650  0.000004028300;
     0.000001351980  0.000000147980  0.000005925700;
     0.000001976540  0.000000216270  0.000008665100;
     0.000002872500  0.000000314200  0.000012596000;
     0.000004149500  0.000000453700  0.000018201000;
     0.000005958600  0.000000651100  0.000026143700;
     0.000008505600  0.000000928800  0.000037330000;
     0.000012068600  0.000001317500  0.000052987000;
     0.000017022600  0.000001857200  0.000074764000;
     0.000023868000  0.000002602000  0.000104870000;
     0.000033266000  0.000003625000  0.000146220000;
     0.000046087000  0.000005019000  0.000202660000;
     0.000063472000  0.000006907000  0.000279230000;
     0.000086892000  0.000009449000  0.000382450000;
     0.000118246000  0.000012848000  0.000520720000;
     0.000159952000  0.000017364000  0.000704776000;
     0.000215080000  0.000023327000  0.000948230000;
     0.000287490000  0.000031150000  0.001268200000;
     0.000381990000  0.000041350000  0.001686100000;
     0.000504550000  0.000054560000  0.002228500000;
     0.000662440000  0.000071560000  0.002927800000;
     0.000864500000  0.000093300000  0.003823700000;
     0.001121500000  0.000120870000  0.004964200000;
     0.001446160000  0.000155640000  0.006406700000;
     0.001853590000  0.000199200000  0.008219300000;
     0.002361600000  0.000253400000  0.010482200000;
     0.002990600000  0.000320200000  0.013289000000;
     0.003764500000  0.000402400000  0.016747000000;
     0.004710200000  0.000502300000  0.020980000000;
     0.005858100000  0.000623200000  0.026127000000;
     0.007242300000  0.000768500000  0.032344000000;
     0.008899600000  0.000941700000  0.039802000000;
     0.010870900000  0.001147800000  0.048691000000;
     0.013198900000  0.001390300000  0.059210000000;
     0.015929200000  0.001674000000  0.071576000000;
     0.019109700000  0.002004400000  0.086010900000;
     0.022788000000  0.002386000000  0.102740000000;
     0.027011000000  0.002822000000  0.122000000000;
     0.031829000000  0.003319000000  0.144020000000;
     0.037278000000  0.003880000000  0.168990000000;
     0.043400000000  0.004509000000  0.197120000000;
     0.050223000000  0.005209000000  0.228570000000;
     0.057764000000  0.005985000000  0.263470000000;
     0.066038000000  0.006833000000  0.301900000000;
     0.075033000000  0.007757000000  0.343870000000;
     0.084736000000  0.008756000000  0.389366000000;
     0.095041000000  0.009816000000  0.437970000000;
     0.105836000000  0.010918000000  0.489220000000;
     0.117066000000  0.012058000000  0.542900000000;
     0.128682000000  0.013237000000  0.598810000000;
     0.140638000000  0.014456000000  0.656760000000;
     0.152893000000  0.015717000000  0.716580000000;
     0.165416000000  0.017025000000  0.778120000000;
     0.178191000000  0.018399000000  0.841310000000;
     0.191214000000  0.019848000000  0.906110000000;
     0.204492000000  0.021391000000  0.972542000000;
     0.217650000000  0.022992000000  1.038900000000;
     0.230267000000  0.024598000000  1.103100000000;
     0.242311000000  0.026213000000  1.165100000000;
     0.253793000000  0.027841000000  1.224900000000;
     0.264737000000  0.029497000000  1.282500000000;
     0.275195000000  0.031195000000  1.338200000000;
     0.285301000000  0.032927000000  1.392600000000;
     0.295143000000  0.034738000000  1.446100000000;
     0.304869000000  0.036654000000  1.499400000000;
     0.314679000000  0.038676000000  1.553480000000;
     0.324355000000  0.040792000000  1.607200000000;
     0.333570000000  0.042946000000  1.658900000000;
     0.342243000000  0.045114000000  1.708200000000;
     0.350312000000  0.047333000000  1.754800000000;
     0.357719000000  0.049602000000  1.798500000000;
     0.364482000000  0.051934000000  1.839200000000;
     0.370493000000  0.054337000000  1.876600000000;
     0.375727000000  0.056822000000  1.910500000000;
     0.380158000000  0.059399000000  1.940800000000;
     0.383734000000  0.062077000000  1.967280000000;
     0.386327000000  0.064737000000  1.989100000000;
     0.387858000000  0.067285000000  2.005700000000;
     0.388396000000  0.069764000000  2.017400000000;
     0.387978000000  0.072218000000  2.024400000000;
     0.386726000000  0.074704000000  2.027300000000;
     0.384696000000  0.077272000000  2.026400000000;
     0.382006000000  0.079979000000  2.022300000000;
     0.378709000000  0.082874000000  2.015300000000;
     0.374915000000  0.086000000000  2.006000000000;
     0.370702000000  0.089456000000  1.994800000000;
     0.366089000000  0.092947000000  1.981400000000;
     0.361045000000  0.096275000000  1.965300000000;
     0.355518000000  0.099535000000  1.946400000000;
     0.349486000000  0.102829000000  1.924800000000;
     0.342957000000  0.106256000000  1.900700000000;
     0.335893000000  0.109901000000  1.874100000000;
     0.328284000000  0.113835000000  1.845100000000;
     0.320150000000  0.118167000000  1.813900000000;
     0.311475000000  0.122932000000  1.780600000000;
     0.302273000000  0.128201000000  1.745370000000;
     0.292858000000  0.133457000000  1.709100000000;
     0.283502000000  0.138323000000  1.672300000000;
     0.274044000000  0.143042000000  1.634700000000;
     0.264263000000  0.147787000000  1.595600000000;
     0.254085000000  0.152761000000  1.554900000000;
     0.243392000000  0.158102000000  1.512200000000;
     0.232187000000  0.163941000000  1.467300000000;
     0.220488000000  0.170362000000  1.419900000000;
     0.208198000000  0.177425000000  1.370000000000;
     0.195618000000  0.185190000000  1.317560000000;
     0.183034000000  0.193025000000  1.262400000000;
     0.170222000000  0.200313000000  1.205000000000;
     0.157348000000  0.207156000000  1.146600000000;
     0.144650000000  0.213644000000  1.088000000000;
     0.132349000000  0.219940000000  1.030200000000;
     0.120584000000  0.226170000000  0.973830000000;
     0.109456000000  0.232467000000  0.919430000000;
     0.099042000000  0.239025000000  0.867460000000;
     0.089388000000  0.245997000000  0.818280000000;
     0.080507000000  0.253589000000  0.772125000000;
     0.072034000000  0.261876000000  0.728290000000;
     0.063710000000  0.270643000000  0.686040000000;
     0.055694000000  0.279645000000  0.645530000000;
     0.048117000000  0.288694000000  0.606850000000;
     0.041072000000  0.297665000000  0.570060000000;
     0.034642000000  0.306469000000  0.535220000000;
     0.028896000000  0.315035000000  0.502340000000;
     0.023876000000  0.323335000000  0.471400000000;
     0.019628000000  0.331366000000  0.442390000000;
     0.016172000000  0.339133000000  0.415254000000;
     0.013300000000  0.347860000000  0.390024000000;
     0.010759000000  0.358326000000  0.366399000000;
     0.008542000000  0.370001000000  0.344015000000;
     0.006661000000  0.382464000000  0.322689000000;
     0.005132000000  0.395379000000  0.302356000000;
     0.003982000000  0.408482000000  0.283036000000;
     0.003239000000  0.421588000000  0.264816000000;
     0.002934000000  0.434619000000  0.247848000000;
     0.003114000000  0.447601000000  0.232318000000;
     0.003816000000  0.460777000000  0.218502000000;
     0.005095000000  0.474340000000  0.205851000000;
     0.006936000000  0.488200000000  0.193596000000;
     0.009299000000  0.502340000000  0.181736000000;
     0.012147000000  0.516740000000  0.170281000000;
     0.015444000000  0.531360000000  0.159249000000;
     0.019156000000  0.546190000000  0.148673000000;
     0.023250000000  0.561180000000  0.138609000000;
     0.027690000000  0.576290000000  0.129096000000;
     0.032444000000  0.591500000000  0.120215000000;
     0.037465000000  0.606741000000  0.112044000000;
     0.042956000000  0.622150000000  0.104710000000;
     0.049114000000  0.637830000000  0.098196000000;
     0.055920000000  0.653710000000  0.092361000000;
     0.063349000000  0.669680000000  0.087088000000;
     0.071358000000  0.685660000000  0.082248000000;
     0.079901000000  0.701550000000  0.077744000000;
     0.088909000000  0.717230000000  0.073456000000;
     0.098293000000  0.732570000000  0.069268000000;
     0.107949000000  0.747460000000  0.065060000000;
     0.117749000000  0.761757000000  0.060709000000;
     0.127839000000  0.775340000000  0.056457000000;
     0.138450000000  0.788220000000  0.052609000000;
     0.149516000000  0.800460000000  0.049122000000;
     0.161041000000  0.812140000000  0.045954000000;
     0.172953000000  0.823330000000  0.043050000000;
     0.185209000000  0.834120000000  0.040368000000;
     0.197755000000  0.844600000000  0.037839000000;
     0.210538000000  0.854870000000  0.035384000000;
     0.223460000000  0.865040000000  0.032949000000;
     0.236491000000  0.875211000000  0.030451000000;
     0.249633000000  0.885370000000  0.028029000000;
     0.262972000000  0.895370000000  0.025862000000;
     0.276515000000  0.905150000000  0.023920000000;
     0.290269000000  0.914650000000  0.022174000000;
     0.304213000000  0.923810000000  0.020584000000;
     0.318361000000  0.932550000000  0.019127000000;
     0.332705000000  0.940810000000  0.017740000000;
     0.347232000000  0.948520000000  0.016403000000;
     0.361926000000  0.955600000000  0.015064000000;
     0.376772000000  0.961988000000  0.013676000000;
     0.391683000000  0.967540000000  0.012308000000;
     0.406594000000  0.972230000000  0.011056000000;
     0.421539000000  0.976170000000  0.009915000000;
     0.436517000000  0.979460000000  0.008872000000;
     0.451584000000  0.982200000000  0.007918000000;
     0.466782000000  0.984520000000  0.007030000000;
     0.482147000000  0.986520000000  0.006223000000;
     0.497738000000  0.988320000000  0.005453000000;
     0.513606000000  0.990020000000  0.004714000000;
     0.529826000000  0.991761000000  0.003988000000;
     0.546440000000  0.993530000000  0.003289000000;
     0.563426000000  0.995230000000  0.002646000000;
     0.580726000000  0.996770000000  0.002063000000;
     0.598290000000  0.998090000000  0.001533000000;
     0.616053000000  0.999110000000  0.001091000000;
     0.633948000000  0.999770000000  0.000711000000;
     0.651901000000  1.000000000000  0.000407000000;
     0.669824000000  0.999710000000  0.000184000000;
     0.687632000000  0.998850000000  0.000047000000;
     0.705224000000  0.997340000000  0.000000000000;
     0.722773000000  0.995260000000  0.000000000000;
     0.740483000000  0.992740000000  0.000000000000;
     0.758273000000  0.989750000000  0.000000000000;
     0.776083000000  0.986300000000  0.000000000000;
     0.793832000000  0.982380000000  0.000000000000;
     0.811436000000  0.977980000000  0.000000000000;
     0.828822000000  0.973110000000  0.000000000000;
     0.845879000000  0.967740000000  0.000000000000;
     0.862525000000  0.961890000000  0.000000000000;
     0.878655000000  0.955552000000  0.000000000000;
     0.894208000000  0.948601000000  0.000000000000;
     0.909206000000  0.940981000000  0.000000000000;
     0.923672000000  0.932798000000  0.000000000000;
     0.937638000000  0.924158000000  0.000000000000;
     0.951162000000  0.915175000000  0.000000000000;
     0.964283000000  0.905954000000  0.000000000000;
     0.977068000000  0.896608000000  0.000000000000;
     0.989590000000  0.887249000000  0.000000000000;
     1.001910000000  0.877986000000  0.000000000000;
     1.014160000000  0.868934000000  0.000000000000;
     1.026500000000  0.860164000000  0.000000000000;
     1.038800000000  0.851519000000  0.000000000000;
     1.051000000000  0.842963000000  0.000000000000;
     1.062900000000  0.834393000000  0.000000000000;
     1.074300000000  0.825623000000  0.000000000000;
     1.085200000000  0.816764000000  0.000000000000;
     1.095200000000  0.807544000000  0.000000000000;
     1.104200000000  0.797947000000  0.000000000000;
     1.112000000000  0.787893000000  0.000000000000;
     1.118520000000  0.777405000000  0.000000000000;
     1.123800000000  0.766490000000  0.000000000000;
     1.128000000000  0.755309000000  0.000000000000;
     1.131100000000  0.743845000000  0.000000000000;
     1.133200000000  0.732190000000  0.000000000000;
     1.134300000000  0.720353000000  0.000000000000;
     1.134300000000  0.708281000000  0.000000000000;
     1.133300000000  0.696055000000  0.000000000000;
     1.131200000000  0.683621000000  0.000000000000;
     1.128100000000  0.671048000000  0.000000000000;
     1.123990000000  0.658341000000  0.000000000000;
     1.118900000000  0.645545000000  0.000000000000;
     1.112900000000  0.632718000000  0.000000000000;
     1.105900000000  0.619815000000  0.000000000000;
     1.098000000000  0.606887000000  0.000000000000;
     1.089100000000  0.593878000000  0.000000000000;
     1.079200000000  0.580781000000  0.000000000000;
     1.068400000000  0.567653000000  0.000000000000;
     1.056700000000  0.554490000000  0.000000000000;
     1.044000000000  0.541228000000  0.000000000000;
     1.030480000000  0.527963000000  0.000000000000;
     1.016000000000  0.514634000000  0.000000000000;
     1.000800000000  0.501363000000  0.000000000000;
     0.984790000000  0.488124000000  0.000000000000;
     0.968080000000  0.474935000000  0.000000000000;
     0.950740000000  0.461834000000  0.000000000000;
     0.932800000000  0.448823000000  0.000000000000;
     0.914340000000  0.435917000000  0.000000000000;
     0.895390000000  0.423153000000  0.000000000000;
     0.876030000000  0.410526000000  0.000000000000;
     0.856297000000  0.398057000000  0.000000000000;
     0.836350000000  0.385835000000  0.000000000000;
     0.816290000000  0.373951000000  0.000000000000;
     0.796050000000  0.362311000000  0.000000000000;
     0.775610000000  0.350863000000  0.000000000000;
     0.754930000000  0.339554000000  0.000000000000;
     0.733990000000  0.328309000000  0.000000000000;
     0.712780000000  0.317118000000  0.000000000000;
     0.691290000000  0.305936000000  0.000000000000;
     0.669520000000  0.294737000000  0.000000000000;
     0.647467000000  0.283493000000  0.000000000000;
     0.625110000000  0.272222000000  0.000000000000;
     0.602520000000  0.260990000000  0.000000000000;
     0.579890000000  0.249877000000  0.000000000000;
     0.557370000000  0.238946000000  0.000000000000;
     0.535110000000  0.228254000000  0.000000000000;
     0.513240000000  0.217853000000  0.000000000000;
     0.491860000000  0.207780000000  0.000000000000;
     0.471080000000  0.198072000000  0.000000000000;
     0.450960000000  0.188748000000  0.000000000000;
     0.431567000000  0.179828000000  0.000000000000;
     0.412870000000  0.171285000000  0.000000000000;
     0.394750000000  0.163059000000  0.000000000000;
     0.377210000000  0.155151000000  0.000000000000;
     0.360190000000  0.147535000000  0.000000000000;
     0.343690000000  0.140211000000  0.000000000000;
     0.327690000000  0.133170000000  0.000000000000;
     0.312170000000  0.126400000000  0.000000000000;
     0.297110000000  0.119892000000  0.000000000000;
     0.282500000000  0.113640000000  0.000000000000;
     0.268329000000  0.107633000000  0.000000000000;
     0.254590000000  0.101870000000  0.000000000000;
     0.241300000000  0.096347000000  0.000000000000;
     0.228480000000  0.091063000000  0.000000000000;
     0.216140000000  0.086010000000  0.000000000000;
     0.204300000000  0.081187000000  0.000000000000;
     0.192950000000  0.076583000000  0.000000000000;
     0.182110000000  0.072198000000  0.000000000000;
     0.171770000000  0.068024000000  0.000000000000;
     0.161920000000  0.064052000000  0.000000000000;
     0.152568000000  0.060281000000  0.000000000000;
     0.143670000000  0.056697000000  0.000000000000;
     0.135200000000  0.053292000000  0.000000000000;
     0.127130000000  0.050059000000  0.000000000000;
     0.119480000000  0.046998000000  0.000000000000;
     0.112210000000  0.044096000000  0.000000000000;
     0.105310000000  0.041345000000  0.000000000000;
     0.098786000000  0.038750700000  0.000000000000;
     0.092610000000  0.036297800000  0.000000000000;
     0.086773000000  0.033983200000  0.000000000000;
     0.081260600000  0.031800400000  0.000000000000;
     0.076048000000  0.029739500000  0.000000000000;
     0.071114000000  0.027791800000  0.000000000000;
     0.066454000000  0.025955100000  0.000000000000;
     0.062062000000  0.024226300000  0.000000000000;
     0.057930000000  0.022601700000  0.000000000000;
     0.054050000000  0.021077900000  0.000000000000;
     0.050412000000  0.019650500000  0.000000000000;
     0.047006000000  0.018315300000  0.000000000000;
     0.043823000000  0.017068600000  0.000000000000;
     0.040850800000  0.015905100000  0.000000000000;
     0.038072000000  0.014818300000  0.000000000000;
     0.035468000000  0.013800800000  0.000000000000;
     0.033031000000  0.012849500000  0.000000000000;
     0.030753000000  0.011960700000  0.000000000000;
     0.028623000000  0.011130300000  0.000000000000;
     0.026635000000  0.010355500000  0.000000000000;
     0.024781000000  0.009633200000  0.000000000000;
     0.023052000000  0.008959900000  0.000000000000;
     0.021441000000  0.008332400000  0.000000000000;
     0.019941300000  0.007748800000  0.000000000000;
     0.018544000000  0.007204600000  0.000000000000;
     0.017241000000  0.006697500000  0.000000000000;
     0.016027000000  0.006225100000  0.000000000000;
     0.014896000000  0.005785000000  0.000000000000;
     0.013842000000  0.005375100000  0.000000000000;
     0.012862000000  0.004994100000  0.000000000000;
     0.011949000000  0.004639200000  0.000000000000;
     0.011100000000  0.004309300000  0.000000000000;
     0.010311000000  0.004002800000  0.000000000000;
     0.009576880000  0.003717740000  0.000000000000;
     0.008894000000  0.003452620000  0.000000000000;
     0.008258100000  0.003205830000  0.000000000000;
     0.007666400000  0.002976230000  0.000000000000;
     0.007116300000  0.002762810000  0.000000000000;
     0.006605200000  0.002564560000  0.000000000000;
     0.006130600000  0.002380480000  0.000000000000;
     0.005690300000  0.002209710000  0.000000000000;
     0.005281900000  0.002051320000  0.000000000000;
     0.004903300000  0.001904490000  0.000000000000;
     0.004552630000  0.001768470000  0.000000000000;
     0.004227500000  0.001642360000  0.000000000000;
     0.003925800000  0.001525350000  0.000000000000;
     0.003645700000  0.001416720000  0.000000000000;
     0.003385900000  0.001315950000  0.000000000000;
     0.003144700000  0.001222390000  0.000000000000;
     0.002920800000  0.001135550000  0.000000000000;
     0.002713000000  0.001054940000  0.000000000000;
     0.002520200000  0.000980140000  0.000000000000;
     0.002341100000  0.000910660000  0.000000000000;
     0.002174960000  0.000846190000  0.000000000000;
     0.002020600000  0.000786290000  0.000000000000;
     0.001877300000  0.000730680000  0.000000000000;
     0.001744100000  0.000678990000  0.000000000000;
     0.001620500000  0.000631010000  0.000000000000;
     0.001505700000  0.000586440000  0.000000000000;
     0.001399200000  0.000545110000  0.000000000000;
     0.001300400000  0.000506720000  0.000000000000;
     0.001208700000  0.000471110000  0.000000000000;
     0.001123600000  0.000438050000  0.000000000000;
     0.001044760000  0.000407410000  0.000000000000;
     0.000971560000  0.000378962000  0.000000000000;
     0.000903600000  0.000352543000  0.000000000000;
     0.000840480000  0.000328001000  0.000000000000;
     0.000781870000  0.000305208000  0.000000000000;
     0.000727450000  0.000284041000  0.000000000000;
     0.000676900000  0.000264375000  0.000000000000;
     0.000629960000  0.000246109000  0.000000000000;
     0.000586370000  0.000229143000  0.000000000000;
     0.000545870000  0.000213376000  0.000000000000;
     0.000508258000  0.000198730000  0.000000000000;
     0.000473300000  0.000185115000  0.000000000000;
     0.000440800000  0.000172454000  0.000000000000;
     0.000410580000  0.000160678000  0.000000000000;
     0.000382490000  0.000149730000  0.000000000000;
     0.000356380000  0.000139550000  0.000000000000;
     0.000332110000  0.000130086000  0.000000000000;
     0.000309550000  0.000121290000  0.000000000000;
     0.000288580000  0.000113106000  0.000000000000;
     0.000269090000  0.000105501000  0.000000000000;
     0.000250969000  0.000098428000  0.000000000000;
     0.000234130000  0.000091853000  0.000000000000;
     0.000218470000  0.000085738000  0.000000000000;
     0.000203910000  0.000080048000  0.000000000000;
     0.000190350000  0.000074751000  0.000000000000;
     0.000177730000  0.000069819000  0.000000000000;
     0.000165970000  0.000065222000  0.000000000000;
     0.000155020000  0.000060939000  0.000000000000;
     0.000144800000  0.000056942000  0.000000000000;
     0.000135280000  0.000053217000  0.000000000000;
     0.000126390000  0.000049737000  0.000000000000;
     0.000118100000  0.000046491000  0.000000000000;
     0.000110370000  0.000043464000  0.000000000000;
     0.000103150000  0.000040635000  0.000000000000;
     0.000096427000  0.000038000000  0.000000000000;
     0.000090151000  0.000035540500  0.000000000000;
     0.000084294000  0.000033244800  0.000000000000;
     0.000078830000  0.000031100600  0.000000000000;
     0.000073729000  0.000029099000  0.000000000000;
     0.000068969000  0.000027230700  0.000000000000;
     0.000064525800  0.000025486000  0.000000000000;
     0.000060376000  0.000023856100  0.000000000000;
     0.000056500000  0.000022333200  0.000000000000;
     0.000052880000  0.000020910400  0.000000000000;
     0.000049498000  0.000019580800  0.000000000000;
     0.000046339000  0.000018338400  0.000000000000;
     0.000043389000  0.000017177700  0.000000000000;
     0.000040634000  0.000016093400  0.000000000000;
     0.000038060000  0.000015080000  0.000000000000;
     0.000035657000  0.000014133600  0.000000000000;
     0.000033411700  0.000013249000  0.000000000000;
     0.000031315000  0.000012422600  0.000000000000;
     0.000029355000  0.000011649900  0.000000000000;
     0.000027524000  0.000010927700  0.000000000000;
     0.000025811000  0.000010251900  0.000000000000;
     0.000024209000  0.000009619600  0.000000000000;
     0.000022711000  0.000009028100  0.000000000000;
     0.000021308000  0.000008474000  0.000000000000;
     0.000019994000  0.000007954800  0.000000000000;
     0.000018764000  0.000007468600  0.000000000000;
     0.000017611500  0.000007012800  0.000000000000;
     0.000016532000  0.000006585800  0.000000000000;
     0.000015521000  0.000006185700  0.000000000000;
     0.000014574000  0.000005810700  0.000000000000;
     0.000013686000  0.000005459000  0.000000000000;
     0.000012855000  0.000005129800  0.000000000000;
     0.000012075000  0.000004820600  0.000000000000;
     0.000011345000  0.000004531200  0.000000000000;
     0.000010659000  0.000004259100  0.000000000000;
     0.000010017000  0.000004004200  0.000000000000;
     0.000009413630  0.000003764730  0.000000000000;
     0.000008847900  0.000003539950  0.000000000000;
     0.000008317100  0.000003329140  0.000000000000;
     0.000007819000  0.000003131150  0.000000000000;
     0.000007351600  0.000002945290  0.000000000000;
     0.000006913000  0.000002770810  0.000000000000;
     0.000006501500  0.000002607050  0.000000000000;
     0.000006115300  0.000002453290  0.000000000000;
     0.000005752900  0.000002308940  0.000000000000;
     0.000005412700  0.000002173380  0.000000000000;
     0.000005093470  0.000002046130  0.000000000000;
     0.000004793800  0.000001926620  0.000000000000;
     0.000004512500  0.000001814400  0.000000000000;
     0.000004248300  0.000001708950  0.000000000000;
     0.000004000200  0.000001609880  0.000000000000;
     0.000003767100  0.000001516770  0.000000000000;
     0.000003548000  0.000001429210  0.000000000000;
     0.000003342100  0.000001346860  0.000000000000;
     0.000003148500  0.000001269450  0.000000000000;
     0.000002966500  0.000001196620  0.000000000000;
     0.000002795310  0.000001128090  0.000000000000;
     0.000002634500  0.000001063680  0.000000000000;
     0.000002483400  0.000001003130  0.000000000000;
     0.000002341400  0.000000946220  0.000000000000;
     0.000002207800  0.000000892630  0.000000000000;
     0.000002082000  0.000000842160  0.000000000000;
     0.000001963600  0.000000794640  0.000000000000;
     0.000001851900  0.000000749780  0.000000000000;
     0.000001746500  0.000000707440  0.000000000000;
     0.000001647100  0.000000667480  0.000000000000;
     0.000001553140  0.000000629700  0.000000000000]


function colormatch(::Type{CIE1931J_CMF}, wavelen::Real)
    return interpolate_table(cie1931j_cmf_table, 370.0, 10.0, wavelen)
end

# Judd adjustment to the CIE 1931 2° CMF, 10nm increments starting at 370nm
const cie1931j_cmf_table =
    [0.0008  0.0001  0.0046;
     0.0045  0.0004  0.0224;
     0.0201  0.0015  0.0925;
     0.0611  0.0045  0.2799;
     0.1267  0.0093  0.5835;
     0.2285  0.0175  1.0622;
     0.3081  0.0273  1.4526;
     0.3312  0.0379  1.6064;
     0.2888  0.0468  1.4717;
     0.2323  0.0600  1.2880;
     0.1745  0.0910  1.1133;
     0.0920  0.1390  0.7552;
     0.0318  0.2080  0.4461;
     0.0048  0.3230  0.2644;
     0.0093  0.5030  0.1541;
     0.0636  0.7100  0.0763;
     0.1668  0.8620  0.0412;
     0.2926  0.9540  0.0200;
     0.4364  0.9950  0.0088;
     0.5970  0.9950  0.0039;
     0.7642  0.9520  0.0020;
     0.9159  0.8700  0.0016;
     1.0225  0.7570  0.0011;
     1.0544  0.6310  0.0007;
     0.9922  0.5030  0.0003;
     0.8432  0.3810  0.0002;
     0.6327  0.2650  0.0001;
     0.4404  0.1750  0.0000;
     0.2787  0.1070  0.0000;
     0.1619  0.0610  0.0000;
     0.0858  0.0320  0.0000;
     0.0459  0.0170  0.0000;
     0.0222  0.0082  0.0000;
     0.0113  0.0041  0.0000;
     0.0057  0.0021  0.0000;
     0.0028  0.0011  0.0000;
     0.0015  0.0005  0.0000;
     0.0005  0.0002  0.0000;
     0.0003  0.0001  0.0000;
     0.0002  0.0001  0.0000;
     0.0001  0.0000  0.0000]



function colormatch(::Type{CIE1931JV_CMF}, wavelen::Real)
    return interpolate_table(cie1931jv_cmf_table, 380.0, 5.0, wavelen)
end

# Judd-Vos adjustment to the CIE 1931 2° CMF, 5nm increments starting at 380nm
const cie1931jv_cmf_table =
    [2.689900e-003  2.000000e-004  1.226000e-002;
     5.310500e-003  3.955600e-004  2.422200e-002;
     1.078100e-002  8.000000e-004  4.925000e-002;
     2.079200e-002  1.545700e-003  9.513500e-002;
     3.798100e-002  2.800000e-003  1.740900e-001;
     6.315700e-002  4.656200e-003  2.901300e-001;
     9.994100e-002  7.400000e-003  4.605300e-001;
     1.582400e-001  1.177900e-002  7.316600e-001;
     2.294800e-001  1.750000e-002  1.065800e+000;
     2.810800e-001  2.267800e-002  1.314600e+000;
     3.109500e-001  2.730000e-002  1.467200e+000;
     3.307200e-001  3.258400e-002  1.579600e+000;
     3.333600e-001  3.790000e-002  1.616600e+000;
     3.167200e-001  4.239100e-002  1.568200e+000;
     2.888200e-001  4.680000e-002  1.471700e+000;
     2.596900e-001  5.212200e-002  1.374000e+000;
     2.327600e-001  6.000000e-002  1.291700e+000;
     2.099900e-001  7.294200e-002  1.235600e+000;
     1.747600e-001  9.098000e-002  1.113800e+000;
     1.328700e-001  1.128400e-001  9.422000e-001;
     9.194400e-002  1.390200e-001  7.559600e-001;
     5.698500e-002  1.698700e-001  5.864000e-001;
     3.173100e-002  2.080200e-001  4.466900e-001;
     1.461300e-002  2.580800e-001  3.411600e-001;
     4.849100e-003  3.230000e-001  2.643700e-001;
     2.321500e-003  4.054000e-001  2.059400e-001;
     9.289900e-003  5.030000e-001  1.544500e-001;
     2.927800e-002  6.081100e-001  1.091800e-001;
     6.379100e-002  7.100000e-001  7.658500e-002;
     1.108100e-001  7.951000e-001  5.622700e-002;
     1.669200e-001  8.620000e-001  4.136600e-002;
     2.276800e-001  9.150500e-001  2.935300e-002;
     2.926900e-001  9.540000e-001  2.004200e-002;
     3.622500e-001  9.800400e-001  1.331200e-002;
     4.363500e-001  9.949500e-001  8.782300e-003;
     5.151300e-001  1.000100e+000  5.857300e-003;
     5.974800e-001  9.950000e-001  4.049300e-003;
     6.812100e-001  9.787500e-001  2.921700e-003;
     7.642500e-001  9.520000e-001  2.277100e-003;
     8.439400e-001  9.155800e-001  1.970600e-003;
     9.163500e-001  8.700000e-001  1.806600e-003;
     9.770300e-001  8.162300e-001  1.544900e-003;
     1.023000e+000  7.570000e-001  1.234800e-003;
     1.051300e+000  6.948300e-001  1.117700e-003;
     1.055000e+000  6.310000e-001  9.056400e-004;
     1.036200e+000  5.665400e-001  6.946700e-004;
     9.923900e-001  5.030000e-001  4.288500e-004;
     9.286100e-001  4.417200e-001  3.181700e-004;
     8.434600e-001  3.810000e-001  2.559800e-004;
     7.398300e-001  3.205200e-001  1.567900e-004;
     6.328900e-001  2.650000e-001  9.769400e-005;
     5.335100e-001  2.170200e-001  6.894400e-005;
     4.406200e-001  1.750000e-001  5.116500e-005;
     3.545300e-001  1.381200e-001  3.601600e-005;
     2.786200e-001  1.070000e-001  2.423800e-005;
     2.148500e-001  8.165200e-002  1.691500e-005;
     1.616100e-001  6.100000e-002  1.190600e-005;
     1.182000e-001  4.432700e-002  8.148900e-006;
     8.575300e-002  3.200000e-002  5.600600e-006;
     6.307700e-002  2.345400e-002  3.954400e-006;
     4.583400e-002  1.700000e-002  2.791200e-006;
     3.205700e-002  1.187200e-002  1.917600e-006;
     2.218700e-002  8.210000e-003  1.313500e-006;
     1.561200e-002  5.772300e-003  9.151900e-007;
     1.109800e-002  4.102000e-003  6.476700e-007;
     7.923300e-003  2.929100e-003  4.635200e-007;
     5.653100e-003  2.091000e-003  3.330400e-007;
     4.003900e-003  1.482200e-003  2.382300e-007;
     2.825300e-003  1.047000e-003  1.702600e-007;
     1.994700e-003  7.401500e-004  1.220700e-007;
     1.399400e-003  5.200000e-004  8.710700e-008;
     9.698000e-004  3.609300e-004  6.145500e-008;
     6.684700e-004  2.492000e-004  4.316200e-008;
     4.614100e-004  1.723100e-004  3.037900e-008;
     3.207300e-004  1.200000e-004  2.155400e-008;
     2.257300e-004  8.462000e-005  1.549300e-008;
     1.597300e-004  6.000000e-005  1.120400e-008;
     1.127500e-004  4.244600e-005  8.087300e-009;
     7.951300e-005  3.000000e-005  5.834000e-009;
     5.608700e-005  2.121000e-005  4.211000e-009;
     3.954100e-005  1.498900e-005  3.038300e-009;
     2.785200e-005  1.058400e-005  2.190700e-009;
     1.959700e-005  7.465600e-006  1.577800e-009;
     1.377000e-005  5.259200e-006  1.134800e-009;
     9.670000e-006  3.702800e-006  8.156500e-010;
     6.791800e-006  2.607600e-006  5.862600e-010;
     4.770600e-006  1.836500e-006  4.213800e-010;
     3.355000e-006  1.295000e-006  3.031900e-010;
     2.353400e-006  9.109200e-007  2.175300e-010;
     1.637700e-006  6.356400e-007  1.547600e-010]

# CIE2006 proposed XYZ CMFs[*]. Yet to be adopted by the CIE.
# The according LMS CMFs got adopted by the CIE.
# The original CIE datasets range from 390 to 830 nm
# The new CIE CMFs are based on a much larger sample of observers,
# and the main difference to the old data is the stronger weighting
# in the short wavelength part of the visual spectrum.
# One goal was to design a dataset where all three CMFs
# integrate to the exact same value. The shortening to the upper limit
# of 780 nm in the version presented here may lead to minor and insignificant
# differences between the three integrals.

# [*] http://cvrl.ioo.ucl.ac.uk/database/text/cienewxyz/cie2012xyz2.htm

# To be tested: differences between shortened and original dataset
#               testing of the effect of extrapolation down to 380 nm

function colormatch(::Type{CIE2006_2_CMF}, wavelen::Real)
    return interpolate_table(cie2006_2deg_xyz_cmf_table, 380.0, 1.0, wavelen)
end

const cie2006_2deg_xyz_cmf_table=
    [0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     3.769647e-03  4.146161e-04  1.847260e-02;
     4.532416e-03  5.028333e-04  2.221101e-02;
     5.446553e-03  6.084991e-04  2.669819e-02;
     6.538868e-03  7.344436e-04  3.206937e-02;
     7.839699e-03  8.837389e-04  3.847832e-02;
     9.382967e-03  1.059646e-03  4.609784e-02;
     1.120608e-02  1.265532e-03  5.511953e-02;
     1.334965e-02  1.504753e-03  6.575257e-02;
     1.585690e-02  1.780493e-03  7.822113e-02;
     1.877286e-02  2.095572e-03  9.276013e-02;
     2.214302e-02  2.452194e-03  1.096090e-01;
     2.601285e-02  2.852216e-03  1.290077e-01;
     3.043036e-02  3.299115e-03  1.512047e-01;
     3.544325e-02  3.797466e-03  1.764441e-01;
     4.109640e-02  4.352768e-03  2.049517e-01;
     4.742986e-02  4.971717e-03  2.369246e-01;
     5.447394e-02  5.661014e-03  2.725123e-01;
     6.223612e-02  6.421615e-03  3.117820e-01;
     7.070048e-02  7.250312e-03  3.547064e-01;
     7.982513e-02  8.140173e-03  4.011473e-01;
     8.953803e-02  9.079860e-03  4.508369e-01;
     9.974848e-02  1.005608e-02  5.034164e-01;
     1.104019e-01  1.106456e-02  5.586361e-01;
     1.214566e-01  1.210522e-02  6.162734e-01;
     1.328741e-01  1.318014e-02  6.760982e-01;
     1.446214e-01  1.429377e-02  7.378822e-01;
     1.566468e-01  1.545004e-02  8.013019e-01;
     1.687901e-01  1.664093e-02  8.655573e-01;
     1.808328e-01  1.785302e-02  9.295791e-01;
     1.925216e-01  1.907018e-02  9.921293e-01;
     2.035729e-01  2.027369e-02  1.051821e+00;
     2.137531e-01  2.144805e-02  1.107509e+00;
     2.231348e-01  2.260041e-02  1.159527e+00;
     2.319245e-01  2.374789e-02  1.208869e+00;
     2.403892e-01  2.491247e-02  1.256834e+00;
     2.488523e-01  2.612106e-02  1.305008e+00;
     2.575896e-01  2.739923e-02  1.354758e+00;
     2.664991e-01  2.874993e-02  1.405594e+00;
     2.753532e-01  3.016909e-02  1.456414e+00;
     2.838921e-01  3.165145e-02  1.505960e+00;
     2.918246e-01  3.319038e-02  1.552826e+00;
     2.989200e-01  3.477912e-02  1.595902e+00;
     3.052993e-01  3.641495e-02  1.635768e+00;
     3.112031e-01  3.809569e-02  1.673573e+00;
     3.169047e-01  3.981843e-02  1.710604e+00;
     3.227087e-01  4.157940e-02  1.748280e+00;
     3.288194e-01  4.337098e-02  1.787504e+00;
     3.349242e-01  4.517180e-02  1.826609e+00;
     3.405452e-01  4.695420e-02  1.863108e+00;
     3.451688e-01  4.868718e-02  1.894332e+00;
     3.482554e-01  5.033657e-02  1.917479e+00;
     3.494153e-01  5.187611e-02  1.930529e+00;
     3.489075e-01  5.332218e-02  1.934819e+00;
     3.471746e-01  5.470603e-02  1.932650e+00;
     3.446705e-01  5.606335e-02  1.926395e+00;
     3.418483e-01  5.743393e-02  1.918437e+00;
     3.390240e-01  5.885107e-02  1.910430e+00;
     3.359926e-01  6.030809e-02  1.901224e+00;
     3.324276e-01  6.178644e-02  1.889000e+00;
     3.280157e-01  6.326570e-02  1.871996e+00;
     3.224637e-01  6.472352e-02  1.848545e+00;
     3.156225e-01  6.614749e-02  1.817792e+00;
     3.078201e-01  6.757256e-02  1.781627e+00;
     2.994771e-01  6.904928e-02  1.742514e+00;
     2.909776e-01  7.063280e-02  1.702749e+00;
     2.826646e-01  7.238339e-02  1.664439e+00;
     2.747962e-01  7.435960e-02  1.629207e+00;
     2.674312e-01  7.659383e-02  1.597360e+00;
     2.605847e-01  7.911436e-02  1.568896e+00;
     2.542749e-01  8.195345e-02  1.543823e+00;
     2.485254e-01  8.514816e-02  1.522157e+00;
     2.433039e-01  8.872657e-02  1.503611e+00;
     2.383414e-01  9.266008e-02  1.486673e+00;
     2.333253e-01  9.689723e-02  1.469595e+00;
     2.279619e-01  1.013746e-01  1.450709e+00;
     2.219781e-01  1.060145e-01  1.428440e+00;
     2.151735e-01  1.107377e-01  1.401587e+00;
     2.075619e-01  1.155111e-01  1.370094e+00;
     1.992183e-01  1.203122e-01  1.334220e+00;
     1.902290e-01  1.251161e-01  1.294275e+00;
     1.806905e-01  1.298957e-01  1.250610e+00;
     1.707154e-01  1.346299e-01  1.203696e+00;
     1.604471e-01  1.393309e-01  1.154316e+00;
     1.500244e-01  1.440235e-01  1.103284e+00;
     1.395705e-01  1.487372e-01  1.051347e+00;
     1.291920e-01  1.535066e-01  9.991789e-01;
     1.189859e-01  1.583644e-01  9.473958e-01;
     1.090615e-01  1.633199e-01  8.966222e-01;
     9.951424e-02  1.683761e-01  8.473981e-01;
     9.041850e-02  1.735365e-01  8.001576e-01;
     8.182895e-02  1.788048e-01  7.552379e-01;
     7.376817e-02  1.841819e-01  7.127879e-01;
     6.619477e-02  1.896559e-01  6.725198e-01;
     5.906380e-02  1.952101e-01  6.340976e-01;
     5.234242e-02  2.008259e-01  5.972433e-01;
     4.600865e-02  2.064828e-01  5.617313e-01;
     4.006154e-02  2.121826e-01  5.274921e-01;
     3.454373e-02  2.180279e-01  4.948809e-01;
     2.949091e-02  2.241586e-01  4.642586e-01;
     2.492140e-02  2.307302e-01  4.358841e-01;
     2.083981e-02  2.379160e-01  4.099313e-01;
     1.723591e-02  2.458706e-01  3.864261e-01;
     1.407924e-02  2.546023e-01  3.650566e-01;
     1.134516e-02  2.640760e-01  3.454812e-01;
     9.019658e-03  2.742490e-01  3.274095e-01;
     7.097731e-03  2.850680e-01  3.105939e-01;
     5.571145e-03  2.964837e-01  2.948102e-01;
     4.394566e-03  3.085010e-01  2.798194e-01;
     3.516303e-03  3.211393e-01  2.654100e-01;
     2.887638e-03  3.344175e-01  2.514084e-01;
     2.461588e-03  3.483536e-01  2.376753e-01;
     2.206348e-03  3.629601e-01  2.241211e-01;
     2.149559e-03  3.782275e-01  2.107484e-01;
     2.337091e-03  3.941359e-01  1.975839e-01;
     2.818931e-03  4.106582e-01  1.846574e-01;
     3.649178e-03  4.277595e-01  1.720018e-01;
     4.891359e-03  4.453993e-01  1.596918e-01;
     6.629364e-03  4.635396e-01  1.479415e-01;
     8.942902e-03  4.821376e-01  1.369428e-01;
     1.190224e-02  5.011430e-01  1.268279e-01;
     1.556989e-02  5.204972e-01  1.176796e-01;
     1.997668e-02  5.401387e-01  1.094970e-01;
     2.504698e-02  5.600208e-01  1.020943e-01;
     3.067530e-02  5.800972e-01  9.527993e-02;
     3.674999e-02  6.003172e-01  8.890075e-02;
     4.315171e-02  6.206256e-01  8.283548e-02;
     4.978584e-02  6.409398e-01  7.700982e-02;
     5.668554e-02  6.610772e-01  7.144001e-02;
     6.391651e-02  6.808134e-01  6.615436e-02;
     7.154352e-02  6.999044e-01  6.117199e-02;
     7.962917e-02  7.180890e-01  5.650407e-02;
     8.821473e-02  7.351593e-01  5.215121e-02;
     9.726978e-02  7.511821e-01  4.809566e-02;
     1.067504e-01  7.663143e-01  4.431720e-02;
     1.166192e-01  7.807352e-01  4.079734e-02;
     1.268468e-01  7.946448e-01  3.751912e-02;
     1.374060e-01  8.082074e-01  3.446846e-02;
     1.482471e-01  8.213817e-01  3.163764e-02;
     1.593076e-01  8.340701e-01  2.901901e-02;
     1.705181e-01  8.461711e-01  2.660364e-02;
     1.818026e-01  8.575799e-01  2.438164e-02;
     1.931090e-01  8.682408e-01  2.234097e-02;
     2.045085e-01  8.783061e-01  2.046415e-02;
     2.161166e-01  8.879907e-01  1.873456e-02;
     2.280650e-01  8.975211e-01  1.713788e-02;
     2.405015e-01  9.071347e-01  1.566174e-02;
     2.535441e-01  9.169947e-01  1.429644e-02;
     2.671300e-01  9.269295e-01  1.303702e-02;
     2.811351e-01  9.366731e-01  1.187897e-02;
     2.954164e-01  9.459482e-01  1.081725e-02;
     3.098117e-01  9.544675e-01  9.846470e-03;
     3.241678e-01  9.619834e-01  8.960687e-03;
     3.384319e-01  9.684390e-01  8.152811e-03;
     3.525786e-01  9.738289e-01  7.416025e-03;
     3.665839e-01  9.781519e-01  6.744115e-03;
     3.804244e-01  9.814106e-01  6.131421e-03;
     3.940988e-01  9.836669e-01  5.572778e-03;
     4.076972e-01  9.852081e-01  5.063463e-03;
     4.213484e-01  9.863813e-01  4.599169e-03;
     4.352003e-01  9.875357e-01  4.175971e-03;
     4.494206e-01  9.890228e-01  3.790291e-03;
     4.641616e-01  9.910811e-01  3.438952e-03;
     4.794395e-01  9.934913e-01  3.119341e-03;
     4.952180e-01  9.959172e-01  2.829038e-03;
     5.114395e-01  9.980205e-01  2.565722e-03;
     5.280233e-01  9.994608e-01  2.327186e-03;
     5.448696e-01  9.999930e-01  2.111280e-03;
     5.618898e-01  9.997557e-01  1.915766e-03;
     5.790137e-01  9.989839e-01  1.738589e-03;
     5.961882e-01  9.979123e-01  1.577920e-03;
     6.133784e-01  9.967737e-01  1.432128e-03;
     6.305897e-01  9.957356e-01  1.299781e-03;
     6.479223e-01  9.947115e-01  1.179667e-03;
     6.654866e-01  9.935534e-01  1.070694e-03;
     6.833782e-01  9.921156e-01  9.718623e-04;
     7.016774e-01  9.902549e-01  8.822531e-04;
     7.204110e-01  9.878596e-01  8.010231e-04;
     7.394495e-01  9.849324e-01  7.273884e-04;
     7.586285e-01  9.815036e-01  6.606347e-04;
     7.777885e-01  9.776035e-01  6.001146e-04;
     7.967750e-01  9.732611e-01  5.452416e-04;
     8.154530e-01  9.684764e-01  4.954847e-04;
     8.337389e-01  9.631369e-01  4.503642e-04;
     8.515493e-01  9.571062e-01  4.094455e-04;
     8.687862e-01  9.502540e-01  3.723345e-04;
     8.853376e-01  9.424569e-01  3.386739e-04;
     9.011588e-01  9.336897e-01  3.081396e-04;
     9.165278e-01  9.242893e-01  2.804370e-04;
     9.318245e-01  9.146707e-01  2.552996e-04;
     9.474524e-01  9.052333e-01  2.324859e-04;
     9.638388e-01  8.963613e-01  2.117772e-04;
     9.812596e-01  8.883069e-01  1.929758e-04;
     9.992953e-01  8.808462e-01  1.759024e-04;
     1.017343e+00  8.736445e-01  1.603947e-04;
     1.034790e+00  8.663755e-01  1.463059e-04;
     1.051011e+00  8.587203e-01  1.335031e-04;
     1.065522e+00  8.504295e-01  1.218660e-04;
     1.078421e+00  8.415047e-01  1.112857e-04;
     1.089944e+00  8.320109e-01  1.016634e-04;
     1.100320e+00  8.220154e-01  9.291003e-05;
     1.109767e+00  8.115868e-01  8.494468e-05;
     1.118438e+00  8.007874e-01  7.769425e-05;
     1.126266e+00  7.896515e-01  7.109247e-05;
     1.133138e+00  7.782053e-01  6.507936e-05;
     1.138952e+00  7.664733e-01  5.960061e-05;
     1.143620e+00  7.544785e-01  5.460706e-05;
     1.147095e+00  7.422473e-01  5.005417e-05;
     1.149464e+00  7.298229e-01  4.590157e-05;
     1.150838e+00  7.172525e-01  4.211268e-05;
     1.151326e+00  7.045818e-01  3.865437e-05;
     1.151033e+00  6.918553e-01  3.549661e-05;
     1.150002e+00  6.791009e-01  3.261220e-05;
     1.148061e+00  6.662846e-01  2.997643e-05;
     1.144998e+00  6.533595e-01  2.756693e-05;
     1.140622e+00  6.402807e-01  2.536339e-05;
     1.134757e+00  6.270066e-01  2.334738e-05;
     1.127298e+00  6.135148e-01  2.150221e-05;
     1.118342e+00  5.998494e-01  1.981268e-05;
     1.108033e+00  5.860682e-01  1.826500e-05;
     1.096515e+00  5.722261e-01  1.684667e-05;
     1.083928e+00  5.583746e-01  1.554631e-05;
     1.070387e+00  5.445535e-01  1.435360e-05;
     1.055934e+00  5.307673e-01  1.325915e-05;
     1.040592e+00  5.170130e-01  1.225443e-05;
     1.024385e+00  5.032889e-01  1.133169e-05;
     1.007344e+00  4.895950e-01  1.048387e-05;
     9.895268e-01  4.759442e-01  0.000000e+00;
     9.711213e-01  4.623958e-01  0.000000e+00;
     9.523257e-01  4.490154e-01  0.000000e+00;
     9.333248e-01  4.358622e-01  0.000000e+00;
     9.142877e-01  4.229897e-01  0.000000e+00;
     8.952798e-01  4.104152e-01  0.000000e+00;
     8.760157e-01  3.980356e-01  0.000000e+00;
     8.561607e-01  3.857300e-01  0.000000e+00;
     8.354235e-01  3.733907e-01  0.000000e+00;
     8.135565e-01  3.609245e-01  0.000000e+00;
     7.904565e-01  3.482860e-01  0.000000e+00;
     7.664364e-01  3.355702e-01  0.000000e+00;
     7.418777e-01  3.228963e-01  0.000000e+00;
     7.171219e-01  3.103704e-01  0.000000e+00;
     6.924717e-01  2.980865e-01  0.000000e+00;
     6.681600e-01  2.861160e-01  0.000000e+00;
     6.442697e-01  2.744822e-01  0.000000e+00;
     6.208450e-01  2.631953e-01  0.000000e+00;
     5.979243e-01  2.522628e-01  0.000000e+00;
     5.755410e-01  2.416902e-01  0.000000e+00;
     5.537296e-01  2.314809e-01  0.000000e+00;
     5.325412e-01  2.216378e-01  0.000000e+00;
     5.120218e-01  2.121622e-01  0.000000e+00;
     4.922070e-01  2.030542e-01  0.000000e+00;
     4.731224e-01  1.943124e-01  0.000000e+00;
     4.547417e-01  1.859227e-01  0.000000e+00;
     4.368719e-01  1.778274e-01  0.000000e+00;
     4.193121e-01  1.699654e-01  0.000000e+00;
     4.018980e-01  1.622841e-01  0.000000e+00;
     3.844986e-01  1.547397e-01  0.000000e+00;
     3.670592e-01  1.473081e-01  0.000000e+00;
     3.497167e-01  1.400169e-01  0.000000e+00;
     3.326305e-01  1.329013e-01  0.000000e+00;
     3.159341e-01  1.259913e-01  0.000000e+00;
     2.997374e-01  1.193120e-01  0.000000e+00;
     2.841189e-01  1.128820e-01  0.000000e+00;
     2.691053e-01  1.067113e-01  0.000000e+00;
     2.547077e-01  1.008052e-01  0.000000e+00;
     2.409319e-01  9.516653e-02  0.000000e+00;
     2.277792e-01  8.979594e-02  0.000000e+00;
     2.152431e-01  8.469044e-02  0.000000e+00;
     2.033010e-01  7.984009e-02  0.000000e+00;
     1.919276e-01  7.523372e-02  0.000000e+00;
     1.810987e-01  7.086061e-02  0.000000e+00;
     1.707914e-01  6.671045e-02  0.000000e+00;
     1.609842e-01  6.277360e-02  0.000000e+00;
     1.516577e-01  5.904179e-02  0.000000e+00;
     1.427936e-01  5.550703e-02  0.000000e+00;
     1.343737e-01  5.216139e-02  0.000000e+00;
     1.263808e-01  4.899699e-02  0.000000e+00;
     1.187979e-01  4.600578e-02  0.000000e+00;
     1.116088e-01  4.317885e-02  0.000000e+00;
     1.047975e-01  4.050755e-02  0.000000e+00;
     9.834835e-02  3.798376e-02  0.000000e+00;
     9.224597e-02  3.559982e-02  0.000000e+00;
     8.647506e-02  3.334856e-02  0.000000e+00;
     8.101986e-02  3.122332e-02  0.000000e+00;
     7.586514e-02  2.921780e-02  0.000000e+00;
     7.099633e-02  2.732601e-02  0.000000e+00;
     6.639960e-02  2.554223e-02  0.000000e+00;
     6.206225e-02  2.386121e-02  0.000000e+00;
     5.797409e-02  2.227859e-02  0.000000e+00;
     5.412533e-02  2.079020e-02  0.000000e+00;
     5.050600e-02  1.939185e-02  0.000000e+00;
     4.710606e-02  1.807939e-02  0.000000e+00;
     4.391411e-02  1.684817e-02  0.000000e+00;
     4.091411e-02  1.569188e-02  0.000000e+00;
     3.809067e-02  1.460446e-02  0.000000e+00;
     3.543034e-02  1.358062e-02  0.000000e+00;
     3.292138e-02  1.261573e-02  0.000000e+00;
     3.055672e-02  1.170696e-02  0.000000e+00;
     2.834146e-02  1.085608e-02  0.000000e+00;
     2.628033e-02  1.006476e-02  0.000000e+00;
     2.437465e-02  9.333376e-03  0.000000e+00;
     2.262306e-02  8.661284e-03  0.000000e+00;
     2.101935e-02  8.046048e-03  0.000000e+00;
     1.954647e-02  7.481130e-03  0.000000e+00;
     1.818727e-02  6.959987e-03  0.000000e+00;
     1.692727e-02  6.477070e-03  0.000000e+00;
     1.575417e-02  6.027677e-03  0.000000e+00;
     1.465854e-02  5.608169e-03  0.000000e+00;
     1.363571e-02  5.216691e-03  0.000000e+00;
     1.268205e-02  4.851785e-03  0.000000e+00;
     1.179394e-02  4.512008e-03  0.000000e+00;
     1.096778e-02  4.195941e-03  0.000000e+00;
     1.019964e-02  3.902057e-03  0.000000e+00;
     9.484317e-03  3.628371e-03  0.000000e+00;
     8.816851e-03  3.373005e-03  0.000000e+00;
     8.192921e-03  3.134315e-03  0.000000e+00;
     7.608750e-03  2.910864e-03  0.000000e+00;
     7.061391e-03  2.701528e-03  0.000000e+00;
     6.549509e-03  2.505796e-03  0.000000e+00;
     6.071970e-03  2.323231e-03  0.000000e+00;
     5.627476e-03  2.153333e-03  0.000000e+00;
     5.214608e-03  1.995557e-03  0.000000e+00;
     4.831848e-03  1.849316e-03  0.000000e+00;
     4.477579e-03  1.713976e-03  0.000000e+00;
     4.150166e-03  1.588899e-03  0.000000e+00;
     3.847988e-03  1.473453e-03  0.000000e+00;
     3.569452e-03  1.367022e-03  0.000000e+00;
     3.312857e-03  1.268954e-03  0.000000e+00;
     3.076022e-03  1.178421e-03  0.000000e+00;
     2.856894e-03  1.094644e-03  0.000000e+00;
     2.653681e-03  1.016943e-03  0.000000e+00;
     2.464821e-03  9.447269e-04  0.000000e+00;
     2.289060e-03  8.775171e-04  0.000000e+00;
     2.125694e-03  8.150438e-04  0.000000e+00;
     1.974121e-03  7.570755e-04  0.000000e+00;
     1.833723e-03  7.033755e-04  0.000000e+00;
     1.703876e-03  6.537050e-04  0.000000e+00;
     1.583904e-03  6.078048e-04  0.000000e+00;
     1.472939e-03  5.653435e-04  0.000000e+00;
     1.370151e-03  5.260046e-04  0.000000e+00;
     1.274803e-03  4.895061e-04  0.000000e+00;
     1.186238e-03  4.555970e-04  0.000000e+00;
     1.103871e-03  4.240548e-04  0.000000e+00;
     1.027194e-03  3.946860e-04  0.000000e+00;
     9.557493e-04  3.673178e-04  0.000000e+00;
     8.891262e-04  3.417941e-04  0.000000e+00;
     8.269535e-04  3.179738e-04  0.000000e+00;
     7.689351e-04  2.957441e-04  0.000000e+00;
     7.149425e-04  2.750558e-04  0.000000e+00;
     6.648590e-04  2.558640e-04  0.000000e+00;
     6.185421e-04  2.381142e-04  0.000000e+00;
     5.758303e-04  2.217445e-04  0.000000e+00;
     5.365046e-04  2.066711e-04  0.000000e+00;
     5.001842e-04  1.927474e-04  0.000000e+00;
     4.665005e-04  1.798315e-04  0.000000e+00;
     4.351386e-04  1.678023e-04  0.000000e+00;
     4.058303e-04  1.565566e-04  0.000000e+00;
     3.783733e-04  1.460168e-04  0.000000e+00;
     3.526892e-04  1.361535e-04  0.000000e+00;
     3.287199e-04  1.269451e-04  0.000000e+00;
     3.063998e-04  1.183671e-04  0.000000e+00;
     2.856577e-04  1.103928e-04  0.000000e+00;
     2.664108e-04  1.029908e-04  0.000000e+00;
     2.485462e-04  9.611836e-05  0.000000e+00;
     2.319529e-04  8.973323e-05  0.000000e+00;
     2.165300e-04  8.379694e-05  0.000000e+00;
     2.021853e-04  7.827442e-05  0.000000e+00;
     1.888338e-04  7.313312e-05  0.000000e+00;
     1.763935e-04  6.834142e-05  0.000000e+00;
     1.647895e-04  6.387035e-05  0.000000e+00;
     1.539542e-04  5.969389e-05  0.000000e+00;
     1.438270e-04  5.578862e-05  0.000000e+00;
     1.343572e-04  5.213509e-05  0.000000e+00;
     1.255141e-04  4.872179e-05  0.000000e+00;
     1.172706e-04  4.553845e-05  0.000000e+00;
     1.095983e-04  4.257443e-05  0.000000e+00;
     1.024685e-04  3.981884e-05  0.000000e+00;
     9.584715e-05  3.725877e-05  0.000000e+00;
     8.968316e-05  3.487467e-05  0.000000e+00;
     8.392734e-05  3.264765e-05  0.000000e+00;
     7.853708e-05  3.056140e-05  0.000000e+00;
     7.347551e-05  2.860175e-05  0.000000e+00;
     6.871576e-05  2.675841e-05  0.000000e+00;
     6.425257e-05  2.502943e-05  0.000000e+00;
     6.008292e-05  2.341373e-05  0.000000e+00;
     5.620098e-05  2.190914e-05  0.000000e+00;
     5.259870e-05  2.051259e-05  0.000000e+00;
     4.926279e-05  1.921902e-05  0.000000e+00;
     4.616623e-05  1.801796e-05  0.000000e+00;
     4.328212e-05  1.689899e-05  0.000000e+00;
     4.058715e-05  1.585309e-05  0.000000e+00;
     3.806114e-05  1.487243e-05  0.000000e+00]


# CIE 2006 10° observer XYZ CMFs. For further information
# see comment section for CIE 2006 2° observer XYZ CMFs
# Transformed from the CIE (2006) 2° LMS cone fundamentals[*]
# [*] http://cvrl.ioo.ucl.ac.uk/database/text/cienewxyz/cie2012xyz10.htm


function colormatch(::Type{CIE2006_10_CMF}, wavelen::Real)
    return interpolate_table(cie2006_10deg_xyz_cmf_table, 380.0, 1.0, wavelen)
end

const cie2006_10deg_xyz_cmf_table=
    [0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     0.000000e+00  0.000000e+00  0.000000e+00;
     2.952420e-03  4.076779e-04  1.318752e-02;
     3.577275e-03  4.977769e-04  1.597879e-02;
     4.332146e-03  6.064754e-04  1.935758e-02;
     5.241609e-03  7.370040e-04  2.343758e-02;
     6.333902e-03  8.929388e-04  2.835021e-02;
     7.641137e-03  1.078166e-03  3.424588e-02;
     9.199401e-03  1.296816e-03  4.129467e-02;
     1.104869e-02  1.553159e-03  4.968641e-02;
     1.323262e-02  1.851463e-03  5.962964e-02;
     1.579791e-02  2.195795e-03  7.134926e-02;
     1.879338e-02  2.589775e-03  8.508254e-02;
     2.226949e-02  3.036799e-03  1.010753e-01;
     2.627978e-02  3.541926e-03  1.195838e-01;
     3.087862e-02  4.111422e-03  1.408647e-01;
     3.611890e-02  4.752618e-03  1.651644e-01;
     4.204986e-02  5.474207e-03  1.927065e-01;
     4.871256e-02  6.285034e-03  2.236782e-01;
     5.612868e-02  7.188068e-03  2.582109e-01;
     6.429866e-02  8.181786e-03  2.963632e-01;
     7.319818e-02  9.260417e-03  3.381018e-01;
     8.277331e-02  1.041303e-02  3.832822e-01;
     9.295327e-02  1.162642e-02  4.316884e-01;
     1.037137e-01  1.289884e-02  4.832440e-01;
     1.150520e-01  1.423442e-02  5.379345e-01;
     1.269771e-01  1.564080e-02  5.957740e-01;
     1.395127e-01  1.712968e-02  6.568187e-01;
     1.526661e-01  1.871265e-02  7.210459e-01;
     1.663054e-01  2.038394e-02  7.878635e-01;
     1.802197e-01  2.212935e-02  8.563391e-01;
     1.941448e-01  2.392985e-02  9.253017e-01;
     2.077647e-01  2.576133e-02  9.933444e-01;
     2.207911e-01  2.760156e-02  1.059178e+00;
     2.332355e-01  2.945513e-02  1.122832e+00;
     2.452462e-01  3.133884e-02  1.184947e+00;
     2.570397e-01  3.327575e-02  1.246476e+00;
     2.688989e-01  3.529554e-02  1.308674e+00;
     2.810677e-01  3.742705e-02  1.372628e+00;
     2.933967e-01  3.967137e-02  1.437661e+00;
     3.055933e-01  4.201998e-02  1.502449e+00;
     3.173165e-01  4.446166e-02  1.565456e+00;
     3.281798e-01  4.698226e-02  1.624940e+00;
     3.378678e-01  4.956742e-02  1.679488e+00;
     3.465097e-01  5.221219e-02  1.729668e+00;
     3.543953e-01  5.491387e-02  1.776755e+00;
     3.618655e-01  5.766919e-02  1.822228e+00;
     3.693084e-01  6.047429e-02  1.867751e+00;
     3.770107e-01  6.332195e-02  1.914504e+00;
     3.846850e-01  6.619271e-02  1.961055e+00;
     3.918591e-01  6.906185e-02  2.005136e+00;
     3.980192e-01  7.190190e-02  2.044296e+00;
     4.026189e-01  7.468288e-02  2.075946e+00;
     4.052637e-01  7.738452e-02  2.098231e+00;
     4.062482e-01  8.003601e-02  2.112591e+00;
     4.060660e-01  8.268524e-02  2.121427e+00;
     4.052283e-01  8.538745e-02  2.127239e+00;
     4.042529e-01  8.820537e-02  2.132574e+00;
     4.034808e-01  9.118925e-02  2.139093e+00;
     4.025362e-01  9.431041e-02  2.144815e+00;
     4.008675e-01  9.751346e-02  2.146832e+00;
     3.979327e-01  1.007349e-01  2.142250e+00;
     3.932139e-01  1.039030e-01  2.128264e+00;
     3.864108e-01  1.069639e-01  2.103205e+00;
     3.779513e-01  1.099676e-01  2.069388e+00;
     3.684176e-01  1.129992e-01  2.030030e+00;
     3.583473e-01  1.161541e-01  1.988178e+00;
     3.482214e-01  1.195389e-01  1.946651e+00;
     3.383830e-01  1.232503e-01  1.907521e+00;
     3.288309e-01  1.273047e-01  1.870689e+00;
     3.194977e-01  1.316964e-01  1.835578e+00;
     3.103345e-01  1.364178e-01  1.801657e+00;
     3.013112e-01  1.414586e-01  1.768440e+00;
     2.923754e-01  1.468003e-01  1.735338e+00;
     2.833273e-01  1.524002e-01  1.701254e+00;
     2.739463e-01  1.582021e-01  1.665053e+00;
     2.640352e-01  1.641400e-01  1.625712e+00;
     2.534221e-01  1.701373e-01  1.582342e+00;
     2.420135e-01  1.761233e-01  1.534439e+00;
     2.299346e-01  1.820896e-01  1.482544e+00;
     2.173617e-01  1.880463e-01  1.427438e+00;
     2.044672e-01  1.940065e-01  1.369876e+00;
     1.914176e-01  1.999859e-01  1.310576e+00;
     1.783672e-01  2.060054e-01  1.250226e+00;
     1.654407e-01  2.120981e-01  1.189511e+00;
     1.527391e-01  2.183041e-01  1.129050e+00;
     1.403439e-01  2.246686e-01  1.069379e+00;
     1.283167e-01  2.312426e-01  1.010952e+00;
     1.167124e-01  2.380741e-01  9.541809e-01;
     1.056121e-01  2.451798e-01  8.995253e-01;
     9.508569e-02  2.525682e-01  8.473720e-01;
     8.518206e-02  2.602479e-01  7.980093e-01;
     7.593120e-02  2.682271e-01  7.516389e-01;
     6.733159e-02  2.765005e-01  7.082645e-01;
     5.932018e-02  2.850035e-01  6.673867e-01;
     5.184106e-02  2.936475e-01  6.284798e-01;
     4.486119e-02  3.023319e-01  5.911174e-01;
     3.836770e-02  3.109438e-01  5.549619e-01;
     3.237296e-02  3.194105e-01  5.198843e-01;
     2.692095e-02  3.278683e-01  4.862772e-01;
     2.204070e-02  3.365263e-01  4.545497e-01;
     1.773951e-02  3.456176e-01  4.249955e-01;
     1.400745e-02  3.554018e-01  3.978114e-01;
     1.082291e-02  3.660893e-01  3.730218e-01;
     8.168996e-03  3.775857e-01  3.502618e-01;
     6.044623e-03  3.896960e-01  3.291407e-01;
     4.462638e-03  4.021947e-01  3.093356e-01;
     3.446810e-03  4.148227e-01  2.905816e-01;
     3.009513e-03  4.273539e-01  2.726773e-01;
     3.090744e-03  4.398206e-01  2.555143e-01;
     3.611221e-03  4.523360e-01  2.390188e-01;
     4.491435e-03  4.650298e-01  2.231335e-01;
     5.652072e-03  4.780482e-01  2.078158e-01;
     7.035322e-03  4.915173e-01  1.930407e-01;
     8.669631e-03  5.054224e-01  1.788089e-01;
     1.060755e-02  5.197057e-01  1.651287e-01;
     1.290468e-02  5.343012e-01  1.520103e-01;
     1.561956e-02  5.491344e-01  1.394643e-01;
     1.881640e-02  5.641302e-01  1.275353e-01;
     2.256923e-02  5.792416e-01  1.163771e-01;
     2.694456e-02  5.944264e-01  1.061161e-01;
     3.199910e-02  6.096388e-01  9.682266e-02;
     3.778185e-02  6.248296e-01  8.852389e-02;
     4.430635e-02  6.399656e-01  8.118263e-02;
     5.146516e-02  6.550943e-01  7.463132e-02;
     5.912224e-02  6.702903e-01  6.870644e-02;
     6.714220e-02  6.856375e-01  6.327834e-02;
     7.538941e-02  7.012292e-01  5.824484e-02;
     8.376697e-02  7.171103e-01  5.353812e-02;
     9.233581e-02  7.330917e-01  4.914863e-02;
     1.011940e-01  7.489041e-01  4.507511e-02;
     1.104362e-01  7.642530e-01  4.131175e-02;
     1.201511e-01  7.788199e-01  3.784916e-02;
     1.303960e-01  7.923410e-01  3.467234e-02;
     1.411310e-01  8.048510e-01  3.175471e-02;
     1.522944e-01  8.164747e-01  2.907029e-02;
     1.638288e-01  8.273520e-01  2.659651e-02;
     1.756832e-01  8.376358e-01  2.431375e-02;
     1.878114e-01  8.474653e-01  2.220677e-02;
     2.001621e-01  8.568868e-01  2.026852e-02;
     2.126822e-01  8.659242e-01  1.849246e-02;
     2.253199e-01  8.746041e-01  1.687084e-02;
     2.380254e-01  8.829552e-01  1.539505e-02;
     2.507787e-01  8.910274e-01  1.405450e-02;
     2.636778e-01  8.989495e-01  1.283354e-02;
     2.768607e-01  9.068753e-01  1.171754e-02;
     2.904792e-01  9.149652e-01  1.069415e-02;
     3.046991e-01  9.233858e-01  9.753000e-03;
     3.196485e-01  9.322325e-01  8.886096e-03;
     3.352447e-01  9.412862e-01  8.089323e-03;
     3.513290e-01  9.502378e-01  7.359131e-03;
     3.677148e-01  9.587647e-01  6.691736e-03;
     3.841856e-01  9.665325e-01  6.083223e-03;
     4.005312e-01  9.732504e-01  5.529423e-03;
     4.166669e-01  9.788415e-01  5.025504e-03;
     4.325420e-01  9.832867e-01  4.566879e-03;
     4.481063e-01  9.865720e-01  4.149405e-03;
     4.633109e-01  9.886887e-01  3.769336e-03;
     4.781440e-01  9.897056e-01  3.423302e-03;
     4.927483e-01  9.899849e-01  3.108313e-03;
     5.073315e-01  9.899624e-01  2.821650e-03;
     5.221315e-01  9.900731e-01  2.560830e-03;
     5.374170e-01  9.907500e-01  2.323578e-03;
     5.534217e-01  9.922826e-01  2.107847e-03;
     5.701242e-01  9.943837e-01  1.911867e-03;
     5.874093e-01  9.966221e-01  1.734006e-03;
     6.051269e-01  9.985649e-01  1.572736e-03;
     6.230892e-01  9.997775e-01  1.426627e-03;
     6.410999e-01  9.999440e-01  1.294325e-03;
     6.590659e-01  9.992200e-01  1.174475e-03;
     6.769436e-01  9.978793e-01  1.065842e-03;
     6.947143e-01  9.961934e-01  9.673215e-04;
     7.123849e-01  9.944304e-01  8.779264e-04;
     7.299978e-01  9.927831e-01  7.967847e-04;
     7.476478e-01  9.911578e-01  7.231502e-04;
     7.654250e-01  9.893925e-01  6.563501e-04;
     7.834009e-01  9.873288e-01  5.957678e-04;
     8.016277e-01  9.848127e-01  5.408385e-04;
     8.201041e-01  9.817253e-01  4.910441e-04;
     8.386843e-01  9.780714e-01  4.459046e-04;
     8.571936e-01  9.738860e-01  4.049826e-04;
     8.754652e-01  9.692028e-01  3.678818e-04;
     8.933408e-01  9.640545e-01  3.342429e-04;
     9.106772e-01  9.584409e-01  3.037407e-04;
     9.273554e-01  9.522379e-01  2.760809e-04;
     9.432502e-01  9.452968e-01  2.509970e-04;
     9.582244e-01  9.374773e-01  2.282474e-04;
     9.721304e-01  9.286495e-01  2.076129e-04;
     9.849237e-01  9.187953e-01  1.888948e-04;
     9.970067e-01  9.083014e-01  1.719127e-04;
     1.008907e+00  8.976352e-01  1.565030e-04;
     1.021163e+00  8.872401e-01  1.425177e-04;
     1.034327e+00  8.775360e-01  1.298230e-04;
     1.048753e+00  8.687920e-01  1.182974e-04;
     1.063937e+00  8.607474e-01  1.078310e-04;
     1.079166e+00  8.530233e-01  9.832455e-05;
     1.093723e+00  8.452535e-01  8.968787e-05;
     1.106886e+00  8.370838e-01  8.183954e-05;
     1.118106e+00  8.282409e-01  7.470582e-05;
     1.127493e+00  8.187320e-01  6.821991e-05;
     1.135317e+00  8.086352e-01  6.232132e-05;
     1.141838e+00  7.980296e-01  5.695534e-05;
     1.147304e+00  7.869950e-01  5.207245e-05;
     1.151897e+00  7.756040e-01  4.762781e-05;
     1.155582e+00  7.638996e-01  4.358082e-05;
     1.158284e+00  7.519157e-01  3.989468e-05;
     1.159934e+00  7.396832e-01  3.653612e-05;
     1.160477e+00  7.272309e-01  3.347499e-05;
     1.159890e+00  7.145878e-01  3.068400e-05;
     1.158259e+00  7.017926e-01  2.813839e-05;
     1.155692e+00  6.888866e-01  2.581574e-05;
     1.152293e+00  6.759103e-01  2.369574e-05;
     1.148163e+00  6.629035e-01  2.175998e-05;
     1.143345e+00  6.498911e-01  1.999179e-05;
     1.137685e+00  6.368410e-01  1.837603e-05;
     1.130993e+00  6.237092e-01  1.689896e-05;
     1.123097e+00  6.104541e-01  1.554815e-05;
     1.113846e+00  5.970375e-01  1.431231e-05;
     1.103152e+00  5.834395e-01  1.318119e-05;
     1.091121e+00  5.697044e-01  1.214548e-05;
     1.077902e+00  5.558892e-01  1.119673e-05;
     1.063644e+00  5.420475e-01  1.032727e-05;
     1.048485e+00  5.282296e-01  9.530130e-06;
     1.032546e+00  5.144746e-01  8.798979e-06;
     1.015870e+00  5.007881e-01  8.128065e-06;
     9.984859e-01  4.871687e-01  7.512160e-06;
     9.804227e-01  4.736160e-01  6.946506e-06;
     9.617111e-01  4.601308e-01  6.426776e-06;
     9.424119e-01  4.467260e-01  0.000000e+00;
     9.227049e-01  4.334589e-01  0.000000e+00;
     9.027804e-01  4.203919e-01  0.000000e+00;
     8.828123e-01  4.075810e-01  0.000000e+00;
     8.629581e-01  3.950755e-01  0.000000e+00;
     8.432731e-01  3.828894e-01  0.000000e+00;
     8.234742e-01  3.709190e-01  0.000000e+00;
     8.032342e-01  3.590447e-01  0.000000e+00;
     7.822715e-01  3.471615e-01  0.000000e+00;
     7.603498e-01  3.351794e-01  0.000000e+00;
     7.373739e-01  3.230562e-01  0.000000e+00;
     7.136470e-01  3.108859e-01  0.000000e+00;
     6.895336e-01  2.987840e-01  0.000000e+00;
     6.653567e-01  2.868527e-01  0.000000e+00;
     6.413984e-01  2.751807e-01  0.000000e+00;
     6.178723e-01  2.638343e-01  0.000000e+00;
     5.948484e-01  2.528330e-01  0.000000e+00;
     5.723600e-01  2.421835e-01  0.000000e+00;
     5.504353e-01  2.318904e-01  0.000000e+00;
     5.290979e-01  2.219564e-01  0.000000e+00;
     5.083728e-01  2.123826e-01  0.000000e+00;
     4.883006e-01  2.031698e-01  0.000000e+00;
     4.689171e-01  1.943179e-01  0.000000e+00;
     4.502486e-01  1.858250e-01  0.000000e+00;
     4.323126e-01  1.776882e-01  0.000000e+00;
     4.150790e-01  1.698926e-01  0.000000e+00;
     3.983657e-01  1.623822e-01  0.000000e+00;
     3.819846e-01  1.550986e-01  0.000000e+00;
     3.657821e-01  1.479918e-01  0.000000e+00;
     3.496358e-01  1.410203e-01  0.000000e+00;
     3.334937e-01  1.341614e-01  0.000000e+00;
     3.174776e-01  1.274401e-01  0.000000e+00;
     3.017298e-01  1.208887e-01  0.000000e+00;
     2.863684e-01  1.145345e-01  0.000000e+00;
     2.714900e-01  1.083996e-01  0.000000e+00;
     2.571632e-01  1.025007e-01  0.000000e+00;
     2.434102e-01  9.684588e-02  0.000000e+00;
     2.302389e-01  9.143944e-02  0.000000e+00;
     2.176527e-01  8.628318e-02  0.000000e+00;
     2.056507e-01  8.137687e-02  0.000000e+00;
     1.942251e-01  7.671708e-02  0.000000e+00;
     1.833530e-01  7.229404e-02  0.000000e+00;
     1.730097e-01  6.809696e-02  0.000000e+00;
     1.631716e-01  6.411549e-02  0.000000e+00;
     1.538163e-01  6.033976e-02  0.000000e+00;
     1.449230e-01  5.676054e-02  0.000000e+00;
     1.364729e-01  5.336992e-02  0.000000e+00;
     1.284483e-01  5.016027e-02  0.000000e+00;
     1.208320e-01  4.712405e-02  0.000000e+00;
     1.136072e-01  4.425383e-02  0.000000e+00;
     1.067579e-01  4.154205e-02  0.000000e+00;
     1.002685e-01  3.898042e-02  0.000000e+00;
     9.412394e-02  3.656091e-02  0.000000e+00;
     8.830929e-02  3.427597e-02  0.000000e+00;
     8.281010e-02  3.211852e-02  0.000000e+00;
     7.761208e-02  3.008192e-02  0.000000e+00;
     7.270064e-02  2.816001e-02  0.000000e+00;
     6.806167e-02  2.634698e-02  0.000000e+00;
     6.368176e-02  2.463731e-02  0.000000e+00;
     5.954815e-02  2.302574e-02  0.000000e+00;
     5.564917e-02  2.150743e-02  0.000000e+00;
     5.197543e-02  2.007838e-02  0.000000e+00;
     4.851788e-02  1.873474e-02  0.000000e+00;
     4.526737e-02  1.747269e-02  0.000000e+00;
     4.221473e-02  1.628841e-02  0.000000e+00;
     3.934954e-02  1.517767e-02  0.000000e+00;
     3.665730e-02  1.413473e-02  0.000000e+00;
     3.412407e-02  1.315408e-02  0.000000e+00;
     3.173768e-02  1.223092e-02  0.000000e+00;
     2.948752e-02  1.136106e-02  0.000000e+00;
     2.736717e-02  1.054190e-02  0.000000e+00;
     2.538113e-02  9.775050e-03  0.000000e+00;
     2.353356e-02  9.061962e-03  0.000000e+00;
     2.182558e-02  8.402962e-03  0.000000e+00;
     2.025590e-02  7.797457e-03  0.000000e+00;
     1.881892e-02  7.243230e-03  0.000000e+00;
     1.749930e-02  6.734381e-03  0.000000e+00;
     1.628167e-02  6.265001e-03  0.000000e+00;
     1.515301e-02  5.830085e-03  0.000000e+00;
     1.410230e-02  5.425391e-03  0.000000e+00;
     1.312106e-02  5.047634e-03  0.000000e+00;
     1.220509e-02  4.695140e-03  0.000000e+00;
     1.135114e-02  4.366592e-03  0.000000e+00;
     1.055593e-02  4.060685e-03  0.000000e+00;
     9.816228e-03  3.776140e-03  0.000000e+00;
     9.128517e-03  3.511578e-03  0.000000e+00;
     8.488116e-03  3.265211e-03  0.000000e+00;
     7.890589e-03  3.035344e-03  0.000000e+00;
     7.332061e-03  2.820496e-03  0.000000e+00;
     6.809147e-03  2.619372e-03  0.000000e+00;
     6.319204e-03  2.430960e-03  0.000000e+00;
     5.861036e-03  2.254796e-03  0.000000e+00;
     5.433624e-03  2.090489e-03  0.000000e+00;
     5.035802e-03  1.937586e-03  0.000000e+00;
     4.666298e-03  1.795595e-03  0.000000e+00;
     4.323750e-03  1.663989e-03  0.000000e+00;
     4.006709e-03  1.542195e-03  0.000000e+00;
     3.713708e-03  1.429639e-03  0.000000e+00;
     3.443294e-03  1.325752e-03  0.000000e+00;
     3.194041e-03  1.229980e-03  0.000000e+00;
     2.964424e-03  1.141734e-03  0.000000e+00;
     2.752492e-03  1.060269e-03  0.000000e+00;
     2.556406e-03  9.848854e-04  0.000000e+00;
     2.374564e-03  9.149703e-04  0.000000e+00;
     2.205568e-03  8.499903e-04  0.000000e+00;
     2.048294e-03  7.895158e-04  0.000000e+00;
     1.902113e-03  7.333038e-04  0.000000e+00;
     1.766485e-03  6.811458e-04  0.000000e+00;
     1.640857e-03  6.328287e-04  0.000000e+00;
     1.524672e-03  5.881375e-04  0.000000e+00;
     1.417322e-03  5.468389e-04  0.000000e+00;
     1.318031e-03  5.086349e-04  0.000000e+00;
     1.226059e-03  4.732403e-04  0.000000e+00;
     1.140743e-03  4.404016e-04  0.000000e+00;
     1.061495e-03  4.098928e-04  0.000000e+00;
     9.877949e-04  3.815137e-04  0.000000e+00;
     9.191847e-04  3.550902e-04  0.000000e+00;
     8.552568e-04  3.304668e-04  0.000000e+00;
     7.956433e-04  3.075030e-04  0.000000e+00;
     7.400120e-04  2.860718e-04  0.000000e+00;
     6.880980e-04  2.660718e-04  0.000000e+00;
     6.397864e-04  2.474586e-04  0.000000e+00;
     5.949726e-04  2.301919e-04  0.000000e+00;
     5.535291e-04  2.142225e-04  0.000000e+00;
     5.153113e-04  1.994949e-04  0.000000e+00;
     4.801234e-04  1.859336e-04  0.000000e+00;
     4.476245e-04  1.734067e-04  0.000000e+00;
     4.174846e-04  1.617865e-04  0.000000e+00;
     3.894221e-04  1.509641e-04  0.000000e+00;
     3.631969e-04  1.408466e-04  0.000000e+00;
     3.386279e-04  1.313642e-04  0.000000e+00;
     3.156452e-04  1.224905e-04  0.000000e+00;
     2.941966e-04  1.142060e-04  0.000000e+00;
     2.742235e-04  1.064886e-04  0.000000e+00;
     2.556624e-04  9.931439e-05  0.000000e+00;
     2.384390e-04  9.265512e-05  0.000000e+00;
     2.224525e-04  8.647225e-05  0.000000e+00;
     2.076036e-04  8.072780e-05  0.000000e+00;
     1.938018e-04  7.538716e-05  0.000000e+00;
     1.809649e-04  7.041878e-05  0.000000e+00;
     1.690167e-04  6.579338e-05  0.000000e+00;
     1.578839e-04  6.148250e-05  0.000000e+00;
     1.474993e-04  5.746008e-05  0.000000e+00;
     1.378026e-04  5.370272e-05  0.000000e+00;
     1.287394e-04  5.018934e-05  0.000000e+00;
     1.202644e-04  4.690245e-05  0.000000e+00;
     1.123502e-04  4.383167e-05  0.000000e+00;
     1.049725e-04  4.096780e-05  0.000000e+00;
     9.810596e-05  3.830123e-05  0.000000e+00;
     9.172477e-05  3.582218e-05  0.000000e+00;
     8.579861e-05  3.351903e-05  0.000000e+00;
     8.028174e-05  3.137419e-05  0.000000e+00;
     7.513013e-05  2.937068e-05  0.000000e+00;
     7.030565e-05  2.749380e-05  0.000000e+00;
     6.577532e-05  2.573083e-05  0.000000e+00;
     6.151508e-05  2.407249e-05  0.000000e+00;
     5.752025e-05  2.251704e-05  0.000000e+00;
     5.378813e-05  2.106350e-05  0.000000e+00;
     5.031350e-05  1.970991e-05  0.000000e+00;
     4.708916e-05  1.845353e-05  0.000000e+00;
     4.410322e-05  1.728979e-05  0.000000e+00;
     4.133150e-05  1.620928e-05  0.000000e+00;
     3.874992e-05  1.520262e-05  0.000000e+00;
     3.633762e-05  1.426169e-05  0.000000e+00;
     3.407653e-05  1.337946e-05  0.000000e+00]
