" Author:   Nate Kane <nathanaelkane AT gmail DOT com>
" Homepage: http://github.com/preservim/vim-indent-guides

"
" Return hex string equivalent to given decimal string or number.
"
" Example: color_helper#dec_to_hex(255, 2)
" Returns: 'FF'
"
" Example: color_helper#dec_to_hex(255, 5)
" Returns: '000FF'
"
function! color_helper#dec_to_hex(arg, padding) abort
  return toupper(printf('%0' . a:padding . 'x', a:arg + 0))
endfunction

"
" Return number equivalent to given hex string ('0x' is optional).
"
" Example: color_helper#hex_to_dec('FF')
" Returns: 255
"
" Example: color_helper#hex_to_dec('88')
" Returns: 136
"
" Example: color_helper#hex_to_dec('00')
" Returns: 0
"
function! color_helper#hex_to_dec(arg) abort
  return (a:arg =~? '^0x') ? a:arg + 0 : ('0x'.a:arg) + 0
endfunction

"
" Converts a given hex color string into an rgb list (eg. [red, green, blue]).
"
" Example: color_helper#hex_color_to_rgb('#0088FF')
" Returns: [0, 136, 255]
"
function! color_helper#hex_color_to_rgb(hex_color) abort
  let l:rgb = []

  if a:hex_color =~ g:indent_guides_color_hex_pattern
    let l:red   = color_helper#hex_to_dec(strpart(a:hex_color, 1, 2))
    let l:green = color_helper#hex_to_dec(strpart(a:hex_color, 3, 2))
    let l:blue  = color_helper#hex_to_dec(strpart(a:hex_color, 5, 2))
    let l:rgb = [l:red, l:green, l:blue]
  end

  return l:rgb
endfunction

"
" Converts a given rgb list (eg. [red, green, blue]) into a hex color string.
"
" Example: color_helper#rgb_color_to_hex([0, 136, 255])
" Returns: '#0088FF'
"
function! color_helper#rgb_color_to_hex(rgb_color) abort
  let l:hex_color  = '#'
  let l:hex_color .= color_helper#dec_to_hex(a:rgb_color[0], 2) " red
  let l:hex_color .= color_helper#dec_to_hex(a:rgb_color[1], 2) " green
  let l:hex_color .= color_helper#dec_to_hex(a:rgb_color[2], 2) " blue

  return l:hex_color
endfunction

"
" Returns a ligtened color using the given color and the percent to lighten it
" by.
"
" Example: color_helper#hex_color_lighten('#000000', 0.10)
" Returns: '#191919'
"
function! color_helper#hex_color_lighten(color, percent) abort
  let l:rgb = color_helper#hex_color_to_rgb(a:color)
  let l:rgb_lightened = []

  for i in l:rgb
    call add(l:rgb_lightened, float2nr(i + ((255 - i) * a:percent)))
  endfor

  return color_helper#rgb_color_to_hex(l:rgb_lightened)
endfunction

"
" Returns a darkened color using the given color and the percent to darken it
" by.
"
" Example: color_helper#hex_color_darken('#FFFFFF', 0.10)
" Returns: '#E5E5E5'
"
function! color_helper#hex_color_darken(color, percent) abort
  let l:rgb = color_helper#hex_color_to_rgb(a:color)
  let l:rgb_darkened = []

  for i in l:rgb
    call add(l:rgb_darkened, float2nr(i * (1 - a:percent)))
  endfor

  return color_helper#rgb_color_to_hex(l:rgb_darkened)
endfunction

"
" Returns a hex color code for the given color name.
"
" Example: color_helper#color_name_to_hex('darkslategray')
" Returns: '#2F4F4F'
"
function! color_helper#color_name_to_hex(color_name) abort
  let l:hex_code   = ''
  let l:color_name = tolower(a:color_name)

  let l:color_list = {
    \ 'alice blue'             : '#F0F8FF',
    \ 'aliceblue'              : '#F0F8FF',
    \ 'antique white'          : '#FAEBD7',
    \ 'antiquewhite'           : '#FAEBD7',
    \ 'antiquewhite1'          : '#FFEFDB',
    \ 'antiquewhite2'          : '#EEDFCC',
    \ 'antiquewhite3'          : '#CDC0B0',
    \ 'antiquewhite4'          : '#8B8378',
    \ 'aquamarine'             : '#7FFFD4',
    \ 'aquamarine1'            : '#7FFFD4',
    \ 'aquamarine2'            : '#76EEC6',
    \ 'aquamarine3'            : '#66CDAA',
    \ 'aquamarine4'            : '#458B74',
    \ 'azure'                  : '#F0FFFF',
    \ 'azure1'                 : '#F0FFFF',
    \ 'azure2'                 : '#E0EEEE',
    \ 'azure3'                 : '#C1CDCD',
    \ 'azure4'                 : '#838B8B',
    \ 'beige'                  : '#F5F5DC',
    \ 'bisque'                 : '#FFE4C4',
    \ 'bisque1'                : '#FFE4C4',
    \ 'bisque2'                : '#EED5B7',
    \ 'bisque3'                : '#CDB79E',
    \ 'bisque4'                : '#8B7D6B',
    \ 'black'                  : '#000000',
    \ 'blanched almond'        : '#FFEBCD',
    \ 'blanchedalmond'         : '#FFEBCD',
    \ 'blue violet'            : '#8A2BE2',
    \ 'blue'                   : '#0000FF',
    \ 'blue1'                  : '#0000FF',
    \ 'blue2'                  : '#0000EE',
    \ 'blue3'                  : '#0000CD',
    \ 'blue4'                  : '#00008B',
    \ 'blueviolet'             : '#8A2BE2',
    \ 'brown'                  : '#A52A2A',
    \ 'brown1'                 : '#FF4040',
    \ 'brown2'                 : '#EE3B3B',
    \ 'brown3'                 : '#CD3333',
    \ 'brown4'                 : '#8B2323',
    \ 'burlywood'              : '#DEB887',
    \ 'burlywood1'             : '#FFD39B',
    \ 'burlywood2'             : '#EEC591',
    \ 'burlywood3'             : '#CDAA7D',
    \ 'burlywood4'             : '#8B7355',
    \ 'cadet blue'             : '#5F9EA0',
    \ 'cadetblue'              : '#5F9EA0',
    \ 'cadetblue1'             : '#98F5FF',
    \ 'cadetblue2'             : '#8EE5EE',
    \ 'cadetblue3'             : '#7AC5CD',
    \ 'cadetblue4'             : '#53868B',
    \ 'chartreuse'             : '#7FFF00',
    \ 'chartreuse1'            : '#7FFF00',
    \ 'chartreuse2'            : '#76EE00',
    \ 'chartreuse3'            : '#66CD00',
    \ 'chartreuse4'            : '#458B00',
    \ 'chocolate'              : '#D2691E',
    \ 'chocolate1'             : '#FF7F24',
    \ 'chocolate2'             : '#EE7621',
    \ 'chocolate3'             : '#CD661D',
    \ 'chocolate4'             : '#8B4513',
    \ 'coral'                  : '#FF7F50',
    \ 'coral1'                 : '#FF7256',
    \ 'coral2'                 : '#EE6A50',
    \ 'coral3'                 : '#CD5B45',
    \ 'coral4'                 : '#8B3E2F',
    \ 'cornflower blue'        : '#6495ED',
    \ 'cornflowerblue'         : '#6495ED',
    \ 'cornsilk'               : '#FFF8DC',
    \ 'cornsilk1'              : '#FFF8DC',
    \ 'cornsilk2'              : '#EEE8CD',
    \ 'cornsilk3'              : '#CDC8B1',
    \ 'cornsilk4'              : '#8B8878',
    \ 'cyan'                   : '#00FFFF',
    \ 'cyan1'                  : '#00FFFF',
    \ 'cyan2'                  : '#00EEEE',
    \ 'cyan3'                  : '#00CDCD',
    \ 'cyan4'                  : '#008B8B',
    \ 'dark blue'              : '#00008B',
    \ 'dark cyan'              : '#008B8B',
    \ 'dark goldenrod'         : '#B8860B',
    \ 'dark gray'              : '#A9A9A9',
    \ 'dark green'             : '#006400',
    \ 'dark grey'              : '#A9A9A9',
    \ 'dark khaki'             : '#BDB76B',
    \ 'dark magenta'           : '#8B008B',
    \ 'dark olive green'       : '#556B2F',
    \ 'dark orange'            : '#FF8C00',
    \ 'dark orchid'            : '#9932CC',
    \ 'dark red'               : '#8B0000',
    \ 'dark salmon'            : '#E9967A',
    \ 'dark sea green'         : '#8FBC8F',
    \ 'dark slate blue'        : '#483D8B',
    \ 'dark slate gray'        : '#2F4F4F',
    \ 'dark slate grey'        : '#2F4F4F',
    \ 'dark turquoise'         : '#00CED1',
    \ 'dark violet'            : '#9400D3',
    \ 'dark yellow'            : '#BBBB00',
    \ 'darkblue'               : '#00008B',
    \ 'darkcyan'               : '#008B8B',
    \ 'darkgoldenrod'          : '#B8860B',
    \ 'darkgoldenrod1'         : '#FFB90F',
    \ 'darkgoldenrod2'         : '#EEAD0E',
    \ 'darkgoldenrod3'         : '#CD950C',
    \ 'darkgoldenrod4'         : '#8B6508',
    \ 'darkgray'               : '#A9A9A9',
    \ 'darkgreen'              : '#006400',
    \ 'darkgrey'               : '#A9A9A9',
    \ 'darkkhaki'              : '#BDB76B',
    \ 'darkmagenta'            : '#8B008B',
    \ 'darkolivegreen'         : '#556B2F',
    \ 'darkolivegreen1'        : '#CAFF70',
    \ 'darkolivegreen2'        : '#BCEE68',
    \ 'darkolivegreen3'        : '#A2CD5A',
    \ 'darkolivegreen4'        : '#6E8B3D',
    \ 'darkorange'             : '#FF8C00',
    \ 'darkorange1'            : '#FF7F00',
    \ 'darkorange2'            : '#EE7600',
    \ 'darkorange3'            : '#CD6600',
    \ 'darkorange4'            : '#8B4500',
    \ 'darkorchid'             : '#9932CC',
    \ 'darkorchid1'            : '#BF3EFF',
    \ 'darkorchid2'            : '#B23AEE',
    \ 'darkorchid3'            : '#9A32CD',
    \ 'darkorchid4'            : '#68228B',
    \ 'darkred'                : '#8B0000',
    \ 'darksalmon'             : '#E9967A',
    \ 'darkseagreen'           : '#8FBC8F',
    \ 'darkseagreen1'          : '#C1FFC1',
    \ 'darkseagreen2'          : '#B4EEB4',
    \ 'darkseagreen3'          : '#9BCD9B',
    \ 'darkseagreen4'          : '#698B69',
    \ 'darkslateblue'          : '#483D8B',
    \ 'darkslategray'          : '#2F4F4F',
    \ 'darkslategray1'         : '#97FFFF',
    \ 'darkslategray2'         : '#8DEEEE',
    \ 'darkslategray3'         : '#79CDCD',
    \ 'darkslategray4'         : '#528B8B',
    \ 'darkslategrey'          : '#2F4F4F',
    \ 'darkturquoise'          : '#00CED1',
    \ 'darkviolet'             : '#9400D3',
    \ 'darkyellow'             : '#BBBB00',
    \ 'deep pink'              : '#FF1493',
    \ 'deep sky blue'          : '#00BFFF',
    \ 'deeppink'               : '#FF1493',
    \ 'deeppink1'              : '#FF1493',
    \ 'deeppink2'              : '#EE1289',
    \ 'deeppink3'              : '#CD1076',
    \ 'deeppink4'              : '#8B0A50',
    \ 'deepskyblue'            : '#00BFFF',
    \ 'deepskyblue1'           : '#00BFFF',
    \ 'deepskyblue2'           : '#00B2EE',
    \ 'deepskyblue3'           : '#009ACD',
    \ 'deepskyblue4'           : '#00688B',
    \ 'dim gray'               : '#696969',
    \ 'dim grey'               : '#696969',
    \ 'dimgray'                : '#696969',
    \ 'dimgrey'                : '#696969',
    \ 'dodger blue'            : '#1E90FF',
    \ 'dodgerblue'             : '#1E90FF',
    \ 'dodgerblue1'            : '#1E90FF',
    \ 'dodgerblue2'            : '#1C86EE',
    \ 'dodgerblue3'            : '#1874CD',
    \ 'dodgerblue4'            : '#104E8B',
    \ 'firebrick'              : '#B22222',
    \ 'firebrick1'             : '#FF3030',
    \ 'firebrick2'             : '#EE2C2C',
    \ 'firebrick3'             : '#CD2626',
    \ 'firebrick4'             : '#8B1A1A',
    \ 'floral white'           : '#FFFAF0',
    \ 'floralwhite'            : '#FFFAF0',
    \ 'forest green'           : '#228B22',
    \ 'forestgreen'            : '#228B22',
    \ 'gainsboro'              : '#DCDCDC',
    \ 'ghost white'            : '#F8F8FF',
    \ 'ghostwhite'             : '#F8F8FF',
    \ 'gold'                   : '#FFD700',
    \ 'gold1'                  : '#FFD700',
    \ 'gold2'                  : '#EEC900',
    \ 'gold3'                  : '#CDAD00',
    \ 'gold4'                  : '#8B7500',
    \ 'goldenrod'              : '#DAA520',
    \ 'goldenrod1'             : '#FFC125',
    \ 'goldenrod2'             : '#EEB422',
    \ 'goldenrod3'             : '#CD9B1D',
    \ 'goldenrod4'             : '#8B6914',
    \ 'gray'                   : '#BEBEBE',
    \ 'gray0'                  : '#000000',
    \ 'gray1'                  : '#030303',
    \ 'gray10'                 : '#1A1A1A',
    \ 'gray100'                : '#FFFFFF',
    \ 'gray11'                 : '#1C1C1C',
    \ 'gray12'                 : '#1F1F1F',
    \ 'gray13'                 : '#212121',
    \ 'gray14'                 : '#242424',
    \ 'gray15'                 : '#262626',
    \ 'gray16'                 : '#292929',
    \ 'gray17'                 : '#2B2B2B',
    \ 'gray18'                 : '#2E2E2E',
    \ 'gray19'                 : '#303030',
    \ 'gray2'                  : '#050505',
    \ 'gray20'                 : '#333333',
    \ 'gray21'                 : '#363636',
    \ 'gray22'                 : '#383838',
    \ 'gray23'                 : '#3B3B3B',
    \ 'gray24'                 : '#3D3D3D',
    \ 'gray25'                 : '#404040',
    \ 'gray26'                 : '#424242',
    \ 'gray27'                 : '#454545',
    \ 'gray28'                 : '#474747',
    \ 'gray29'                 : '#4A4A4A',
    \ 'gray3'                  : '#080808',
    \ 'gray30'                 : '#4D4D4D',
    \ 'gray31'                 : '#4F4F4F',
    \ 'gray32'                 : '#525252',
    \ 'gray33'                 : '#545454',
    \ 'gray34'                 : '#575757',
    \ 'gray35'                 : '#595959',
    \ 'gray36'                 : '#5C5C5C',
    \ 'gray37'                 : '#5E5E5E',
    \ 'gray38'                 : '#616161',
    \ 'gray39'                 : '#636363',
    \ 'gray4'                  : '#0A0A0A',
    \ 'gray40'                 : '#666666',
    \ 'gray41'                 : '#696969',
    \ 'gray42'                 : '#6B6B6B',
    \ 'gray43'                 : '#6E6E6E',
    \ 'gray44'                 : '#707070',
    \ 'gray45'                 : '#737373',
    \ 'gray46'                 : '#757575',
    \ 'gray47'                 : '#787878',
    \ 'gray48'                 : '#7A7A7A',
    \ 'gray49'                 : '#7D7D7D',
    \ 'gray5'                  : '#0D0D0D',
    \ 'gray50'                 : '#7F7F7F',
    \ 'gray51'                 : '#828282',
    \ 'gray52'                 : '#858585',
    \ 'gray53'                 : '#878787',
    \ 'gray54'                 : '#8A8A8A',
    \ 'gray55'                 : '#8C8C8C',
    \ 'gray56'                 : '#8F8F8F',
    \ 'gray57'                 : '#919191',
    \ 'gray58'                 : '#949494',
    \ 'gray59'                 : '#969696',
    \ 'gray6'                  : '#0F0F0F',
    \ 'gray60'                 : '#999999',
    \ 'gray61'                 : '#9C9C9C',
    \ 'gray62'                 : '#9E9E9E',
    \ 'gray63'                 : '#A1A1A1',
    \ 'gray64'                 : '#A3A3A3',
    \ 'gray65'                 : '#A6A6A6',
    \ 'gray66'                 : '#A8A8A8',
    \ 'gray67'                 : '#ABABAB',
    \ 'gray68'                 : '#ADADAD',
    \ 'gray69'                 : '#B0B0B0',
    \ 'gray7'                  : '#121212',
    \ 'gray70'                 : '#B3B3B3',
    \ 'gray71'                 : '#B5B5B5',
    \ 'gray72'                 : '#B8B8B8',
    \ 'gray73'                 : '#BABABA',
    \ 'gray74'                 : '#BDBDBD',
    \ 'gray75'                 : '#BFBFBF',
    \ 'gray76'                 : '#C2C2C2',
    \ 'gray77'                 : '#C4C4C4',
    \ 'gray78'                 : '#C7C7C7',
    \ 'gray79'                 : '#C9C9C9',
    \ 'gray8'                  : '#141414',
    \ 'gray80'                 : '#CCCCCC',
    \ 'gray81'                 : '#CFCFCF',
    \ 'gray82'                 : '#D1D1D1',
    \ 'gray83'                 : '#D4D4D4',
    \ 'gray84'                 : '#D6D6D6',
    \ 'gray85'                 : '#D9D9D9',
    \ 'gray86'                 : '#DBDBDB',
    \ 'gray87'                 : '#DEDEDE',
    \ 'gray88'                 : '#E0E0E0',
    \ 'gray89'                 : '#E3E3E3',
    \ 'gray9'                  : '#171717',
    \ 'gray90'                 : '#E5E5E5',
    \ 'gray91'                 : '#E8E8E8',
    \ 'gray92'                 : '#EBEBEB',
    \ 'gray93'                 : '#EDEDED',
    \ 'gray94'                 : '#F0F0F0',
    \ 'gray95'                 : '#F2F2F2',
    \ 'gray96'                 : '#F5F5F5',
    \ 'gray97'                 : '#F7F7F7',
    \ 'gray98'                 : '#FAFAFA',
    \ 'gray99'                 : '#FCFCFC',
    \ 'green yellow'           : '#ADFF2F',
    \ 'green'                  : '#00FF00',
    \ 'green1'                 : '#00FF00',
    \ 'green2'                 : '#00EE00',
    \ 'green3'                 : '#00CD00',
    \ 'green4'                 : '#008B00',
    \ 'greenyellow'            : '#ADFF2F',
    \ 'grey'                   : '#BEBEBE',
    \ 'grey0'                  : '#000000',
    \ 'grey1'                  : '#030303',
    \ 'grey10'                 : '#1A1A1A',
    \ 'grey100'                : '#FFFFFF',
    \ 'grey11'                 : '#1C1C1C',
    \ 'grey12'                 : '#1F1F1F',
    \ 'grey13'                 : '#212121',
    \ 'grey14'                 : '#242424',
    \ 'grey15'                 : '#262626',
    \ 'grey16'                 : '#292929',
    \ 'grey17'                 : '#2B2B2B',
    \ 'grey18'                 : '#2E2E2E',
    \ 'grey19'                 : '#303030',
    \ 'grey2'                  : '#050505',
    \ 'grey20'                 : '#333333',
    \ 'grey21'                 : '#363636',
    \ 'grey22'                 : '#383838',
    \ 'grey23'                 : '#3B3B3B',
    \ 'grey24'                 : '#3D3D3D',
    \ 'grey25'                 : '#404040',
    \ 'grey26'                 : '#424242',
    \ 'grey27'                 : '#454545',
    \ 'grey28'                 : '#474747',
    \ 'grey29'                 : '#4A4A4A',
    \ 'grey3'                  : '#080808',
    \ 'grey30'                 : '#4D4D4D',
    \ 'grey31'                 : '#4F4F4F',
    \ 'grey32'                 : '#525252',
    \ 'grey33'                 : '#545454',
    \ 'grey34'                 : '#575757',
    \ 'grey35'                 : '#595959',
    \ 'grey36'                 : '#5C5C5C',
    \ 'grey37'                 : '#5E5E5E',
    \ 'grey38'                 : '#616161',
    \ 'grey39'                 : '#636363',
    \ 'grey4'                  : '#0A0A0A',
    \ 'grey40'                 : '#666666',
    \ 'grey41'                 : '#696969',
    \ 'grey42'                 : '#6B6B6B',
    \ 'grey43'                 : '#6E6E6E',
    \ 'grey44'                 : '#707070',
    \ 'grey45'                 : '#737373',
    \ 'grey46'                 : '#757575',
    \ 'grey47'                 : '#787878',
    \ 'grey48'                 : '#7A7A7A',
    \ 'grey49'                 : '#7D7D7D',
    \ 'grey5'                  : '#0D0D0D',
    \ 'grey50'                 : '#7F7F7F',
    \ 'grey51'                 : '#828282',
    \ 'grey52'                 : '#858585',
    \ 'grey53'                 : '#878787',
    \ 'grey54'                 : '#8A8A8A',
    \ 'grey55'                 : '#8C8C8C',
    \ 'grey56'                 : '#8F8F8F',
    \ 'grey57'                 : '#919191',
    \ 'grey58'                 : '#949494',
    \ 'grey59'                 : '#969696',
    \ 'grey6'                  : '#0F0F0F',
    \ 'grey60'                 : '#999999',
    \ 'grey61'                 : '#9C9C9C',
    \ 'grey62'                 : '#9E9E9E',
    \ 'grey63'                 : '#A1A1A1',
    \ 'grey64'                 : '#A3A3A3',
    \ 'grey65'                 : '#A6A6A6',
    \ 'grey66'                 : '#A8A8A8',
    \ 'grey67'                 : '#ABABAB',
    \ 'grey68'                 : '#ADADAD',
    \ 'grey69'                 : '#B0B0B0',
    \ 'grey7'                  : '#121212',
    \ 'grey70'                 : '#B3B3B3',
    \ 'grey71'                 : '#B5B5B5',
    \ 'grey72'                 : '#B8B8B8',
    \ 'grey73'                 : '#BABABA',
    \ 'grey74'                 : '#BDBDBD',
    \ 'grey75'                 : '#BFBFBF',
    \ 'grey76'                 : '#C2C2C2',
    \ 'grey77'                 : '#C4C4C4',
    \ 'grey78'                 : '#C7C7C7',
    \ 'grey79'                 : '#C9C9C9',
    \ 'grey8'                  : '#141414',
    \ 'grey80'                 : '#CCCCCC',
    \ 'grey81'                 : '#CFCFCF',
    \ 'grey82'                 : '#D1D1D1',
    \ 'grey83'                 : '#D4D4D4',
    \ 'grey84'                 : '#D6D6D6',
    \ 'grey85'                 : '#D9D9D9',
    \ 'grey86'                 : '#DBDBDB',
    \ 'grey87'                 : '#DEDEDE',
    \ 'grey88'                 : '#E0E0E0',
    \ 'grey89'                 : '#E3E3E3',
    \ 'grey9'                  : '#171717',
    \ 'grey90'                 : '#E5E5E5',
    \ 'grey91'                 : '#E8E8E8',
    \ 'grey92'                 : '#EBEBEB',
    \ 'grey93'                 : '#EDEDED',
    \ 'grey94'                 : '#F0F0F0',
    \ 'grey95'                 : '#F2F2F2',
    \ 'grey96'                 : '#F5F5F5',
    \ 'grey97'                 : '#F7F7F7',
    \ 'grey98'                 : '#FAFAFA',
    \ 'grey99'                 : '#FCFCFC',
    \ 'honeydew'               : '#F0FFF0',
    \ 'honeydew1'              : '#F0FFF0',
    \ 'honeydew2'              : '#E0EEE0',
    \ 'honeydew3'              : '#C1CDC1',
    \ 'honeydew4'              : '#838B83',
    \ 'hot pink'               : '#FF69B4',
    \ 'hotpink'                : '#FF69B4',
    \ 'hotpink1'               : '#FF6EB4',
    \ 'hotpink2'               : '#EE6AA7',
    \ 'hotpink3'               : '#CD6090',
    \ 'hotpink4'               : '#8B3A62',
    \ 'indian red'             : '#CD5C5C',
    \ 'indianred'              : '#CD5C5C',
    \ 'indianred1'             : '#FF6A6A',
    \ 'indianred2'             : '#EE6363',
    \ 'indianred3'             : '#CD5555',
    \ 'indianred4'             : '#8B3A3A',
    \ 'ivory'                  : '#FFFFF0',
    \ 'ivory1'                 : '#FFFFF0',
    \ 'ivory2'                 : '#EEEEE0',
    \ 'ivory3'                 : '#CDCDC1',
    \ 'ivory4'                 : '#8B8B83',
    \ 'khaki'                  : '#F0E68C',
    \ 'khaki1'                 : '#FFF68F',
    \ 'khaki2'                 : '#EEE685',
    \ 'khaki3'                 : '#CDC673',
    \ 'khaki4'                 : '#8B864E',
    \ 'lavender blush'         : '#FFF0F5',
    \ 'lavender'               : '#E6E6FA',
    \ 'lavenderblush'          : '#FFF0F5',
    \ 'lavenderblush1'         : '#FFF0F5',
    \ 'lavenderblush2'         : '#EEE0E5',
    \ 'lavenderblush3'         : '#CDC1C5',
    \ 'lavenderblush4'         : '#8B8386',
    \ 'lawn green'             : '#7CFC00',
    \ 'lawngreen'              : '#7CFC00',
    \ 'lemon chiffon'          : '#FFFACD',
    \ 'lemonchiffon'           : '#FFFACD',
    \ 'lemonchiffon1'          : '#FFFACD',
    \ 'lemonchiffon2'          : '#EEE9BF',
    \ 'lemonchiffon3'          : '#CDC9A5',
    \ 'lemonchiffon4'          : '#8B8970',
    \ 'light blue'             : '#ADD8E6',
    \ 'light coral'            : '#F08080',
    \ 'light cyan'             : '#E0FFFF',
    \ 'light goldenrod yellow' : '#FAFAD2',
    \ 'light goldenrod'        : '#EEDD82',
    \ 'light gray'             : '#D3D3D3',
    \ 'light green'            : '#90EE90',
    \ 'light grey'             : '#D3D3D3',
    \ 'light magenta'          : '#FFBBFF',
    \ 'light pink'             : '#FFB6C1',
    \ 'light red'              : '#FFBBBB',
    \ 'light salmon'           : '#FFA07A',
    \ 'light sea green'        : '#20B2AA',
    \ 'light sky blue'         : '#87CEFA',
    \ 'light slate blue'       : '#8470FF',
    \ 'light slate gray'       : '#778899',
    \ 'light slate grey'       : '#778899',
    \ 'light steel blue'       : '#B0C4DE',
    \ 'light yellow'           : '#FFFFE0',
    \ 'lightblue'              : '#ADD8E6',
    \ 'lightblue1'             : '#BFEFFF',
    \ 'lightblue2'             : '#B2DFEE',
    \ 'lightblue3'             : '#9AC0CD',
    \ 'lightblue4'             : '#68838B',
    \ 'lightcoral'             : '#F08080',
    \ 'lightcyan'              : '#E0FFFF',
    \ 'lightcyan1'             : '#E0FFFF',
    \ 'lightcyan2'             : '#D1EEEE',
    \ 'lightcyan3'             : '#B4CDCD',
    \ 'lightcyan4'             : '#7A8B8B',
    \ 'lightgoldenrod'         : '#EEDD82',
    \ 'lightgoldenrod1'        : '#FFEC8B',
    \ 'lightgoldenrod2'        : '#EEDC82',
    \ 'lightgoldenrod3'        : '#CDBE70',
    \ 'lightgoldenrod4'        : '#8B814C',
    \ 'lightgoldenrodyellow'   : '#FAFAD2',
    \ 'lightgray'              : '#D3D3D3',
    \ 'lightgreen'             : '#90EE90',
    \ 'lightgrey'              : '#D3D3D3',
    \ 'lightmagenta'           : '#FFBBFF',
    \ 'lightpink'              : '#FFB6C1',
    \ 'lightpink1'             : '#FFAEB9',
    \ 'lightpink2'             : '#EEA2AD',
    \ 'lightpink3'             : '#CD8C95',
    \ 'lightpink4'             : '#8B5F65',
    \ 'lightred'               : '#FFBBBB',
    \ 'lightsalmon'            : '#FFA07A',
    \ 'lightsalmon1'           : '#FFA07A',
    \ 'lightsalmon2'           : '#EE9572',
    \ 'lightsalmon3'           : '#CD8162',
    \ 'lightsalmon4'           : '#8B5742',
    \ 'lightseagreen'          : '#20B2AA',
    \ 'lightskyblue'           : '#87CEFA',
    \ 'lightskyblue1'          : '#B0E2FF',
    \ 'lightskyblue2'          : '#A4D3EE',
    \ 'lightskyblue3'          : '#8DB6CD',
    \ 'lightskyblue4'          : '#607B8B',
    \ 'lightslateblue'         : '#8470FF',
    \ 'lightslategray'         : '#778899',
    \ 'lightslategrey'         : '#778899',
    \ 'lightsteelblue'         : '#B0C4DE',
    \ 'lightsteelblue1'        : '#CAE1FF',
    \ 'lightsteelblue2'        : '#BCD2EE',
    \ 'lightsteelblue3'        : '#A2B5CD',
    \ 'lightsteelblue4'        : '#6E7B8B',
    \ 'lightyellow'            : '#FFFFE0',
    \ 'lightyellow1'           : '#FFFFE0',
    \ 'lightyellow2'           : '#EEEED1',
    \ 'lightyellow3'           : '#CDCDB4',
    \ 'lightyellow4'           : '#8B8B7A',
    \ 'lime green'             : '#32CD32',
    \ 'limegreen'              : '#32CD32',
    \ 'linen'                  : '#FAF0E6',
    \ 'magenta'                : '#FF00FF',
    \ 'magenta1'               : '#FF00FF',
    \ 'magenta2'               : '#EE00EE',
    \ 'magenta3'               : '#CD00CD',
    \ 'magenta4'               : '#8B008B',
    \ 'maroon'                 : '#B03060',
    \ 'maroon1'                : '#FF34B3',
    \ 'maroon2'                : '#EE30A7',
    \ 'maroon3'                : '#CD2990',
    \ 'maroon4'                : '#8B1C62',
    \ 'medium aquamarine'      : '#66CDAA',
    \ 'medium blue'            : '#0000CD',
    \ 'medium orchid'          : '#BA55D3',
    \ 'medium purple'          : '#9370DB',
    \ 'medium sea green'       : '#3CB371',
    \ 'medium slate blue'      : '#7B68EE',
    \ 'medium spring green'    : '#00FA9A',
    \ 'medium turquoise'       : '#48D1CC',
    \ 'medium violet red'      : '#C71585',
    \ 'mediumaquamarine'       : '#66CDAA',
    \ 'mediumblue'             : '#0000CD',
    \ 'mediumorchid'           : '#BA55D3',
    \ 'mediumorchid1'          : '#E066FF',
    \ 'mediumorchid2'          : '#D15FEE',
    \ 'mediumorchid3'          : '#B452CD',
    \ 'mediumorchid4'          : '#7A378B',
    \ 'mediumpurple'           : '#9370DB',
    \ 'mediumpurple1'          : '#AB82FF',
    \ 'mediumpurple2'          : '#9F79EE',
    \ 'mediumpurple3'          : '#8968CD',
    \ 'mediumpurple4'          : '#5D478B',
    \ 'mediumseagreen'         : '#3CB371',
    \ 'mediumslateblue'        : '#7B68EE',
    \ 'mediumspringgreen'      : '#00FA9A',
    \ 'mediumturquoise'        : '#48D1CC',
    \ 'mediumvioletred'        : '#C71585',
    \ 'midnight blue'          : '#191970',
    \ 'midnightblue'           : '#191970',
    \ 'mint cream'             : '#F5FFFA',
    \ 'mintcream'              : '#F5FFFA',
    \ 'misty rose'             : '#FFE4E1',
    \ 'mistyrose'              : '#FFE4E1',
    \ 'mistyrose1'             : '#FFE4E1',
    \ 'mistyrose2'             : '#EED5D2',
    \ 'mistyrose3'             : '#CDB7B5',
    \ 'mistyrose4'             : '#8B7D7B',
    \ 'moccasin'               : '#FFE4B5',
    \ 'navajo white'           : '#FFDEAD',
    \ 'navajowhite'            : '#FFDEAD',
    \ 'navajowhite1'           : '#FFDEAD',
    \ 'navajowhite2'           : '#EECFA1',
    \ 'navajowhite3'           : '#CDB38B',
    \ 'navajowhite4'           : '#8B795E',
    \ 'navy blue'              : '#000080',
    \ 'navy'                   : '#000080',
    \ 'navyblue'               : '#000080',
    \ 'old lace'               : '#FDF5E6',
    \ 'oldlace'                : '#FDF5E6',
    \ 'olive drab'             : '#6B8E23',
    \ 'olivedrab'              : '#6B8E23',
    \ 'olivedrab1'             : '#C0FF3E',
    \ 'olivedrab2'             : '#B3EE3A',
    \ 'olivedrab3'             : '#9ACD32',
    \ 'olivedrab4'             : '#698B22',
    \ 'orange red'             : '#FF4500',
    \ 'orange'                 : '#FFA500',
    \ 'orange1'                : '#FFA500',
    \ 'orange2'                : '#EE9A00',
    \ 'orange3'                : '#CD8500',
    \ 'orange4'                : '#8B5A00',
    \ 'orangered'              : '#FF4500',
    \ 'orangered1'             : '#FF4500',
    \ 'orangered2'             : '#EE4000',
    \ 'orangered3'             : '#CD3700',
    \ 'orangered4'             : '#8B2500',
    \ 'orchid'                 : '#DA70D6',
    \ 'orchid1'                : '#FF83FA',
    \ 'orchid2'                : '#EE7AE9',
    \ 'orchid3'                : '#CD69C9',
    \ 'orchid4'                : '#8B4789',
    \ 'pale goldenrod'         : '#EEE8AA',
    \ 'pale green'             : '#98FB98',
    \ 'pale turquoise'         : '#AFEEEE',
    \ 'pale violet red'        : '#DB7093',
    \ 'palegoldenrod'          : '#EEE8AA',
    \ 'palegreen'              : '#98FB98',
    \ 'palegreen1'             : '#9AFF9A',
    \ 'palegreen2'             : '#90EE90',
    \ 'palegreen3'             : '#7CCD7C',
    \ 'palegreen4'             : '#548B54',
    \ 'paleturquoise'          : '#AFEEEE',
    \ 'paleturquoise1'         : '#BBFFFF',
    \ 'paleturquoise2'         : '#AEEEEE',
    \ 'paleturquoise3'         : '#96CDCD',
    \ 'paleturquoise4'         : '#668B8B',
    \ 'palevioletred'          : '#DB7093',
    \ 'palevioletred1'         : '#FF82AB',
    \ 'palevioletred2'         : '#EE799F',
    \ 'palevioletred3'         : '#CD6889',
    \ 'palevioletred4'         : '#8B475D',
    \ 'papaya whip'            : '#FFEFD5',
    \ 'papayawhip'             : '#FFEFD5',
    \ 'peach puff'             : '#FFDAB9',
    \ 'peachpuff'              : '#FFDAB9',
    \ 'peachpuff1'             : '#FFDAB9',
    \ 'peachpuff2'             : '#EECBAD',
    \ 'peachpuff3'             : '#CDAF95',
    \ 'peachpuff4'             : '#8B7765',
    \ 'peru'                   : '#CD853F',
    \ 'pink'                   : '#FFC0CB',
    \ 'pink1'                  : '#FFB5C5',
    \ 'pink2'                  : '#EEA9B8',
    \ 'pink3'                  : '#CD919E',
    \ 'pink4'                  : '#8B636C',
    \ 'plum'                   : '#DDA0DD',
    \ 'plum1'                  : '#FFBBFF',
    \ 'plum2'                  : '#EEAEEE',
    \ 'plum3'                  : '#CD96CD',
    \ 'plum4'                  : '#8B668B',
    \ 'powder blue'            : '#B0E0E6',
    \ 'powderblue'             : '#B0E0E6',
    \ 'purple'                 : '#A020F0',
    \ 'purple1'                : '#9B30FF',
    \ 'purple2'                : '#912CEE',
    \ 'purple3'                : '#7D26CD',
    \ 'purple4'                : '#551A8B',
    \ 'red'                    : '#FF0000',
    \ 'red1'                   : '#FF0000',
    \ 'red2'                   : '#EE0000',
    \ 'red3'                   : '#CD0000',
    \ 'red4'                   : '#8B0000',
    \ 'rosy brown'             : '#BC8F8F',
    \ 'rosybrown'              : '#BC8F8F',
    \ 'rosybrown1'             : '#FFC1C1',
    \ 'rosybrown2'             : '#EEB4B4',
    \ 'rosybrown3'             : '#CD9B9B',
    \ 'rosybrown4'             : '#8B6969',
    \ 'royal blue'             : '#4169E1',
    \ 'royalblue'              : '#4169E1',
    \ 'royalblue1'             : '#4876FF',
    \ 'royalblue2'             : '#436EEE',
    \ 'royalblue3'             : '#3A5FCD',
    \ 'royalblue4'             : '#27408B',
    \ 'saddle brown'           : '#8B4513',
    \ 'saddlebrown'            : '#8B4513',
    \ 'salmon'                 : '#FA8072',
    \ 'salmon1'                : '#FF8C69',
    \ 'salmon2'                : '#EE8262',
    \ 'salmon3'                : '#CD7054',
    \ 'salmon4'                : '#8B4C39',
    \ 'sandy brown'            : '#F4A460',
    \ 'sandybrown'             : '#F4A460',
    \ 'sea green'              : '#2E8B57',
    \ 'seagreen'               : '#2E8B57',
    \ 'seagreen1'              : '#54FF9F',
    \ 'seagreen2'              : '#4EEE94',
    \ 'seagreen3'              : '#43CD80',
    \ 'seagreen4'              : '#2E8B57',
    \ 'seashell'               : '#FFF5EE',
    \ 'seashell1'              : '#FFF5EE',
    \ 'seashell2'              : '#EEE5DE',
    \ 'seashell3'              : '#CDC5BF',
    \ 'seashell4'              : '#8B8682',
    \ 'sienna'                 : '#A0522D',
    \ 'sienna1'                : '#FF8247',
    \ 'sienna2'                : '#EE7942',
    \ 'sienna3'                : '#CD6839',
    \ 'sienna4'                : '#8B4726',
    \ 'sky blue'               : '#87CEEB',
    \ 'skyblue'                : '#87CEEB',
    \ 'skyblue1'               : '#87CEFF',
    \ 'skyblue2'               : '#7EC0EE',
    \ 'skyblue3'               : '#6CA6CD',
    \ 'skyblue4'               : '#4A708B',
    \ 'slate blue'             : '#6A5ACD',
    \ 'slate gray'             : '#708090',
    \ 'slate grey'             : '#708090',
    \ 'slateblue'              : '#6A5ACD',
    \ 'slateblue1'             : '#836FFF',
    \ 'slateblue2'             : '#7A67EE',
    \ 'slateblue3'             : '#6959CD',
    \ 'slateblue4'             : '#473C8B',
    \ 'slategray'              : '#708090',
    \ 'slategray1'             : '#C6E2FF',
    \ 'slategray2'             : '#B9D3EE',
    \ 'slategray3'             : '#9FB6CD',
    \ 'slategray4'             : '#6C7B8B',
    \ 'slategrey'              : '#708090',
    \ 'snow'                   : '#FFFAFA',
    \ 'snow1'                  : '#FFFAFA',
    \ 'snow2'                  : '#EEE9E9',
    \ 'snow3'                  : '#CDC9C9',
    \ 'snow4'                  : '#8B8989',
    \ 'spring green'           : '#00FF7F',
    \ 'springgreen'            : '#00FF7F',
    \ 'springgreen1'           : '#00FF7F',
    \ 'springgreen2'           : '#00EE76',
    \ 'springgreen3'           : '#00CD66',
    \ 'springgreen4'           : '#008B45',
    \ 'steel blue'             : '#4682B4',
    \ 'steelblue'              : '#4682B4',
    \ 'steelblue1'             : '#63B8FF',
    \ 'steelblue2'             : '#5CACEE',
    \ 'steelblue3'             : '#4F94CD',
    \ 'steelblue4'             : '#36648B',
    \ 'tan'                    : '#D2B48C',
    \ 'tan1'                   : '#FFA54F',
    \ 'tan2'                   : '#EE9A49',
    \ 'tan3'                   : '#CD853F',
    \ 'tan4'                   : '#8B5A2B',
    \ 'thistle'                : '#D8BFD8',
    \ 'thistle1'               : '#FFE1FF',
    \ 'thistle2'               : '#EED2EE',
    \ 'thistle3'               : '#CDB5CD',
    \ 'thistle4'               : '#8B7B8B',
    \ 'tomato'                 : '#FF6347',
    \ 'tomato1'                : '#FF6347',
    \ 'tomato2'                : '#EE5C42',
    \ 'tomato3'                : '#CD4F39',
    \ 'tomato4'                : '#8B3626',
    \ 'turquoise'              : '#40E0D0',
    \ 'turquoise1'             : '#00F5FF',
    \ 'turquoise2'             : '#00E5EE',
    \ 'turquoise3'             : '#00C5CD',
    \ 'turquoise4'             : '#00868B',
    \ 'violet red'             : '#D02090',
    \ 'violet'                 : '#EE82EE',
    \ 'violetred'              : '#D02090',
    \ 'violetred1'             : '#FF3E96',
    \ 'violetred2'             : '#EE3A8C',
    \ 'violetred3'             : '#CD3278',
    \ 'violetred4'             : '#8B2252',
    \ 'wheat'                  : '#F5DEB3',
    \ 'wheat1'                 : '#FFE7BA',
    \ 'wheat2'                 : '#EED8AE',
    \ 'wheat3'                 : '#CDBA96',
    \ 'wheat4'                 : '#8B7E66',
    \ 'white smoke'            : '#F5F5F5',
    \ 'white'                  : '#FFFFFF',
    \ 'whitesmoke'             : '#F5F5F5',
    \ 'yellow green'           : '#9ACD32',
    \ 'yellow'                 : '#FFFF00',
    \ 'yellow1'                : '#FFFF00',
    \ 'yellow2'                : '#EEEE00',
    \ 'yellow3'                : '#CDCD00',
    \ 'yellow4'                : '#8B8B00',
    \ 'yellowgreen'            : '#9ACD32',
  \}

  if has_key(l:color_list, l:color_name)
    let l:hex_code = l:color_list[l:color_name]
  endif

  return l:hex_code
endfunction
