# 7 "parsing_cocci/lexer_cocci.mll"
 
open Parser_cocci_menhir
module D = Data
module Ast = Ast_cocci
module Ast0 = Ast0_cocci
module P = Parse_aux
module FC = Flag_parsing_cocci
exception Lexical of string
let tok = Lexing.lexeme

let line = ref 1
let file = ref ""
let logical_line = ref 0

(* ---------------------------------------------------------------------- *)
(* control codes *)

(* Defined in data.ml
type line_type = MINUS | OPTMINUS | PLUS | CONTEXT | OPT
*)

let current_line_type = ref (D.CONTEXT,!line,!logical_line)

let prev_plus = ref false
let line_start = ref 0 (* offset of the beginning of the line *)
let get_current_line_type lexbuf =
  let (c,l,ll) = !current_line_type in
  let lex_start = Lexing.lexeme_start lexbuf in
  let preceeding_spaces =
    if !line_start < 0 then 0 else lex_start - !line_start in
  (*line_start := -1;*)
  prev_plus := (c = D.PLUS) || (c = D.PLUSPLUS);
  (c,l,ll,ll,lex_start,preceeding_spaces,[],[],[],"")
let current_line_started = ref false
let col_zero = ref true

let contextify (c,l,ll,lle,lex_start,preceeding_spaces,bef,aft,pos,ws) =
  (D.CONTEXT,l,ll,lle,lex_start,preceeding_spaces,bef,aft,pos,ws)

let reset_line lexbuf =
  line := !line + 1;
  current_line_type := (D.CONTEXT,!line,!logical_line);
  current_line_started := false;
  (if not(!Data.in_meta || !Data.in_rule_name) then col_zero := true);
  line_start := Lexing.lexeme_start lexbuf + 1

let started_line = ref (-1)

let start_line seen_char =
  current_line_started := true;
  col_zero := false;
  (if seen_char && not(!line = !started_line)
  then
    begin
      started_line := !line;
      logical_line := !logical_line + 1
    end)

let pass_zero _ = col_zero := false

let lexerr s1 s2 = raise (Lexical (Printf.sprintf "%s%s" s1 s2))

let opt_reverse_token token =
  if !FC.interpret_inverted
  then match token with
         D.MINUS        -> D.PLUSPLUS  (* maybe too liberal *)
       | D.OPTMINUS     -> lexerr "cannot invert token ?- (an optional minus line), which is needed for reversing the patch" ""
       | D.PLUS         -> D.MINUS
       | D.PLUSPLUS     -> D.MINUS (* may not be sufficient *)
       | _              -> token
  else token

let add_current_line_type x =
  match (opt_reverse_token x,!current_line_type) with
    (D.MINUS,(D.CONTEXT,ln,lln))  ->
      current_line_type := (D.MINUS,ln,lln)
  | (D.MINUS,(D.OPT,ln,lln))      ->
      current_line_type := (D.OPTMINUS,ln,lln)
  | (D.PLUS,(D.CONTEXT,ln,lln))   ->
      current_line_type := (D.PLUS,ln,lln)
  | (D.PLUSPLUS,(D.CONTEXT,ln,lln))   ->
      current_line_type := (D.PLUSPLUS,ln,lln)
  | (D.OPT,(D.CONTEXT,ln,lln))    ->
      current_line_type := (D.OPT,ln,lln)
  | _ -> lexerr "invalid control character combination" ""

let check_minus_context_linetype s =
  match !current_line_type with
    (D.PLUS,_,_) | (D.PLUSPLUS,_,_) -> lexerr "invalid in a + context: " s
  | _ -> ()

let check_context_linetype s =
  match !current_line_type with
    (D.CONTEXT,_,_) -> ()
  | _ -> lexerr "invalid in a nonempty context: " s

let check_plus_linetype s =
  match !current_line_type with
    (D.PLUS,_,_) | (D.PLUSPLUS,_,_) -> ()
  | _ -> lexerr "invalid in a non + context: " s

let check_arity_context_linetype s =
  match !current_line_type with
    (D.CONTEXT,_,_) | (D.PLUS,_,_) | (D.PLUSPLUS,_,_)
  | (D.OPT,_,_) -> ()
  | _ -> lexerr "invalid in a nonempty context: " s

let check_comment s =
  if not !current_line_started
  then lexerr "+ expected at the beginning of the line" s

let process_include start finish str =
  (match !current_line_type with
    (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
      (try
	let _ = Str.search_forward (Str.regexp "\\.\\.\\.") str start in
	lexerr "... not allowed in + include" ""
      with Not_found -> ())
  | _ -> ());
  String.sub str (start + 1) (finish - start - 1)

(* ---------------------------------------------------------------------- *)
type pm = PATCH | MATCH | UNKNOWN

let pm = ref UNKNOWN

let patch_or_match = function
    PATCH ->
      if not !D.ignore_patch_or_match
      then
	(match !pm with
	  MATCH ->
	    lexerr "- or + not allowed in the first column for a match" ""
	| PATCH -> ()
	| UNKNOWN -> Flag.sgrep_mode2 := false; pm := PATCH)
  | MATCH ->
      if not !D.ignore_patch_or_match
      then
	(match !pm with
	  PATCH -> lexerr "* not allowed in the first column for a patch" ""
	| MATCH -> ()
	| UNKNOWN -> Flag.sgrep_mode2 := true; pm := MATCH)
  | _ -> failwith "unexpected argument"

(* ---------------------------------------------------------------------- *)
(* identifiers, including metavariables *)

let metavariables = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let all_metavariables =
  (Hashtbl.create(100) : (string,(string * (D.clt -> token)) list) Hashtbl.t)

let type_names = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let attr_names = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let declarer_names = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let iterator_names = (Hashtbl.create(100) : (string, D.clt -> token) Hashtbl.t)

let symbol_names = (Hashtbl.create(15) : (string, D.clt -> token) Hashtbl.t)

let rule_names = (Hashtbl.create(100) : (string, unit) Hashtbl.t)

let check_var s linetype =
  let fail _ =
    if (!Data.in_prolog || !Data.in_rule_name) &&
      Str.string_match (Str.regexp "<.*>") s 0
    then TPathIsoFile s
    else
      try (Hashtbl.find metavariables s) linetype
      with Not_found ->
	(try (Hashtbl.find type_names s) linetype
	with Not_found ->
	  (try (Hashtbl.find attr_names s) linetype
	  with Not_found ->
	    (try (Hashtbl.find declarer_names s) linetype
	    with Not_found ->
	      (try (Hashtbl.find iterator_names s) linetype
	      with Not_found ->
		(try (Hashtbl.find symbol_names s) linetype
		with Not_found ->
                  TIdent (s,linetype)))))) in
  if !Data.in_meta || !Data.in_rule_name
  then (try Hashtbl.find rule_names s; TRuleName s with Not_found -> fail())
  else fail()

let id_tokens lexbuf =
  let s = tok lexbuf in
  let linetype = get_current_line_type lexbuf in
  let in_rule_name = !Data.in_rule_name in
  let in_meta = !Data.in_meta && not !Data.saw_struct in
  let in_iso = !Data.in_iso in
  let in_prolog = !Data.in_prolog in
  (if s = "identifer" && in_meta
  then Common.pr2 "Warning: should identifer be identifier?");
  match s with
    "metavariable" when in_meta -> check_arity_context_linetype s; TMetavariable
  | "identifier" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TIdentifier
  | "type" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TType
  | "parameter" when in_meta ->  check_arity_context_linetype s; TParameter
  | "operator" when in_meta ->   check_arity_context_linetype s; TOperator
  | "binary" when in_meta ->   check_arity_context_linetype s; TBinary
  | "assignment" when in_meta ->   check_arity_context_linetype s; TAssignment
  | "pragmainfo" when in_meta ->   check_arity_context_linetype s; TPragmaInfo
  | "constant"  when in_meta ->  check_arity_context_linetype s; TConstant
  | "generated" when in_rule_name && not (!Flag.make_hrule = None) ->
      check_arity_context_linetype s; TGenerated
  | "expression" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TExpression
  | "declaration" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TDeclaration
  | "field" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TField
  | "initialiser" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TInitialiser
  | "initializer" when in_meta || in_rule_name ->
      check_arity_context_linetype s; TInitialiser
  | "idexpression" when in_meta ->
      check_arity_context_linetype s; TIdExpression
  | "statement" when in_meta ->  check_arity_context_linetype s; TStatement
  | "function"  when in_meta ->  check_arity_context_linetype s; TFunction
  | "local" when in_meta ->      check_arity_context_linetype s; TLocal
  | "global" when in_meta ->     check_arity_context_linetype s; TGlobal
  | "list" when in_meta ->       check_arity_context_linetype s; Tlist
  | "fresh" when in_meta ->      check_arity_context_linetype s; TFresh
  | "typedef" when in_meta ->    check_arity_context_linetype s; TTypedef
  | "attribute" when in_meta ->  check_arity_context_linetype s; TAttribute
  | "declarer" when in_meta ->   check_arity_context_linetype s; TDeclarer
  | "iterator" when in_meta ->   check_arity_context_linetype s; TIterator
  | "name" when in_meta ->       check_arity_context_linetype s; TName
  | "position" when in_meta ->   check_arity_context_linetype s; TPosition
  | "comments" when in_meta ->   check_arity_context_linetype s; TComments
  | "format" when in_meta ->     check_arity_context_linetype s; TFormat
  | "analysis" when in_meta ->   check_arity_context_linetype s; TAnalysis
  | "any" when in_meta ->        check_arity_context_linetype s; TPosAny
  | "pure" when in_meta && in_iso ->
      check_arity_context_linetype s; TPure
  | "context" when in_meta && in_iso ->
      check_arity_context_linetype s; TContext
  | "error" when in_meta ->      check_arity_context_linetype s; TError
  | "words" when in_meta ->      check_context_linetype s; TWords
  | "symbol" when in_meta ->     check_arity_context_linetype s; TSymbol

  | "using" when in_rule_name || in_prolog ->
     check_context_linetype s; TIsoUsing
  | "using" when !Flag.c_plus_plus <> Flag.Off -> start_line true; TUsing linetype
  | "namespace" when !Flag.c_plus_plus <> Flag.Off -> start_line true; TNamespace linetype
  | "virtual" when in_prolog || in_rule_name || in_meta ->
      (* don't want to allow virtual as a rule name *)
      check_context_linetype s; TVirtual
  | "merge" when in_prolog || in_rule_name || in_meta ->
      check_context_linetype s; TMerge
  | "disable" when in_rule_name ->  check_context_linetype s; TDisable
  | "extends" when in_rule_name -> check_context_linetype s; TExtends
  | "depends" when in_rule_name -> check_context_linetype s; TDepends
  | "on" when in_rule_name      -> check_context_linetype s; TOn
  | "ever" when in_rule_name    -> check_context_linetype s; TEver
  | "never" when in_rule_name   -> check_context_linetype s; TNever
  | "file" when in_rule_name    -> check_context_linetype s; TFile
  | "in" when in_rule_name      -> check_context_linetype s; TIn
  (* exists and forall for when are reparsed in parse_cocci.ml *)
  | "exists" when in_rule_name  -> check_context_linetype s; TExists
  | "forall" when in_rule_name  -> check_context_linetype s; TForall
  | "script" when (in_rule_name || in_meta) ->
      check_context_linetype s; TScript (!file, !line)
  | "initialize" when in_rule_name -> check_context_linetype s; TInitialize
  | "finalize" when in_rule_name   -> check_context_linetype s; TFinalize

  | "char" ->       Tchar     linetype
  | "short" ->      Tshort    linetype
  | "int" ->        Tint      linetype
  | "double" ->     Tdouble   linetype
  | "float" ->      Tfloat    linetype
  | "complex" ->    if !Flag.c_plus_plus == Flag.Off
		    then (*C*) Tcomplex  linetype
		    else (*C++*) TTypeId ("complex",linetype)
  | "long" ->       Tlong     linetype
  | "void" ->       Tvoid     linetype
  | "size_t" ->     if !Flag.c_plus_plus == Flag.Off
		    then (*C*) Tsize_t linetype
		    else (*C++*) TTypeId ("size_t",linetype)
  | "ssize_t" ->    if !Flag.c_plus_plus == Flag.Off
		    then (*C*) Tssize_t linetype
		    else (*C++*) TTypeId ("ssize_t",linetype)
  | "ptrdiff_t" ->  if !Flag.c_plus_plus == Flag.Off
		    then (*C*) Tptrdiff_t linetype
		    else (*C++*) TTypeId ("ptrdiff_t",linetype)
  (* in_meta is only for the first keyword; drop it now to allow any type
     name *)
  | "struct" ->     Data.saw_struct := true; Tstruct   linetype
  | "class" ->      Data.saw_struct := true; Tclass    linetype
  | "typename" ->   Data.saw_struct := true; Ttypename  linetype
  | "union" ->      Data.saw_struct := true; Tunion    linetype
  | "enum" ->       Data.saw_struct := true; Tenum     linetype
  | "unsigned" ->   Tunsigned linetype
  | "signed" ->     Tsigned   linetype
  | "decimal" when !Flag.ibm -> Tdecimal linetype
  | "EXEC" when !Flag.ibm -> Texec linetype

  | "delete"  ->
	if !Flag.c_plus_plus = Flag.Off
	then check_var s linetype
	else Tdelete linetype
  | "auto"  ->
      begin match !Flag.c_plus_plus with
        Flag.On None ->
          raise (
            Semantic_cocci.Semantic (
              "auto has different meaning in different versions of C++. Please specify a version using --c++=<version>"))
      | Flag.On (Some i) ->
          if i >= 2011
          then TautoType linetype
          else Tauto linetype
      | Flag.Off -> Tauto linetype end
  | "register" ->   Tregister linetype
  | "extern" ->     Textern   linetype
  | "alignas" ->    Talignas  linetype
  | "static" ->     Tstatic   linetype
  | "inline" ->     Tinline   linetype
  | "typedef" ->    Ttypedef  linetype

  | "const" ->      Tconst    linetype
  | "volatile" ->   Tvolatile linetype

  | "if" ->         TIf       linetype
  | "else" ->       TElse     linetype
  | "while" ->      TWhile    linetype
  | "do" ->         TDo       linetype
  | "for" ->        TFor      linetype
  | "switch" ->     TSwitch   linetype
  | "case" ->       TCase     linetype
  | "default" ->    TDefault  linetype
  | "return" ->     TReturn   linetype
  | "break" ->      TBreak    linetype
  | "continue" ->   TContinue linetype
  | "goto" ->       TGoto     linetype

  | "sizeof" ->     TSizeof   linetype
  | "typeof" ->     TTypeof   ("typeof", linetype)
  | "typeof_uequal" when !Flag.c_plus_plus <> Flag.Off ->
      TTypeof   ("typeof_unequal", linetype)
  | "decltype" when !Flag.c_plus_plus <> Flag.Off ->
      TTypeof   ("decltype", linetype)

  | "new" when !Flag.c_plus_plus != Flag.Off -> TNew linetype

  | "__attribute__" -> TAttr_ linetype

  | "Expression"       when !Data.in_iso -> TIsoExpression
  | "ArgExpression"    when !Data.in_iso -> TIsoArgExpression
  | "TestExpression"   when !Data.in_iso -> TIsoTestExpression
  | "ToTestExpression" when !Data.in_iso -> TIsoToTestExpression
  | "Statement"        when !Data.in_iso -> TIsoStatement
  | "Declaration"      when !Data.in_iso -> TIsoDeclaration
  | "Type"             when !Data.in_iso -> TIsoType
  | "TopLevel"         when !Data.in_iso -> TIsoTopLevel

  | "_" when !Data.in_meta -> TUnderscore

  | s -> check_var s linetype

let mkassign op lexbuf =
  TOpAssign (op, (get_current_line_type lexbuf))

let spinit _ = (* per semantic patch *)
  Hashtbl.clear Data.non_local_script_constraints;
  Data.clear_special_names()

let init _ = (* per file, first .cocci then iso *)
  line := 1;
  logical_line := 0;
  prev_plus := false;
  line_start := 0;
  current_line_started := false;
  current_line_type := (D.CONTEXT,0,0);
  col_zero := true;
  pm := UNKNOWN;
  Data.in_rule_name := false;
  Data.in_meta := false;
  Data.in_prolog := false;
  Data.saw_struct := false;
  Data.inheritable_positions := [];
  Hashtbl.clear all_metavariables;
  Hashtbl.clear Data.all_metadecls;
  Hashtbl.clear metavariables;
  Hashtbl.clear type_names;
  Hashtbl.clear attr_names;
  Hashtbl.clear rule_names;
  Hashtbl.clear iterator_names;
  Hashtbl.clear declarer_names;
  Hashtbl.clear symbol_names;
  let get_name (_,x) = x in
  Data.add_meta_meta :=
    (fun name cstr pure ->
      let fn clt = TMeta(name,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_id_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaId(name,constraints,Ast.NoVal,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_virt_id_meta_found :=
    (fun name vl ->
      let fn clt = TIdent(vl,clt) in
      Hashtbl.replace metavariables name fn);
  Data.add_virt_id_meta_not_found :=
    (fun name pure ->
      let fn clt = TMetaId(name,Ast.CstrTrue,Ast.NoVal,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_fresh_id_meta :=
    (fun name seed ->
      let fn clt = TMetaId(name,Ast.CstrTrue,seed,Ast0.Impure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_type_meta :=
    (fun name cstr pure ->
      let fn clt = TMetaType(name,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_init_meta :=
    (fun name cstr pure ->
      let fn clt = TMetaInit(name,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_initlist_meta :=
    (fun name lenname cstr pure ->
      let fn clt = TMetaInitList(name,lenname,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_param_meta :=
    (fun name cstr pure ->
      let fn clt = TMetaParam(name,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_paramlist_meta :=
    (fun name lenname cstr pure ->
      let fn clt = TMetaParamList(name,lenname,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_const_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaConst(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_err_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaErr(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_exp_meta :=
    (fun tyopt name constraints pure bitfield ->
      let fn clt = TMetaExp(name,constraints,pure,tyopt,clt,bitfield) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_idexp_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaIdExp(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_local_idexp_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaLocalIdExp(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_global_idexp_meta :=
    (fun tyopt name constraints pure ->
      let fn clt = TMetaGlobalIdExp(name,constraints,pure,tyopt,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_explist_meta :=
    (fun name lenname cstr pure ->
      let fn clt = TMetaExpList(name,lenname,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_decl_meta :=
    (fun name cstr pure ->
      let fn clt = TMetaDecl(name,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_field_meta :=
    (fun name cstr pure ->
      let fn clt = TMetaField(name,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_field_list_meta :=
    (fun name lenname cstr pure ->
      let fn clt = TMetaFieldList(name,lenname,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_fmt_meta :=
    (function name -> function iconstraints ->
      let fn clt = failwith "format metavariable only allowed in a string" in
      Data.format_metavariables :=
	(get_name name,(name,iconstraints)) :: !Data.format_metavariables;
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_fmtlist_meta :=
    (fun name constraints lenname ->
      let fn clt =
	failwith "format list metavariable only allowed in a string" in
      Data.format_list_metavariables :=
	(get_name name,(name,lenname,constraints))
	:: !Data.format_list_metavariables;
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_stm_meta :=
    (fun name cstr pure ->
      let fn clt = TMetaStm(name,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_stmlist_meta :=
    (fun name lenname cstr pure ->
      let fn clt = TMetaStmList(name,lenname,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_dparamlist_meta :=
    (fun name lenname cstr pure ->
      let fn clt = TMetaDParamList(name,lenname,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_func_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaFunc(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_local_func_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaLocalFunc(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_iterator_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaIterator(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_declarer_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaDeclarer(name,constraints,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_pos_meta :=
    (fun name constraints any ->
      let fn ((d,ln,_,_,_,_,_,_,_,_) as clt) =
	(if d = Data.PLUS
	then
	  failwith
	    (Printf.sprintf "%d: positions only allowed in minus code" ln));
	TMetaPos(name,constraints,any,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_com_meta :=
    (fun name constraints ->
      let fn ((d,ln,_,_,_,_,_,_,_,_) as clt) =
	(if d = Data.PLUS
	then
	  failwith
	    (Printf.sprintf "%d: comment variables only allowed in minus code" ln));
	TMetaCom(name,constraints,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_assignOp_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaAssignOp (name, constraints, pure, clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_binaryOp_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaBinaryOp (name, constraints, pure, clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_pragmainfo_meta :=
    (fun name constraints pure ->
      let fn clt = TMetaPragmaInfo (name, constraints, pure, clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_type_name :=
    (function name ->
      let fn clt = TTypeId(name,clt) in
      Data.add_special_name name Data.Type;
      Hashtbl.replace type_names name fn);
  Data.add_attribute :=
    (function name ->
      let fn clt = Tattr (name, clt) in
      Data.add_special_name name Data.Attr;
      Hashtbl.replace attr_names name fn);
  Data.add_attribute_meta :=
    (fun name cstr pure ->
      let fn ((d,ln,_,_,_,_,_,_,_,_) as clt) =
        if (match d with (Data.PLUS | Data.PLUSPLUS) -> true | _ -> false)
        then
          (* TODO support meta attributes in plus code *)
	  failwith
	    (Printf.sprintf
             "%d: meta attributes currently only allowed in context/minus code"
             ln);
        TMetaAttribute(name,cstr,pure,clt) in
      Hashtbl.replace metavariables (get_name name) fn);
  Data.add_declarer_name :=
    (function name ->
      let fn clt = TDeclarerId(name,clt) in
      Data.add_special_name name Data.Declarer;
      Hashtbl.replace declarer_names name fn);
  Data.add_iterator_name :=
    (function name ->
      let fn clt = TIteratorId(name,clt) in
      Data.add_special_name name Data.Iterator;
      Hashtbl.replace iterator_names name fn);
  Data.add_symbol_meta :=
    (function name ->
      let fn clt = TSymId (name,clt) in
      Hashtbl.replace symbol_names name fn);
  Data.init_rule := (function _ -> Hashtbl.clear metavariables);
  Data.install_bindings :=
    (function parent ->
      List.iter (function (name,fn) -> Hashtbl.add metavariables name fn)
	(Hashtbl.find all_metavariables parent))

let post_init _ =
  Stdcompat.Hashtbl.reset all_metavariables;
  Stdcompat.Hashtbl.reset Data.all_metadecls;
  Stdcompat.Hashtbl.reset metavariables;
  Stdcompat.Hashtbl.reset type_names;
  Stdcompat.Hashtbl.reset attr_names;
  Stdcompat.Hashtbl.reset rule_names;
  Stdcompat.Hashtbl.reset iterator_names;
  Stdcompat.Hashtbl.reset declarer_names;
  Stdcompat.Hashtbl.reset symbol_names

(* initialization for each cocci rule *)
let reinit _ =
  Data.format_metavariables := [];
  Data.format_list_metavariables := []

(* the following is needed to properly tokenize include files.  Because an
include file is included after seeing a @, so current_line_started is true.
Current_line_started is not important for parsing the name of a rule, so we
don't have to reset this value to true after parsing an included file. *)
let include_init _ =
  current_line_started := false

let drop_spaces s =
  let len = String.length s in
  let rec loop n =
    if n = len
    then n
    else
      if List.mem (String.get s n) [' ';'\t']
      then loop (n+1)
      else n in
  let start = loop 0 in
  String.sub s start (len - start)

# 627 "parsing_cocci/lexer_cocci.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\152\255\153\255\081\000\119\000\166\255\167\255\192\000\
    \023\001\112\001\201\001\034\002\121\002\035\000\208\002\039\003\
    \090\000\078\000\126\003\216\255\080\000\083\000\222\255\224\255\
    \084\000\227\255\228\255\085\000\230\255\057\000\233\255\213\003\
    \091\000\192\000\238\255\044\004\092\000\240\255\116\000\117\000\
    \148\000\121\004\132\004\219\004\090\000\247\004\170\000\255\255\
    \077\005\205\000\108\000\150\000\044\000\171\000\153\000\061\000\
    \045\000\065\000\079\000\181\000\249\004\174\000\172\255\209\255\
    \081\000\108\000\090\000\079\000\095\000\087\000\183\255\084\005\
    \099\000\100\000\108\000\220\000\155\005\230\005\119\000\118\000\
    \114\000\124\000\052\001\088\006\171\006\180\255\129\000\131\000\
    \126\000\153\000\053\001\029\007\143\007\251\004\241\000\100\005\
    \154\000\146\000\138\000\183\000\183\000\085\005\239\000\003\001\
    \036\001\086\005\080\001\178\255\004\001\177\255\242\000\176\255\
    \071\001\072\001\160\001\161\001\076\001\163\001\252\001\077\001\
    \205\000\223\000\042\001\222\000\224\000\165\001\166\001\234\000\
    \013\001\255\001\005\001\003\001\082\002\009\001\048\001\083\002\
    \250\255\230\007\061\008\148\008\235\008\066\009\116\003\106\001\
    \032\002\108\001\093\001\245\255\243\255\110\001\223\255\085\005\
    \202\003\033\004\096\001\143\001\098\001\231\255\159\001\160\001\
    \162\001\246\255\190\001\192\001\244\255\203\255\193\255\202\255\
    \155\255\212\255\194\001\226\255\085\002\211\255\199\001\086\002\
    \198\255\153\009\240\009\071\010\158\010\245\010\004\000\173\255\
    \185\255\232\255\234\255\237\255\190\255\206\255\076\011\163\011\
    \250\011\081\012\168\012\255\012\086\013\229\255\208\255\210\255\
    \197\255\182\001\184\001\204\255\192\255\201\255\173\013\004\014\
    \091\014\178\014\154\255\199\255\191\255\207\255\009\015\096\015\
    \183\015\014\016\101\016\188\016\019\017\106\017\193\017\205\255\
    \024\018\111\018\198\018\029\019\116\019\164\255\165\255\162\255\
    \163\255\160\255\161\255\205\019\159\255\156\255\071\005\217\001\
    \024\020\117\005\056\006\078\006\051\020\088\006\247\006\121\020\
    \050\002\157\255\042\002\215\020\193\255\039\021\077\021\204\255\
    \205\255\151\021\238\021\071\022\160\022\249\022\211\001\220\255\
    \080\023\167\023\254\023\186\001\121\002\085\024\036\002\233\001\
    \187\001\236\001\018\002\107\002\239\255\240\255\001\007\118\002\
    \243\255\244\255\245\255\246\255\247\255\248\255\120\002\122\002\
    \017\002\207\002\255\255\148\005\078\002\004\003\209\255\216\255\
    \252\255\236\255\251\255\235\255\250\255\219\255\242\255\019\007\
    \110\007\029\007\128\007\234\255\067\002\211\255\217\255\218\255\
    \215\255\225\255\069\002\210\255\214\255\222\255\172\024\003\025\
    \090\025\177\025\008\026\095\026\182\026\213\255\221\255\212\255\
    \013\027\100\027\187\027\018\028\105\028\192\028\023\029\110\029\
    \197\029\028\030\115\030\202\030\033\031\120\031\208\255\202\255\
    \203\255\200\255\201\255\198\255\199\255\209\031\197\255\194\255\
    \011\020\109\002\028\032\220\007\089\020\053\032\086\032\050\008\
    \137\008\156\032\139\002\195\255\179\003\255\255\112\020\252\255\
    \226\032\100\005\154\005\254\255\249\032\253\255\010\004\254\255\
    \191\020\255\255\251\255\031\033\224\008\055\009\253\255\054\033\
    \252\255\101\020\253\255\064\032\254\255\255\255\250\255\092\033\
    \142\009\229\009\252\255\115\033\251\255\008\004\252\255\253\255\
    \254\255\115\002\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\098\000\098\000\255\255\255\255\087\000\
    \086\000\086\000\086\000\086\000\086\000\071\000\086\000\086\000\
    \068\000\042\000\086\000\255\255\061\000\034\000\255\255\255\255\
    \038\000\255\255\255\255\055\000\255\255\024\000\255\255\086\000\
    \019\000\103\000\255\255\086\000\016\000\255\255\014\000\013\000\
    \060\000\030\000\086\000\086\000\006\000\103\000\035\000\255\255\
    \001\000\255\255\255\255\002\000\255\255\255\255\003\000\255\255\
    \255\255\255\255\255\255\004\000\004\000\004\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\073\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\074\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\076\000\076\000\081\000\
    \255\255\255\255\255\255\255\255\255\255\080\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \081\000\081\000\081\000\081\000\081\000\081\000\081\000\081\000\
    \255\255\255\255\255\255\255\255\255\255\081\000\081\000\255\255\
    \255\255\081\000\255\255\255\255\081\000\255\255\255\255\081\000\
    \255\255\086\000\086\000\007\000\086\000\086\000\097\000\255\255\
    \008\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \097\000\255\255\059\000\067\000\036\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\041\000\255\255\084\000\255\255\040\000\085\000\
    \255\255\086\000\016\000\086\000\086\000\057\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\086\000\086\000\
    \086\000\086\000\020\000\086\000\069\000\255\255\255\255\255\255\
    \255\255\066\000\037\000\255\255\255\255\255\255\086\000\086\000\
    \086\000\039\000\255\255\255\255\255\255\255\255\086\000\064\000\
    \086\000\086\000\048\000\065\000\086\000\086\000\049\000\255\255\
    \086\000\071\000\086\000\086\000\050\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\086\000\255\255\255\255\098\000\098\000\
    \255\255\097\000\255\255\255\255\098\000\097\000\255\255\098\000\
    \098\000\255\255\098\000\255\255\255\255\060\000\060\000\255\255\
    \255\255\049\000\048\000\048\000\048\000\048\000\062\000\255\255\
    \048\000\048\000\048\000\029\000\028\000\048\000\027\000\032\000\
    \025\000\022\000\018\000\031\000\255\255\255\255\014\000\023\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\062\000\006\000\
    \062\000\024\000\255\255\001\000\255\255\002\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\059\000\
    \255\255\059\000\255\255\255\255\017\000\255\255\255\255\255\255\
    \255\255\255\255\026\000\255\255\255\255\255\255\048\000\048\000\
    \048\000\048\000\048\000\027\000\028\000\255\255\255\255\255\255\
    \048\000\029\000\048\000\048\000\043\000\048\000\033\000\048\000\
    \048\000\041\000\034\000\048\000\048\000\042\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\048\000\255\255\255\255\
    \060\000\060\000\255\255\059\000\255\255\255\255\060\000\059\000\
    \255\255\060\000\060\000\255\255\255\255\255\255\000\000\255\255\
    \003\000\001\000\001\000\255\255\002\000\255\255\255\255\255\255\
    \001\000\255\255\255\255\004\000\002\000\002\000\255\255\003\000\
    \255\255\255\255\255\255\002\000\255\255\255\255\255\255\005\000\
    \003\000\003\000\255\255\004\000\255\255\255\255\255\255\255\255\
    \255\255\004\000\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\255\255\255\255\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\000\000\000\000\
    \255\255\000\000\000\000\255\255\000\000\255\255\000\000\255\255\
    \255\255\255\255\000\000\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\051\000\255\255\255\255\054\000\255\255\
    \255\255\255\255\255\255\255\255\061\000\061\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\094\000\094\000\112\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\108\000\
    \106\000\255\255\106\000\000\000\108\000\000\000\255\255\000\000\
    \112\000\112\000\112\000\112\000\116\000\112\000\112\000\119\000\
    \255\255\255\255\255\255\255\255\255\255\125\000\126\000\255\255\
    \255\255\129\000\255\255\255\255\132\000\255\255\255\255\135\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\000\000\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\000\000\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\172\000\000\000\255\255\175\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\255\255\255\255\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\252\000\000\000\255\255\255\255\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\037\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\000\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\109\001\000\000\111\001\000\000\
    \255\255\255\255\255\255\000\000\255\255\000\000\119\001\000\000\
    \122\001\000\000\000\000\255\255\255\255\255\255\000\000\255\255\
    \000\000\130\001\000\000\134\001\000\000\000\000\000\000\255\255\
    \255\255\255\255\000\000\255\255\000\000\142\001\000\000\000\000\
    \000\000\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\048\000\047\000\047\000\047\000\047\000\183\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \048\000\036\000\005\000\045\000\007\000\024\000\016\000\006\000\
    \034\000\030\000\021\000\038\000\023\000\039\000\041\000\046\000\
    \004\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\027\000\022\000\040\000\017\000\020\000\037\000\
    \044\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\011\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\010\000\008\000\043\000\
    \008\000\008\000\008\000\029\000\033\000\028\000\013\000\008\000\
    \223\000\015\000\031\000\018\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\035\000\014\000\
    \008\000\008\000\008\000\008\000\008\000\009\000\008\000\042\000\
    \012\000\008\000\008\000\026\000\032\000\025\000\019\000\241\000\
    \212\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\211\000\210\000\200\000\201\000\202\000\
    \199\000\198\000\025\000\028\000\197\000\237\000\242\000\213\000\
    \189\000\176\000\136\000\051\000\055\000\238\000\056\000\174\000\
    \255\255\057\000\170\000\255\255\058\000\241\000\239\000\244\000\
    \244\000\244\000\244\000\244\000\244\000\244\000\244\000\243\000\
    \243\000\173\000\169\000\171\000\053\000\237\000\242\000\059\000\
    \255\255\026\000\133\000\237\000\242\000\238\000\060\000\158\000\
    \095\000\086\000\159\000\238\000\078\000\072\000\239\000\073\000\
    \096\000\074\000\183\000\053\000\239\000\182\000\157\000\240\000\
    \155\000\154\000\075\000\156\000\062\000\060\000\053\000\188\000\
    \122\000\051\000\121\000\237\000\242\000\079\000\120\000\080\000\
    \081\000\082\000\087\000\238\000\007\000\076\000\184\000\063\000\
    \187\000\185\000\088\000\089\000\239\000\053\000\054\000\240\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\090\000\255\255\076\000\097\000\098\000\099\000\
    \002\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\100\000\101\000\110\000\052\000\007\000\
    \111\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\186\000\083\000\091\000\130\000\
    \052\000\255\255\109\000\127\000\126\000\125\000\255\255\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\255\255\255\255\128\000\083\000\091\000\255\255\255\255\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\107\000\129\000\131\000\132\000\008\000\134\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\233\000\123\000\007\000\135\000\255\255\234\000\
    \144\000\255\255\149\000\148\000\150\000\124\000\168\000\165\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \235\000\008\000\255\255\255\255\117\000\255\255\255\255\255\255\
    \255\255\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\166\000\167\000\162\000\160\000\008\000\
    \161\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\231\000\163\000\007\000\164\000\172\000\
    \232\000\255\255\175\000\205\000\204\000\203\000\086\001\071\001\
    \056\001\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\255\255\255\255\115\000\255\255\116\000\
    \118\000\255\255\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\255\255\248\000\057\001\058\001\
    \008\000\055\001\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\229\000\248\000\007\000\255\255\
    \255\255\230\000\061\001\146\000\255\255\255\255\145\000\054\001\
    \255\255\040\001\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\255\255\255\255\147\000\255\255\
    \255\255\060\001\119\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\037\001\249\000\249\000\
    \053\001\008\000\059\001\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\007\000\249\000\249\000\
    \255\255\255\255\146\001\255\255\000\000\255\255\255\255\052\001\
    \051\001\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\045\001\046\001\043\001\069\001\041\001\
    \000\000\106\001\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\107\001\
    \008\000\106\001\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \224\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\007\000\070\001\044\001\107\001\
    \042\001\038\001\000\000\000\000\255\255\000\000\037\001\255\255\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\039\001\000\000\255\255\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\008\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\219\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\007\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\008\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\214\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\007\000\000\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\151\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\151\000\000\000\000\000\000\000\008\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\206\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\007\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\255\255\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\110\001\
    \000\000\000\000\144\001\144\001\144\001\144\001\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\121\001\145\001\143\001\008\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\190\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \007\000\152\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\120\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\177\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\143\000\
    \007\000\142\000\142\000\142\000\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\255\255\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\140\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\007\000\
    \071\000\000\000\060\000\255\255\093\000\255\255\000\000\000\000\
    \255\255\000\000\255\255\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\071\000\
    \054\000\060\000\070\000\093\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\137\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\008\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\048\000\047\000\
    \047\000\047\000\047\000\068\000\065\000\071\000\105\000\105\000\
    \066\000\000\000\000\000\064\000\000\000\000\000\000\000\067\000\
    \000\000\000\000\052\000\000\000\069\000\048\000\255\255\000\000\
    \049\000\000\000\000\000\000\000\071\000\105\000\105\000\104\000\
    \104\000\000\000\000\000\000\000\050\000\000\000\000\000\000\000\
    \153\000\000\000\153\000\102\000\102\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\152\000\152\000\152\000\000\000\
    \000\000\103\000\103\000\250\000\114\001\114\001\114\001\114\001\
    \114\001\114\001\114\001\114\001\249\000\035\001\034\001\034\001\
    \034\001\034\001\000\000\000\000\076\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\000\000\
    \000\000\000\000\000\000\250\000\035\001\000\000\000\000\000\000\
    \068\000\065\000\151\000\076\000\249\000\066\000\000\000\000\000\
    \064\000\000\000\000\000\036\001\067\000\000\000\000\000\052\000\
    \114\000\069\000\115\001\115\001\115\001\115\001\115\001\115\001\
    \115\001\115\001\113\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\151\000\000\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\000\000\000\000\
    \000\000\255\255\077\000\255\255\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\000\000\000\000\000\000\000\000\077\000\000\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\083\000\000\000\246\000\255\255\246\000\000\000\000\000\
    \245\000\245\000\245\000\245\000\245\000\245\000\245\000\245\000\
    \245\000\245\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \083\000\000\000\000\000\000\000\241\000\000\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \245\000\245\000\245\000\245\000\245\000\245\000\245\000\245\000\
    \245\000\245\000\000\000\242\000\000\000\000\000\000\000\000\000\
    \000\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\242\000\000\000\000\000\000\000\084\000\
    \000\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\085\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\000\000\000\000\
    \000\000\000\000\084\000\000\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\091\000\245\000\
    \245\000\245\000\245\000\245\000\245\000\245\000\245\000\245\000\
    \245\000\047\001\047\001\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\000\000\000\000\091\000\000\000\000\000\
    \000\000\000\000\000\000\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\047\001\047\001\049\001\049\001\049\001\
    \049\001\049\001\049\001\049\001\049\001\049\001\049\001\000\000\
    \048\001\000\000\000\000\000\000\000\000\000\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \048\001\000\000\000\000\000\000\092\000\000\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \093\000\050\001\000\000\050\001\000\000\000\000\049\001\049\001\
    \049\001\049\001\049\001\049\001\049\001\049\001\049\001\049\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\093\000\
    \049\001\049\001\049\001\049\001\049\001\049\001\049\001\049\001\
    \049\001\049\001\000\000\000\000\000\000\000\000\000\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\000\000\000\000\000\000\000\000\092\000\000\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\007\000\000\000\099\001\099\001\099\001\099\001\
    \099\001\099\001\099\001\099\001\099\001\099\001\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\048\001\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\138\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\048\001\000\000\000\000\000\000\008\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\007\000\103\001\103\001\103\001\103\001\103\001\103\001\
    \103\001\103\001\103\001\103\001\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\139\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\008\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \007\000\103\001\103\001\103\001\103\001\103\001\103\001\103\001\
    \103\001\103\001\103\001\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\007\000\
    \125\001\125\001\125\001\125\001\125\001\125\001\125\001\125\001\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\008\000\000\000\008\000\008\000\008\000\008\000\
    \141\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\007\000\126\001\
    \126\001\126\001\126\001\126\001\126\001\126\001\126\001\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\008\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \139\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\007\000\137\001\137\001\
    \137\001\137\001\137\001\137\001\137\001\137\001\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \008\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\178\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\007\000\138\001\138\001\138\001\
    \138\001\138\001\138\001\138\001\138\001\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\179\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\007\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\008\000\000\000\
    \008\000\008\000\008\000\008\000\180\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\007\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\008\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\181\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\007\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\008\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \007\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \191\000\008\000\008\000\008\000\008\000\008\000\008\000\007\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\008\000\000\000\192\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\193\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\007\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\008\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \195\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\007\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \008\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\194\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\007\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\008\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\007\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\008\000\000\000\
    \008\000\008\000\008\000\196\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\007\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\008\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\007\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\008\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\207\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \007\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\208\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\007\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\008\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\209\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\007\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\008\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\007\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \008\000\000\000\008\000\008\000\008\000\215\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\007\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\216\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\007\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\008\000\000\000\
    \008\000\008\000\008\000\008\000\217\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\007\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\008\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\218\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\007\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\008\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \007\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\220\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\007\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\008\000\000\000\008\000\008\000\008\000\008\000\
    \221\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\007\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\000\000\000\000\000\000\
    \000\000\008\000\000\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\222\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\007\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\000\000\000\000\000\000\000\000\
    \008\000\000\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\007\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\000\000\000\000\000\000\000\000\008\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\225\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\007\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\000\000\000\000\000\000\000\000\226\000\000\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\007\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\000\000\000\000\000\000\000\000\008\000\000\000\008\000\
    \008\000\008\000\008\000\227\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\007\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\008\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\228\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \007\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\236\000\
    \000\000\007\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \000\000\000\000\000\000\000\000\008\000\000\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \247\000\247\000\247\000\247\000\247\000\247\000\247\000\247\000\
    \247\000\247\000\000\000\000\000\000\000\000\000\000\000\107\001\
    \000\000\247\000\247\000\247\000\247\000\247\000\247\000\000\000\
    \107\001\241\000\000\000\244\000\244\000\244\000\244\000\244\000\
    \244\000\244\000\244\000\243\000\243\000\000\000\000\000\132\001\
    \132\001\132\001\132\001\000\000\000\000\000\000\000\000\107\001\
    \242\000\247\000\247\000\247\000\247\000\247\000\247\000\238\000\
    \107\001\000\000\000\000\000\000\104\001\000\000\104\001\133\001\
    \239\000\103\001\103\001\103\001\103\001\103\001\103\001\103\001\
    \103\001\103\001\103\001\000\000\000\000\000\000\000\000\000\000\
    \242\000\000\000\000\000\000\000\000\000\000\000\000\000\238\000\
    \113\001\113\001\113\001\113\001\113\001\113\001\113\001\113\001\
    \239\000\247\000\247\000\247\000\247\000\247\000\247\000\247\000\
    \247\000\247\000\247\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\247\000\247\000\247\000\247\000\247\000\247\000\
    \000\000\131\001\000\000\000\000\000\000\238\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\239\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\247\000\247\000\247\000\247\000\247\000\247\000\
    \035\001\034\001\034\001\034\001\034\001\238\000\000\000\000\000\
    \112\001\000\000\000\000\000\000\000\000\000\000\239\000\124\001\
    \124\001\124\001\124\001\124\001\124\001\124\001\124\001\035\001\
    \030\001\255\000\006\001\001\001\016\001\014\001\000\001\029\001\
    \028\001\018\001\017\001\021\001\023\001\022\001\033\001\254\000\
    \253\000\253\000\253\000\253\000\253\000\253\000\253\000\253\000\
    \253\000\007\001\020\001\019\001\031\001\015\001\000\000\032\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\005\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\004\001\002\001\002\001\002\001\
    \002\001\002\001\027\001\000\000\026\001\011\001\002\001\123\001\
    \009\001\013\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\008\001\002\001\
    \002\001\002\001\002\001\002\001\003\001\002\001\002\001\010\001\
    \002\001\002\001\025\001\012\001\024\001\099\001\000\000\253\000\
    \253\000\253\000\253\000\253\000\253\000\253\000\253\000\253\000\
    \253\000\000\000\000\000\000\000\000\000\255\255\000\000\000\000\
    \000\000\000\000\000\000\095\001\100\001\000\000\000\000\000\000\
    \255\255\000\000\000\000\096\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\099\001\097\001\102\001\102\001\102\001\
    \102\001\102\001\102\001\102\001\102\001\101\001\101\001\000\000\
    \000\000\000\000\000\000\095\001\100\001\000\000\000\000\000\000\
    \000\000\095\001\100\001\096\001\000\000\000\000\000\000\000\000\
    \000\000\096\001\000\000\000\000\097\001\000\000\000\000\000\000\
    \000\000\000\000\097\001\000\000\000\000\098\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\001\100\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\096\001\000\000\001\001\000\000\000\000\000\000\255\255\
    \000\000\000\000\097\001\000\000\000\000\098\001\000\000\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\000\000\000\000\000\000\000\000\000\000\000\000\255\255\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\000\000\000\000\000\000\000\000\001\001\000\000\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\000\000\000\000\000\000\000\000\002\001\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\091\001\000\000\001\001\000\000\000\000\092\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\093\001\
    \002\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\002\001\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\089\001\000\000\001\001\000\000\000\000\090\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\002\001\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\087\001\000\000\001\001\000\000\000\000\
    \088\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\000\000\000\000\000\000\000\000\
    \002\001\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\001\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\002\001\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\082\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\001\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\002\001\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\077\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\001\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\000\000\000\000\000\000\000\000\002\001\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\072\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\001\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \000\000\000\000\000\000\000\000\002\001\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\062\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \001\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\000\000\
    \000\000\000\000\000\000\002\001\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \063\001\002\001\002\001\002\001\002\001\002\001\002\001\001\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\000\000\000\000\
    \000\000\000\000\002\001\000\000\065\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\064\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\001\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\000\000\000\000\000\000\
    \000\000\002\001\000\000\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\068\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\001\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\000\000\000\000\000\000\000\000\
    \002\001\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\066\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\001\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\002\001\
    \000\000\002\001\002\001\002\001\067\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\001\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\002\001\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\001\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\000\000\000\000\000\000\000\000\002\001\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\001\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \000\000\000\000\000\000\000\000\002\001\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\073\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \001\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\000\000\
    \000\000\000\000\000\000\074\001\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\001\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\000\000\000\000\
    \000\000\000\000\002\001\000\000\002\001\002\001\002\001\002\001\
    \075\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\001\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\000\000\000\000\000\000\
    \000\000\002\001\000\000\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\076\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\001\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\000\000\000\000\000\000\000\000\
    \002\001\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\001\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\002\001\
    \000\000\002\001\002\001\002\001\078\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\001\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\079\001\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\001\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\000\000\000\000\000\000\000\000\002\001\000\000\002\001\
    \002\001\002\001\002\001\080\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\001\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \000\000\000\000\000\000\000\000\002\001\000\000\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\081\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \001\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\000\000\
    \000\000\000\000\000\000\002\001\000\000\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\001\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\000\000\000\000\
    \000\000\000\000\083\001\000\000\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\001\001\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\000\000\000\000\000\000\
    \000\000\002\001\000\000\002\001\002\001\002\001\002\001\084\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\001\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\000\000\000\000\000\000\000\000\
    \002\001\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\085\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\001\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\002\001\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\094\001\000\000\001\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\000\000\000\000\000\000\000\000\
    \002\001\000\000\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\105\001\105\001\105\001\105\001\
    \105\001\105\001\105\001\105\001\105\001\105\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\105\001\105\001\105\001\
    \105\001\105\001\105\001\099\001\000\000\101\001\101\001\101\001\
    \101\001\101\001\101\001\101\001\101\001\101\001\101\001\000\000\
    \136\001\136\001\136\001\136\001\136\001\136\001\136\001\136\001\
    \000\000\000\000\100\001\000\000\000\000\105\001\105\001\105\001\
    \105\001\105\001\105\001\000\000\099\001\000\000\102\001\102\001\
    \102\001\102\001\102\001\102\001\102\001\102\001\101\001\101\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\100\001\100\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\096\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\097\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \135\001\000\000\000\000\100\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\096\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\097\001\105\001\105\001\105\001\105\001\
    \105\001\105\001\105\001\105\001\105\001\105\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\105\001\105\001\105\001\
    \105\001\105\001\105\001\000\000\000\000\000\000\000\000\000\000\
    \096\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\097\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\105\001\105\001\105\001\
    \105\001\105\001\105\001\000\000\000\000\000\000\000\000\000\000\
    \096\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\097\001\116\001\116\001\116\001\116\001\116\001\116\001\
    \116\001\116\001\116\001\116\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\116\001\116\001\116\001\116\001\116\001\
    \116\001\117\001\117\001\117\001\117\001\117\001\117\001\117\001\
    \117\001\117\001\117\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\117\001\117\001\117\001\117\001\117\001\117\001\
    \255\255\000\000\000\000\116\001\116\001\116\001\116\001\116\001\
    \116\001\000\000\000\000\000\000\000\000\000\000\000\000\127\001\
    \127\001\127\001\127\001\127\001\127\001\127\001\127\001\127\001\
    \127\001\000\000\117\001\117\001\117\001\117\001\117\001\117\001\
    \127\001\127\001\127\001\127\001\127\001\127\001\128\001\128\001\
    \128\001\128\001\128\001\128\001\128\001\128\001\128\001\128\001\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\128\001\
    \128\001\128\001\128\001\128\001\128\001\000\000\000\000\000\000\
    \127\001\127\001\127\001\127\001\127\001\127\001\000\000\000\000\
    \000\000\000\000\000\000\000\000\139\001\139\001\139\001\139\001\
    \139\001\139\001\139\001\139\001\139\001\139\001\000\000\128\001\
    \128\001\128\001\128\001\128\001\128\001\139\001\139\001\139\001\
    \139\001\139\001\139\001\140\001\140\001\140\001\140\001\140\001\
    \140\001\140\001\140\001\140\001\140\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\140\001\140\001\140\001\140\001\
    \140\001\140\001\000\000\000\000\000\000\139\001\139\001\139\001\
    \139\001\139\001\139\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\140\001\140\001\140\001\140\001\
    \140\001\140\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\000\000\000\000\000\000\182\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \013\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \016\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\017\000\017\000\020\000\020\000\020\000\
    \021\000\024\000\024\000\027\000\029\000\003\000\003\000\016\000\
    \032\000\036\000\044\000\050\000\052\000\003\000\055\000\038\000\
    \051\000\056\000\039\000\054\000\057\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\038\000\039\000\039\000\053\000\003\000\003\000\058\000\
    \061\000\040\000\064\000\004\000\004\000\003\000\059\000\040\000\
    \066\000\067\000\040\000\004\000\068\000\069\000\003\000\072\000\
    \066\000\073\000\033\000\053\000\004\000\033\000\040\000\004\000\
    \040\000\040\000\074\000\040\000\046\000\059\000\049\000\032\000\
    \065\000\046\000\065\000\004\000\004\000\078\000\065\000\079\000\
    \080\000\081\000\086\000\004\000\007\000\075\000\033\000\046\000\
    \033\000\033\000\087\000\088\000\004\000\049\000\049\000\004\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\089\000\094\000\075\000\096\000\097\000\098\000\
    \000\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\099\000\100\000\102\000\053\000\007\000\
    \110\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\008\000\033\000\082\000\090\000\120\000\
    \049\000\103\000\108\000\121\000\123\000\124\000\104\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\112\000\113\000\127\000\082\000\090\000\116\000\119\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\106\000\128\000\130\000\131\000\008\000\133\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\009\000\122\000\009\000\134\000\051\000\009\000\
    \143\000\054\000\145\000\146\000\149\000\122\000\154\000\156\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\114\000\115\000\113\000\117\000\061\000\125\000\
    \126\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\155\000\155\000\158\000\159\000\009\000\
    \160\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\010\000\162\000\010\000\163\000\170\000\
    \010\000\094\000\174\000\201\000\201\000\202\000\006\001\011\001\
    \016\001\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\103\000\108\000\114\000\118\000\115\000\
    \117\000\129\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\104\000\239\000\015\001\015\001\
    \010\000\017\001\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\011\000\239\000\011\000\112\000\
    \113\000\011\000\014\001\144\000\116\000\119\000\144\000\018\001\
    \106\000\032\001\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\132\000\135\000\144\000\172\000\
    \175\000\014\001\118\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\036\001\248\000\250\000\
    \052\001\011\000\058\001\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\012\000\248\000\250\000\
    \114\000\115\000\145\001\117\000\255\255\125\000\126\000\019\001\
    \019\001\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\023\001\023\001\030\001\012\001\031\001\
    \255\255\097\001\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\255\255\255\255\255\255\106\001\
    \012\000\097\001\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\014\000\012\001\030\001\106\001\
    \031\001\033\001\255\255\255\255\118\000\255\255\033\001\129\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\255\255\255\255\033\001\255\255\037\001\255\255\
    \255\255\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\255\255\255\255\255\255\255\255\014\000\
    \255\255\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\015\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\132\000\135\000\255\255\172\000\175\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\255\255\255\255\255\255\255\255\015\000\255\255\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\015\000\015\000\015\000\015\000\
    \015\000\015\000\018\000\255\255\142\000\142\000\142\000\142\000\
    \142\000\142\000\142\000\142\000\142\000\142\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \255\255\142\000\255\255\255\255\255\255\255\255\255\255\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\142\000\255\255\255\255\255\255\018\000\255\255\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\018\000\018\000\018\000\018\000\018\000\018\000\018\000\
    \018\000\031\000\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\037\001\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\108\001\
    \255\255\255\255\141\001\141\001\141\001\141\001\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \255\255\118\001\141\001\141\001\031\000\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \035\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\255\255\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\118\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\255\255\
    \255\255\255\255\255\255\035\000\255\255\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\041\000\
    \042\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\108\001\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\255\255\
    \255\255\255\255\255\255\042\000\255\255\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\043\000\
    \045\000\255\255\060\000\060\000\093\000\093\000\255\255\255\255\
    \141\001\255\255\118\001\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\255\255\255\255\045\000\
    \045\000\060\000\045\000\093\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\255\255\255\255\
    \255\255\255\255\043\000\255\255\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\048\000\048\000\
    \048\000\048\000\048\000\045\000\045\000\071\000\101\000\105\000\
    \045\000\255\255\255\255\045\000\255\255\255\255\255\255\045\000\
    \255\255\255\255\045\000\255\255\045\000\048\000\095\000\255\255\
    \048\000\255\255\255\255\255\255\071\000\101\000\105\000\101\000\
    \105\000\255\255\255\255\255\255\048\000\255\255\255\255\255\255\
    \151\000\255\255\151\000\101\000\105\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\255\255\
    \255\255\101\000\105\000\238\000\113\001\113\001\113\001\113\001\
    \113\001\113\001\113\001\113\001\238\000\035\001\035\001\035\001\
    \035\001\035\001\255\255\255\255\076\000\241\000\241\000\241\000\
    \241\000\241\000\241\000\241\000\241\000\241\000\241\000\255\255\
    \255\255\255\255\255\255\238\000\035\001\255\255\255\255\255\255\
    \071\000\071\000\241\000\076\000\238\000\071\000\255\255\255\255\
    \071\000\255\255\255\255\035\001\071\000\255\255\255\255\071\000\
    \095\000\071\000\114\001\114\001\114\001\114\001\114\001\114\001\
    \114\001\114\001\095\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\241\000\255\255\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\255\255\255\255\
    \255\255\060\000\076\000\093\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\255\255\255\255\255\255\255\255\077\000\255\255\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\083\000\255\255\242\000\095\000\242\000\255\255\255\255\
    \242\000\242\000\242\000\242\000\242\000\242\000\242\000\242\000\
    \242\000\242\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \083\000\255\255\255\255\255\255\243\000\255\255\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \245\000\245\000\245\000\245\000\245\000\245\000\245\000\245\000\
    \245\000\245\000\255\255\243\000\255\255\255\255\255\255\255\255\
    \255\255\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\243\000\255\255\255\255\255\255\083\000\
    \255\255\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\084\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\255\255\255\255\
    \255\255\255\255\084\000\255\255\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\091\000\246\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\246\000\246\000\
    \246\000\022\001\022\001\022\001\022\001\022\001\022\001\022\001\
    \022\001\022\001\022\001\255\255\255\255\091\000\255\255\255\255\
    \255\255\255\255\255\255\047\001\047\001\047\001\047\001\047\001\
    \047\001\047\001\047\001\047\001\047\001\049\001\049\001\049\001\
    \049\001\049\001\049\001\049\001\049\001\049\001\049\001\255\255\
    \047\001\255\255\255\255\255\255\255\255\255\255\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \047\001\255\255\255\255\255\255\091\000\255\255\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \092\000\048\001\255\255\048\001\255\255\255\255\048\001\048\001\
    \048\001\048\001\048\001\048\001\048\001\048\001\048\001\048\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\092\000\
    \050\001\050\001\050\001\050\001\050\001\050\001\050\001\050\001\
    \050\001\050\001\255\255\255\255\255\255\255\255\255\255\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\255\255\255\255\255\255\255\255\092\000\255\255\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\137\000\255\255\099\001\099\001\099\001\099\001\
    \099\001\099\001\099\001\099\001\099\001\099\001\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \255\255\099\001\255\255\255\255\255\255\255\255\255\255\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\099\001\255\255\255\255\255\255\137\000\255\255\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\137\000\137\000\137\000\137\000\137\000\137\000\137\000\
    \137\000\138\000\103\001\103\001\103\001\103\001\103\001\103\001\
    \103\001\103\001\103\001\103\001\255\255\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \255\255\255\255\255\255\255\255\138\000\255\255\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \138\000\138\000\138\000\138\000\138\000\138\000\138\000\138\000\
    \139\000\104\001\104\001\104\001\104\001\104\001\104\001\104\001\
    \104\001\104\001\104\001\255\255\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\255\255\
    \255\255\255\255\255\255\139\000\255\255\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\139\000\
    \139\000\139\000\139\000\139\000\139\000\139\000\139\000\140\000\
    \124\001\124\001\124\001\124\001\124\001\124\001\124\001\124\001\
    \255\255\255\255\255\255\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\255\255\255\255\
    \255\255\255\255\140\000\255\255\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\140\000\140\000\
    \140\000\140\000\140\000\140\000\140\000\140\000\141\000\125\001\
    \125\001\125\001\125\001\125\001\125\001\125\001\125\001\255\255\
    \255\255\255\255\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\255\255\255\255\255\255\
    \255\255\141\000\255\255\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\177\000\136\001\136\001\
    \136\001\136\001\136\001\136\001\136\001\136\001\255\255\255\255\
    \255\255\177\000\177\000\177\000\177\000\177\000\177\000\177\000\
    \177\000\177\000\177\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\177\000\177\000\177\000\177\000\177\000\177\000\
    \177\000\177\000\177\000\177\000\177\000\177\000\177\000\177\000\
    \177\000\177\000\177\000\177\000\177\000\177\000\177\000\177\000\
    \177\000\177\000\177\000\177\000\255\255\255\255\255\255\255\255\
    \177\000\255\255\177\000\177\000\177\000\177\000\177\000\177\000\
    \177\000\177\000\177\000\177\000\177\000\177\000\177\000\177\000\
    \177\000\177\000\177\000\177\000\177\000\177\000\177\000\177\000\
    \177\000\177\000\177\000\177\000\178\000\137\001\137\001\137\001\
    \137\001\137\001\137\001\137\001\137\001\255\255\255\255\255\255\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\255\255\255\255\255\255\255\255\178\000\
    \255\255\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\178\000\178\000\178\000\178\000\178\000\
    \178\000\178\000\178\000\179\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\255\255\255\255\255\255\255\255\179\000\255\255\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\180\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\255\255\255\255\255\255\255\255\180\000\255\255\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\180\000\180\000\180\000\180\000\180\000\180\000\180\000\
    \180\000\181\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\181\000\181\000\181\000\
    \181\000\181\000\181\000\181\000\181\000\181\000\181\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\181\000\181\000\
    \181\000\181\000\181\000\181\000\181\000\181\000\181\000\181\000\
    \181\000\181\000\181\000\181\000\181\000\181\000\181\000\181\000\
    \181\000\181\000\181\000\181\000\181\000\181\000\181\000\181\000\
    \255\255\255\255\255\255\255\255\181\000\255\255\181\000\181\000\
    \181\000\181\000\181\000\181\000\181\000\181\000\181\000\181\000\
    \181\000\181\000\181\000\181\000\181\000\181\000\181\000\181\000\
    \181\000\181\000\181\000\181\000\181\000\181\000\181\000\181\000\
    \190\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\255\255\
    \255\255\255\255\255\255\190\000\255\255\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\190\000\190\000\190\000\190\000\190\000\190\000\191\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\255\255\255\255\
    \255\255\255\255\191\000\255\255\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\191\000\191\000\192\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\255\255\255\255\255\255\
    \255\255\192\000\255\255\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\192\000\193\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\255\255\255\255\255\255\255\255\
    \193\000\255\255\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\194\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\255\255\255\255\255\255\255\255\194\000\
    \255\255\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\195\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\255\255\255\255\255\255\255\255\195\000\255\255\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\195\000\195\000\195\000\195\000\195\000\195\000\
    \195\000\195\000\196\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\255\255\255\255\255\255\255\255\196\000\255\255\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\196\000\196\000\196\000\196\000\196\000\196\000\196\000\
    \196\000\206\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \255\255\255\255\255\255\255\255\206\000\255\255\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \207\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\255\255\
    \255\255\255\255\255\255\207\000\255\255\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\207\000\
    \207\000\207\000\207\000\207\000\207\000\207\000\207\000\208\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\208\000\208\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\208\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\255\255\255\255\
    \255\255\255\255\208\000\255\255\208\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\208\000\208\000\
    \208\000\208\000\208\000\208\000\208\000\208\000\209\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\255\255\255\255\255\255\
    \255\255\209\000\255\255\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\209\000\209\000\209\000\
    \209\000\209\000\209\000\209\000\209\000\214\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\255\255\255\255\255\255\255\255\
    \214\000\255\255\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\214\000\214\000\214\000\214\000\
    \214\000\214\000\214\000\214\000\215\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\255\255\255\255\255\255\255\255\215\000\
    \255\255\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\215\000\215\000\215\000\215\000\215\000\
    \215\000\215\000\215\000\216\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\216\000\
    \216\000\216\000\216\000\216\000\216\000\216\000\216\000\216\000\
    \216\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \216\000\216\000\216\000\216\000\216\000\216\000\216\000\216\000\
    \216\000\216\000\216\000\216\000\216\000\216\000\216\000\216\000\
    \216\000\216\000\216\000\216\000\216\000\216\000\216\000\216\000\
    \216\000\216\000\255\255\255\255\255\255\255\255\216\000\255\255\
    \216\000\216\000\216\000\216\000\216\000\216\000\216\000\216\000\
    \216\000\216\000\216\000\216\000\216\000\216\000\216\000\216\000\
    \216\000\216\000\216\000\216\000\216\000\216\000\216\000\216\000\
    \216\000\216\000\217\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\255\255\255\255\255\255\255\255\217\000\255\255\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\217\000\217\000\217\000\217\000\217\000\217\000\217\000\
    \217\000\218\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \255\255\255\255\255\255\255\255\218\000\255\255\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \218\000\218\000\218\000\218\000\218\000\218\000\218\000\218\000\
    \219\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\255\255\
    \255\255\255\255\255\255\219\000\255\255\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\219\000\
    \219\000\219\000\219\000\219\000\219\000\219\000\219\000\220\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\255\255\255\255\
    \255\255\255\255\220\000\255\255\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\220\000\220\000\
    \220\000\220\000\220\000\220\000\220\000\220\000\221\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\255\255\255\255\255\255\
    \255\255\221\000\255\255\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\221\000\221\000\221\000\
    \221\000\221\000\221\000\221\000\221\000\222\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\255\255\255\255\255\255\255\255\
    \222\000\255\255\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\222\000\222\000\222\000\222\000\
    \222\000\222\000\222\000\222\000\224\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \224\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\255\255\255\255\255\255\255\255\224\000\
    \255\255\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\224\000\224\000\224\000\224\000\224\000\
    \224\000\224\000\224\000\225\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\225\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\225\000\255\255\255\255\255\255\255\255\225\000\255\255\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\225\000\225\000\225\000\225\000\225\000\225\000\225\000\
    \225\000\225\000\226\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\226\000\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \226\000\255\255\255\255\255\255\255\255\226\000\255\255\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \226\000\226\000\226\000\226\000\226\000\226\000\226\000\226\000\
    \226\000\227\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\227\000\227\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\227\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \255\255\255\255\255\255\255\255\227\000\255\255\227\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \227\000\227\000\227\000\227\000\227\000\227\000\227\000\227\000\
    \228\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\228\000\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\228\000\255\255\
    \255\255\255\255\255\255\228\000\255\255\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\228\000\228\000\
    \228\000\228\000\228\000\228\000\228\000\228\000\228\000\235\000\
    \255\255\235\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\235\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\235\000\235\000\
    \255\255\255\255\255\255\255\255\235\000\255\255\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\235\000\235\000\235\000\235\000\
    \240\000\240\000\240\000\240\000\240\000\240\000\240\000\240\000\
    \240\000\240\000\255\255\255\255\255\255\255\255\255\255\096\001\
    \255\255\240\000\240\000\240\000\240\000\240\000\240\000\255\255\
    \096\001\244\000\255\255\244\000\244\000\244\000\244\000\244\000\
    \244\000\244\000\244\000\244\000\244\000\255\255\255\255\129\001\
    \129\001\129\001\129\001\255\255\255\255\255\255\255\255\096\001\
    \244\000\240\000\240\000\240\000\240\000\240\000\240\000\244\000\
    \096\001\255\255\255\255\255\255\100\001\255\255\100\001\129\001\
    \244\000\100\001\100\001\100\001\100\001\100\001\100\001\100\001\
    \100\001\100\001\100\001\255\255\255\255\255\255\255\255\255\255\
    \244\000\255\255\255\255\255\255\255\255\255\255\255\255\244\000\
    \110\001\110\001\110\001\110\001\110\001\110\001\110\001\110\001\
    \244\000\247\000\247\000\247\000\247\000\247\000\247\000\247\000\
    \247\000\247\000\247\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\247\000\247\000\247\000\247\000\247\000\247\000\
    \255\255\129\001\255\255\255\255\255\255\247\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\247\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\247\000\247\000\247\000\247\000\247\000\247\000\
    \251\000\251\000\251\000\251\000\251\000\247\000\255\255\255\255\
    \110\001\255\255\255\255\255\255\255\255\255\255\247\000\120\001\
    \120\001\120\001\120\001\120\001\120\001\120\001\120\001\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\255\255\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\255\255\251\000\251\000\251\000\120\001\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\251\000\251\000\251\000\251\000\251\000\
    \251\000\251\000\251\000\251\000\251\000\253\000\255\255\253\000\
    \253\000\253\000\253\000\253\000\253\000\253\000\253\000\253\000\
    \253\000\255\255\255\255\255\255\255\255\129\001\255\255\255\255\
    \255\255\255\255\255\255\253\000\253\000\255\255\255\255\255\255\
    \110\001\255\255\255\255\253\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\254\000\253\000\254\000\254\000\254\000\
    \254\000\254\000\254\000\254\000\254\000\254\000\254\000\255\255\
    \255\255\255\255\255\255\253\000\253\000\255\255\255\255\255\255\
    \255\255\254\000\254\000\253\000\255\255\255\255\255\255\255\255\
    \255\255\254\000\255\255\255\255\253\000\255\255\255\255\255\255\
    \255\255\255\255\254\000\255\255\255\255\254\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\254\000\254\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\254\000\255\255\001\001\255\255\255\255\255\255\120\001\
    \255\255\255\255\254\000\255\255\255\255\254\000\255\255\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\255\255\255\255\255\255\255\255\255\255\255\255\251\000\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\255\255\255\255\255\255\255\255\001\001\255\255\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\001\
    \001\001\001\001\002\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\255\255\255\255\255\255\255\255\002\001\255\255\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\002\001\002\001\002\001\002\001\002\001\002\001\002\001\
    \002\001\003\001\255\255\003\001\255\255\255\255\003\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\003\001\
    \003\001\003\001\003\001\003\001\003\001\003\001\003\001\003\001\
    \003\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \003\001\003\001\003\001\003\001\003\001\003\001\003\001\003\001\
    \003\001\003\001\003\001\003\001\003\001\003\001\003\001\003\001\
    \003\001\003\001\003\001\003\001\003\001\003\001\003\001\003\001\
    \003\001\003\001\255\255\255\255\255\255\255\255\003\001\255\255\
    \003\001\003\001\003\001\003\001\003\001\003\001\003\001\003\001\
    \003\001\003\001\003\001\003\001\003\001\003\001\003\001\003\001\
    \003\001\003\001\003\001\003\001\003\001\003\001\003\001\003\001\
    \003\001\003\001\004\001\255\255\004\001\255\255\255\255\004\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \004\001\004\001\004\001\004\001\004\001\004\001\004\001\004\001\
    \004\001\004\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\004\001\004\001\004\001\004\001\004\001\004\001\004\001\
    \004\001\004\001\004\001\004\001\004\001\004\001\004\001\004\001\
    \004\001\004\001\004\001\004\001\004\001\004\001\004\001\004\001\
    \004\001\004\001\004\001\255\255\255\255\255\255\255\255\004\001\
    \255\255\004\001\004\001\004\001\004\001\004\001\004\001\004\001\
    \004\001\004\001\004\001\004\001\004\001\004\001\004\001\004\001\
    \004\001\004\001\004\001\004\001\004\001\004\001\004\001\004\001\
    \004\001\004\001\004\001\005\001\255\255\005\001\255\255\255\255\
    \005\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\005\001\005\001\005\001\005\001\005\001\005\001\005\001\
    \005\001\005\001\005\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\005\001\005\001\005\001\005\001\005\001\005\001\
    \005\001\005\001\005\001\005\001\005\001\005\001\005\001\005\001\
    \005\001\005\001\005\001\005\001\005\001\005\001\005\001\005\001\
    \005\001\005\001\005\001\005\001\255\255\255\255\255\255\255\255\
    \005\001\255\255\005\001\005\001\005\001\005\001\005\001\005\001\
    \005\001\005\001\005\001\005\001\005\001\005\001\005\001\005\001\
    \005\001\005\001\005\001\005\001\005\001\005\001\005\001\005\001\
    \005\001\005\001\005\001\005\001\008\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \008\001\008\001\008\001\008\001\008\001\008\001\008\001\008\001\
    \008\001\008\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\008\001\008\001\008\001\008\001\008\001\008\001\008\001\
    \008\001\008\001\008\001\008\001\008\001\008\001\008\001\008\001\
    \008\001\008\001\008\001\008\001\008\001\008\001\008\001\008\001\
    \008\001\008\001\008\001\255\255\255\255\255\255\255\255\008\001\
    \255\255\008\001\008\001\008\001\008\001\008\001\008\001\008\001\
    \008\001\008\001\008\001\008\001\008\001\008\001\008\001\008\001\
    \008\001\008\001\008\001\008\001\008\001\008\001\008\001\008\001\
    \008\001\008\001\008\001\009\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\009\001\
    \009\001\009\001\009\001\009\001\009\001\009\001\009\001\009\001\
    \009\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \009\001\009\001\009\001\009\001\009\001\009\001\009\001\009\001\
    \009\001\009\001\009\001\009\001\009\001\009\001\009\001\009\001\
    \009\001\009\001\009\001\009\001\009\001\009\001\009\001\009\001\
    \009\001\009\001\255\255\255\255\255\255\255\255\009\001\255\255\
    \009\001\009\001\009\001\009\001\009\001\009\001\009\001\009\001\
    \009\001\009\001\009\001\009\001\009\001\009\001\009\001\009\001\
    \009\001\009\001\009\001\009\001\009\001\009\001\009\001\009\001\
    \009\001\009\001\010\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\010\001\010\001\
    \010\001\010\001\010\001\010\001\010\001\010\001\010\001\010\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\010\001\
    \010\001\010\001\010\001\010\001\010\001\010\001\010\001\010\001\
    \010\001\010\001\010\001\010\001\010\001\010\001\010\001\010\001\
    \010\001\010\001\010\001\010\001\010\001\010\001\010\001\010\001\
    \010\001\255\255\255\255\255\255\255\255\010\001\255\255\010\001\
    \010\001\010\001\010\001\010\001\010\001\010\001\010\001\010\001\
    \010\001\010\001\010\001\010\001\010\001\010\001\010\001\010\001\
    \010\001\010\001\010\001\010\001\010\001\010\001\010\001\010\001\
    \010\001\013\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\013\001\013\001\013\001\
    \013\001\013\001\013\001\013\001\013\001\013\001\013\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\013\001\013\001\
    \013\001\013\001\013\001\013\001\013\001\013\001\013\001\013\001\
    \013\001\013\001\013\001\013\001\013\001\013\001\013\001\013\001\
    \013\001\013\001\013\001\013\001\013\001\013\001\013\001\013\001\
    \255\255\255\255\255\255\255\255\013\001\255\255\013\001\013\001\
    \013\001\013\001\013\001\013\001\013\001\013\001\013\001\013\001\
    \013\001\013\001\013\001\013\001\013\001\013\001\013\001\013\001\
    \013\001\013\001\013\001\013\001\013\001\013\001\013\001\013\001\
    \062\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\062\001\062\001\062\001\062\001\
    \062\001\062\001\062\001\062\001\062\001\062\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\062\001\062\001\062\001\
    \062\001\062\001\062\001\062\001\062\001\062\001\062\001\062\001\
    \062\001\062\001\062\001\062\001\062\001\062\001\062\001\062\001\
    \062\001\062\001\062\001\062\001\062\001\062\001\062\001\255\255\
    \255\255\255\255\255\255\062\001\255\255\062\001\062\001\062\001\
    \062\001\062\001\062\001\062\001\062\001\062\001\062\001\062\001\
    \062\001\062\001\062\001\062\001\062\001\062\001\062\001\062\001\
    \062\001\062\001\062\001\062\001\062\001\062\001\062\001\063\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\255\255\255\255\
    \255\255\255\255\063\001\255\255\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\063\001\063\001\
    \063\001\063\001\063\001\063\001\063\001\063\001\064\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\064\001\064\001\064\001\064\001\064\001\064\001\
    \064\001\064\001\064\001\064\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\064\001\064\001\064\001\064\001\064\001\
    \064\001\064\001\064\001\064\001\064\001\064\001\064\001\064\001\
    \064\001\064\001\064\001\064\001\064\001\064\001\064\001\064\001\
    \064\001\064\001\064\001\064\001\064\001\255\255\255\255\255\255\
    \255\255\064\001\255\255\064\001\064\001\064\001\064\001\064\001\
    \064\001\064\001\064\001\064\001\064\001\064\001\064\001\064\001\
    \064\001\064\001\064\001\064\001\064\001\064\001\064\001\064\001\
    \064\001\064\001\064\001\064\001\064\001\065\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\065\001\065\001\065\001\065\001\065\001\065\001\065\001\
    \065\001\065\001\065\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\065\001\065\001\065\001\065\001\065\001\065\001\
    \065\001\065\001\065\001\065\001\065\001\065\001\065\001\065\001\
    \065\001\065\001\065\001\065\001\065\001\065\001\065\001\065\001\
    \065\001\065\001\065\001\065\001\255\255\255\255\255\255\255\255\
    \065\001\255\255\065\001\065\001\065\001\065\001\065\001\065\001\
    \065\001\065\001\065\001\065\001\065\001\065\001\065\001\065\001\
    \065\001\065\001\065\001\065\001\065\001\065\001\065\001\065\001\
    \065\001\065\001\065\001\065\001\066\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\255\255\255\255\255\255\255\255\066\001\
    \255\255\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\066\001\066\001\066\001\066\001\066\001\
    \066\001\066\001\066\001\067\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\255\255\255\255\255\255\255\255\067\001\255\255\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\067\001\067\001\067\001\067\001\067\001\067\001\
    \067\001\067\001\068\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\068\001\068\001\
    \068\001\068\001\068\001\068\001\068\001\068\001\068\001\068\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\068\001\
    \068\001\068\001\068\001\068\001\068\001\068\001\068\001\068\001\
    \068\001\068\001\068\001\068\001\068\001\068\001\068\001\068\001\
    \068\001\068\001\068\001\068\001\068\001\068\001\068\001\068\001\
    \068\001\255\255\255\255\255\255\255\255\068\001\255\255\068\001\
    \068\001\068\001\068\001\068\001\068\001\068\001\068\001\068\001\
    \068\001\068\001\068\001\068\001\068\001\068\001\068\001\068\001\
    \068\001\068\001\068\001\068\001\068\001\068\001\068\001\068\001\
    \068\001\072\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \255\255\255\255\255\255\255\255\072\001\255\255\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \072\001\072\001\072\001\072\001\072\001\072\001\072\001\072\001\
    \073\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\073\001\073\001\073\001\073\001\
    \073\001\073\001\073\001\073\001\073\001\073\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\073\001\073\001\073\001\
    \073\001\073\001\073\001\073\001\073\001\073\001\073\001\073\001\
    \073\001\073\001\073\001\073\001\073\001\073\001\073\001\073\001\
    \073\001\073\001\073\001\073\001\073\001\073\001\073\001\255\255\
    \255\255\255\255\255\255\073\001\255\255\073\001\073\001\073\001\
    \073\001\073\001\073\001\073\001\073\001\073\001\073\001\073\001\
    \073\001\073\001\073\001\073\001\073\001\073\001\073\001\073\001\
    \073\001\073\001\073\001\073\001\073\001\073\001\073\001\074\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\074\001\074\001\074\001\074\001\074\001\
    \074\001\074\001\074\001\074\001\074\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\074\001\074\001\074\001\074\001\
    \074\001\074\001\074\001\074\001\074\001\074\001\074\001\074\001\
    \074\001\074\001\074\001\074\001\074\001\074\001\074\001\074\001\
    \074\001\074\001\074\001\074\001\074\001\074\001\255\255\255\255\
    \255\255\255\255\074\001\255\255\074\001\074\001\074\001\074\001\
    \074\001\074\001\074\001\074\001\074\001\074\001\074\001\074\001\
    \074\001\074\001\074\001\074\001\074\001\074\001\074\001\074\001\
    \074\001\074\001\074\001\074\001\074\001\074\001\075\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\075\001\075\001\075\001\075\001\075\001\075\001\
    \075\001\075\001\075\001\075\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\075\001\075\001\075\001\075\001\075\001\
    \075\001\075\001\075\001\075\001\075\001\075\001\075\001\075\001\
    \075\001\075\001\075\001\075\001\075\001\075\001\075\001\075\001\
    \075\001\075\001\075\001\075\001\075\001\255\255\255\255\255\255\
    \255\255\075\001\255\255\075\001\075\001\075\001\075\001\075\001\
    \075\001\075\001\075\001\075\001\075\001\075\001\075\001\075\001\
    \075\001\075\001\075\001\075\001\075\001\075\001\075\001\075\001\
    \075\001\075\001\075\001\075\001\075\001\076\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\076\001\076\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\076\001\076\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\076\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\076\001\076\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\076\001\076\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\076\001\076\001\076\001\255\255\255\255\255\255\255\255\
    \076\001\255\255\076\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\076\001\076\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\076\001\076\001\076\001\076\001\076\001\076\001\076\001\
    \076\001\076\001\076\001\076\001\077\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \077\001\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \077\001\077\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \077\001\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \077\001\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \077\001\077\001\077\001\255\255\255\255\255\255\255\255\077\001\
    \255\255\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \077\001\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \077\001\077\001\077\001\077\001\077\001\077\001\077\001\077\001\
    \077\001\077\001\077\001\078\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\078\001\
    \078\001\078\001\078\001\078\001\078\001\078\001\078\001\078\001\
    \078\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \078\001\078\001\078\001\078\001\078\001\078\001\078\001\078\001\
    \078\001\078\001\078\001\078\001\078\001\078\001\078\001\078\001\
    \078\001\078\001\078\001\078\001\078\001\078\001\078\001\078\001\
    \078\001\078\001\255\255\255\255\255\255\255\255\078\001\255\255\
    \078\001\078\001\078\001\078\001\078\001\078\001\078\001\078\001\
    \078\001\078\001\078\001\078\001\078\001\078\001\078\001\078\001\
    \078\001\078\001\078\001\078\001\078\001\078\001\078\001\078\001\
    \078\001\078\001\079\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\255\255\255\255\255\255\255\255\079\001\255\255\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\080\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\080\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\080\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\080\001\080\001\
    \255\255\255\255\255\255\255\255\080\001\255\255\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\080\001\080\001\
    \081\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\081\001\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\081\001\255\255\
    \255\255\255\255\255\255\081\001\255\255\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\081\001\081\001\081\001\082\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\082\001\082\001\082\001\082\001\082\001\
    \082\001\082\001\082\001\082\001\082\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\082\001\082\001\082\001\082\001\
    \082\001\082\001\082\001\082\001\082\001\082\001\082\001\082\001\
    \082\001\082\001\082\001\082\001\082\001\082\001\082\001\082\001\
    \082\001\082\001\082\001\082\001\082\001\082\001\255\255\255\255\
    \255\255\255\255\082\001\255\255\082\001\082\001\082\001\082\001\
    \082\001\082\001\082\001\082\001\082\001\082\001\082\001\082\001\
    \082\001\082\001\082\001\082\001\082\001\082\001\082\001\082\001\
    \082\001\082\001\082\001\082\001\082\001\082\001\083\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\083\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\083\001\255\255\255\255\255\255\
    \255\255\083\001\255\255\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\083\001\083\001\083\001\083\001\
    \083\001\083\001\083\001\083\001\083\001\084\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\084\001\084\001\084\001\084\001\084\001\084\001\084\001\
    \084\001\084\001\084\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\084\001\084\001\084\001\084\001\084\001\084\001\
    \084\001\084\001\084\001\084\001\084\001\084\001\084\001\084\001\
    \084\001\084\001\084\001\084\001\084\001\084\001\084\001\084\001\
    \084\001\084\001\084\001\084\001\255\255\255\255\255\255\255\255\
    \084\001\255\255\084\001\084\001\084\001\084\001\084\001\084\001\
    \084\001\084\001\084\001\084\001\084\001\084\001\084\001\084\001\
    \084\001\084\001\084\001\084\001\084\001\084\001\084\001\084\001\
    \084\001\084\001\084\001\084\001\085\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \085\001\085\001\085\001\085\001\085\001\085\001\085\001\085\001\
    \085\001\085\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\085\001\085\001\085\001\085\001\085\001\085\001\085\001\
    \085\001\085\001\085\001\085\001\085\001\085\001\085\001\085\001\
    \085\001\085\001\085\001\085\001\085\001\085\001\085\001\085\001\
    \085\001\085\001\085\001\255\255\255\255\255\255\255\255\085\001\
    \255\255\085\001\085\001\085\001\085\001\085\001\085\001\085\001\
    \085\001\085\001\085\001\085\001\085\001\085\001\085\001\085\001\
    \085\001\085\001\085\001\085\001\085\001\085\001\085\001\085\001\
    \085\001\085\001\085\001\093\001\255\255\093\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\093\001\093\001\093\001\093\001\093\001\093\001\093\001\
    \093\001\093\001\093\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\093\001\093\001\093\001\093\001\093\001\093\001\
    \093\001\093\001\093\001\093\001\093\001\093\001\093\001\093\001\
    \093\001\093\001\093\001\093\001\093\001\093\001\093\001\093\001\
    \093\001\093\001\093\001\093\001\255\255\255\255\255\255\255\255\
    \093\001\255\255\093\001\093\001\093\001\093\001\093\001\093\001\
    \093\001\093\001\093\001\093\001\093\001\093\001\093\001\093\001\
    \093\001\093\001\093\001\093\001\093\001\093\001\093\001\093\001\
    \093\001\093\001\093\001\093\001\098\001\098\001\098\001\098\001\
    \098\001\098\001\098\001\098\001\098\001\098\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\098\001\098\001\098\001\
    \098\001\098\001\098\001\101\001\255\255\101\001\101\001\101\001\
    \101\001\101\001\101\001\101\001\101\001\101\001\101\001\255\255\
    \131\001\131\001\131\001\131\001\131\001\131\001\131\001\131\001\
    \255\255\255\255\101\001\255\255\255\255\098\001\098\001\098\001\
    \098\001\098\001\098\001\255\255\102\001\255\255\102\001\102\001\
    \102\001\102\001\102\001\102\001\102\001\102\001\102\001\102\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\101\001\102\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\102\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\102\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \131\001\255\255\255\255\102\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\102\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\102\001\105\001\105\001\105\001\105\001\
    \105\001\105\001\105\001\105\001\105\001\105\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\105\001\105\001\105\001\
    \105\001\105\001\105\001\255\255\255\255\255\255\255\255\255\255\
    \105\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\105\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\105\001\105\001\105\001\
    \105\001\105\001\105\001\255\255\255\255\255\255\255\255\255\255\
    \105\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\105\001\112\001\112\001\112\001\112\001\112\001\112\001\
    \112\001\112\001\112\001\112\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\112\001\112\001\112\001\112\001\112\001\
    \112\001\116\001\116\001\116\001\116\001\116\001\116\001\116\001\
    \116\001\116\001\116\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\116\001\116\001\116\001\116\001\116\001\116\001\
    \131\001\255\255\255\255\112\001\112\001\112\001\112\001\112\001\
    \112\001\255\255\255\255\255\255\255\255\255\255\255\255\123\001\
    \123\001\123\001\123\001\123\001\123\001\123\001\123\001\123\001\
    \123\001\255\255\116\001\116\001\116\001\116\001\116\001\116\001\
    \123\001\123\001\123\001\123\001\123\001\123\001\127\001\127\001\
    \127\001\127\001\127\001\127\001\127\001\127\001\127\001\127\001\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\127\001\
    \127\001\127\001\127\001\127\001\127\001\255\255\255\255\255\255\
    \123\001\123\001\123\001\123\001\123\001\123\001\255\255\255\255\
    \255\255\255\255\255\255\255\255\135\001\135\001\135\001\135\001\
    \135\001\135\001\135\001\135\001\135\001\135\001\255\255\127\001\
    \127\001\127\001\127\001\127\001\127\001\135\001\135\001\135\001\
    \135\001\135\001\135\001\139\001\139\001\139\001\139\001\139\001\
    \139\001\139\001\139\001\139\001\139\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\139\001\139\001\139\001\139\001\
    \139\001\139\001\255\255\255\255\255\255\135\001\135\001\135\001\
    \135\001\135\001\135\001\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\139\001\139\001\139\001\139\001\
    \139\001\139\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\002\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\004\000\005\000\000\000\043\000\000\000\000\000\
    \000\000\000\000\006\000\038\000\152\000\007\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\000\000\011\000\000\000\
    \000\000\000\000\000\000\014\000\014\000\014\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\023\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\038\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\060\000\060\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\001\000\020\000\020\000\033\000\033\000\051\000\
    \057\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\001\000\020\000\020\000\033\000\033\000\051\000\057\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\051\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\051\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\048\000\000\000\000\000\000\000\028\000\017\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\000\000\000\000\000\000\000\000\054\000\000\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\057\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \057\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\000\000\000\000\000\000\000\000\054\000\
    \000\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\048\000\075\000\076\000\082\000\083\000\090\000\
    \093\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\048\000\075\000\076\000\082\000\083\000\090\000\093\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\091\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\091\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\089\000\255\255\255\255\255\255\081\000\074\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\255\255\255\255\255\255\255\255\091\000\255\255\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\092\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \092\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\255\255\255\255\255\255\255\255\092\000\
    \255\255\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255";
  Lexing.lex_code =
   "\255\006\255\005\255\004\255\255\000\004\255\000\005\255\000\006\
    \255\007\255\255\008\255\255\000\007\001\008\255\010\255\009\255\
    \255\012\255\011\255\255\000\009\001\011\255\000\010\001\012\255\
    \013\255\255\014\255\255\015\255\255\016\255\255\003\016\001\014\
    \000\013\002\015\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 17 (-1);(* L=3 [6] <- p ; [5] <- p ; [4] <- p ;  *)
  lexbuf.Lexing.lex_mem.(6) <- lexbuf.Lexing.lex_curr_pos ;
  lexbuf.Lexing.lex_mem.(5) <- lexbuf.Lexing.lex_curr_pos ;
  lexbuf.Lexing.lex_mem.(4) <- lexbuf.Lexing.lex_curr_pos ;
 __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 655 "parsing_cocci/lexer_cocci.mll"
    ( let cls = !current_line_started in
      if not cls
      then
	begin
	  match !current_line_type with
	    (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
	      (* increment the logical line even though nothing seen *)
	      start_line true;
	      let info = get_current_line_type lexbuf in
	      reset_line lexbuf;
	      TDirective (Ast.Noindent "", info)
	  | _ -> reset_line lexbuf; token lexbuf
	end
      else (reset_line lexbuf; token lexbuf) )
# 3287 "parsing_cocci/lexer_cocci.ml"

  | 1 ->
let
# 670 "parsing_cocci/lexer_cocci.mll"
                      w
# 3293 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 670 "parsing_cocci/lexer_cocci.mll"
                         ( (* collect whitespaces only when inside a rule *)
    start_line false;
    if !Data.in_rule_name || !Data.in_prolog || !Data.in_iso
    then token lexbuf
    else TWhitespace w )
# 3301 "parsing_cocci/lexer_cocci.ml"

  | 2 ->
let
# 676 "parsing_cocci/lexer_cocci.mll"
                                       after
# 3307 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 676 "parsing_cocci/lexer_cocci.mll"
                                              (
    match !current_line_type with
      (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
	let str =
	  if !current_line_started
	  then (tok lexbuf)
	  else after in
	start_line true;
	TDirective (Ast.Indent str, get_current_line_type lexbuf)
    | _ -> start_line false; token lexbuf )
# 3320 "parsing_cocci/lexer_cocci.ml"

  | 3 ->
let
# 687 "parsing_cocci/lexer_cocci.mll"
                                       after
# 3326 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 687 "parsing_cocci/lexer_cocci.mll"
                                              (
     check_context_linetype after;
     start_line false; token lexbuf )
# 3332 "parsing_cocci/lexer_cocci.ml"

  | 4 ->
let
# 691 "parsing_cocci/lexer_cocci.mll"
                                                                                   spatch
# 3338 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 691 "parsing_cocci/lexer_cocci.mll"
                                                                                           (
     check_context_linetype spatch;
     start_line false; token lexbuf )
# 3344 "parsing_cocci/lexer_cocci.ml"

  | 5 ->
# 695 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TArobArob )
# 3349 "parsing_cocci/lexer_cocci.ml"

  | 6 ->
# 696 "parsing_cocci/lexer_cocci.mll"
         ( pass_zero();
	   if !Data.in_rule_name || not !current_line_started
	   then (start_line true; TArob)
	   else (check_minus_context_linetype "@";
		 TPArob (get_current_line_type lexbuf)) )
# 3358 "parsing_cocci/lexer_cocci.ml"

  | 7 ->
# 703 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; check_minus_context_linetype (tok lexbuf);
	TWhen (get_current_line_type lexbuf) )
# 3364 "parsing_cocci/lexer_cocci.ml"

  | 8 ->
# 707 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; check_minus_context_linetype (tok lexbuf);
	TEllipsis (get_current_line_type lexbuf) )
# 3370 "parsing_cocci/lexer_cocci.ml"

  | 9 ->
# 709 "parsing_cocci/lexer_cocci.mll"
           ( start_line true; check_context_linetype (tok lexbuf);
	     TOEllipsis (get_current_line_type lexbuf) )
# 3376 "parsing_cocci/lexer_cocci.ml"

  | 10 ->
# 711 "parsing_cocci/lexer_cocci.mll"
           ( start_line true; check_context_linetype (tok lexbuf);
	     TCEllipsis (get_current_line_type lexbuf) )
# 3382 "parsing_cocci/lexer_cocci.ml"

  | 11 ->
# 713 "parsing_cocci/lexer_cocci.mll"
            ( start_line true; check_minus_context_linetype (tok lexbuf);
	     TPOEllipsis (get_current_line_type lexbuf) )
# 3388 "parsing_cocci/lexer_cocci.ml"

  | 12 ->
# 715 "parsing_cocci/lexer_cocci.mll"
            ( start_line true; check_minus_context_linetype (tok lexbuf);
	     TPCEllipsis (get_current_line_type lexbuf) )
# 3394 "parsing_cocci/lexer_cocci.ml"

  | 13 ->
# 717 "parsing_cocci/lexer_cocci.mll"
        ( pass_zero();
	  if !current_line_started
	  then (start_line true; TMinus (get_current_line_type lexbuf))
          else (patch_or_match PATCH;
		add_current_line_type D.MINUS; token lexbuf) )
# 3403 "parsing_cocci/lexer_cocci.ml"

  | 14 ->
# 722 "parsing_cocci/lexer_cocci.mll"
        ( pass_zero();
	  if !current_line_started
	  then (start_line true; TPlus (get_current_line_type lexbuf))
          else (patch_or_match PATCH;
		add_current_line_type D.PLUS; token lexbuf) )
# 3412 "parsing_cocci/lexer_cocci.ml"

  | 15 ->
# 727 "parsing_cocci/lexer_cocci.mll"
        ( pass_zero();
	  if !current_line_started
	  then (start_line true; TWhy (get_current_line_type lexbuf))
          else (add_current_line_type D.OPT; token lexbuf) )
# 3420 "parsing_cocci/lexer_cocci.ml"

  | 16 ->
# 732 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; TBang (tok lexbuf,get_current_line_type lexbuf) )
# 3425 "parsing_cocci/lexer_cocci.ml"

  | 17 ->
# 733 "parsing_cocci/lexer_cocci.mll"
        ( if not !col_zero
	  then (start_line true; TOPar (get_current_line_type lexbuf))
          else
            (start_line true; check_context_linetype (tok lexbuf);
	     TOPar0 ("(",get_current_line_type lexbuf)))
# 3434 "parsing_cocci/lexer_cocci.ml"

  | 18 ->
# 738 "parsing_cocci/lexer_cocci.mll"
          ( start_line true;
	    TOPar0 ("\\(",contextify(get_current_line_type lexbuf)) )
# 3440 "parsing_cocci/lexer_cocci.ml"

  | 19 ->
# 740 "parsing_cocci/lexer_cocci.mll"
        ( if not (!col_zero)
	  then (start_line true; TOr(tok lexbuf,get_current_line_type lexbuf))
          else (start_line true;
		check_context_linetype (tok lexbuf);
		TMid0 ("|",get_current_line_type lexbuf)))
# 3449 "parsing_cocci/lexer_cocci.ml"

  | 20 ->
# 745 "parsing_cocci/lexer_cocci.mll"
            ( start_line true; TOr(tok lexbuf, get_current_line_type lexbuf) )
# 3454 "parsing_cocci/lexer_cocci.ml"

  | 21 ->
# 746 "parsing_cocci/lexer_cocci.mll"
          ( start_line true;
	    TMid0 ("\\|",contextify(get_current_line_type lexbuf)) )
# 3460 "parsing_cocci/lexer_cocci.ml"

  | 22 ->
# 748 "parsing_cocci/lexer_cocci.mll"
        ( if not !col_zero
	  then (start_line true; TCPar (get_current_line_type lexbuf))
          else
            (start_line true; check_context_linetype (tok lexbuf);
	     TCPar0 (")",get_current_line_type lexbuf)))
# 3469 "parsing_cocci/lexer_cocci.ml"

  | 23 ->
# 753 "parsing_cocci/lexer_cocci.mll"
          ( start_line true;
	    TCPar0 ("\\)",contextify(get_current_line_type lexbuf)) )
# 3475 "parsing_cocci/lexer_cocci.ml"

  | 24 ->
# 756 "parsing_cocci/lexer_cocci.mll"
               ( start_line true; TOCro(tok lexbuf, get_current_line_type lexbuf)   )
# 3480 "parsing_cocci/lexer_cocci.ml"

  | 25 ->
# 757 "parsing_cocci/lexer_cocci.mll"
               ( start_line true; TCCro(tok lexbuf, get_current_line_type lexbuf)   )
# 3485 "parsing_cocci/lexer_cocci.ml"

  | 26 ->
# 758 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TOCroCro (tok lexbuf,get_current_line_type lexbuf)   )
# 3490 "parsing_cocci/lexer_cocci.ml"

  | 27 ->
# 759 "parsing_cocci/lexer_cocci.mll"
               ( start_line true; TOBrace(tok lexbuf, get_current_line_type lexbuf) )
# 3495 "parsing_cocci/lexer_cocci.ml"

  | 28 ->
# 760 "parsing_cocci/lexer_cocci.mll"
               ( start_line true; TCBrace(tok lexbuf, get_current_line_type lexbuf) )
# 3500 "parsing_cocci/lexer_cocci.ml"

  | 29 ->
# 762 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TPtrOp (get_current_line_type lexbuf)  )
# 3505 "parsing_cocci/lexer_cocci.ml"

  | 30 ->
# 763 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TDot (get_current_line_type lexbuf)    )
# 3510 "parsing_cocci/lexer_cocci.ml"

  | 31 ->
# 764 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TComma (get_current_line_type lexbuf)  )
# 3515 "parsing_cocci/lexer_cocci.ml"

  | 32 ->
# 765 "parsing_cocci/lexer_cocci.mll"
               ( start_line true; TVAEllipsis (get_current_line_type lexbuf) )
# 3520 "parsing_cocci/lexer_cocci.ml"

  | 33 ->
# 766 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TPtVirg (";",get_current_line_type lexbuf) )
# 3525 "parsing_cocci/lexer_cocci.ml"

  | 34 ->
# 769 "parsing_cocci/lexer_cocci.mll"
                   ( pass_zero();
		     if !current_line_started
		     then
		       (start_line true; TMul ("*", get_current_line_type lexbuf))
		     else
		       (patch_or_match MATCH;
			add_current_line_type D.MINUS; token lexbuf) )
# 3536 "parsing_cocci/lexer_cocci.ml"

  | 35 ->
# 776 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TDmOp (Ast.Div,get_current_line_type lexbuf) )
# 3542 "parsing_cocci/lexer_cocci.ml"

  | 36 ->
# 778 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true;
		     TDmOp (Ast.Min,get_current_line_type lexbuf) )
# 3548 "parsing_cocci/lexer_cocci.ml"

  | 37 ->
# 780 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true;
		     TDmOp (Ast.Max,get_current_line_type lexbuf) )
# 3554 "parsing_cocci/lexer_cocci.ml"

  | 38 ->
# 782 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TDmOp (Ast.Mod,get_current_line_type lexbuf) )
# 3560 "parsing_cocci/lexer_cocci.ml"

  | 39 ->
# 784 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;  TTilde (tok lexbuf, get_current_line_type lexbuf) )
# 3565 "parsing_cocci/lexer_cocci.ml"

  | 40 ->
# 786 "parsing_cocci/lexer_cocci.mll"
                   ( pass_zero();
 		     if !current_line_started
 		     then
 		       (start_line true; TInc (get_current_line_type lexbuf))
 		     else (patch_or_match PATCH;
 			   add_current_line_type D.PLUSPLUS; token lexbuf) )
# 3575 "parsing_cocci/lexer_cocci.ml"

  | 41 ->
# 792 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;  TDec (get_current_line_type lexbuf) )
# 3580 "parsing_cocci/lexer_cocci.ml"

  | 42 ->
# 794 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TEq (get_current_line_type lexbuf) )
# 3585 "parsing_cocci/lexer_cocci.ml"

  | 43 ->
# 796 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Minus lexbuf )
# 3590 "parsing_cocci/lexer_cocci.ml"

  | 44 ->
# 797 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Plus lexbuf )
# 3595 "parsing_cocci/lexer_cocci.ml"

  | 45 ->
# 799 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Mul lexbuf )
# 3600 "parsing_cocci/lexer_cocci.ml"

  | 46 ->
# 800 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Div lexbuf )
# 3605 "parsing_cocci/lexer_cocci.ml"

  | 47 ->
# 801 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Mod lexbuf )
# 3610 "parsing_cocci/lexer_cocci.ml"

  | 48 ->
# 803 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true; mkassign (Ast.And(tok lexbuf)) lexbuf )
# 3615 "parsing_cocci/lexer_cocci.ml"

  | 49 ->
# 804 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true; mkassign (Ast.Or(tok lexbuf)) lexbuf )
# 3620 "parsing_cocci/lexer_cocci.ml"

  | 50 ->
# 805 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true; mkassign (Ast.Xor(tok lexbuf)) lexbuf )
# 3625 "parsing_cocci/lexer_cocci.ml"

  | 51 ->
# 806 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true; mkassign Ast.Max lexbuf )
# 3630 "parsing_cocci/lexer_cocci.ml"

  | 52 ->
# 807 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true; mkassign Ast.Min lexbuf )
# 3635 "parsing_cocci/lexer_cocci.ml"

  | 53 ->
# 809 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.DecLeft lexbuf )
# 3640 "parsing_cocci/lexer_cocci.ml"

  | 54 ->
# 810 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.DecRight lexbuf )
# 3645 "parsing_cocci/lexer_cocci.ml"

  | 55 ->
# 812 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TDotDot (":",get_current_line_type lexbuf) )
# 3650 "parsing_cocci/lexer_cocci.ml"

  | 56 ->
# 814 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TEqEq    (get_current_line_type lexbuf) )
# 3655 "parsing_cocci/lexer_cocci.ml"

  | 57 ->
# 815 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TNotEq   (tok lexbuf,get_current_line_type lexbuf) )
# 3660 "parsing_cocci/lexer_cocci.ml"

  | 58 ->
# 816 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TLogOp(Ast.SupEq,get_current_line_type lexbuf) )
# 3666 "parsing_cocci/lexer_cocci.ml"

  | 59 ->
# 818 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TLogOp(Ast.InfEq,get_current_line_type lexbuf) )
# 3672 "parsing_cocci/lexer_cocci.ml"

  | 60 ->
# 820 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TLogOp(Ast.Inf,get_current_line_type lexbuf) )
# 3678 "parsing_cocci/lexer_cocci.ml"

  | 61 ->
# 822 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TLogOp(Ast.Sup,get_current_line_type lexbuf) )
# 3684 "parsing_cocci/lexer_cocci.ml"

  | 62 ->
# 824 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TInf3(get_current_line_type lexbuf) )
# 3690 "parsing_cocci/lexer_cocci.ml"

  | 63 ->
# 826 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TSup3(get_current_line_type lexbuf) )
# 3696 "parsing_cocci/lexer_cocci.ml"

  | 64 ->
# 829 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TAndLog (tok lexbuf, get_current_line_type lexbuf) )
# 3702 "parsing_cocci/lexer_cocci.ml"

  | 65 ->
# 831 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TOrLog  (tok lexbuf, get_current_line_type lexbuf) )
# 3708 "parsing_cocci/lexer_cocci.ml"

  | 66 ->
# 834 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TShROp(Ast.DecRight,get_current_line_type lexbuf) )
# 3714 "parsing_cocci/lexer_cocci.ml"

  | 67 ->
# 836 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TShLOp(Ast.DecLeft,get_current_line_type lexbuf) )
# 3720 "parsing_cocci/lexer_cocci.ml"

  | 68 ->
# 839 "parsing_cocci/lexer_cocci.mll"
                   ( if not (!col_zero)
                     then (start_line true;
			   TAnd("&", get_current_line_type lexbuf))
                     else (start_line true;
	                   check_context_linetype (tok lexbuf);
		           TAnd0 ("&",get_current_line_type lexbuf)))
# 3730 "parsing_cocci/lexer_cocci.ml"

  | 69 ->
# 845 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TAnd(tok lexbuf, get_current_line_type lexbuf) )
# 3736 "parsing_cocci/lexer_cocci.ml"

  | 70 ->
# 847 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
	             TAnd0 ("\\&",contextify(get_current_line_type lexbuf)) )
# 3742 "parsing_cocci/lexer_cocci.ml"

  | 71 ->
# 850 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TXor(tok lexbuf, get_current_line_type lexbuf) )
# 3748 "parsing_cocci/lexer_cocci.ml"

  | 72 ->
# 853 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true; TCppConcatOp )
# 3753 "parsing_cocci/lexer_cocci.ml"

  | 73 ->
let
# 854 "parsing_cocci/lexer_cocci.mll"
                                               wss
# 3759 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 854 "parsing_cocci/lexer_cocci.mll"
                                                        def
# 3764 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(1)
and
# 855 "parsing_cocci/lexer_cocci.mll"
                                  ident
# 3769 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_curr_pos in
# 856 "parsing_cocci/lexer_cocci.mll"
      ( start_line true;
	let (arity,line,lline,llend,offset,col,strbef,straft,pos,ws) as lt =
	  get_current_line_type lexbuf in
	let off = String.length def in
	(* -1 in the code below because the ident is not at the line start *)
	TUndef
	  (lt,
	   check_var ident
	     (arity,line,lline,llend,offset+off,col+off,[],[],[],wss)) )
# 3781 "parsing_cocci/lexer_cocci.ml"

  | 74 ->
let
# 865 "parsing_cocci/lexer_cocci.mll"
                                                  wss
# 3787 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 865 "parsing_cocci/lexer_cocci.mll"
                                                            def
# 3792 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(1)
and
# 866 "parsing_cocci/lexer_cocci.mll"
                                   ident
# 3797 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_curr_pos in
# 867 "parsing_cocci/lexer_cocci.mll"
      ( start_line true;
	let (arity,line,lline,llend,offset,col,strbef,straft,pos,ws) as lt =
	  get_current_line_type lexbuf in
	let off = String.length def in
	(* -1 in the code below because the ident is not at the line start *)
	TDefine
	  (lt,
	   check_var ident
	     (arity,line,lline,llend,offset+off,col+off,[],[],[],wss)) )
# 3809 "parsing_cocci/lexer_cocci.ml"

  | 75 ->
let
# 876 "parsing_cocci/lexer_cocci.mll"
                                                  wss
# 3815 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 876 "parsing_cocci/lexer_cocci.mll"
                                                            def
# 3820 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(1)
and
# 877 "parsing_cocci/lexer_cocci.mll"
                                    ident
# 3825 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) (lexbuf.Lexing.lex_curr_pos + -1) in
# 879 "parsing_cocci/lexer_cocci.mll"
      ( start_line true;
	let (arity,line,lline,llend,offset,col,strbef,straft,pos,ws) as lt =
	  get_current_line_type lexbuf in
	let off = String.length def in
	TDefineParam
        (lt,
	 check_var ident
	   (* why pos here but not above? *)
	   (arity,line,lline,llend,offset+off,col+off,strbef,straft,pos,wss),
	 offset + off + (String.length ident),
	 col + off + (String.length ident)) )
# 3839 "parsing_cocci/lexer_cocci.ml"

  | 76 ->
let
# 890 "parsing_cocci/lexer_cocci.mll"
                                    prag
# 3845 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 890 "parsing_cocci/lexer_cocci.mll"
                                                          wss1
# 3850 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 891 "parsing_cocci/lexer_cocci.mll"
                                    ident
# 3855 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(2)
and
# 891 "parsing_cocci/lexer_cocci.mll"
                                                           wss2
# 3860 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(2) lexbuf.Lexing.lex_mem.(3)
and
# 891 "parsing_cocci/lexer_cocci.mll"
                                                                               rest
# 3865 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_curr_pos in
# 892 "parsing_cocci/lexer_cocci.mll"
      ( start_line true;
	let (arity,line,lline,llend,offset,col,strbef,straft,pos,ws) as lt =
	  get_current_line_type lexbuf in
	let off1 = String.length prag + String.length wss1 in
	let off2 = off1 + String.length ident + String.length wss2 in
	TPragma(lt,
		check_var ident
	        (* why pos here but not above? *)
		  (arity,line,lline,llend,offset+off1,col+off1,strbef,straft,
		   pos,wss1),
                check_var rest
		  (arity,line,lline,llend,offset+off2,col+off2,strbef,straft,
		   pos,wss2),
		rest,
		(arity,line,lline,llend,offset+off2,col+off2,strbef,straft,
		 pos,wss2)) )
# 3884 "parsing_cocci/lexer_cocci.ml"

  | 77 ->
# 911 "parsing_cocci/lexer_cocci.mll"
      ( TIncludeL
	  (let str = tok lexbuf in
	  let start = String.index str '\"' in
	  let finish = String.rindex str '\"' in
	  start_line true;
	  (process_include start finish str, get_current_line_type lexbuf)) )
# 3894 "parsing_cocci/lexer_cocci.ml"

  | 78 ->
# 918 "parsing_cocci/lexer_cocci.mll"
      ( TIncludeNL
	  (let str = tok lexbuf in
	  let start = String.index str '<' in
	  let finish = String.rindex str '>' in
	  start_line true;
	  (process_include start finish str,get_current_line_type lexbuf)) )
# 3904 "parsing_cocci/lexer_cocci.ml"

  | 79 ->
# 925 "parsing_cocci/lexer_cocci.mll"
      ( check_minus_context_linetype (tok lexbuf);
	start_line true;
	TIncludeAny("...",get_current_line_type lexbuf) )
# 3911 "parsing_cocci/lexer_cocci.ml"

  | 80 ->
# 929 "parsing_cocci/lexer_cocci.mll"
      ( check_context_linetype (tok lexbuf);
	start_line true;
	TInclude(get_current_line_type lexbuf) )
# 3918 "parsing_cocci/lexer_cocci.ml"

  | 81 ->
# 940 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; check_plus_linetype (tok lexbuf);
	TDirective (Ast.Noindent(tok lexbuf), get_current_line_type lexbuf) )
# 3924 "parsing_cocci/lexer_cocci.ml"

  | 82 ->
# 944 "parsing_cocci/lexer_cocci.mll"
      ( start_line true;
	let res = TCppEscapedNewline (get_current_line_type lexbuf) in
	reset_line lexbuf;
	res )
# 3932 "parsing_cocci/lexer_cocci.ml"

  | 83 ->
# 950 "parsing_cocci/lexer_cocci.mll"
      (
       match !current_line_type with
        (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
        start_line true;
	(* second argument to TDirective is not quite right, because
	   it represents only the first token of the comment, but that
	   should be good enough *)
	TDirective (Ast.Indent("/*"^(comment check_comment lexbuf)),
		 get_current_line_type lexbuf)
      |	_ -> let _ = comment (fun _ -> ()) lexbuf in token lexbuf )
# 3946 "parsing_cocci/lexer_cocci.ml"

  | 84 ->
# 961 "parsing_cocci/lexer_cocci.mll"
      ( (if !current_line_started
      then lexerr "--- must be at the beginning of the line" "");
	start_line true;
	TMinusFile
	  (let str = tok lexbuf in
	  (drop_spaces(String.sub str 3 (String.length str - 3)),
	   (get_current_line_type lexbuf))) )
# 3957 "parsing_cocci/lexer_cocci.ml"

  | 85 ->
# 969 "parsing_cocci/lexer_cocci.mll"
      ( (if !current_line_started
      then lexerr "+++ must be at the beginning of the line" "");
	start_line true;
	TPlusFile
	  (let str = tok lexbuf in
	  (drop_spaces(String.sub str 3 (String.length str - 3)),
	   (get_current_line_type lexbuf))) )
# 3968 "parsing_cocci/lexer_cocci.ml"

  | 86 ->
# 978 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; id_tokens lexbuf )
# 3973 "parsing_cocci/lexer_cocci.ml"

  | 87 ->
# 982 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; id_tokens lexbuf )
# 3978 "parsing_cocci/lexer_cocci.ml"

  | 88 ->
# 986 "parsing_cocci/lexer_cocci.mll"
        ( start_line true;
	  TChar(char lexbuf,Ast.IsChar,get_current_line_type lexbuf) )
# 3984 "parsing_cocci/lexer_cocci.ml"

  | 89 ->
# 988 "parsing_cocci/lexer_cocci.mll"
         ( start_line true;
	  TString(string lexbuf,Ast.IsChar,(get_current_line_type lexbuf)) )
# 3990 "parsing_cocci/lexer_cocci.ml"

  | 90 ->
# 990 "parsing_cocci/lexer_cocci.mll"
            ( start_line true;
	  TChar(char lexbuf,Ast.IsWchar,get_current_line_type lexbuf) )
# 3996 "parsing_cocci/lexer_cocci.ml"

  | 91 ->
# 992 "parsing_cocci/lexer_cocci.mll"
             ( start_line true;
	  TString(string lexbuf,Ast.IsWchar,(get_current_line_type lexbuf)) )
# 4002 "parsing_cocci/lexer_cocci.ml"

  | 92 ->
# 994 "parsing_cocci/lexer_cocci.mll"
            ( start_line true;
	  TChar(char lexbuf,Ast.IsUchar,get_current_line_type lexbuf) )
# 4008 "parsing_cocci/lexer_cocci.ml"

  | 93 ->
# 996 "parsing_cocci/lexer_cocci.mll"
             ( start_line true;
	  TString(string lexbuf,Ast.IsUchar,(get_current_line_type lexbuf)) )
# 4014 "parsing_cocci/lexer_cocci.ml"

  | 94 ->
# 998 "parsing_cocci/lexer_cocci.mll"
            ( start_line true;
	  TChar(char lexbuf,Ast.Isuchar,get_current_line_type lexbuf) )
# 4020 "parsing_cocci/lexer_cocci.ml"

  | 95 ->
# 1000 "parsing_cocci/lexer_cocci.mll"
             ( start_line true;
	  TString(string lexbuf,Ast.Isuchar,(get_current_line_type lexbuf)) )
# 4026 "parsing_cocci/lexer_cocci.ml"

  | 96 ->
# 1002 "parsing_cocci/lexer_cocci.mll"
              ( start_line true;
	  TString(string lexbuf,Ast.Isu8char,(get_current_line_type lexbuf)) )
# 4032 "parsing_cocci/lexer_cocci.ml"

  | 97 ->
let
# 1004 "parsing_cocci/lexer_cocci.mll"
             x
# 4038 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1004 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TFloat(x,(get_current_line_type lexbuf)) )
# 4043 "parsing_cocci/lexer_cocci.ml"

  | 98 ->
let
# 1015 "parsing_cocci/lexer_cocci.mll"
         x
# 4049 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1015 "parsing_cocci/lexer_cocci.mll"
            ( start_line true; TInt(x,(get_current_line_type lexbuf)) )
# 4053 "parsing_cocci/lexer_cocci.ml"

  | 99 ->
let
# 1017 "parsing_cocci/lexer_cocci.mll"
                           x
# 4059 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1018 "parsing_cocci/lexer_cocci.mll"
      ( if !Flag.ibm
      then
	begin
	  start_line true;
	  let len = string_of_int(String.length x - 1) in
          TDecimalCst(x,len,"0",(get_current_line_type lexbuf))
	end
      else failwith "unrecognized constant modifier d/D" )
# 4070 "parsing_cocci/lexer_cocci.ml"

  | 100 ->
# 1027 "parsing_cocci/lexer_cocci.mll"
                   ( TIso )
# 4075 "parsing_cocci/lexer_cocci.ml"

  | 101 ->
# 1028 "parsing_cocci/lexer_cocci.mll"
                   ( TRightIso )
# 4080 "parsing_cocci/lexer_cocci.ml"

  | 102 ->
# 1030 "parsing_cocci/lexer_cocci.mll"
                   ( EOF )
# 4085 "parsing_cocci/lexer_cocci.ml"

  | 103 ->
# 1032 "parsing_cocci/lexer_cocci.mll"
      ( lexerr "unrecognised symbol, in token rule: " (tok lexbuf) )
# 4090 "parsing_cocci/lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and metavariable_decl_token lexbuf =
   __ocaml_lex_metavariable_decl_token_rec lexbuf 251
and __ocaml_lex_metavariable_decl_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1037 "parsing_cocci/lexer_cocci.mll"
    ( reset_line lexbuf; metavariable_decl_token lexbuf )
# 4102 "parsing_cocci/lexer_cocci.ml"

  | 1 ->
# 1039 "parsing_cocci/lexer_cocci.mll"
                  (
    start_line false; metavariable_decl_token lexbuf )
# 4108 "parsing_cocci/lexer_cocci.ml"

  | 2 ->
# 1042 "parsing_cocci/lexer_cocci.mll"
                                   (
    start_line false; metavariable_decl_token lexbuf )
# 4114 "parsing_cocci/lexer_cocci.ml"

  | 3 ->
# 1045 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TArobArob )
# 4119 "parsing_cocci/lexer_cocci.ml"

  | 4 ->
# 1047 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TTildeEq (get_current_line_type lexbuf) )
# 4124 "parsing_cocci/lexer_cocci.ml"

  | 5 ->
# 1048 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TTildeExclEq (get_current_line_type lexbuf) )
# 4129 "parsing_cocci/lexer_cocci.ml"

  | 6 ->
# 1049 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TEq (get_current_line_type lexbuf) )
# 4134 "parsing_cocci/lexer_cocci.ml"

  | 7 ->
# 1050 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TOPar (get_current_line_type lexbuf) )
# 4139 "parsing_cocci/lexer_cocci.ml"

  | 8 ->
# 1051 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TCPar (get_current_line_type lexbuf) )
# 4144 "parsing_cocci/lexer_cocci.ml"

  | 9 ->
# 1053 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TOCro (tok lexbuf,get_current_line_type lexbuf)   )
# 4149 "parsing_cocci/lexer_cocci.ml"

  | 10 ->
# 1054 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TCCro (tok lexbuf,get_current_line_type lexbuf)   )
# 4154 "parsing_cocci/lexer_cocci.ml"

  | 11 ->
# 1055 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TOBrace (tok lexbuf,get_current_line_type lexbuf) )
# 4159 "parsing_cocci/lexer_cocci.ml"

  | 12 ->
# 1056 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TCBrace (tok lexbuf,get_current_line_type lexbuf) )
# 4164 "parsing_cocci/lexer_cocci.ml"

  | 13 ->
# 1058 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TPtrOp (get_current_line_type lexbuf)  )
# 4169 "parsing_cocci/lexer_cocci.ml"

  | 14 ->
# 1059 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TDot (get_current_line_type lexbuf)    )
# 4174 "parsing_cocci/lexer_cocci.ml"

  | 15 ->
# 1060 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TComma (get_current_line_type lexbuf)  )
# 4179 "parsing_cocci/lexer_cocci.ml"

  | 16 ->
# 1061 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TMPtVirg (* works better with tokens_all *) )
# 4185 "parsing_cocci/lexer_cocci.ml"

  | 17 ->
# 1063 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true;
		     TShLOp(Ast.DecLeft,get_current_line_type lexbuf) )
# 4191 "parsing_cocci/lexer_cocci.ml"

  | 18 ->
# 1066 "parsing_cocci/lexer_cocci.mll"
                   ( pass_zero();
		     if !current_line_started
		     then
		       (start_line true; TMul ("*", get_current_line_type lexbuf))
		     else
		       (patch_or_match MATCH;
			add_current_line_type D.MINUS;
			metavariable_decl_token lexbuf) )
# 4203 "parsing_cocci/lexer_cocci.ml"

  | 19 ->
# 1075 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TEqEq    (get_current_line_type lexbuf) )
# 4208 "parsing_cocci/lexer_cocci.ml"

  | 20 ->
# 1076 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TNotEq   (tok lexbuf,get_current_line_type lexbuf) )
# 4213 "parsing_cocci/lexer_cocci.ml"

  | 21 ->
# 1077 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TSub     (get_current_line_type lexbuf) )
# 4218 "parsing_cocci/lexer_cocci.ml"

  | 22 ->
# 1078 "parsing_cocci/lexer_cocci.mll"
        ( (start_line true; TPlus (get_current_line_type lexbuf)) )
# 4223 "parsing_cocci/lexer_cocci.ml"

  | 23 ->
# 1079 "parsing_cocci/lexer_cocci.mll"
        ( (start_line true; TMinus (get_current_line_type lexbuf)) )
# 4228 "parsing_cocci/lexer_cocci.ml"

  | 24 ->
# 1080 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TDmOp (Ast.Div,get_current_line_type lexbuf) )
# 4233 "parsing_cocci/lexer_cocci.ml"

  | 25 ->
# 1081 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TDmOp (Ast.Mod,get_current_line_type lexbuf) )
# 4238 "parsing_cocci/lexer_cocci.ml"

  | 26 ->
# 1082 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TShROp(Ast.DecRight,get_current_line_type lexbuf) )
# 4243 "parsing_cocci/lexer_cocci.ml"

  | 27 ->
# 1083 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TAnd(tok lexbuf, get_current_line_type lexbuf) )
# 4248 "parsing_cocci/lexer_cocci.ml"

  | 28 ->
# 1084 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TOr (tok lexbuf, get_current_line_type lexbuf) )
# 4253 "parsing_cocci/lexer_cocci.ml"

  | 29 ->
# 1085 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; TXor(tok lexbuf, get_current_line_type lexbuf) )
# 4258 "parsing_cocci/lexer_cocci.ml"

  | 30 ->
# 1086 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TLogOp(Ast.SupEq,get_current_line_type lexbuf) )
# 4263 "parsing_cocci/lexer_cocci.ml"

  | 31 ->
# 1087 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TLogOp(Ast.Inf,get_current_line_type lexbuf) )
# 4268 "parsing_cocci/lexer_cocci.ml"

  | 32 ->
# 1088 "parsing_cocci/lexer_cocci.mll"
        ( start_line true; TLogOp(Ast.Sup,get_current_line_type lexbuf) )
# 4273 "parsing_cocci/lexer_cocci.ml"

  | 33 ->
# 1089 "parsing_cocci/lexer_cocci.mll"
                 ( start_line true; TAndLog (tok lexbuf, get_current_line_type lexbuf) )
# 4278 "parsing_cocci/lexer_cocci.ml"

  | 34 ->
# 1090 "parsing_cocci/lexer_cocci.mll"
                 ( start_line true; TOrLog  (tok lexbuf, get_current_line_type lexbuf) )
# 4283 "parsing_cocci/lexer_cocci.ml"

  | 35 ->
# 1091 "parsing_cocci/lexer_cocci.mll"
         ( start_line true; TDotDot (":",get_current_line_type lexbuf) )
# 4288 "parsing_cocci/lexer_cocci.ml"

  | 36 ->
# 1092 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Minus lexbuf )
# 4293 "parsing_cocci/lexer_cocci.ml"

  | 37 ->
# 1093 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Plus lexbuf )
# 4298 "parsing_cocci/lexer_cocci.ml"

  | 38 ->
# 1094 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Mul lexbuf )
# 4303 "parsing_cocci/lexer_cocci.ml"

  | 39 ->
# 1095 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Div lexbuf )
# 4308 "parsing_cocci/lexer_cocci.ml"

  | 40 ->
# 1096 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.Mod lexbuf )
# 4313 "parsing_cocci/lexer_cocci.ml"

  | 41 ->
# 1097 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign (Ast.And (tok lexbuf)) lexbuf )
# 4318 "parsing_cocci/lexer_cocci.ml"

  | 42 ->
# 1098 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign (Ast.Or (tok lexbuf)) lexbuf )
# 4323 "parsing_cocci/lexer_cocci.ml"

  | 43 ->
# 1099 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign (Ast.Xor (tok lexbuf)) lexbuf )
# 4328 "parsing_cocci/lexer_cocci.ml"

  | 44 ->
# 1100 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.DecLeft lexbuf )
# 4333 "parsing_cocci/lexer_cocci.ml"

  | 45 ->
# 1101 "parsing_cocci/lexer_cocci.mll"
                   ( start_line true; mkassign Ast.DecRight lexbuf )
# 4338 "parsing_cocci/lexer_cocci.ml"

  | 46 ->
# 1103 "parsing_cocci/lexer_cocci.mll"
      (match !current_line_type with
        (D.PLUS,_,_) | (D.PLUSPLUS,_,_) ->
        start_line true;
	(* second argument to TDirective is not quite right, because
	   it represents only the first token of the comment, but that
	   should be good enough *)
	TDirective (Ast.Indent("/*"^(comment check_comment lexbuf)),
		 get_current_line_type lexbuf)
      |	_ -> let _ = comment (fun _ -> ()) lexbuf in
	     metavariable_decl_token lexbuf )
# 4352 "parsing_cocci/lexer_cocci.ml"

  | 47 ->
# 1114 "parsing_cocci/lexer_cocci.mll"
                    ( start_line true; TCppConcatOp (* for fresh vars *) )
# 4357 "parsing_cocci/lexer_cocci.ml"

  | 48 ->
# 1117 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; id_tokens lexbuf )
# 4362 "parsing_cocci/lexer_cocci.ml"

  | 49 ->
# 1121 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; id_tokens lexbuf )
# 4367 "parsing_cocci/lexer_cocci.ml"

  | 50 ->
# 1125 "parsing_cocci/lexer_cocci.mll"
        ( start_line true;
	  TChar(char lexbuf,Ast.IsChar,get_current_line_type lexbuf) )
# 4373 "parsing_cocci/lexer_cocci.ml"

  | 51 ->
# 1127 "parsing_cocci/lexer_cocci.mll"
         ( start_line true;
	  TString(string lexbuf,Ast.IsChar,(get_current_line_type lexbuf)) )
# 4379 "parsing_cocci/lexer_cocci.ml"

  | 52 ->
# 1129 "parsing_cocci/lexer_cocci.mll"
            ( start_line true;
	  TChar(char lexbuf,Ast.IsWchar,get_current_line_type lexbuf) )
# 4385 "parsing_cocci/lexer_cocci.ml"

  | 53 ->
# 1131 "parsing_cocci/lexer_cocci.mll"
             ( start_line true;
	  TString(string lexbuf,Ast.IsWchar,(get_current_line_type lexbuf)) )
# 4391 "parsing_cocci/lexer_cocci.ml"

  | 54 ->
# 1133 "parsing_cocci/lexer_cocci.mll"
            ( start_line true;
	  TChar(char lexbuf,Ast.IsUchar,get_current_line_type lexbuf) )
# 4397 "parsing_cocci/lexer_cocci.ml"

  | 55 ->
# 1135 "parsing_cocci/lexer_cocci.mll"
             ( start_line true;
	  TString(string lexbuf,Ast.IsUchar,(get_current_line_type lexbuf)) )
# 4403 "parsing_cocci/lexer_cocci.ml"

  | 56 ->
# 1137 "parsing_cocci/lexer_cocci.mll"
            ( start_line true;
	  TChar(char lexbuf,Ast.Isuchar,get_current_line_type lexbuf) )
# 4409 "parsing_cocci/lexer_cocci.ml"

  | 57 ->
# 1139 "parsing_cocci/lexer_cocci.mll"
             ( start_line true;
	  TString(string lexbuf,Ast.Isuchar,(get_current_line_type lexbuf)) )
# 4415 "parsing_cocci/lexer_cocci.ml"

  | 58 ->
# 1141 "parsing_cocci/lexer_cocci.mll"
              ( start_line true;
	  TString(string lexbuf,Ast.Isu8char,(get_current_line_type lexbuf)) )
# 4421 "parsing_cocci/lexer_cocci.ml"

  | 59 ->
let
# 1143 "parsing_cocci/lexer_cocci.mll"
             x
# 4427 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1143 "parsing_cocci/lexer_cocci.mll"
                   ( TFloat(x,(get_current_line_type lexbuf)) )
# 4431 "parsing_cocci/lexer_cocci.ml"

  | 60 ->
let
# 1152 "parsing_cocci/lexer_cocci.mll"
         x
# 4437 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1152 "parsing_cocci/lexer_cocci.mll"
            ( start_line true; TInt(x,(get_current_line_type lexbuf)) )
# 4441 "parsing_cocci/lexer_cocci.ml"

  | 61 ->
let
# 1154 "parsing_cocci/lexer_cocci.mll"
                           x
# 4447 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1155 "parsing_cocci/lexer_cocci.mll"
      ( if !Flag.ibm
      then
	begin
	  start_line true;
	  let len = string_of_int(String.length x - 1) in
          TDecimalCst(x,len,"0",(get_current_line_type lexbuf))
	end
      else failwith "unrecognized constant modifier d/D" )
# 4458 "parsing_cocci/lexer_cocci.ml"

  | 62 ->
# 1164 "parsing_cocci/lexer_cocci.mll"
      ( lexerr
	 "metavariables: unrecognised symbol in metavariable_decl_token rule: "
	 (tok lexbuf) )
# 4465 "parsing_cocci/lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_metavariable_decl_token_rec lexbuf __ocaml_lex_state

and char lexbuf =
   __ocaml_lex_char_rec lexbuf 364
and __ocaml_lex_char_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 1170 "parsing_cocci/lexer_cocci.mll"
          x
# 4478 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 1170 "parsing_cocci/lexer_cocci.mll"
                                       ( String.make 1 x ^ restchars lexbuf )
# 4482 "parsing_cocci/lexer_cocci.ml"

  | 1 ->
let
# 1172 "parsing_cocci/lexer_cocci.mll"
                                             x
# 4488 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1172 "parsing_cocci/lexer_cocci.mll"
                                                     ( x ^ restchars lexbuf )
# 4492 "parsing_cocci/lexer_cocci.ml"

  | 2 ->
let
# 1176 "parsing_cocci/lexer_cocci.mll"
                                  x
# 4498 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1176 "parsing_cocci/lexer_cocci.mll"
                                                     ( x ^ restchars lexbuf )
# 4502 "parsing_cocci/lexer_cocci.ml"

  | 3 ->
let
# 1177 "parsing_cocci/lexer_cocci.mll"
                 v
# 4508 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 1177 "parsing_cocci/lexer_cocci.mll"
                                  x
# 4513 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 1178 "parsing_cocci/lexer_cocci.mll"
 (
          (match v with (* Machine specific ? *)
          | 'n' -> ()  | 't' -> ()   | 'v' -> ()  | 'b' -> () | 'r' -> ()
          | 'f' -> () | 'a' -> ()
	  | '\\' -> () | '?'  -> () | '\'' -> ()  | '\"' -> ()
          | 'e' -> () (* linuxext: ? *)
	  | _ ->
              Common.pr2 ("LEXER: unrecognised symbol in char:"^tok lexbuf);
	  );
          x ^ restchars lexbuf
	)
# 4527 "parsing_cocci/lexer_cocci.ml"

  | 4 ->
# 1190 "parsing_cocci/lexer_cocci.mll"
      ( Common.pr2 ("LEXER: unrecognised symbol in char:"^tok lexbuf);
        tok lexbuf ^ restchars lexbuf
      )
# 4534 "parsing_cocci/lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_char_rec lexbuf __ocaml_lex_state

and restchars lexbuf =
   __ocaml_lex_restchars_rec lexbuf 374
and __ocaml_lex_restchars_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1195 "parsing_cocci/lexer_cocci.mll"
                                       ( "" )
# 4546 "parsing_cocci/lexer_cocci.ml"

  | 1 ->
let
# 1196 "parsing_cocci/lexer_cocci.mll"
          x
# 4552 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 1196 "parsing_cocci/lexer_cocci.mll"
                                       ( String.make 1 x ^ restchars lexbuf )
# 4556 "parsing_cocci/lexer_cocci.ml"

  | 2 ->
let
# 1198 "parsing_cocci/lexer_cocci.mll"
                                             x
# 4562 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1198 "parsing_cocci/lexer_cocci.mll"
                                                     ( x ^ restchars lexbuf )
# 4566 "parsing_cocci/lexer_cocci.ml"

  | 3 ->
let
# 1202 "parsing_cocci/lexer_cocci.mll"
                                  x
# 4572 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1202 "parsing_cocci/lexer_cocci.mll"
                                                     ( x ^ restchars lexbuf )
# 4576 "parsing_cocci/lexer_cocci.ml"

  | 4 ->
let
# 1203 "parsing_cocci/lexer_cocci.mll"
                 v
# 4582 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 1203 "parsing_cocci/lexer_cocci.mll"
                                  x
# 4587 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 1204 "parsing_cocci/lexer_cocci.mll"
 (
          (match v with (* Machine specific ? *)
          | 'n' -> ()  | 't' -> ()   | 'v' -> ()  | 'b' -> () | 'r' -> ()
          | 'f' -> () | 'a' -> ()
	  | '\\' -> () | '?'  -> () | '\'' -> ()  | '\"' -> ()
          | 'e' -> () (* linuxext: ? *)
	  | _ ->
              Common.pr2 ("LEXER: unrecognised symbol in char:"^tok lexbuf);
	  );
          x ^ restchars lexbuf
	)
# 4601 "parsing_cocci/lexer_cocci.ml"

  | 5 ->
# 1216 "parsing_cocci/lexer_cocci.mll"
      ( Common.pr2 ("LEXER: unrecognised symbol in char:"^tok lexbuf);
        tok lexbuf ^ restchars lexbuf
      )
# 4608 "parsing_cocci/lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_restchars_rec lexbuf __ocaml_lex_state

and string lexbuf =
   __ocaml_lex_string_rec lexbuf 385
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1221 "parsing_cocci/lexer_cocci.mll"
                                               ( "" )
# 4620 "parsing_cocci/lexer_cocci.ml"

  | 1 ->
let
# 1222 "parsing_cocci/lexer_cocci.mll"
                                 x
# 4626 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 1223 "parsing_cocci/lexer_cocci.mll"
    ( line := !line + 1; (Printf.sprintf "%c" x) ^ string lexbuf )
# 4630 "parsing_cocci/lexer_cocci.ml"

  | 2 ->
let
# 1224 "parsing_cocci/lexer_cocci.mll"
          x
# 4636 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 1224 "parsing_cocci/lexer_cocci.mll"
                               ( Common.string_of_char x ^ string lexbuf )
# 4640 "parsing_cocci/lexer_cocci.ml"

  | 3 ->
let
# 1225 "parsing_cocci/lexer_cocci.mll"
                                            x
# 4646 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1225 "parsing_cocci/lexer_cocci.mll"
                                              ( x ^ string lexbuf )
# 4650 "parsing_cocci/lexer_cocci.ml"

  | 4 ->
let
# 1226 "parsing_cocci/lexer_cocci.mll"
                               x
# 4656 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 1226 "parsing_cocci/lexer_cocci.mll"
                                              ( x ^ string lexbuf )
# 4660 "parsing_cocci/lexer_cocci.ml"

  | 5 ->
let
# 1227 "parsing_cocci/lexer_cocci.mll"
                v
# 4666 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 1227 "parsing_cocci/lexer_cocci.mll"
                       x
# 4671 "parsing_cocci/lexer_cocci.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 1228 "parsing_cocci/lexer_cocci.mll"
       (
        let others _ =
          match v with
	    | 'n' -> ()  | 't' -> ()   | 'v' -> ()  | 'b' -> () | 'r' -> ()
	    | 'f' -> () | 'a' -> ()
	    | '\\' -> () | '?'  -> () | '\'' -> ()  | '\"' -> ()
	    | 'e' -> ()
	    | '\n' -> ()
	    | '(' -> () | '|' -> () | ')' -> ()
	    | _ -> lexerr "unrecognised symbol:" (tok lexbuf) in
	if !Data.in_meta
	then
	  (if List.mem v ['$';'^';'\\';'.';'*';'+';'?';'[';']'] (* for regexps *)
	  then ()
	  else others())
	else others();
        x ^ string lexbuf
       )
# 4692 "parsing_cocci/lexer_cocci.ml"

  | 6 ->
# 1246 "parsing_cocci/lexer_cocci.mll"
      ( lexerr "unrecognised symbol: " (tok lexbuf) )
# 4697 "parsing_cocci/lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_rec lexbuf __ocaml_lex_state

and comment check_comment lexbuf =
   __ocaml_lex_comment_rec check_comment lexbuf 397
and __ocaml_lex_comment_rec check_comment lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 1249 "parsing_cocci/lexer_cocci.mll"
         ( let s = tok lexbuf in check_comment s; start_line true; s )
# 4709 "parsing_cocci/lexer_cocci.ml"

  | 1 ->
# 1251 "parsing_cocci/lexer_cocci.mll"
      ( let s = tok lexbuf in
        (* even blank line should have a + *)
        check_comment s;
        reset_line lexbuf; s ^ comment check_comment lexbuf )
# 4717 "parsing_cocci/lexer_cocci.ml"

  | 2 ->
# 1255 "parsing_cocci/lexer_cocci.mll"
        ( pass_zero();
	  if !current_line_started
	  then (start_line true;
		let s = tok lexbuf in s^(comment check_comment lexbuf))
	  else (start_line true; comment check_comment lexbuf) )
# 4726 "parsing_cocci/lexer_cocci.ml"

  | 3 ->
# 1262 "parsing_cocci/lexer_cocci.mll"
      ( let s = tok lexbuf in
        check_comment s; start_line true; s ^ comment check_comment lexbuf )
# 4732 "parsing_cocci/lexer_cocci.ml"

  | 4 ->
# 1265 "parsing_cocci/lexer_cocci.mll"
      ( let s = tok lexbuf in
        check_comment s; start_line true; s ^ comment check_comment lexbuf )
# 4738 "parsing_cocci/lexer_cocci.ml"

  | 5 ->
# 1268 "parsing_cocci/lexer_cocci.mll"
      ( start_line true; let s = tok lexbuf in
        Common.pr2 ("LEXER: unrecognised symbol in comment:"^s);
        s ^ comment check_comment lexbuf
      )
# 4746 "parsing_cocci/lexer_cocci.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec check_comment lexbuf __ocaml_lex_state

;;

