"use strict";
// Copyright 2022, 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardCommandParametersDescription = void 0;
exports.describeCommandParameters = describeCommandParameters;
exports.tuple = tuple;
const typescript_result_1 = require("@gnuxie/typescript-result");
const RestParameterDescription_1 = require("./RestParameterDescription");
const KeywordParameterDescription_1 = require("./KeywordParameterDescription");
const ParseErrors_1 = require("./ParseErrors");
const TextPresentationRenderer_1 = require("../TextReader/TextPresentationRenderer");
const PresentationSchema_1 = require("./PresentationSchema");
class StandardCommandParametersDescription {
    constructor(descriptions, keywords, rest) {
        this.descriptions = descriptions;
        this.keywords = keywords;
        this.rest = rest;
    }
    parse(partialCommand) {
        var _a, _b;
        const keywordsParser = this.keywords.getParser();
        const stream = partialCommand.stream;
        const immediateArguments = [];
        for (const parameter of this.descriptions) {
            // it eats any keywords at any point in the stream
            // as they can appear at any point technically.
            const keywordResult = keywordsParser.parseKeywords(partialCommand);
            if ((0, typescript_result_1.isError)(keywordResult)) {
                return keywordResult;
            }
            const nextItem = stream.peekItem();
            if (nextItem === undefined) {
                if (parameter.prompt) {
                    return ParseErrors_1.PromptRequiredError.Result(`An argument for the parameter ${parameter.name} was expected but was not provided. A prompt is available for this parameter.`, {
                        promptParameter: parameter,
                        partialCommand: partialCommand,
                    });
                }
                else {
                    return ParseErrors_1.ArgumentParseError.Result(`An argument for the parameter ${parameter.name} was expected but was not provided.`, { parameter, partialCommand: partialCommand });
                }
            }
            const acceptedPresentation = (0, PresentationSchema_1.acceptPresentation)(parameter.acceptor, partialCommand.commandTable, nextItem);
            if (acceptedPresentation === undefined) {
                return ParseErrors_1.ArgumentParseError.Result(`Was expecting a match for the presentation type: ${(0, PresentationSchema_1.printPresentationSchema)(parameter.acceptor)} but got ${TextPresentationRenderer_1.TextPresentationRenderer.render(nextItem)}.`, {
                    parameter: parameter,
                    partialCommand: partialCommand,
                });
            }
            stream.readItem(); // disopose of argument.
            immediateArguments.push(acceptedPresentation);
        }
        const restResult = keywordsParser.parseRest(partialCommand, this.rest);
        if ((0, typescript_result_1.isError)(restResult)) {
            return restResult;
        }
        return (0, typescript_result_1.Ok)({
            description: partialCommand.description,
            immediateArguments: immediateArguments.map((p) => p.object),
            keywords: keywordsParser.getKeywords(),
            rest: (_b = (_a = restResult.ok) === null || _a === void 0 ? void 0 : _a.map((p) => p.object)) !== null && _b !== void 0 ? _b : [],
            designator: partialCommand.designator,
            isPartial: false,
            commandTable: partialCommand.commandTable,
            toPartialCommand() {
                return {
                    description: partialCommand.description,
                    stream: stream,
                    isPartial: true,
                    designator: partialCommand.designator,
                    commandTable: partialCommand.commandTable,
                };
            },
        });
    }
}
exports.StandardCommandParametersDescription = StandardCommandParametersDescription;
function describeCommandParameters(options) {
    return new StandardCommandParametersDescription(parameterDescriptionsFromParameterOptions(options.parameters), options.keywords === undefined
        ? (0, KeywordParameterDescription_1.describeKeywordParameters)({
            keywordDescriptions: {},
            allowOtherKeys: false,
        })
        : (0, KeywordParameterDescription_1.describeKeywordParameters)(options.keywords), options.rest === undefined
        ? undefined
        : (0, RestParameterDescription_1.describeRestParameters)(options.rest));
}
function parameterDescriptionsFromParameterOptions(descriptions) {
    return descriptions.map(describeParameter);
}
function describeParameter(description) {
    if ("schemaType" in description.acceptor) {
        return description;
    }
    else {
        return {
            ...description,
            acceptor: {
                schemaType: PresentationSchema_1.PresentationSchemaType.Single,
                presentationType: description.acceptor,
            },
        };
    }
}
/**
 * For some reason typescript really struggles to infer tuples.
 * So we have to use a function to guide the inference.
 * This is supposed to be used on parameter descriptions.
 */
function tuple(...args) {
    return args;
}
//# sourceMappingURL=ParameterParsing.js.map