"use strict";
// SPDX-FileCopyrightText: 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.readCommand = readCommand;
const super_cool_stream_1 = require("@gnuxie/super-cool-stream");
const typescript_result_1 = require("@gnuxie/typescript-result");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const Keyword_1 = require("../Command/Keyword");
const TextPresentationTypes_1 = require("./TextPresentationTypes");
/** Whitespace we want to nom. */
const WHITESPACE = [" ", "\r", "\f", "\v", "\n", "\t"];
/**
 * Transforms a command from a string to a list of `Presentation`s.
 * The reader works by reading the command word by word (\S),
 * producing a Presentation for each word.
 * The reader doesn't produce an AST because there isn't any syntax that can make a tree,
 * only a flat list can be produced.
 * This allows commands to be dispatched based on `Presentation` and allows
 * for more efficient (in terms of loc) parsing of arguments.
 *
 * @param string The command.
 * @returns Presentations that have been read from this command.
 */
function readCommand(string) {
    return readCommandFromStream(new super_cool_stream_1.StringStream(string));
}
function readCommandFromStream(stream) {
    const words = [];
    eatWhitespace(stream);
    while (stream.peekChar() !== undefined) {
        words.push(readItem(stream));
        eatWhitespace(stream);
    }
    return words.map(applyPostReadTransformersToReadItem);
}
function eatWhitespace(stream) {
    readUntil(/\S/, stream, []);
}
function readWord(stream) {
    const word = [stream.readChar()];
    readUntil(/\s/, stream, word);
    return word.join("");
}
/**
 * Read a single "Item".
 * @param stream Stream to read the item from, must be at the beginning of a word not be EOF or whitespace.
 * @returns A single ReadItem.
 */
function readItem(stream) {
    if (stream.peekChar() === undefined) {
        throw new TypeError("EOF");
    }
    if (WHITESPACE.includes(stream.peekChar())) {
        throw new TypeError("whitespace should have been stripped");
    }
    const dispatchCharacter = stream.peekChar();
    if (dispatchCharacter === undefined) {
        throw new TypeError(`There should be a dispatch character and if there isn't then the code is wrong`);
    }
    const macro = WORD_DISPATCH_CHARACTERS.get(dispatchCharacter);
    if (macro) {
        return macro(stream);
    }
    else {
        // Then read a normal word.
        return readWord(stream);
    }
}
const WORD_DISPATCH_CHARACTERS = new Map();
/**
 * Defines a read macro to produce a read item.
 * @param dispatchCharacter A character at the start of a word that `readItem`
 * should use this macro to produce a `ReadItem` with.
 * @param macro A function that reads a stream and produces a `ReadItem`
 */
function defineReadItem(dispatchCharacter, macro) {
    if (WORD_DISPATCH_CHARACTERS.has(dispatchCharacter)) {
        throw new TypeError(`Read macro already defined for this dispatch character: ${dispatchCharacter}`);
    }
    WORD_DISPATCH_CHARACTERS.set(dispatchCharacter, macro);
}
const POST_READ_TRANSFORMERS = new Map();
/**
 * Define a function that will be applied to ReadItem's that are strings that
 * also match the regex.
 * If the regex matches, the transformer function will be called with the read item
 * and given the oppertunity to return a new version of the item.
 *
 * This is mainly used to transform URLs into a MatrixRoomReference.
 */
function definePostReadReplace(regex, transformer) {
    if (POST_READ_TRANSFORMERS.has(regex.source)) {
        throw new TypeError(`A transformer has already been defined for the regexp ${regex.source}`);
    }
    POST_READ_TRANSFORMERS.set(regex.source, { regex, transformer });
}
function applyPostReadTransformersToReadItem(item) {
    if (typeof item === "string") {
        for (const [, { regex, transformer }] of POST_READ_TRANSFORMERS) {
            if (regex.test(item)) {
                const transformResult = transformer(item);
                if (typeof transformResult !== "string") {
                    return transformResult;
                }
            }
        }
        return TextPresentationTypes_1.StringPresentationType.wrap(item);
    }
    return item;
}
/**
 * Helper that consumes from `stream` and appends to `output` until a character is peeked matching `regex`.
 * @param regex A regex for a character to stop at.
 * @param stream A stream to consume from.
 * @param output An array of characters.
 * @returns `output`.
 */
function readUntil(regex, stream, output) {
    while (stream.peekChar() !== undefined && !regex.test(stream.peekChar())) {
        output.push(stream.readChar());
    }
    return output;
}
/**
 * Produce a `MatrixRoomReference` from the stream from a room alias or id.
 * Returns a string if the room id or alias is malformed (and thus representing something else).
 * @param stream The stream to consume the room reference from.
 * @returns A MatrixRoomReference or string if what has been read does not represent a room.
 */
function readRoomIDOrAlias(stream) {
    const word = [stream.readChar()];
    readUntil(/\s/, stream, word);
    const wholeWord = word.join("");
    if ((0, matrix_basic_types_1.isStringRoomID)(wholeWord)) {
        return TextPresentationTypes_1.MatrixRoomIDPresentationType.wrap(new matrix_basic_types_1.MatrixRoomID(wholeWord));
    }
    else if ((0, matrix_basic_types_1.isStringRoomAlias)(wholeWord)) {
        return TextPresentationTypes_1.MatrixRoomAliasPresentationType.wrap(new matrix_basic_types_1.MatrixRoomAlias(wholeWord));
    }
    else {
        return wholeWord;
    }
}
/**
 * Read the word as an alias if it is an alias, otherwise it will just return a string token.
 */
defineReadItem("#", readRoomIDOrAlias);
defineReadItem("!", readRoomIDOrAlias);
/**
 * Read the word as a UserID, otherwise return a string if what has been read doesn not represent a user.
 */
defineReadItem("@", (stream) => {
    const word = [stream.readChar()];
    readUntil(/[:\s]/, stream, word);
    if (stream.peekChar() === undefined ||
        WHITESPACE.includes(stream.peekChar())) {
        return word.join("");
    }
    readUntil(/\s/, stream, word);
    const maybeUserID = word.join("");
    if ((0, matrix_basic_types_1.isStringUserID)(maybeUserID)) {
        return TextPresentationTypes_1.MatrixUserIDPresentationType.wrap(new matrix_basic_types_1.MatrixUserID(maybeUserID));
    }
    else {
        return maybeUserID;
    }
});
function maybeReadInteger(stream) {
    const isNegative = stream.peekChar() === "-";
    if (isNegative) {
        stream.readChar();
    }
    let word = "";
    while (stream.peekChar() !== undefined && /^[0-9]$/.test(stream.peekChar())) {
        const char = stream.readChar();
        if (char === undefined) {
            throw new TypeError("Shouldn't be possible cos we're checking if it's undefined in the loop clause");
        }
        word += char;
    }
    const number = isNegative ? -Number.parseInt(word) : Number.parseInt(word);
    return word.length > 0 ? TextPresentationTypes_1.NumberPresentationType.wrap(number) : undefined;
}
function readKeywordOrNegativeInteger(stream) {
    const maybeInteger = stream.savingPositionIf({
        predicate: (t) => t === undefined,
        body: (stream) => maybeReadInteger(stream),
    });
    if (maybeInteger !== undefined) {
        return maybeInteger;
    }
    else {
        return readKeyword(stream);
    }
}
/**
 * Read a keyword frorm the stream, throws away all of the prefixing `[:-]` characters
 * when producing the keyword designator.
 * @param stream A stream to consume the keyword from.
 * @returns A `Keyword`
 */
function readKeyword(stream) {
    readUntil(/[^-:]/, stream, []);
    if (stream.peekChar() === undefined) {
        return TextPresentationTypes_1.KeywordPresentationType.wrap(new Keyword_1.Keyword(""));
    }
    const word = [stream.readChar()];
    readUntil(/[\s]/, stream, word);
    return TextPresentationTypes_1.KeywordPresentationType.wrap(new Keyword_1.Keyword(word.join("")));
}
defineReadItem("-", readKeywordOrNegativeInteger);
defineReadItem(":", readKeywordOrNegativeInteger);
function maybeReadQuotedString(stream) {
    if (stream.peekChar() !== '"') {
        return undefined;
    }
    stream.readChar();
    const word = [];
    do {
        readUntil(/"/, stream, word);
        if (stream.peekChar() === '"' && word.at(-1) === "\\") {
            word.pop();
            word.push('"');
            stream.readChar();
            if (stream.peekChar() === undefined) {
                return TextPresentationTypes_1.StringPresentationType.wrap(word.join(""));
            }
            else {
                continue;
            }
        }
        else if (stream.peekChar() === '"') {
            stream.readChar();
            // wrap to stop post processing of the string
            // which is the reason they are quoting in the first place.
            return TextPresentationTypes_1.StringPresentationType.wrap(word.join(""));
        }
        else if (stream.peekChar() === undefined) {
            // the only reason we don't error here is because we don't have the infrastructure
            // for how reader errors will work and not look confusing as shit.
            return undefined;
        }
        // eslint-disable-next-line no-constant-condition
    } while (true);
}
function readString(stream) {
    const quotedString = stream.savingPositionIf({
        predicate: (t) => t === undefined,
        body: (stream) => maybeReadQuotedString(stream),
    });
    if (quotedString !== undefined) {
        return quotedString;
    }
    else {
        // we want these to be transformable read items.
        return readWord(stream);
    }
}
defineReadItem('"', readString);
definePostReadReplace(/^https:\/\/matrix\.to/, (input) => {
    const parseResult = matrix_basic_types_1.Permalinks.parseUrl(input);
    if ((0, typescript_result_1.isError)(parseResult)) {
        // it's an invalid URI.
        return input;
    }
    const url = parseResult.ok;
    if (url.eventID !== undefined) {
        const eventResult = matrix_basic_types_1.MatrixEventReference.fromPermalink(input);
        if ((0, typescript_result_1.isError)(eventResult)) {
            return input;
        }
        else {
            return TextPresentationTypes_1.MatrixEventReferencePresentationType.wrap(eventResult.ok);
        }
    }
    else if (url.userID !== undefined) {
        return TextPresentationTypes_1.MatrixUserIDPresentationType.wrap(new matrix_basic_types_1.MatrixUserID(url.userID));
    }
    else {
        const roomResult = matrix_basic_types_1.MatrixRoomReference.fromPermalink(input);
        if ((0, typescript_result_1.isError)(roomResult)) {
            return input;
        }
        else {
            if (roomResult.ok instanceof matrix_basic_types_1.MatrixRoomID) {
                return TextPresentationTypes_1.MatrixRoomIDPresentationType.wrap(roomResult.ok);
            }
            else {
                return TextPresentationTypes_1.MatrixRoomAliasPresentationType.wrap(roomResult.ok);
            }
        }
    }
});
definePostReadReplace(/^[0-9]+$/, (input) => {
    return TextPresentationTypes_1.NumberPresentationType.wrap(Number.parseInt(input));
});
definePostReadReplace(/^true|false$/, (input) => {
    return TextPresentationTypes_1.BooleanPresentationType.wrap(input === "true");
});
//# sourceMappingURL=TextCommandReader.js.map