"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardClientsInRoomMap = void 0;
const MembershipEvent_1 = require("../MatrixTypes/MembershipEvent");
const Value_1 = require("../Interface/Value");
const StandardClientRooms_1 = require("./StandardClientRooms");
const Action_1 = require("../Interface/Action");
class StandardClientsInRoomMap {
    constructor() {
        this.userIDByRoom = new Map();
        this.clientRoomsByUserID = new Map();
        this.userRevisionListener = this.userRevisionListenerMethod.bind(this);
    }
    addUserToRoom(roomID, userID) {
        const entry = this.userIDByRoom.get(roomID);
        if (entry === undefined) {
            this.userIDByRoom.set(roomID, [userID]);
        }
        else {
            entry.push(userID);
        }
    }
    removeUserFromRoom(roomID, userID) {
        const entry = this.userIDByRoom.get(roomID);
        if (entry == undefined) {
            return;
        }
        const nextEntry = entry.filter((user) => user !== userID);
        if (nextEntry.length === 0) {
            this.userIDByRoom.delete(roomID);
        }
        else {
            this.userIDByRoom.set(roomID, nextEntry);
        }
    }
    userRevisionListenerMethod(revision, changes) {
        for (const joinRoomID of changes.joined) {
            this.addUserToRoom(joinRoomID, revision.clientUserID);
        }
        for (const preemptivelyJoinedRoomID of changes.preemptivelyJoined) {
            this.addUserToRoom(preemptivelyJoinedRoomID, revision.clientUserID);
        }
        for (const partRoomID of changes.parted) {
            this.removeUserFromRoom(partRoomID, revision.clientUserID);
        }
        for (const partRoomID of changes.failedPreemptiveJoins) {
            this.removeUserFromRoom(partRoomID, revision.clientUserID);
        }
    }
    addClientRooms(client) {
        for (const roomID of client.currentRevision.allJoinedRooms) {
            this.addUserToRoom(roomID, client.clientUserID);
        }
        for (const roomID of client.allPreemptedRooms) {
            this.addUserToRoom(roomID, client.clientUserID);
        }
        this.clientRoomsByUserID.set(client.clientUserID, client);
        client.on('revision', this.userRevisionListener);
    }
    async makeClientRooms(userID, joinedRoomsThunk) {
        // if for whatever reason a client needs this class to use a different
        // implemetnation of ClientRooms, then we should add a dedicated factory
        // as an argument to this class.
        const existingClientRooms = this.getClientRooms(userID);
        if (existingClientRooms !== undefined) {
            return (0, Action_1.Ok)(existingClientRooms);
        }
        const clientRooms = await StandardClientRooms_1.StandardClientRooms.makeClientRooms(userID, joinedRoomsThunk);
        if ((0, Action_1.isError)(clientRooms)) {
            return clientRooms;
        }
        else {
            this.addClientRooms(clientRooms.ok);
            return clientRooms;
        }
    }
    removeClientRooms(client) {
        for (const roomID of client.currentRevision.allJoinedRooms) {
            this.removeUserFromRoom(roomID, client.clientUserID);
        }
        for (const roomID of client.allPreemptedRooms) {
            this.removeUserFromRoom(roomID, client.clientUserID);
        }
        this.clientRoomsByUserID.delete(client.clientUserID);
        client.off('revision', this.userRevisionListener);
    }
    removeClient(clientUserID) {
        const clientRooms = this.getClientRooms(clientUserID);
        if (clientRooms === undefined) {
            return;
        }
        this.removeClientRooms(clientRooms);
    }
    isClientInRoom(userID, roomID) {
        const entry = this.clientRoomsByUserID.get(userID);
        if (entry === undefined) {
            return false;
        }
        else {
            return entry.isJoinedRoom(roomID);
        }
    }
    isClientPreemptivelyInRoom(userID, roomID) {
        const entry = this.clientRoomsByUserID.get(userID);
        if (entry === undefined) {
            return false;
        }
        else {
            return entry.isPreemptivelyJoinedRoom(roomID);
        }
    }
    preemptTimelineJoin(userID, roomID) {
        const entry = this.getClientRooms(userID);
        if (entry === undefined) {
            throw new TypeError(`Unable to preempt a join for an unknown client ${userID}`);
        }
        entry.preemptTimelineJoin(roomID);
    }
    getClientRooms(userID) {
        return this.clientRoomsByUserID.get(userID);
    }
    getManagedUsersInRoom(roomID) {
        var _a;
        return (_a = this.userIDByRoom.get(roomID)) !== null && _a !== void 0 ? _a : [];
    }
    handleTimelineEvent(roomID, event) {
        const usersInRoom = this.getManagedUsersInRoom(roomID);
        for (const user of usersInRoom) {
            const clientRooms = this.getClientRooms(user);
            clientRooms === null || clientRooms === void 0 ? void 0 : clientRooms.handleTimelineEvent(roomID, event);
        }
        if (event.type === 'm.room.member' && Value_1.Value.Check(MembershipEvent_1.MembershipEvent, event)) {
            // only inform if we already informed the client about this event.
            if (!usersInRoom.includes(event.state_key)) {
                const clientRooms = this.getClientRooms(event.state_key);
                clientRooms === null || clientRooms === void 0 ? void 0 : clientRooms.handleTimelineEvent(roomID, event);
            }
        }
    }
}
exports.StandardClientsInRoomMap = StandardClientsInRoomMap;
//# sourceMappingURL=ClientsInRoomMap.js.map