"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardPersistentConfigData = void 0;
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const typescript_result_1 = require("@gnuxie/typescript-result");
const ConfigParseError_1 = require("./ConfigParseError");
const Value_1 = require("../Interface/Value");
class StandardPersistentConfigData {
    constructor(description, backend) {
        this.description = description;
        this.backend = backend;
        // nothing to do.
    }
    async requestParsedConfig() {
        const loadResult = await this.backend.requestUnparsedConfig();
        if ((0, typescript_result_1.isError)(loadResult)) {
            return loadResult;
        }
        if (loadResult.ok === undefined) {
            return (0, typescript_result_1.Ok)(undefined);
        }
        return this.addRecoveryOptionsToResult(loadResult.ok, this.description.parseConfig(loadResult.ok));
    }
    async saveConfig(config) {
        const encodeResult = Value_1.Value.Encode(this.description.schema, config);
        if ((0, typescript_result_1.isError)(encodeResult)) {
            return encodeResult;
        }
        return this.backend.saveEncodedConfig(encodeResult.ok);
    }
    makeRecoveryOptionForConfig() {
        return {
            description: 'Reset the configuration to its default values.',
            recover: async () => {
                const newConfig = this.description.getDefaultConfig();
                return await this.saveConfig(newConfig);
            },
        };
    }
    makeRecoveryOptionsForProperty(config, key) {
        return [
            {
                description: `Reset the property "${key}" to its default value.`,
                recover: async () => {
                    const newConfig = this.description
                        .toMirror()
                        .removeProperty(key, config);
                    return await this.backend.saveEncodedConfig(newConfig);
                },
            },
            this.makeRecoveryOptionForConfig(),
        ];
    }
    makeRecoveryOptionsForPropertyItem(config, key, index) {
        var _a;
        return [
            {
                description: `Remove the item "${(_a = config[key]) === null || _a === void 0 ? void 0 : _a[index]}" from the property "${key}".`,
                recover: async () => {
                    const newConfig = this.description
                        .toMirror()
                        .removeItem(config, key, index);
                    return await this.backend.saveEncodedConfig(newConfig);
                },
            },
            ...this.makeRecoveryOptionsForProperty(config, key),
        ];
    }
    makeRecoveryOptionsForError(config, error) {
        if (error instanceof ConfigParseError_1.ConfigParseError) {
            const mostRelevantError = error.errors[0];
            if (mostRelevantError === undefined) {
                return [this.makeRecoveryOptionForConfig()];
            }
            else {
                return this.makeRecoveryOptionsForError(config, mostRelevantError);
            }
        }
        else if (error instanceof ConfigParseError_1.ConfigPropertyError) {
            if (error.diagnosis === ConfigParseError_1.ConfigErrorDiagnosis.ProblematicArrayItem) {
                return this.makeRecoveryOptionsForPropertyItem(config, error.topLevelProperty(), error.itemIndex());
            }
            else {
                return this.makeRecoveryOptionsForProperty(config, error.topLevelProperty());
            }
        }
        else {
            return [];
        }
    }
    addRecoveryOptionsToResult(config, result) {
        if ((0, typescript_result_1.isOk)(result)) {
            return result;
        }
        else {
            const options = this.makeRecoveryOptionsForError(config, result.error);
            if (options.length > 0) {
                result.error.addRecoveryOptions(options);
            }
            return result;
        }
    }
    async reportUseError(message, options) {
        const loadResult = await this.backend.requestUnparsedConfig();
        if ((0, typescript_result_1.isError)(loadResult)) {
            return loadResult;
        }
        if (loadResult.ok === undefined) {
            throw new TypeError('The config defaults must be broken');
        }
        return this.addRecoveryOptionsToResult(loadResult.ok, ConfigParseError_1.ConfigPropertyUseError.Result(message, {
            ...options,
            description: this.description,
        }));
    }
}
exports.StandardPersistentConfigData = StandardPersistentConfigData;
//# sourceMappingURL=PersistentConfigData.js.map