"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Value = exports.DecodeException = void 0;
// Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const typebox_1 = require("@sinclair/typebox");
const compiler_1 = require("@sinclair/typebox/compiler");
const Action_1 = require("./Action");
const ActionException_1 = require("./ActionException");
const Logger_1 = require("../Logging/Logger");
class DecodeException extends ActionException_1.ActionException {
    constructor(message, exception, errors, suppressLog) {
        super(ActionException_1.ActionExceptionKind.Unknown, exception, message, {
            suppressLog: suppressLog !== null && suppressLog !== void 0 ? suppressLog : false,
        });
        this.errors = errors;
        if (!suppressLog) {
            DecodeException.log.error(this.uuid, ...this.errors);
        }
    }
}
exports.DecodeException = DecodeException;
DecodeException.log = new Logger_1.Logger('DecodeException');
class Value {
    static Compile(schema) {
        const entry = this.compiledSchema.get(schema);
        if (entry === undefined) {
            const compiledCheck = compiler_1.TypeCompiler.Compile(schema);
            this.compiledSchema.set(schema, compiledCheck);
            return compiledCheck;
        }
        return entry;
    }
    static Decode(schema, value, { suppressLogOnError } = {}) {
        const decoder = this.Compile(schema);
        try {
            return (0, Action_1.Ok)(decoder.Decode(value));
        }
        catch (e) {
            if (e instanceof typebox_1.TypeBoxError) {
                const errors = [...decoder.Errors(value)];
                return (0, Action_1.ResultError)(new DecodeException('Unable to decode an event', e, errors, suppressLogOnError));
            }
            else {
                throw e;
            }
        }
    }
    static Check(schema, value) {
        const decoder = this.Compile(schema);
        return decoder.Check(value);
    }
    static Encode(schema, value) {
        return this.resultify(schema, (encoder) => encoder.Encode(value));
    }
    static Errors(schema, value) {
        const decoder = this.Compile(schema);
        return decoder.Errors(value);
    }
    static resultify(schema, continuation) {
        try {
            const decoder = this.Compile(schema);
            return (0, Action_1.Ok)(continuation(decoder));
        }
        catch (e) {
            if (!(e instanceof typebox_1.TypeBoxError)) {
                throw e;
            }
            else {
                return ActionException_1.ActionException.Result(`Unable to decode schema from value`, {
                    exception: e,
                    exceptionKind: ActionException_1.ActionExceptionKind.Unknown,
                });
            }
        }
    }
}
exports.Value = Value;
Value.compiledSchema = new Map();
//# sourceMappingURL=Value.js.map