"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardPolicyListRevision = void 0;
exports.groupChangesByScope = groupChangesByScope;
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
const PolicyEvents_1 = require("../MatrixTypes/PolicyEvents");
const PolicyRule_1 = require("./PolicyRule");
const PolicyRuleChange_1 = require("./PolicyRuleChange");
const Revision_1 = require("./Revision");
const immutable_1 = require("immutable");
const crypto_js_1 = require("crypto-js");
const enc_base64_1 = __importDefault(require("crypto-js/enc-base64"));
const Logger_1 = require("../Logging/Logger");
const log = new Logger_1.Logger('StandardPolicyListRevision');
/**
 * A standard implementation of a `PolicyListRevision` using immutable's persistent maps.
 */
class StandardPolicyListRevision {
    /**
     * Use {@link StandardPolicyListRevision.blankRevision} to get started.
     * Only use this constructor if you are implementing a variant of PolicyListRevision.
     * @param revisionID A revision ID to represent this revision.
     * @param policyRules A map containing the rules for this revision by state type and then state key.
     * @param policyRuleByEventId A map containing the rules ofr this revision by event id.
     */
    constructor(revisionID, 
    /**
     * Allow us to detect whether we have updated the state for this event.
     */
    policyRuleByType, policyRuleScopes) {
        this.revisionID = revisionID;
        this.policyRuleByType = policyRuleByType;
        this.policyRuleScopes = policyRuleScopes;
    }
    /**
     * @returns An empty revision.
     */
    static blankRevision() {
        return new StandardPolicyListRevision(new Revision_1.Revision(), (0, immutable_1.Map)(), (0, immutable_1.Map)());
    }
    isBlankRevision() {
        return this.policyRuleByType.isEmpty();
    }
    allRules() {
        return [...this.policyRuleByType.values()]
            .map((byEventId) => [...byEventId.values()])
            .flat();
    }
    allRulesMatchingEntity(entity, { recommendation, type: ruleKind, searchHashedRules, }) {
        var _a;
        const ruleTypeOf = (entityPart) => {
            if (ruleKind) {
                return ruleKind;
            }
            else if (entityPart.startsWith('!') || entityPart.startsWith('#')) {
                return PolicyEvents_1.PolicyRuleType.Room;
            }
            else if (entity.startsWith('@')) {
                return PolicyEvents_1.PolicyRuleType.User;
            }
            else {
                return PolicyEvents_1.PolicyRuleType.Server;
            }
        };
        if (recommendation !== undefined) {
            const scope = (_a = this.policyRuleScopes
                .get(ruleTypeOf(entity))) === null || _a === void 0 ? void 0 : _a.get(recommendation);
            if (scope === undefined) {
                return [];
            }
            return scope.allRulesMatchingEntity(entity, Boolean(searchHashedRules));
        }
        return this.allRulesOfType(ruleTypeOf(entity), recommendation).filter((rule) => rule.matchType !== PolicyRule_1.PolicyRuleMatchType.HashedLiteral &&
            rule.isMatch(entity));
    }
    findRulesMatchingHash(hash, algorithm, { type, recommendation, }) {
        if (algorithm !== 'sha256') {
            throw new TypeError('Unimplemented hash algorithm');
        }
        const allScopesForType = this.policyRuleScopes.get(type);
        if (allScopesForType === undefined) {
            return [];
        }
        const rules = [];
        const scopesToCheck = (() => {
            if (recommendation !== undefined) {
                const recommendationScope = allScopesForType.get(recommendation);
                if (recommendationScope === undefined) {
                    return [];
                }
                else {
                    return [recommendationScope];
                }
            }
            else {
                return [...allScopesForType.values()];
            }
        })();
        for (const scope of scopesToCheck) {
            rules.push(...scope.findHashRules(hash));
        }
        return rules;
    }
    findRuleMatchingEntity(entity, { recommendation, type, searchHashedRules }) {
        var _a;
        const scope = (_a = this.policyRuleScopes.get(type)) === null || _a === void 0 ? void 0 : _a.get(recommendation);
        if (scope === undefined) {
            return undefined;
        }
        else {
            return scope.findRuleMatchingEntity(entity, searchHashedRules);
        }
    }
    allRulesOfType(type, recommendation) {
        const rules = [];
        const eventIdMap = this.policyRuleByType.get(type);
        if (eventIdMap) {
            for (const rule of eventIdMap.values()) {
                if (rule.kind === type) {
                    if (recommendation === undefined) {
                        rules.push(rule);
                    }
                    else if (rule.recommendation === recommendation) {
                        rules.push(rule);
                    }
                }
            }
        }
        return rules;
    }
    reviseFromChanges(changes) {
        var _a;
        let nextPolicyRulesByType = this.policyRuleByType;
        const setPolicyRule = (stateType, rule) => {
            var _a;
            const byEventTable = (_a = nextPolicyRulesByType.get(stateType)) !== null && _a !== void 0 ? _a : (0, immutable_1.Map)();
            nextPolicyRulesByType = nextPolicyRulesByType.set(stateType, byEventTable.set(rule.sourceEvent.event_id, rule));
        };
        const removePolicyRule = (rule) => {
            const byEventTable = nextPolicyRulesByType.get(rule.kind);
            if (byEventTable === undefined) {
                throw new TypeError(`Cannot find a rule for ${rule.sourceEvent.event_id}, this should be impossible`);
            }
            nextPolicyRulesByType = nextPolicyRulesByType.set(rule.kind, byEventTable.delete(rule.sourceEvent.event_id));
        };
        for (const change of changes) {
            if (change.changeType === PolicyRuleChange_1.PolicyRuleChangeType.Added ||
                change.changeType === PolicyRuleChange_1.PolicyRuleChangeType.Modified) {
                setPolicyRule(change.rule.kind, change.rule);
            }
            else if (change.changeType === PolicyRuleChange_1.PolicyRuleChangeType.RevealedLiteral) {
                if ((_a = this.policyRuleByType
                    .get(change.rule.kind)) === null || _a === void 0 ? void 0 : _a.get(change.rule.sourceEvent.event_id)) {
                    setPolicyRule(change.rule.kind, change.rule);
                }
                else {
                    // We need to discount revealed literals for rules we don't know about... because otherwise we could be interning removed rules.
                    log.error('got a RevealedLiteral for an unknown policy rule', change.rule);
                }
                // The code base could change, and then we'd be screwed:
                // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
            }
            else if (change.changeType === PolicyRuleChange_1.PolicyRuleChangeType.Removed) {
                removePolicyRule(change.rule);
            }
            else {
                throw new TypeError(`Unknown change type ${change.changeType}`);
            }
        }
        const nextRevisionID = new Revision_1.Revision();
        const changesByScope = groupChangesByScope(changes);
        const nextPolicyRuleScopes = flattenChangesByScope(changesByScope).reduce((map, [policyRuleType, recommendation, changes]) => {
            const scopeEntry = map.getIn([policyRuleType, recommendation], undefined);
            const byEventMap = nextPolicyRulesByType.get(policyRuleType, (0, immutable_1.Map)());
            if (scopeEntry === undefined) {
                return map.setIn([policyRuleType, recommendation], PolicyRuleScope.blankScope(nextRevisionID, policyRuleType, recommendation).reviseFromChanges(nextRevisionID, changes, byEventMap));
            }
            else {
                return map.setIn([policyRuleType, recommendation], scopeEntry.reviseFromChanges(nextRevisionID, changes, byEventMap));
            }
        }, this.policyRuleScopes);
        return new StandardPolicyListRevision(nextRevisionID, nextPolicyRulesByType, nextPolicyRuleScopes);
    }
    hasEvent(eventId) {
        return ([...this.policyRuleByType.values()].find((byEvent) => byEvent.has(eventId)) !== undefined);
    }
    hasPolicy(eventID) {
        return this.hasEvent(eventID);
    }
    getPolicy(eventID) {
        const map = [...this.policyRuleByType.values()].find((byEvent) => byEvent.has(eventID));
        return map === null || map === void 0 ? void 0 : map.get(eventID);
    }
}
exports.StandardPolicyListRevision = StandardPolicyListRevision;
function groupChangesByScope(changes) {
    const changesByScope = new Map();
    const addChange = (change) => {
        const policyTypeEntry = changesByScope.get(change.rule.kind);
        if (policyTypeEntry === undefined) {
            const map = new Map();
            map.set(change.rule.recommendation, [change]);
            changesByScope.set(change.rule.kind, map);
        }
        else {
            const recommendationEntry = policyTypeEntry.get(change.rule.recommendation);
            if (recommendationEntry === undefined) {
                policyTypeEntry.set(change.rule.recommendation, [change]);
            }
            else {
                recommendationEntry.push(change);
            }
        }
    };
    for (const change of changes) {
        addChange(change);
    }
    return changesByScope;
}
function flattenChangesByScope(scopes) {
    const flatChanges = [];
    for (const [policyRuleType, changeByRecommendation] of scopes.entries()) {
        for (const [recommendation, changes] of changeByRecommendation.entries()) {
            flatChanges.push([policyRuleType, recommendation, changes]);
        }
    }
    return flatChanges;
}
/**
 * A scope is a collection of rules that are scoped to a single entity type and
 * recommendation. So for the most basic policy list, there will usually be
 * a scope for all the `m.policy.rule.user` events that have the recommendation
 * `m.ban`.
 *
 * Scopes are built, quite painfully, to make rule lookup convienant and quick.
 * We accept this because revisions are few and far between, and if they are
 * frequent, will have a very small number of change events.
 */
class PolicyRuleScope {
    static blankScope(revisionID, ruleType, recommendation) {
        return new PolicyRuleScope(revisionID, ruleType, recommendation, (0, immutable_1.Map)(), (0, immutable_1.Map)(), (0, immutable_1.Map)());
    }
    constructor(revisionID, 
    /**
     * The entity type that this cache is for e.g. RULE_USER.
     */
    entityType, 
    /**
     * The recommendation that this cache is for e.g. m.ban (RECOMMENDATION_BAN).
     */
    recommendation, 
    /**
     * Glob rules always have to be scanned against every entity.
     */
    globRules, 
    /**
     * This table allows us to skip matching an entity against every literal.
     */
    literalRules, 
    /**
     * Hashed literal rules. This tables allows us to quickly find hashed rules.
     */
    sha256HashedLiteralRules) {
        this.revisionID = revisionID;
        this.entityType = entityType;
        this.recommendation = recommendation;
        this.globRules = globRules;
        this.literalRules = literalRules;
        this.sha256HashedLiteralRules = sha256HashedLiteralRules;
        // nothing to do.
    }
    reviseFromChanges(revision, changes, rulesByEventID) {
        const addRuleToMap = (map, rule) => {
            var _a;
            const rules = (_a = map.get(rule.entity)) !== null && _a !== void 0 ? _a : (0, immutable_1.List)();
            return map.set(rule.entity, rules.push(rule));
        };
        const removeRuleFromMap = (map, ruleToRemove) => {
            var _a;
            const rules = ((_a = map.get(ruleToRemove.entity)) !== null && _a !== void 0 ? _a : (0, immutable_1.List)()).filter((rule) => rule.sourceEvent.event_id !== ruleToRemove.sourceEvent.event_id);
            if (rules.size === 0) {
                return map.delete(ruleToRemove.entity);
            }
            else {
                return map.set(ruleToRemove.entity, rules);
            }
        };
        let nextGlobRules = this.globRules;
        let nextLiteralRules = this.literalRules;
        let nextSha256LiteralRules = this.sha256HashedLiteralRules;
        const addRule = (rule) => {
            var _a;
            if (rule.matchType === PolicyRule_1.PolicyRuleMatchType.Glob) {
                nextGlobRules = addRuleToMap(nextGlobRules, rule);
            }
            else if (rule.matchType === PolicyRule_1.PolicyRuleMatchType.Literal) {
                nextLiteralRules = addRuleToMap(nextLiteralRules, rule);
            }
            else {
                const sha256 = rule.hashes['sha256'];
                if (sha256) {
                    nextSha256LiteralRules = ((rules) => nextSha256LiteralRules.set(sha256, rules.push(rule)))((_a = nextSha256LiteralRules.get(sha256)) !== null && _a !== void 0 ? _a : (0, immutable_1.List)());
                }
            }
        };
        const removeRule = (rule) => {
            var _a;
            if (rule.matchType === PolicyRule_1.PolicyRuleMatchType.Glob) {
                nextGlobRules = removeRuleFromMap(nextGlobRules, rule);
            }
            else if (rule.matchType === PolicyRule_1.PolicyRuleMatchType.Literal) {
                nextLiteralRules = removeRuleFromMap(nextLiteralRules, rule);
            }
            else {
                const sha256 = rule.hashes['sha256'];
                if (sha256) {
                    const rules = ((_a = nextSha256LiteralRules.get(sha256)) !== null && _a !== void 0 ? _a : (0, immutable_1.List)()).filter((existingRule) => existingRule.sourceEvent.event_id !== rule.sourceEvent.event_id);
                    if (rules.size === 0) {
                        nextSha256LiteralRules = nextSha256LiteralRules.delete(sha256);
                    }
                    else {
                        nextSha256LiteralRules = nextSha256LiteralRules.set(sha256, rules);
                    }
                }
            }
        };
        for (const change of changes) {
            if (change.rule.kind !== this.entityType ||
                change.rule.recommendation !== this.recommendation) {
                continue;
            }
            switch (change.changeType) {
                case PolicyRuleChange_1.PolicyRuleChangeType.Added:
                case PolicyRuleChange_1.PolicyRuleChangeType.Modified:
                    addRule(change.rule);
                    break;
                case PolicyRuleChange_1.PolicyRuleChangeType.RevealedLiteral:
                    // We have to only add the rule if we know it is currently valid.. otherwise we could accidentally add a removed rule.
                    if (rulesByEventID.has(change.event.event_id)) {
                        addRule(change.rule);
                    }
                    break;
                case PolicyRuleChange_1.PolicyRuleChangeType.Removed:
                    removeRule(change.rule);
            }
        }
        return new PolicyRuleScope(revision, this.entityType, this.recommendation, nextGlobRules, nextLiteralRules, nextSha256LiteralRules);
    }
    literalRulesMatchingEntity(entity) {
        var _a;
        return [...((_a = this.literalRules.get(entity)) !== null && _a !== void 0 ? _a : [])];
    }
    globRulesMatchingEntity(entity) {
        return [...this.globRules.values()]
            .filter((rules) => {
            const [firstRule] = rules;
            if (firstRule === undefined) {
                throw new TypeError(`The code is wrong and so is my understanding of everything`);
            }
            return firstRule.isMatch(entity);
        })
            .map((rules) => [...rules])
            .flat();
    }
    hashedRulesMatchingEntity(entity) {
        return [
            ...this.sha256HashedLiteralRules.get(enc_base64_1.default.stringify((0, crypto_js_1.SHA256)(entity)), []),
        ];
    }
    allRulesMatchingEntity(entity, searchHashedRules) {
        return [
            ...this.literalRulesMatchingEntity(entity),
            ...this.globRulesMatchingEntity(entity),
            ...(searchHashedRules ? this.hashedRulesMatchingEntity(entity) : []),
        ];
    }
    findRuleMatchingEntity(entity, searchHashedRules) {
        const literalRule = this.literalRules.get(entity);
        if (literalRule !== undefined) {
            return literalRule.get(0);
        }
        if (searchHashedRules) {
            const hashedRules = this.sha256HashedLiteralRules.get(enc_base64_1.default.stringify((0, crypto_js_1.SHA256)(entity)));
            if (hashedRules !== undefined) {
                return hashedRules.get(0);
            }
        }
        const globRules = this.globRulesMatchingEntity(entity);
        if (globRules.length === 0) {
            return undefined;
        }
        else {
            return globRules.at(0);
        }
    }
    findHashRules(hash) {
        return [...this.sha256HashedLiteralRules.get(hash, [])];
    }
}
//# sourceMappingURL=StandardPolicyListRevision.js.map