"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardUserConsequences = void 0;
const PowerLevelsMirror_1 = require("../../../Client/PowerLevelsMirror");
const Action_1 = require("../../../Interface/Action");
const MembershipChange_1 = require("../../../Membership/MembershipChange");
const CapabilityProvider_1 = require("../CapabilityProvider");
const RoomSetResult_1 = require("./RoomSetResult");
const UserConsequences_1 = require("./UserConsequences");
require("./UserConsequences"); // we need this so the interface is loaded.
class StandardUserConsequences {
    constructor(roomBanner, roomUnbanner, setMembership) {
        this.roomBanner = roomBanner;
        this.roomUnbanner = roomUnbanner;
        this.setMembership = setMembership;
        this.requiredPermissions = [PowerLevelsMirror_1.PowerLevelPermission.Ban];
        this.requiredEventPermissions = [];
        this.requiredStatePermissions = [];
        // nothing to do.
    }
    static async applyPolicyMatchesToRoom(matches, roomMembershipRevision, consequenceProviderCB) {
        const resultBuilder = new RoomSetResult_1.ResultForUsersInRoomBuilder();
        for (const match of matches) {
            const membership = roomMembershipRevision.membershipForUser(match.userID);
            if (membership === undefined ||
                membership.membership === MembershipChange_1.Membership.Ban) {
                continue;
            }
            const consequenceResult = await consequenceProviderCB(membership.roomID, membership.userID, (0, UserConsequences_1.targetReason)(match));
            resultBuilder.addResult(match.userID, consequenceResult);
        }
        return resultBuilder.getResult();
    }
    static async applyPolicyMatchesToSetMembership(matches, setMembership, consequenceProviderCB) {
        const resultBuilder = new RoomSetResult_1.ResultForUsersInSetBuilder();
        for (const membershipRevision of setMembership.allRooms) {
            const results = await StandardUserConsequences.applyPolicyMatchesToRoom(matches, membershipRevision, consequenceProviderCB);
            for (const [userID, result] of results.map) {
                resultBuilder.addResult(userID, membershipRevision.room.toRoomIDOrAlias(), result);
            }
        }
        return resultBuilder.getResult();
    }
    async consequenceForUserInRoom(roomID, user, reason) {
        return await this.roomBanner.banUser(roomID, user, reason);
    }
    async consequenceForUsersInRoomSet(targets) {
        return (0, Action_1.Ok)(await StandardUserConsequences.applyPolicyMatchesToSetMembership(targets, this.setMembership, this.roomBanner.banUser.bind(this.roomBanner)));
    }
    async consequenceForUsersInRoom(roomID, targets) {
        const membershipRevision = this.setMembership.getRevision(roomID);
        if (membershipRevision === undefined) {
            return Action_1.ActionError.Result(`Unable to find a membership revision for the room ${roomID}`);
        }
        return (0, Action_1.Ok)(await StandardUserConsequences.applyPolicyMatchesToRoom(targets, membershipRevision, this.roomBanner.banUser.bind(this.roomBanner)));
    }
    async unbanUserFromRoomSet(userID, reason) {
        const resultBuilder = new RoomSetResult_1.RoomSetResultBuilder();
        for (const membershipRevision of this.setMembership.allRooms) {
            const membership = membershipRevision.membershipForUser(userID);
            if (membership !== undefined &&
                membership.membership === MembershipChange_1.Membership.Ban) {
                resultBuilder.addResult(membershipRevision.room.toRoomIDOrAlias(), await this.roomUnbanner.unbanUser(membershipRevision.room, userID, reason));
            }
        }
        return (0, Action_1.Ok)(resultBuilder.getResult());
    }
}
exports.StandardUserConsequences = StandardUserConsequences;
(0, CapabilityProvider_1.describeCapabilityProvider)({
    name: 'StandardUserConsequences',
    description: 'Issues room level bans and unbans for users.',
    interface: 'UserConsequences',
    factory(_description, context) {
        return new StandardUserConsequences(context.roomBanner, context.roomUnbanner, context.setMembership);
    },
});
//# sourceMappingURL=StandardUserConsequences.js.map