"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynapseAdminUserSuspensionCapability = void 0;
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const typescript_result_1 = require("@gnuxie/typescript-result");
const HomeserverUserPurgingDeactivate_1 = require("./HomeserverUserPurgingDeactivate");
require("./UserRestrictionCapability");
const log = new matrix_protection_suite_1.Logger("SynapseAdminUserSuspensionCapability");
class SynapseAdminUserSuspensionCapability {
    constructor(synapseAdminClient, userAuditLog) {
        this.synapseAdminClient = synapseAdminClient;
        this.userAuditLog = userAuditLog;
        this.requiredPermissions = [];
        this.requiredStatePermissions = [];
        this.requiredEventPermissions = [];
        // nothing to do here.
    }
    async isUserRestricted(userID) {
        const userDetails = await this.synapseAdminClient.getUserDetails(userID);
        if ((0, typescript_result_1.isError)(userDetails)) {
            return userDetails;
        }
        if (!userDetails.ok) {
            return typescript_result_1.ResultError.Result(`Synapse cannot find details for the user ${userID}`);
        }
        const isUserRestricted = (0, HomeserverUserPurgingDeactivate_1.isUserAccountRestricted)(userDetails.ok);
        if (!isUserRestricted) {
            return (0, typescript_result_1.Ok)(false);
        }
        // We intentionally update the audit log here to keep our local information
        // accurate.
        const auditResult = await this.userAuditLog.isUserRestricted(userID);
        if ((0, typescript_result_1.isError)(auditResult)) {
            log.error("Failed to check if user is restricted", userID);
            return (0, typescript_result_1.Ok)(isUserRestricted);
        }
        else if (auditResult.ok) {
            log.debug("Recording missing user restriction", userID);
            const logResult = await this.userAuditLog.recordExistingUserRestriction(userID, matrix_protection_suite_for_matrix_bot_sdk_1.AccountRestriction.Suspended);
            if ((0, typescript_result_1.isError)(logResult)) {
                log.error("Failed to audit a missing user restriction", userID);
            }
        }
        return (0, typescript_result_1.Ok)(isUserRestricted);
    }
    async restrictUser(userID, options) {
        const suspendResult = await this.synapseAdminClient.suspendUser(userID);
        if ((0, typescript_result_1.isError)(suspendResult)) {
            return suspendResult;
        }
        const logResult = await this.userAuditLog.recordUserRestriction(userID, matrix_protection_suite_for_matrix_bot_sdk_1.AccountRestriction.Suspended, options);
        if ((0, typescript_result_1.isError)(logResult)) {
            log.error("Failed to audit a suspension", userID);
            return logResult.elaborate("Failed to audit the suspension");
        }
        return (0, typescript_result_1.Ok)(matrix_protection_suite_for_matrix_bot_sdk_1.AccountRestriction.Suspended);
    }
    async unrestrictUser(userID, sender) {
        const unsuspendResult = await this.synapseAdminClient.unrestrictUser(userID);
        if ((0, typescript_result_1.isError)(unsuspendResult)) {
            return unsuspendResult;
        }
        const auditResult = await this.userAuditLog.recordUserRestriction(userID, unsuspendResult.ok, { sender, rule: null });
        if ((0, typescript_result_1.isError)(auditResult)) {
            return auditResult.elaborate("Failed to audit the unsuspension of a user");
        }
        return (0, typescript_result_1.Ok)(undefined);
    }
}
exports.SynapseAdminUserSuspensionCapability = SynapseAdminUserSuspensionCapability;
(0, matrix_protection_suite_1.describeCapabilityProvider)({
    name: SynapseAdminUserSuspensionCapability.name,
    description: `A capability to suspend users on the homeserver`,
    interface: "UserRestrictionCapability",
    factory(description, draupnir) {
        if (draupnir.synapseAdminClient === undefined ||
            draupnir.stores.userRestrictionAuditLog === undefined) {
            throw new TypeError("This capability requires the SynapseAdminClient and the user restriction audit log to be available to draupnir, and they are not in your configuration.");
        }
        return new SynapseAdminUserSuspensionCapability(draupnir.synapseAdminClient, draupnir.stores.userRestrictionAuditLog);
    },
});
//# sourceMappingURL=UserSuspensionCapability.js.map