"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2020 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirKickCommand = void 0;
const matrix_bot_sdk_1 = require("matrix-bot-sdk");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
function addUserToKick(map, roomID, userID) {
    const userEntry = map.get(userID) ?? ((entry) => (map.set(userID, entry), entry))([]);
    userEntry.push(roomID);
    return map;
}
function renderUsersToKick(usersToKick) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
                "Kicking ",
                usersToKick.size,
                " unique users from protected rooms."),
            [...usersToKick.entries()].map(([userID, rooms]) => (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
                    "Kicking ",
                    userID,
                    " from ",
                    rooms.length,
                    " rooms."),
                interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, rooms.map((room) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null, room))))))))));
}
exports.DraupnirKickCommand = (0, interface_manager_1.describeCommand)({
    summary: "Kicks a user or all of those matching a glob in a particular room or all protected rooms. `--glob` must be provided to use globs. Can be scoped to a specific room with `--room`. Can be dry run with `--dry-run`.",
    parameters: (0, interface_manager_1.tuple)({
        name: "user",
        acceptor: interface_manager_1.MatrixUserIDPresentationType,
    }),
    keywords: {
        keywordDescriptions: {
            "dry-run": {
                isFlag: true,
                description: "Runs the kick command without actually removing any users.",
            },
            glob: {
                isFlag: true,
                description: "Allows globs to be used to kick several users from rooms.",
            },
            room: {
                acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
                description: "Allows the command to be scoped to just one protected room.",
            },
        },
    },
    rest: {
        name: "reason",
        acceptor: interface_manager_1.StringPresentationType,
    },
    async executor({ roomKicker, roomResolver, setMembership, taskQueue, noop, }, _info, keywords, reasonParts, user) {
        const restrictToRoomReference = keywords.getKeywordValue("room", undefined);
        const isDryRun = noop || keywords.getKeywordValue("dry-run", false);
        const allowGlob = keywords.getKeywordValue("glob", false);
        const isGlob = user.toString().includes("*") || user.toString().includes("?");
        if (isGlob && !allowGlob) {
            return matrix_protection_suite_1.ActionError.Result("Wildcard bans require an additional argument `--glob` to confirm");
        }
        const restrictToRoom = restrictToRoomReference
            ? await roomResolver.resolveRoom(restrictToRoomReference)
            : undefined;
        if (restrictToRoom !== undefined && (0, matrix_protection_suite_1.isError)(restrictToRoom)) {
            return restrictToRoom;
        }
        const restrictToRoomRevision = restrictToRoom === undefined
            ? undefined
            : setMembership.getRevision(restrictToRoom.ok.toRoomIDOrAlias());
        const roomsToKickWithin = restrictToRoomRevision !== undefined
            ? [restrictToRoomRevision]
            : setMembership.allRooms;
        const reason = reasonParts.join(" ");
        const kickRule = new matrix_bot_sdk_1.MatrixGlob(user.toString());
        const usersToKick = new Map();
        for (const revision of roomsToKickWithin) {
            for (const member of revision.members()) {
                switch (member.membership) {
                    case matrix_protection_suite_1.Membership.Join:
                    case matrix_protection_suite_1.Membership.Invite:
                    case matrix_protection_suite_1.Membership.Knock: {
                        if (kickRule.test(member.userID)) {
                            addUserToKick(usersToKick, revision.room.toRoomIDOrAlias(), member.userID);
                            if (!isDryRun) {
                                taskQueue.push(() => {
                                    return roomKicker.kickUser(revision.room.toRoomIDOrAlias(), member.userID, reason);
                                });
                            }
                        }
                    }
                }
            }
        }
        return (0, matrix_protection_suite_1.Ok)(usersToKick);
    },
});
DraupnirCommandPrerequisites_1.DraupnirContextToCommandContextTranslator.registerTranslation(exports.DraupnirKickCommand, function (draupnir) {
    return {
        roomKicker: draupnir.clientPlatform.toRoomKicker(),
        roomResolver: draupnir.clientPlatform.toRoomResolver(),
        setMembership: draupnir.protectedRoomsSet.setRoomMembership,
        taskQueue: draupnir.taskQueue,
        noop: draupnir.config.noop,
    };
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirKickCommand, {
    JSXRenderer(result) {
        if ((0, matrix_protection_suite_1.isError)(result)) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        return (0, matrix_protection_suite_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, renderUsersToKick(result.ok)));
    },
});
//# sourceMappingURL=KickCommand.js.map