"use strict";
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirStatusCommand = void 0;
exports.groupWatchedPolicyRoomsByProtectionStatus = groupWatchedPolicyRoomsByProtectionStatus;
exports.draupnirStatusInfo = draupnirStatusInfo;
exports.renderPolicyList = renderPolicyList;
exports.renderStatusInfo = renderStatusInfo;
const config_1 = require("../config");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const PolicyChangeNotification_1 = require("../protections/PolicyChangeNotification");
const RoomTakedownProtection_1 = require("../protections/RoomTakedown/RoomTakedownProtection");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
exports.DraupnirStatusCommand = (0, interface_manager_1.describeCommand)({
    summary: "Show the status of the bot.",
    parameters: [],
    async executor(draupnir) {
        return (0, matrix_protection_suite_1.Ok)(draupnirStatusInfo(draupnir));
    },
});
function groupWatchedPolicyRoomsByProtectionStatus(watchedPolicyRooms, allJoinedRooms, protectedRooms) {
    const watchedListProfiles = watchedPolicyRooms.allRooms;
    const subscribedAndProtectedLists = watchedListProfiles.filter((profile) => allJoinedRooms.includes(profile.room.toRoomIDOrAlias()) &&
        protectedRooms.find((protectedRoom) => protectedRoom.toRoomIDOrAlias() === profile.room.toRoomIDOrAlias()));
    const subscribedLists = watchedListProfiles.filter((profile) => allJoinedRooms.includes(profile.room.toRoomIDOrAlias()) &&
        !protectedRooms.find((protectedRoom) => protectedRoom.toRoomIDOrAlias() === profile.room.toRoomIDOrAlias()));
    const subscribedButPartedLists = watchedListProfiles.filter((profile) => !allJoinedRooms.includes(profile.room.toRoomIDOrAlias()));
    return {
        subscribedLists: subscribedLists,
        subscribedAndProtectedLists: subscribedAndProtectedLists,
        subscribedButPartedLists,
    };
}
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirStatusCommand, {
    JSXRenderer(result) {
        if ((0, matrix_protection_suite_1.isError)(result)) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        return (0, matrix_protection_suite_1.Ok)(renderStatusInfo(result.ok));
    },
});
function extractProtectionNotificationRooms(draupnir) {
    return {
        policyChangeNotificationRoomID: draupnir.protectedRoomsSet.protections.findEnabledProtection(PolicyChangeNotification_1.PolicyChangeNotification.name)?.notificationRoomID,
        roomDiscoveryNotificationRoomID: draupnir.protectedRoomsSet.protections.findEnabledProtection(RoomTakedownProtection_1.RoomTakedownProtection.name)?.discoveryNotificationRoom,
    };
}
// FIXME: need a shoutout to dependencies in here and NOTICE info.
function draupnirStatusInfo(draupnir) {
    const watchedListInfo = groupWatchedPolicyRoomsByProtectionStatus(draupnir.protectedRoomsSet.watchedPolicyRooms, draupnir.clientRooms.currentRevision.allJoinedRooms, draupnir.protectedRoomsSet.allProtectedRooms);
    return {
        numberOfProtectedRooms: draupnir.protectedRoomsSet.allProtectedRooms.length,
        numberOfUniqueMembers: draupnir.protectedRoomsSet.setMembership.currentRevision.uniqueMemberCount(),
        subscribedLists: watchedListInfo.subscribedLists,
        subscribedAndProtectedLists: watchedListInfo.subscribedAndProtectedLists,
        subscribedButPartedLists: watchedListInfo.subscribedButPartedLists,
        documentationURL: config_1.DOCUMENTATION_URL,
        version: config_1.SOFTWARE_VERSION,
        repository: config_1.PACKAGE_JSON["repository"] ?? "Unknown",
        ...extractProtectionNotificationRooms(draupnir),
    };
}
function renderPolicyList(list) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: list.revision.room.toPermalink() }, list.revision.room.toRoomIDOrAlias()),
        " ",
        "  (",
        list.revision.shortcode ?? "<no shortcode>",
        ") propagation:",
        " ",
        list.propagation,
        "   (rules:",
        " ",
        list.revision.allRulesOfType(matrix_protection_suite_1.PolicyRuleType.Server).length,
        " servers,",
        " ",
        list.revision.allRulesOfType(matrix_protection_suite_1.PolicyRuleType.User).length,
        " users,",
        " ",
        list.revision.allRulesOfType(matrix_protection_suite_1.PolicyRuleType.Room).length,
        " rooms) (last update:",
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, new Date(list.revision.revisionID.time).toLocaleString()),
        ")"));
}
function renderNotificationRooms(info) {
    if (info.roomDiscoveryNotificationRoomID === undefined &&
        info.policyChangeNotificationRoomID === undefined) {
        return interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null);
    }
    const renderNotificationRoom = (name, roomID) => {
        if (roomID === undefined) {
            return interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null);
        }
        return (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
            name,
            ": ",
            (0, mps_interface_adaptor_1.renderRoomPill)(matrix_basic_types_1.MatrixRoomReference.fromRoomID(roomID))));
    };
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Notification rooms:"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            renderNotificationRoom("Policy change", info.policyChangeNotificationRoomID),
            renderNotificationRoom("Room discovery", info.roomDiscoveryNotificationRoomID))));
}
function renderStatusInfo(info) {
    const renderPolicyLists = (header, lists) => {
        const renderedLists = lists.map(renderPolicyList);
        return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, header),
            interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
            interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, renderedLists.length === 0 ? (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("i", null, "None"))) : (renderedLists))));
    };
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Protected Rooms: "),
        info.numberOfProtectedRooms,
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Protected Users: "),
        info.numberOfUniqueMembers,
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        renderPolicyLists("Subscribed policy rooms", info.subscribedLists),
        renderPolicyLists("Subscribed and protected policy rooms", info.subscribedAndProtectedLists),
        renderNotificationRooms(info),
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Version: "),
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, info.version),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Repository: "),
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, info.repository),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        interface_manager_1.DeadDocumentJSX.JSXFactory("b", null, "Documentation: "),
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("a", { href: info.documentationURL }, info.documentationURL),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null)));
}
//# sourceMappingURL=StatusCommand.js.map