"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardConfigDescription = exports.UnknownConfig = void 0;
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const typebox_1 = require("@sinclair/typebox");
const typescript_result_1 = require("@gnuxie/typescript-result");
const value_1 = require("@sinclair/typebox/value");
const ConfigParseError_1 = require("./ConfigParseError");
const ConfigMirror_1 = require("./ConfigMirror");
exports.UnknownConfig = typebox_1.Type.Object({}, { additionalProperties: true });
class StandardConfigDescription {
    constructor(schema) {
        this.schema = schema;
    }
    parseConfig(config) {
        // withDefaults will be fine as long as no one defaults with post-transformed values.
        const withDefaults = value_1.Value.Default(this.schema, config);
        const errors = [...value_1.Value.Errors(this.schema, withDefaults)];
        if (errors.length > 0) {
            return ConfigParseError_1.ConfigParseError.Result('Unable to parse this config', {
                errors: errors.map((error) => new ConfigParseError_1.ConfigPropertyError(error.message, this, error.path, error.value)),
                config: withDefaults,
                // We have a contravariance issue on the `toMirror` method because
                // the mirror accepts specific config shapes.
                description: this,
            });
        }
        else {
            return (0, typescript_result_1.Ok)(value_1.Value.Decode(this.schema, withDefaults));
        }
    }
    parseJSONConfig(config) {
        return (0, typescript_result_1.Ok)(config);
    }
    properties() {
        return Object.entries(this.schema.properties).map(([name, schema]) => ({
            name,
            path: '/' + name,
            description: schema.description,
            default: schema.default,
            isUniqueItems: 'uniqueItems' in schema && schema.uniqueItems === true,
            isArray: 'items' in schema,
        }));
    }
    getPropertyDescription(key) {
        const schema = this.schema.properties[key];
        if (schema === undefined) {
            throw new TypeError(`Property ${key} does not exist on this schema`);
        }
        return {
            name: key,
            path: '/' + key,
            description: schema.description,
            default: schema.default,
            isUniqueItems: 'uniqueItems' in schema && schema.uniqueItems === true,
            isArray: 'items' in schema,
        };
    }
    toMirror() {
        return new ConfigMirror_1.StandardConfigMirror(this);
    }
    getDefaultConfig() {
        return value_1.Value.Default(this.schema, {});
    }
}
exports.StandardConfigDescription = StandardConfigDescription;
//# sourceMappingURL=ConfigDescription.js.map