"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigPropertyUseError = exports.ConfigPropertyError = exports.ConfigParseError = exports.ConfigErrorDiagnosis = exports.ConfigRecoverableError = void 0;
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const typescript_result_1 = require("@gnuxie/typescript-result");
class ConfigRecoverableError extends typescript_result_1.ResultError {
    constructor(message, configDescription) {
        super(message);
        this.configDescription = configDescription;
        this.recoveryOptions = [];
    }
    addRecoveryOptions(options) {
        this.recoveryOptions.push(...options);
        return this;
    }
}
exports.ConfigRecoverableError = ConfigRecoverableError;
// others that could be missing: Missing porperties, completely different schema?
// We call them problematic because we can get errors once they are used too rather
// than just during parsing.
var ConfigErrorDiagnosis;
(function (ConfigErrorDiagnosis) {
    ConfigErrorDiagnosis["ProblematicValue"] = "ProblematicValue";
    ConfigErrorDiagnosis["ProblematicArrayItem"] = "ProblematicArrayItem";
})(ConfigErrorDiagnosis || (exports.ConfigErrorDiagnosis = ConfigErrorDiagnosis = {}));
class ConfigParseError extends ConfigRecoverableError {
    constructor(message, description, errors, config) {
        super(message, description);
        this.errors = errors;
        this.config = config;
    }
    static Result(message, options) {
        return (0, typescript_result_1.Err)(new ConfigParseError(message, options.description, options.errors, options.config));
    }
}
exports.ConfigParseError = ConfigParseError;
// This doesn't have to appear just during parsing, it can appear
// later on while processing the configuration file to display a problem
// with a particular property.
class ConfigPropertyError extends ConfigRecoverableError {
    constructor(message, description, path, value) {
        super(message, description);
        this.path = path;
        this.value = value;
        if (/\d+$/.test(path)) {
            this.diagnosis = ConfigErrorDiagnosis.ProblematicArrayItem;
        }
        else {
            this.diagnosis = ConfigErrorDiagnosis.ProblematicValue;
        }
    }
    static Result(message, options) {
        return (0, typescript_result_1.Err)(new ConfigPropertyError(message, options.description, options.path, options.value));
    }
    toReadableString() {
        return `Property at ${this.path} has the following diagnosis: ${this.diagnosis}, problem: ${this.message}, and value: ${String(this.value)}`;
    }
    itemIndex() {
        var _a;
        const match = (_a = this.path.match(/\/(\d+)$/)) === null || _a === void 0 ? void 0 : _a[1];
        if (match === undefined) {
            throw new TypeError('Invalid path was given to ConfigPropertyError');
        }
        return parseInt(match, 10);
    }
    topLevelProperty() {
        const key = this.path.split('/')[1];
        if (key === undefined) {
            throw new TypeError('Invalid path was given to ConfigPropertyError');
        }
        return key;
    }
}
exports.ConfigPropertyError = ConfigPropertyError;
class ConfigPropertyUseError extends ConfigPropertyError {
    constructor(message, description, path, value, cause) {
        super(message, description, path, value);
        this.cause = cause;
    }
    static Result(message, options) {
        return (0, typescript_result_1.Err)(new ConfigPropertyUseError(message, options.description, options.path, options.value, options.cause));
    }
    toReadableString() {
        return `${super.toReadableString()}\ncaused by: ${this.cause.toReadableString()}`;
    }
}
exports.ConfigPropertyUseError = ConfigPropertyUseError;
//# sourceMappingURL=ConfigParseError.js.map