"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// SPDX-FileCopyrightText: 2023-2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const DeclareRoomState_1 = require("../StateTracking/DeclareRoomState");
const EventGeneration_1 = require("../TestUtilities/EventGeneration");
const MembershipChange_1 = require("./MembershipChange");
test('Unseen memberships are classified as joins', function () {
    expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
        sender: (0, EventGeneration_1.randomUserID)(),
        membership: MembershipChange_1.Membership.Join,
    }), undefined)).toBe(MembershipChange_1.MembershipChangeType.Joined);
});
test('Invited and knocked users joining are classified as joins', function () {
    const sender = (0, EventGeneration_1.randomUserID)();
    const memberships = [MembershipChange_1.Membership.Invite, MembershipChange_1.Membership.Knock];
    for (const membership of memberships) {
        expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership: MembershipChange_1.Membership.Join,
        }), (0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership,
        }).content)).toBe(MembershipChange_1.MembershipChangeType.Joined);
    }
});
test('Previously discovered memberships are classified as Rejoins', function () {
    const sender = (0, EventGeneration_1.randomUserID)();
    const memberships = [
        MembershipChange_1.Membership.Leave,
        // we have to allow Ban -> Join because we might not have complete information
        // we might skip the "unban/leave" event.
        MembershipChange_1.Membership.Ban,
    ];
    for (const membership of memberships) {
        expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership: MembershipChange_1.Membership.Join,
        }), (0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership,
        }).content)).toBe(MembershipChange_1.MembershipChangeType.Rejoined);
    }
});
// my god there are some depressing shortfalls in MembershipChangeType.
// I don't like it.
test(`Leaving the room by a user's own admission is classified as leaving`, function () {
    const sender = (0, EventGeneration_1.randomUserID)();
    // Invite and Knock leaves should really be rejections and (whatever leaving on knock is supposed to mean).
    const memberships = [MembershipChange_1.Membership.Invite, MembershipChange_1.Membership.Knock, MembershipChange_1.Membership.Join];
    for (const membership of memberships) {
        expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership: MembershipChange_1.Membership.Leave,
        }), (0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership,
        }).content)).toBe(MembershipChange_1.MembershipChangeType.Left);
    }
    expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
        sender,
        target: sender,
        membership: MembershipChange_1.Membership.Leave,
    }))).toBe(MembershipChange_1.MembershipChangeType.Left);
});
test(`Being kicked by the room admin is classified as being kicked`, function () {
    const target = (0, EventGeneration_1.randomUserID)();
    const memberships = [MembershipChange_1.Membership.Invite, MembershipChange_1.Membership.Knock, MembershipChange_1.Membership.Join];
    for (const membership of memberships) {
        expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
            target,
            sender: target,
            membership: MembershipChange_1.Membership.Leave,
        }), (0, DeclareRoomState_1.describeRoomMember)({
            sender: (0, EventGeneration_1.randomUserID)(),
            target,
            membership,
        }).content)).toBe(MembershipChange_1.MembershipChangeType.Left);
    }
    expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
        target,
        sender: (0, EventGeneration_1.randomUserID)(),
        membership: MembershipChange_1.Membership.Leave,
    }))).toBe(MembershipChange_1.MembershipChangeType.Kicked);
});
test(`Being banned by the room admin is classified as being banned`, function () {
    const target = (0, EventGeneration_1.randomUserID)();
    const memberships = [
        MembershipChange_1.Membership.Invite,
        MembershipChange_1.Membership.Knock,
        MembershipChange_1.Membership.Join,
        MembershipChange_1.Membership.Leave,
    ];
    for (const membership of memberships) {
        expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
            target,
            sender: target,
            membership: MembershipChange_1.Membership.Ban,
        }), (0, DeclareRoomState_1.describeRoomMember)({
            sender: (0, EventGeneration_1.randomUserID)(),
            target,
            membership,
        }).content)).toBe(MembershipChange_1.MembershipChangeType.Banned);
    }
    expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
        target,
        sender: (0, EventGeneration_1.randomUserID)(),
        membership: MembershipChange_1.Membership.Ban,
    }))).toBe(MembershipChange_1.MembershipChangeType.Banned);
});
test(`Being unbanned by the room admin is classified as being unbanned`, function () {
    // it's important to remember that transitions that get skipped, will be
    // classified as kicked instead...
    const target = (0, EventGeneration_1.randomUserID)();
    expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
        target,
        sender: (0, EventGeneration_1.randomUserID)(),
        membership: MembershipChange_1.Membership.Leave,
    }), (0, DeclareRoomState_1.describeRoomMember)({
        sender: target,
        target,
        membership: MembershipChange_1.Membership.Ban,
    }).content)).toBe(MembershipChange_1.MembershipChangeType.Unbanned);
});
test(`Knocking on the room as a user is classified as knocking`, function () {
    const sender = (0, EventGeneration_1.randomUserID)();
    expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
        sender,
        membership: MembershipChange_1.Membership.Knock,
    }))).toBe(MembershipChange_1.MembershipChangeType.Knocked);
});
test(`Reknocking on the room as a user is classified as reknocking`, function () {
    const sender = (0, EventGeneration_1.randomUserID)();
    const memberships = [
        MembershipChange_1.Membership.Invite,
        MembershipChange_1.Membership.Join,
        MembershipChange_1.Membership.Leave,
        MembershipChange_1.Membership.Ban,
    ];
    for (const membership of memberships) {
        expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership: MembershipChange_1.Membership.Knock,
        }), (0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership,
        }).content)).toBe(MembershipChange_1.MembershipChangeType.Reknocked);
    }
});
test(`Inviting an external user to the room will be classified as invited`, function () {
    const sender = (0, EventGeneration_1.randomUserID)();
    expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
        sender,
        target: (0, EventGeneration_1.randomUserID)(),
        membership: MembershipChange_1.Membership.Invite,
    }))).toBe(MembershipChange_1.MembershipChangeType.Invited);
});
test(`Reinviting a user to the room classified as reinviting them`, function () {
    const sender = (0, EventGeneration_1.randomUserID)();
    const memberships = [
        MembershipChange_1.Membership.Join,
        MembershipChange_1.Membership.Leave,
        MembershipChange_1.Membership.Ban,
        MembershipChange_1.Membership.Knock,
    ];
    for (const membership of memberships) {
        expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership: MembershipChange_1.Membership.Invite,
        }), (0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership,
        }).content)).toBe(MembershipChange_1.MembershipChangeType.Invited);
    }
});
test(`The same states will be classified as no change`, function () {
    const sender = (0, EventGeneration_1.randomUserID)();
    const memberships = [
        MembershipChange_1.Membership.Ban,
        MembershipChange_1.Membership.Invite,
        MembershipChange_1.Membership.Join,
        MembershipChange_1.Membership.Knock,
        MembershipChange_1.Membership.Leave,
    ];
    for (const membership of memberships) {
        expect((0, MembershipChange_1.membershipChangeType)((0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership,
        }), (0, DeclareRoomState_1.describeRoomMember)({
            sender,
            membership,
        }).content)).toBe(MembershipChange_1.MembershipChangeType.NoChange);
    }
});
//# sourceMappingURL=MembershipChange.test.js.map