"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardProtectedRoomsManager = void 0;
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const events_1 = require("events");
const Action_1 = require("../../Interface/Action");
const SetRoomMembership_1 = require("../../Membership/SetRoomMembership");
const SetRoomState_1 = require("../../StateTracking/SetRoomState");
const ProtectedRoomsManager_1 = require("./ProtectedRoomsManager");
const SetMembershipRevisionIssuer_1 = require("../../Membership/SetMembershipRevisionIssuer");
function makeJoinAndAdd(roomJoiner, roomStateManager, roomMembershipManager, addCB) {
    return async function joinAndAdd(room) {
        const joinResult = await roomJoiner.joinRoom(room);
        if ((0, Action_1.isError)(joinResult)) {
            return joinResult;
        }
        const stateIssuer = await roomStateManager.getRoomStateRevisionIssuer(room);
        if ((0, Action_1.isError)(stateIssuer)) {
            return stateIssuer;
        }
        const membershipIssuer = await roomMembershipManager.getRoomMembershipRevisionIssuer(room);
        if ((0, Action_1.isError)(membershipIssuer)) {
            return membershipIssuer;
        }
        addCB(room, stateIssuer.ok, membershipIssuer.ok);
        return (0, Action_1.Ok)(undefined);
    };
}
class StandardProtectedRoomsManager extends events_1.EventEmitter {
    constructor(protectedRoomsConfig, setRoomMembership, setRoomState, roomStateManager, roomMembershipManager, roomJoiner) {
        super();
        this.protectedRoomsConfig = protectedRoomsConfig;
        this.setRoomMembership = setRoomMembership;
        this.setRoomState = setRoomState;
        this.roomStateManager = roomStateManager;
        this.roomMembershipManager = roomMembershipManager;
        this.roomJoiner = roomJoiner;
        this.protectedRooms = new Map();
        for (const room of protectedRoomsConfig.getProtectedRooms()) {
            this.protectedRooms.set(room.toRoomIDOrAlias(), room);
        }
        this.setMembership = new SetMembershipRevisionIssuer_1.StandardSetMembershipRevisionIssuer(setRoomMembership);
    }
    /**
     * Creates a StandaardProtectedRoomsManager from blank setMembership/State
     * objects. Will initialise the setMembership/State to contain the correct
     * state for the entire protected rooms set.
     */
    static async create(protectedRoomsConfig, roomStateManager, roomMembershipManager, roomJoiner, blankSetMembership, blankSetRoomState) {
        const joinAndAdd = makeJoinAndAdd(roomJoiner, roomStateManager, roomMembershipManager, function (room, stateIssuer, membershipIssuer) {
            SetRoomState_1.SetRoomStateMirror.addRoom(blankSetRoomState, room, stateIssuer);
            SetRoomMembership_1.SetRoomMembershipMirror.addRoom(blankSetMembership, room, membershipIssuer);
        });
        for (const room of protectedRoomsConfig.getProtectedRooms()) {
            const result = await joinAndAdd(room);
            if ((0, Action_1.isError)(result)) {
                return await protectedRoomsConfig.reportUseError(`Unable to join and add configured protected room ${room.toPermalink()}`, room, result.error);
            }
        }
        return (0, Action_1.Ok)(new StandardProtectedRoomsManager(protectedRoomsConfig, blankSetMembership, blankSetRoomState, roomStateManager, roomMembershipManager, roomJoiner));
    }
    get allProtectedRooms() {
        return [...this.protectedRooms.values()];
    }
    isProtectedRoom(roomID) {
        return this.protectedRooms.has(roomID);
    }
    getProtectedRoom(roomID) {
        return this.protectedRooms.get(roomID);
    }
    async addRoom(room) {
        const joinResult = await this.roomJoiner.joinRoom(room);
        if ((0, Action_1.isError)(joinResult)) {
            return joinResult;
        }
        if (this.isProtectedRoom(room.toRoomIDOrAlias())) {
            return (0, Action_1.Ok)(undefined);
        }
        const stateIssuer = await this.roomStateManager.getRoomStateRevisionIssuer(room);
        if ((0, Action_1.isError)(stateIssuer)) {
            return stateIssuer;
        }
        const membershipIssuer = await this.roomMembershipManager.getRoomMembershipRevisionIssuer(room);
        if ((0, Action_1.isError)(membershipIssuer)) {
            return membershipIssuer;
        }
        const storeResult = await this.protectedRoomsConfig.addRoom(room);
        if ((0, Action_1.isError)(storeResult)) {
            return storeResult;
        }
        // we must mark the room as protected first, so that the emitters for the set changes
        // will know the room is now protected and be able to act accordingly.
        this.protectedRooms.set(room.toRoomIDOrAlias(), room);
        // I don't like that these emitters will work while there's still inconsistent state
        // in the other set. The emit should be placed on the mirrors and called here too!
        // Then the sets should be moved to the ProtectedRoomsManager subfolder.
        // I also don't know whether it will matter when the state emitter is called
        // before the membership emitter.
        SetRoomState_1.SetRoomStateMirror.addRoom(this.setRoomState, room, stateIssuer.ok);
        SetRoomMembership_1.SetRoomMembershipMirror.addRoom(this.setRoomMembership, room, membershipIssuer.ok);
        this.emit('change', room, ProtectedRoomsManager_1.ProtectedRoomChangeType.Added);
        return (0, Action_1.Ok)(undefined);
    }
    async removeRoom(room) {
        const storeResult = await this.protectedRoomsConfig.removeRoom(room);
        if ((0, Action_1.isError)(storeResult)) {
            return storeResult;
        }
        if (this.isProtectedRoom(room.toRoomIDOrAlias())) {
            SetRoomState_1.SetRoomStateMirror.removeRoom(this.setRoomState, room);
            SetRoomMembership_1.SetRoomMembershipMirror.removeRoom(this.setRoomMembership, room);
            this.protectedRooms.delete(room.toRoomIDOrAlias());
            this.emit('change', room, ProtectedRoomsManager_1.ProtectedRoomChangeType.Removed);
        }
        return (0, Action_1.Ok)(undefined);
    }
    unregisterListeners() {
        // doesn't seem to be anything we need to do.
    }
}
exports.StandardProtectedRoomsManager = StandardProtectedRoomsManager;
//# sourceMappingURL=StandardProtectedRoomsManager.js.map