"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SafeMembershipEventMirror = exports.UnsafeContentKey = void 0;
// Copyright 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
const Action_1 = require("../Interface/Action");
const Value_1 = require("../Interface/Value");
const MembershipEvent_1 = require("../MatrixTypes/MembershipEvent");
const value_1 = require("@sinclair/typebox/value");
/**
 * Used by the `SafeMembershipEventMirror` to extract unsafe content from an event.
 */
exports.UnsafeContentKey = Symbol('unsafeContent');
/**
 * Used by the `SafeMembershipEventMirror` to determine if an object is `SafeMembershipEventContent`.
 */
const SafeMembershipEventContentKey = Symbol('SafeMembershipEventContent');
exports.SafeMembershipEventMirror = Object.freeze({
    getUnsafeContent(content) {
        return content[exports.UnsafeContentKey];
    },
    isSafeContent(content) {
        return (typeof content === 'object' &&
            content !== null &&
            SafeMembershipEventContentKey in content);
    },
    /**
     * Create `SafeMembershipEventContent` from valid content and unsafe content.
     */
    create(content, { unsafeContent = undefined, } = {}) {
        return {
            ...content,
            ...{
                [SafeMembershipEventContentKey]: true,
            },
            ...(unsafeContent === undefined
                ? {}
                : { [exports.UnsafeContentKey]: unsafeContent }),
        };
    },
    /**
     * Parse unknown membership content into safe membership content, if possible.
     * @param unknownContent unknown content.
     * @returns An ActionResult with the safe content, or a reason why safe content cannot be created.
     */
    parse(unknownContent) {
        const decodeResult = Value_1.Value.Decode(MembershipEvent_1.MembershipEventContent, unknownContent, {
            suppressLogOnError: true,
        });
        if ((0, Action_1.isOk)(decodeResult)) {
            return (0, Action_1.Ok)(this.create(decodeResult.ok));
        }
        else {
            const unsafePropertyKeys = decodeResult.error.errors.map((error) => value_1.ValuePointer.Format(error.path).next().value);
            if (unsafePropertyKeys.includes('membership')) {
                // this is a legitimatly unsafe event.
                return decodeResult;
            }
            const safeContent = Object.fromEntries(Object.entries(unknownContent).filter(([key]) => !unsafePropertyKeys.includes(key)));
            const unsafeContent = Object.fromEntries(Object.entries(unknownContent).filter(([key]) => unsafePropertyKeys.includes(key)));
            return (0, Action_1.Ok)(this.create(safeContent, {
                unsafeContent,
            }));
        }
    },
    parseEvent(event) {
        const baseEventResult = Value_1.Value.Decode(MembershipEvent_1.BaseMembershipEvent, event);
        if ((0, Action_1.isError)(baseEventResult)) {
            return baseEventResult;
        }
        const safeContentResult = exports.SafeMembershipEventMirror.parse(baseEventResult.ok.content);
        if ((0, Action_1.isError)(safeContentResult)) {
            return safeContentResult;
        }
        const completeEvent = baseEventResult.ok;
        completeEvent.content = safeContentResult.ok;
        return (0, Action_1.Ok)(completeEvent);
    },
});
//# sourceMappingURL=SafeMembershipEvent.js.map