"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynapseRoomListScanner = exports.SynapseRoomListRoomExplorer = void 0;
const events_1 = __importDefault(require("events"));
const matrix_protection_suite_1 = require("matrix-protection-suite");
const typescript_result_1 = require("@gnuxie/typescript-result");
const log = new matrix_protection_suite_1.Logger("SynapseRoomListExplorer");
// The way this should work is by having a gate.
// each time the gate is called it will run a scan
// unless the last gate was 1 minute ago
// then in the background we call the same gate every
// 15 minutes or so or when the protection is created.
class SynapseRoomListRoomExplorer extends events_1.default {
    createScanLoop() {
        return new matrix_protection_suite_1.ConstantPeriodBatch(() => {
            this.gate.enqueueOpen();
            this.backgroundScan = this.createScanLoop();
        }, this.scanIntervalMS);
    }
    constructor(cooldownMS, scanIntervalMS, scanner) {
        super();
        this.cooldownMS = cooldownMS;
        this.scanIntervalMS = scanIntervalMS;
        this.scanner = scanner;
        this.gate = new matrix_protection_suite_1.StandardTimedGate(this.scanRoomDirectory.bind(this), this.cooldownMS);
        this.gate.enqueueOpen();
        this.backgroundScan = this.createScanLoop();
    }
    unregisterListeners() {
        this.gate.destroy();
        this.backgroundScan.cancel();
    }
    handlePolicyChange(change) {
        // we check for all rule types because they could be server and user
        // rules that we want to discover rooms for.
        // We also check for all match types because they might have just gotten
        // an invitation from a given sender and we haven't discovered that room yet.
        if (change.some((c) => c.changeType === matrix_protection_suite_1.PolicyRuleChangeType.Added ||
            c.changeType === matrix_protection_suite_1.PolicyRuleChangeType.Modified)) {
            this.gate.enqueueOpen();
        }
    }
    async scanRoomDirectory(taskTracker) {
        await this.scanner.scanRoomDirectory(taskTracker);
    }
}
exports.SynapseRoomListRoomExplorer = SynapseRoomListRoomExplorer;
class SynapseRoomListScanner {
    constructor(roomDiscovery, synapseAdminClient) {
        this.roomDiscovery = roomDiscovery;
        this.synapseAdminClient = synapseAdminClient;
        // nothing to do.
    }
    async dealWithPage(page) {
        return (await this.roomDiscovery.checkRoomsDiscovered(page.rooms
            .filter((room) => !this.roomDiscovery.isRoomDiscovered(room.room_id))
            .map((room) => ({
            roomID: room.room_id,
            details: {
                room_id: room.room_id,
                creator: room.creator ?? undefined,
                name: room.name ?? undefined,
            },
        }))));
    }
    async scanRoomDirectory(taskTracker) {
        let nextToken = undefined;
        let totalRooms = 0;
        log.info("Starting to scan the synapse room directory");
        do {
            const roomPageResult = await this.synapseAdminClient.listRooms({
                limit: 250,
                ...(nextToken ? { from: nextToken } : {}),
            });
            if ((0, typescript_result_1.isError)(roomPageResult)) {
                log.error("Couldn't paginate the synapse room list", roomPageResult.error);
                return;
            }
            const pageProcessResult = await this.dealWithPage(roomPageResult.ok);
            if ((0, typescript_result_1.isError)(pageProcessResult)) {
                log.error("Couldn't process the synapse room list page", pageProcessResult.error);
                return;
            }
            nextToken = roomPageResult.ok.next_batch ?? undefined;
            totalRooms = roomPageResult.ok.total_rooms ?? 0;
        } while (nextToken !== undefined && !taskTracker.isCancelled());
        log.info("Ended scan, number of rooms processed:", totalRooms);
    }
}
exports.SynapseRoomListScanner = SynapseRoomListScanner;
//# sourceMappingURL=SynapseRoomListRoomExplorerer.js.map