const tslib = require('tslib');
const node = require('@sentry/node');
const utils = require('@sentry/utils');
const electron = require('electron');
const electronNormalize = require('./electron-normalize.js');

function getRendererName(contents) {
    var _a, _b;
    const options = (_a = node.getCurrentHub().getClient()) === null || _a === void 0 ? void 0 : _a.getOptions();
    return (_b = options === null || options === void 0 ? void 0 : options.getRendererName) === null || _b === void 0 ? void 0 : _b.call(options, contents);
}
function sendRendererAnrEvent(contents, blockedMs, frames) {
    const rendererName = getRendererName(contents) || 'renderer';
    const event = {
        level: 'error',
        exception: {
            values: [
                {
                    type: 'ApplicationNotResponding',
                    value: `Application Not Responding for at least ${blockedMs} ms`,
                    stacktrace: { frames },
                    mechanism: {
                        // This ensures the UI doesn't say 'Crashed in' for the stack trace
                        type: 'ANR',
                    },
                },
            ],
        },
        tags: {
            'event.process': rendererName,
        },
    };
    node.captureEvent(event);
}
function rendererDebugger(contents, pausedStack) {
    contents.debugger.attach('1.3');
    const messageHandler = utils.createDebugPauseMessageHandler((cmd) => contents.debugger.sendCommand(cmd), node.getModuleFromFilename, pausedStack);
    contents.debugger.on('message', (_, method, params) => {
        messageHandler({ method, params });
    });
    // In node, we enable just before pausing but for Chrome, the debugger must be enabled before he ANR event occurs
    void contents.debugger.sendCommand('Debugger.enable');
    return () => {
        return contents.debugger.sendCommand('Debugger.pause');
    };
}
let rendererWatchdogTimers;
/** Creates a renderer ANR status hook */
function createRendererAnrStatusHook() {
    function log(message, ...args) {
        utils.logger.log(`[Renderer ANR] ${message}`, ...args);
    }
    return (message, contents) => {
        rendererWatchdogTimers = rendererWatchdogTimers || new Map();
        let watchdog = rendererWatchdogTimers.get(contents);
        if (watchdog === undefined) {
            log('Renderer sent first status message', message.config);
            let pauseAndCapture;
            if (message.config.captureStackTrace) {
                log('Connecting to debugger');
                pauseAndCapture = rendererDebugger(contents, (frames) => {
                    log('Event captured with stack frames');
                    sendRendererAnrEvent(contents, message.config.anrThreshold, frames);
                });
            }
            watchdog = utils.watchdogTimer(100, message.config.anrThreshold, () => tslib.__awaiter(this, void 0, void 0, function* () {
                log('Watchdog timeout');
                if (pauseAndCapture) {
                    log('Pausing debugger to capture stack trace');
                    pauseAndCapture();
                }
                else {
                    log('Capturing event');
                    sendRendererAnrEvent(contents, message.config.anrThreshold);
                }
            }));
            contents.once('destroyed', () => {
                rendererWatchdogTimers === null || rendererWatchdogTimers === void 0 ? void 0 : rendererWatchdogTimers.delete(contents);
            });
            rendererWatchdogTimers.set(contents, watchdog);
        }
        watchdog.poll();
        if (message.status !== 'alive') {
            log('Renderer visibility changed', message.status);
            watchdog.enabled(message.status === 'visible');
        }
    };
}
/**
 * **Note** This feature is still in beta so there may be breaking changes in future releases.
 *
 * Starts a child process that detects Application Not Responding (ANR) errors.
 *
 * It's important to await on the returned promise before your app code to ensure this code does not run in the ANR
 * child process.
 *
 * ```js
 * import { init, enableMainProcessAnrDetection } from '@sentry/electron';
 *
 * init({ dsn: "__DSN__" });
 *
 * // with ESM + Electron v28+
 * await enableMainProcessAnrDetection({ captureStackTrace: true });
 * runApp();
 *
 * // with CJS
 * enableMainProcessAnrDetection({ captureStackTrace: true }).then(() => {
 *   runApp();
 * });
 * ```
 */
function enableMainProcessAnrDetection(options) {
    if (electronNormalize.ELECTRON_MAJOR_VERSION < 4) {
        throw new Error('Main process ANR detection is only supported on Electron v4+');
    }
    const mainOptions = Object.assign({ entryScript: electron.app.getAppPath() }, options);
    return node.enableAnrDetection(mainOptions);
}

exports.createRendererAnrStatusHook = createRendererAnrStatusHook;
exports.enableMainProcessAnrDetection = enableMainProcessAnrDetection;
//# sourceMappingURL=anr.js.map
