#include "derandomizer.h"

#include <util/error.h>

namespace decoder {

Derandomizer::Derandomizer() {
  // Build de-randomization table (per CCSDS standard).
  // The pseudo random sequence is generated by the polynomial:
  //
  //   h(x) = x^8 + x^7 + x^5 + x^3 + 1.
  //
  // The LRIT documentation refers to CCSDS 101.0-B-6-S (October 2002).
  // Newer versions of the CCSDS Blue Book series contain identical
  // guidelines for de-randomization.
  //
  // You can find a copy of the referred blue book here:
  // https://public.ccsds.org/Pubs/101x0b6s.pdf
  //
  uint8_t lfsr = 0xff;
  uint8_t bit;
  for (unsigned i = 0; i < table_.size(); i++) {
    table_[i] = 0;
    for (unsigned j = 0; j < 8; j++) {
      table_[i] <<= 1;
      table_[i] |= lfsr & 0x1;
      bit = ((lfsr >> 7) ^ (lfsr >> 5) ^ (lfsr >> 3) ^ (lfsr >> 0)) & 0x1;
      lfsr = (lfsr >> 1) | (bit << 7);
    }
  }
}

void Derandomizer::run(uint8_t* data, size_t len) {
  ASSERT(len == table_.size());
  for (unsigned i = 0; i < table_.size(); i++) {
    data[i] ^= table_[i];
  }
}

} // namespace decoder
