#include "rrc.h"

#include <cstring>

#ifdef __ARM_NEON
#include <arm_neon.h>
#endif

#include <util/error.h>

namespace {

// Implementation of RRC filter definition as found on Wikipedia.
// Manually cross checked results against taps generated by GNU Radio.
std::vector<float> taps(int sampleRate, int symbolRate) {
  const double sps = (double) sampleRate / (double) symbolRate;
  constexpr int ntaps = RRC::NTAPS;
  constexpr double beta = 0.5f;
  std::vector<float> taps(ntaps);
  for (int i = 0; i < (int) ntaps; i++) {
    int t = i - (ntaps / 2);
    double z = t / sps;
    double r = 1 / sps;

    // Deal with t = 0
    if (t == 0) {
      taps[i] = r * (1.0f + (4 * beta / M_PI) - beta);
      continue;
    }

    // Special case for denominator equal to zero in next branch
    double tmp = 4 * beta * z;
    tmp = 1 - tmp * tmp;
    if (fabs(tmp * tmp) < 1e-5) {
      double t1 = (1 + 2 / M_PI);
      double t2 = sinf(M_PI / (4 * beta));
      double t3 = (1 - 2 / M_PI);
      double t4 = cosf(M_PI / (4 * beta));
      taps[i] = r * beta / sqrtf(2.0f) * ((t1 * t2) + (t3 * t4));
      continue;
    }

    // Final catch all case
    double t1 = sinf(M_PI * z * (1 - beta));
    double t2 = cosf(M_PI * z * (1 + beta));
    double t3 = 4 * beta * z;
    double t4 = 1 / (M_PI * z * (1 - (16 * beta * beta * z * z)));
    taps[i] = r * t4 * (t1 + t2 * t3);
  }

  // Normalize
  double sum = 0.0;
  for (int i = 0; i < (int) ntaps; i++) {
    sum += taps[i];
  }
  for (int i = 0; i < (int) ntaps; i++) {
    taps[i] /= sum;
  }

  return taps;
}

} // namespace

RRC::RRC(int decimation, int sampleRate, int symbolRate) :
    decimation_(decimation) {
  taps_ = taps(sampleRate, symbolRate);

  // Last tap is zero (so it is a multiple of 4)
  static_assert((NTAPS + 1) % 4 == 0, "NTAPS + 1 not a multiple of 4");
  taps_.resize(NTAPS + 1);
  taps_[NTAPS] = 0.0f;

  // Seed the delay line with zeroes
  tmp_.resize(NTAPS);
}

#ifdef __ARM_NEON

void RRC::work(
    size_t nsamples,
    std::complex<float>* fi,
    std::complex<float>* fo) {
  // Load taps
  float32x4_t taps[(NTAPS + 1) / 4];
  for (size_t i = 0; i < (NTAPS + 1); i += 4) {
    taps[i / 4] = vld1q_f32(&taps_[i]);
  }

  for (size_t i = 0; i < (nsamples / decimation_); i++) {
    float32x4x2_t acc;
    acc.val[0] = vdupq_n_f32(0.0f);
    acc.val[1] = vdupq_n_f32(0.0f);

    // The compiler should unroll this
    for (size_t j = 0; j < (NTAPS + 1); j += 4) {
      float32x4x2_t val = vld2q_f32((const float32_t*) &fi[j]);
      acc.val[0] = vmlaq_f32(acc.val[0], val.val[0], taps[j / 4]);
      acc.val[1] = vmlaq_f32(acc.val[1], val.val[1], taps[j / 4]);
    }

    // Sum accumulators
    auto acci = vpadd_f32(vget_low_f32(acc.val[0]), vget_high_f32(acc.val[0]));
    auto accq = vpadd_f32(vget_low_f32(acc.val[1]), vget_high_f32(acc.val[1]));
    acci = vpadd_f32(acci, acci);
    accq = vpadd_f32(accq, accq);
    fo->real(vget_lane_f32(acci, 0));
    fo->imag(vget_lane_f32(accq, 0));

    // Advance input/output cursors
    fi += decimation_;
    fo += 1;
  }
}

#else

void RRC::work(
    size_t nsamples,
    std::complex<float>* fi,
    std::complex<float>* fo) {
  for (size_t i = 0; i < (nsamples / decimation_); i++) {
    *fo = 0.0f;
    for (size_t j = 0; j < (NTAPS + 1); j++) {
      *fo += fi[j] * taps_[j];
    }

    // Advance input/output cursors
    fi += decimation_;
    fo += 1;
  }
}

#endif

void RRC::work(
    const std::shared_ptr<Queue<Samples> >& qin,
    const std::shared_ptr<Queue<Samples> >& qout) {
  auto input = qin->popForRead();
  if (!input) {
    qout->close();
    return;
  }

  auto output = qout->popForWrite();
  auto nsamples = input->size();
  ASSERT((nsamples % decimation_) == 0);
  output->resize(nsamples / decimation_);
  tmp_.insert(tmp_.end(), input->begin(), input->end());
  ASSERT(tmp_.size() == input->size() + NTAPS);

  // Return read buffer (it has been copied into tmp_)
  qin->pushRead(std::move(input));

  // Do actual work
  std::complex<float>* fi = tmp_.data();
  std::complex<float>* fo = output->data();
  work(nsamples, fi, fo);

  // Keep final NTAPS samples around
  tmp_.erase(tmp_.begin(), tmp_.end() - NTAPS);
  ASSERT(tmp_.size() == NTAPS);

  // Publish output if applicable
  if (samplePublisher_) {
    samplePublisher_->publish(*output);
  }

  // Return output buffer
  qout->pushWrite(std::move(output));
}
