"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReportItemType = void 0;
const time_1 = require("../time");
const BaseItem_1 = require("../models/BaseItem");
const Alarm_1 = require("../models/Alarm");
const Folder_1 = require("../models/Folder");
const Note_1 = require("../models/Note");
const BaseModel_1 = require("../BaseModel");
const DecryptionWorker_1 = require("./DecryptionWorker");
const ResourceFetcher_1 = require("./ResourceFetcher");
const Resource_1 = require("../models/Resource");
const locale_1 = require("../locale");
const { toTitleCase } = require('../string-utils.js');
var CanRetryType;
(function (CanRetryType) {
    CanRetryType["E2EE"] = "e2ee";
    CanRetryType["ResourceDownload"] = "resourceDownload";
    CanRetryType["ItemSync"] = "itemSync";
})(CanRetryType || (CanRetryType = {}));
var ReportItemType;
(function (ReportItemType) {
    ReportItemType["OpenList"] = "openList";
    ReportItemType["CloseList"] = "closeList";
})(ReportItemType || (exports.ReportItemType = ReportItemType = {}));
class ReportService {
    csvEscapeCell(cell) {
        cell = this.csvValueToString(cell);
        const output = cell.replace(/"/, '""');
        if (this.csvCellRequiresQuotes(cell, ',')) {
            return `"${output}"`;
        }
        return output;
    }
    csvCellRequiresQuotes(cell, delimiter) {
        if (cell.indexOf('\n') >= 0)
            return true;
        if (cell.indexOf('"') >= 0)
            return true;
        if (cell.indexOf(delimiter) >= 0)
            return true;
        return false;
    }
    csvValueToString(v) {
        if (v === undefined || v === null)
            return '';
        return v.toString();
    }
    csvCreateLine(row) {
        for (let i = 0; i < row.length; i++) {
            row[i] = this.csvEscapeCell(row[i]);
        }
        return row.join(',');
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    csvCreate(rows) {
        const output = [];
        for (let i = 0; i < rows.length; i++) {
            output.push(this.csvCreateLine(rows[i]));
        }
        return output.join('\n');
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
    async basicItemList(option = null) {
        if (!option)
            option = {};
        if (!option.format)
            option.format = 'array';
        const itemTypes = BaseItem_1.default.syncItemTypes();
        const output = [];
        output.push(['type', 'id', 'updated_time', 'sync_time', 'is_conflict']);
        for (let i = 0; i < itemTypes.length; i++) {
            const itemType = itemTypes[i];
            const ItemClass = BaseItem_1.default.getClassByItemType(itemType);
            const items = await ItemClass.modelSelectAll(`SELECT items.id, items.updated_time, sync_items.sync_time FROM ${ItemClass.tableName()} items JOIN sync_items ON sync_items.item_id = items.id`);
            for (let j = 0; j < items.length; j++) {
                const item = items[j];
                const row = [itemType, item.id, item.updated_time, item.sync_time];
                row.push('is_conflict' in item ? item.is_conflict : '');
                output.push(row);
            }
        }
        return option.format === 'csv' ? this.csvCreate(output) : output;
    }
    async syncStatus(syncTarget) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        const output = {
            items: {},
            total: {},
        };
        let itemCount = 0;
        let syncedCount = 0;
        for (let i = 0; i < BaseItem_1.default.syncItemDefinitions_.length; i++) {
            const d = BaseItem_1.default.syncItemDefinitions_[i];
            // ref: https://github.com/laurent22/joplin/issues/7940#issuecomment-1473709148
            if (d.className === 'MasterKey')
                continue;
            const ItemClass = BaseItem_1.default.getClass(d.className);
            const o = {
                total: await ItemClass.count(),
                synced: await ItemClass.syncedCount(syncTarget),
            };
            output.items[d.className] = o;
            itemCount += o.total;
            syncedCount += o.synced;
        }
        const conflictedCount = await Note_1.default.conflictedCount();
        output.total = {
            total: itemCount - conflictedCount,
            synced: syncedCount,
        };
        output.toDelete = {
            total: await BaseItem_1.default.deletedItemCount(syncTarget),
        };
        output.conflicted = {
            total: await Note_1.default.conflictedCount(),
        };
        output.items['Note'].total -= output.conflicted.total;
        return output;
    }
    addRetryAllHandler(section) {
        // eslint-disable-next-line @typescript-eslint/ban-types -- Old code before rule was applied
        const retryHandlers = [];
        for (let i = 0; i < section.body.length; i++) {
            const item = section.body[i];
            if (typeof item !== 'string' && item.canRetry) {
                retryHandlers.push(item.retryHandler);
            }
        }
        if (retryHandlers.length) {
            section.canRetryAll = true;
            section.retryAllHandler = async () => {
                for (const retryHandler of retryHandlers) {
                    await retryHandler();
                }
            };
        }
        return section;
    }
    async status(syncTarget) {
        const r = await this.syncStatus(syncTarget);
        const sections = [];
        let section = null;
        const disabledItems = await BaseItem_1.default.syncDisabledItems(syncTarget);
        if (disabledItems.length) {
            section = { title: (0, locale_1._)('Items that cannot be synchronised'), body: [] };
            section.body.push((0, locale_1._)('These items will remain on the device but will not be uploaded to the sync target. In order to find these items, either search for the title or the ID (which is displayed in brackets above).'));
            const processRow = (row) => {
                let msg = '';
                if (row.location === BaseItem_1.default.SYNC_ITEM_LOCATION_LOCAL) {
                    msg = (0, locale_1._)('%s (%s) could not be uploaded: %s', row.item.title, row.item.id, row.syncInfo.sync_disabled_reason);
                }
                else {
                    msg = (0, locale_1._)('Item "%s" could not be downloaded: %s', row.syncInfo.item_id, row.syncInfo.sync_disabled_reason);
                }
                // row.item may be undefined when location !== SYNC_ITEM_LOCATION_LOCAL
                const item = { type_: row.syncInfo.item_type, id: row.syncInfo.item_id };
                section.body.push({
                    text: msg,
                    canRetry: true,
                    canRetryType: CanRetryType.ItemSync,
                    retryHandler: async () => {
                        await BaseItem_1.default.saveSyncEnabled(item.type_, item.id);
                    },
                    canIgnore: !row.warning_ignored,
                    ignoreHandler: async () => {
                        await BaseItem_1.default.ignoreItemSyncWarning(syncTarget, item);
                    },
                });
            };
            section.body.push({ type: ReportItemType.OpenList, key: 'disabledSyncItems' });
            let hasIgnoredItems = false;
            let hasUnignoredItems = false;
            for (const row of disabledItems) {
                if (!row.warning_ignored) {
                    processRow(row);
                    hasUnignoredItems = true;
                }
                else {
                    hasIgnoredItems = true;
                }
            }
            if (!hasUnignoredItems) {
                section.body.push((0, locale_1._)('All item sync failures have been marked as "ignored".'));
            }
            section.body.push({ type: ReportItemType.CloseList });
            section = this.addRetryAllHandler(section);
            sections.push(section);
            if (hasIgnoredItems) {
                section = { title: (0, locale_1._)('Ignored items that cannot be synchronised'), body: [] };
                section.body.push((0, locale_1._)('These items failed to sync, but have been marked as "ignored". They won\'t cause the sync warning to appear, but still aren\'t synced. To unignore, click "retry".'));
                section.body.push({ type: ReportItemType.OpenList, key: 'ignoredDisabledItems' });
                for (const row of disabledItems) {
                    if (row.warning_ignored) {
                        processRow(row);
                    }
                }
                section.body.push({ type: ReportItemType.CloseList });
                sections.push(section);
            }
        }
        const decryptionDisabledItems = await DecryptionWorker_1.default.instance().decryptionDisabledItems();
        if (decryptionDisabledItems.length) {
            section = { title: (0, locale_1._)('Items that cannot be decrypted'), body: [], name: 'failedDecryption', canRetryAll: false, retryAllHandler: null };
            section.body.push((0, locale_1._)('Joplin failed to decrypt these items multiple times, possibly because they are corrupted or too large. These items will remain on the device but Joplin will no longer attempt to decrypt them.'));
            section.body.push('');
            const errorMessagesToItems = new Map();
            for (let i = 0; i < decryptionDisabledItems.length; i++) {
                const row = decryptionDisabledItems[i];
                const resourceTypeName = toTitleCase(BaseModel_1.default.modelTypeToName(row.type_));
                const message = (0, locale_1._)('%s: %s', resourceTypeName, row.id);
                const item = {
                    text: message,
                    canRetry: true,
                    canRetryType: CanRetryType.E2EE,
                    retryHandler: async () => {
                        await DecryptionWorker_1.default.instance().clearDisabledItem(row.type_, row.id);
                        void DecryptionWorker_1.default.instance().scheduleStart();
                    },
                };
                const itemError = row.reason;
                if (itemError) {
                    // If the error message is known, postpone adding the report item.
                    // Instead, add it under the error message as a heading
                    if (errorMessagesToItems.has(itemError)) {
                        errorMessagesToItems.get(itemError).push(item);
                    }
                    else {
                        errorMessagesToItems.set(itemError, [item]);
                    }
                }
                else {
                    // If there's no known error, add directly:
                    section.body.push(item);
                }
            }
            // Categorize any items under each known error:
            let errorIdx = 0;
            for (const itemError of errorMessagesToItems.keys()) {
                section.body.push((0, locale_1._)('Items with error: %s', itemError));
                errorIdx++;
                section.body.push({ type: ReportItemType.OpenList, key: `itemsWithError${errorIdx}` });
                // Add all items associated with the header
                section.body.push(...errorMessagesToItems.get(itemError));
                section.body.push({ type: ReportItemType.CloseList });
            }
            section = this.addRetryAllHandler(section);
            sections.push(section);
        }
        {
            section = { title: (0, locale_1._)('Attachments'), body: [], name: 'resources' };
            const statuses = [Resource_1.default.FETCH_STATUS_IDLE, Resource_1.default.FETCH_STATUS_STARTED, Resource_1.default.FETCH_STATUS_DONE, Resource_1.default.FETCH_STATUS_ERROR];
            for (const status of statuses) {
                if (status === Resource_1.default.FETCH_STATUS_DONE) {
                    const downloadedButEncryptedBlobCount = await Resource_1.default.downloadedButEncryptedBlobCount();
                    const downloadedCount = await Resource_1.default.downloadStatusCounts(Resource_1.default.FETCH_STATUS_DONE);
                    const createdLocallyCount = await Resource_1.default.createdLocallyCount();
                    section.body.push((0, locale_1._)('%s: %d', (0, locale_1._)('Downloaded and decrypted'), downloadedCount - downloadedButEncryptedBlobCount));
                    section.body.push((0, locale_1._)('%s: %d', (0, locale_1._)('Downloaded and encrypted'), downloadedButEncryptedBlobCount));
                    section.body.push((0, locale_1._)('%s: %d', (0, locale_1._)('Created locally'), createdLocallyCount));
                }
                else {
                    const count = await Resource_1.default.downloadStatusCounts(status);
                    section.body.push((0, locale_1._)('%s: %d', Resource_1.default.fetchStatusToLabel(status), count));
                }
            }
            sections.push(section);
        }
        const resourceErrorFetchStatuses = await Resource_1.default.errorFetchStatuses();
        if (resourceErrorFetchStatuses.length) {
            section = { title: (0, locale_1._)('Attachments that could not be downloaded'), body: [], name: 'failedResourceDownload' };
            for (let i = 0; i < resourceErrorFetchStatuses.length; i++) {
                const row = resourceErrorFetchStatuses[i];
                section.body.push({
                    text: (0, locale_1._)('%s (%s): %s', row.resource_title, row.resource_id, row.fetch_error),
                    canRetry: true,
                    canRetryType: CanRetryType.ResourceDownload,
                    retryHandler: async () => {
                        await Resource_1.default.resetFetchErrorStatus(row.resource_id);
                        void ResourceFetcher_1.default.instance().autoAddResources();
                    },
                });
            }
            section = this.addRetryAllHandler(section);
            sections.push(section);
        }
        section = { title: (0, locale_1._)('Sync status (synced items / total items)'), body: [] };
        for (const n in r.items) {
            if (!r.items.hasOwnProperty(n))
                continue;
            section.body.push((0, locale_1._)('%s: %d/%d', n, r.items[n].synced, r.items[n].total));
        }
        section.body.push((0, locale_1._)('Total: %d/%d', r.total.synced, r.total.total));
        section.body.push('');
        section.body.push((0, locale_1._)('Conflicted: %d', r.conflicted.total));
        section.body.push((0, locale_1._)('To delete: %d', r.toDelete.total));
        sections.push(section);
        section = { title: (0, locale_1._)('Notebooks'), body: [] };
        const folders = await Folder_1.default.all({
            order: [{ by: 'title', dir: 'ASC' }],
            caseInsensitive: true,
        });
        for (let i = 0; i < folders.length; i++) {
            const noteCount = await Folder_1.default.noteCount(folders[i].id);
            section.body.push((0, locale_1._n)('%s: %d note', '%s: %d notes', noteCount, folders[i].title, noteCount));
        }
        sections.push(section);
        const alarms = await Alarm_1.default.allDue();
        if (alarms.length) {
            section = { title: (0, locale_1._)('Coming alarms'), body: [] };
            for (let i = 0; i < alarms.length; i++) {
                const alarm = alarms[i];
                const note = await Note_1.default.load(alarm.note_id);
                section.body.push((0, locale_1._)('On %s: %s', time_1.default.formatMsToLocal(alarm.trigger_time), note.title));
            }
            sections.push(section);
        }
        return sections;
    }
}
exports.default = ReportService;
//# sourceMappingURL=ReportService.js.map