"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ItemChange_1 = require("../models/ItemChange");
const Note_1 = require("../models/Note");
const Folder_1 = require("../models/Folder");
const Setting_1 = require("../models/Setting");
const Revision_1 = require("../models/Revision");
const BaseModel_1 = require("../BaseModel");
const ItemChangeUtils_1 = require("./ItemChangeUtils");
const shim_1 = require("../shim");
const BaseService_1 = require("./BaseService");
const locale_1 = require("../locale");
const Logger_1 = require("@joplin/utils/Logger");
const renderer_1 = require("../../renderer");
const { substrWithEllipsis } = require('../string-utils');
const { sprintf } = require('sprintf-js');
const { wrapError } = require('../errorUtils');
const logger = Logger_1.default.create('RevisionService');
class RevisionService extends BaseService_1.default {
    constructor() {
        super(...arguments);
        // An "old note" is one that has been created before the revision service existed. These
        // notes never benefited from revisions so the first time they are modified, a copy of
        // the original note is saved. The goal is to have at least one revision in case the note
        // is deleted or modified as a result of a bug or user mistake.
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        this.isOldNotesCache_ = {};
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        this.maintenanceCalls_ = [];
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        this.maintenanceTimer1_ = null;
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        this.maintenanceTimer2_ = null;
        this.isCollecting_ = false;
        this.isRunningInBackground_ = false;
    }
    static instance() {
        if (this.instance_)
            return this.instance_;
        this.instance_ = new RevisionService();
        return this.instance_;
    }
    oldNoteCutOffDate_() {
        return Date.now() - Setting_1.default.value('revisionService.oldNoteInterval');
    }
    async isOldNote(noteId) {
        if (noteId in this.isOldNotesCache_)
            return this.isOldNotesCache_[noteId];
        const isOld = await Note_1.default.noteIsOlderThan(noteId, this.oldNoteCutOffDate_());
        this.isOldNotesCache_[noteId] = isOld;
        return isOld;
    }
    noteMetadata_(note) {
        const excludedFields = ['type_', 'title', 'body', 'created_time', 'updated_time', 'encryption_applied', 'encryption_cipher_text', 'is_conflict'];
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        const md = {};
        for (const k in note) {
            if (excludedFields.indexOf(k) >= 0)
                continue;
            // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
            md[k] = note[k];
        }
        if (note.user_updated_time === note.updated_time)
            delete md.user_updated_time;
        if (note.user_created_time === note.created_time)
            delete md.user_created_time;
        return md;
    }
    async createNoteRevision_(note, parentRevId = null) {
        try {
            const parentRev = parentRevId ? await Revision_1.default.load(parentRevId) : await Revision_1.default.latestRevision(BaseModel_1.default.TYPE_NOTE, note.id);
            const output = {
                parent_id: '',
                item_type: BaseModel_1.default.TYPE_NOTE,
                item_id: note.id,
                item_updated_time: note.updated_time,
            };
            const noteMd = this.noteMetadata_(note);
            const noteTitle = note.title ? note.title : '';
            const noteBody = note.body ? note.body : '';
            if (!parentRev) {
                output.title_diff = Revision_1.default.createTextPatch('', noteTitle);
                output.body_diff = Revision_1.default.createTextPatch('', noteBody);
                output.metadata_diff = Revision_1.default.createObjectPatch({}, noteMd);
            }
            else {
                if (Date.now() - parentRev.updated_time < Setting_1.default.value('revisionService.intervalBetweenRevisions'))
                    return null;
                const merged = await Revision_1.default.mergeDiffs(parentRev);
                output.parent_id = parentRev.id;
                output.title_diff = Revision_1.default.createTextPatch(merged.title, noteTitle);
                output.body_diff = Revision_1.default.createTextPatch(merged.body, noteBody);
                output.metadata_diff = Revision_1.default.createObjectPatch(merged.metadata, noteMd);
            }
            if (Revision_1.default.isEmptyRevision(output))
                return null;
            return Revision_1.default.save(output);
        }
        catch (error) {
            const newError = wrapError(`Could not create revision for note: ${note.id}`, error);
            throw newError;
        }
    }
    async collectRevisions() {
        if (this.isCollecting_)
            return;
        this.isCollecting_ = true;
        await ItemChange_1.default.waitForAllSaved();
        const doneNoteIds = [];
        try {
            while (true) {
                // See synchronizer test units to see why changes coming
                // from sync are skipped.
                const changes = await ItemChange_1.default.modelSelectAll(`
					SELECT id, item_id, type, before_change_item
					FROM item_changes
					WHERE item_type = ?
					AND source != ?
					AND source != ?
					AND id > ?
					ORDER BY id ASC
					LIMIT 10
				`, [BaseModel_1.default.TYPE_NOTE, ItemChange_1.default.SOURCE_SYNC, ItemChange_1.default.SOURCE_DECRYPTION, Setting_1.default.value('revisionService.lastProcessedChangeId')]);
                if (!changes.length)
                    break;
                const noteIds = changes.map((a) => a.item_id);
                const notes = await Note_1.default.modelSelectAll(`SELECT * FROM notes WHERE is_conflict = 0 AND encryption_applied = 0 AND id IN (${Note_1.default.escapeIdsForSql(noteIds)})`);
                for (let i = 0; i < changes.length; i++) {
                    const change = changes[i];
                    const noteId = change.item_id;
                    try {
                        if (change.type === ItemChange_1.default.TYPE_UPDATE && doneNoteIds.indexOf(noteId) < 0) {
                            const note = BaseModel_1.default.byId(notes, noteId);
                            const oldNote = change.before_change_item ? JSON.parse(change.before_change_item) : null;
                            if (note) {
                                if (oldNote && oldNote.updated_time < this.oldNoteCutOffDate_()) {
                                    // This is where we save the original version of this old note
                                    const rev = await this.createNoteRevision_(oldNote);
                                    if (rev)
                                        logger.debug(sprintf('collectRevisions: Saved revision %s (old note)', rev.id));
                                }
                                const rev = await this.createNoteRevision_(note);
                                if (rev)
                                    logger.debug(sprintf('collectRevisions: Saved revision %s (Last rev was more than %d ms ago)', rev.id, Setting_1.default.value('revisionService.intervalBetweenRevisions')));
                                doneNoteIds.push(noteId);
                                this.isOldNotesCache_[noteId] = false;
                            }
                        }
                        if (change.type === ItemChange_1.default.TYPE_DELETE && !!change.before_change_item) {
                            const note = JSON.parse(change.before_change_item);
                            const revExists = await Revision_1.default.revisionExists(BaseModel_1.default.TYPE_NOTE, note.id, note.updated_time);
                            if (!revExists) {
                                const rev = await this.createNoteRevision_(note);
                                if (rev)
                                    logger.debug(sprintf('collectRevisions: Saved revision %s (for deleted note)', rev.id));
                            }
                            doneNoteIds.push(noteId);
                        }
                    }
                    catch (error) {
                        if (error.code === 'revision_encrypted') {
                            throw error;
                        }
                        else {
                            // If any revision creation fails, we continue
                            // processing the other changes. It seems a rare bug
                            // in diff-match-patch can cause the creation of
                            // revisions to fail in some case. It should be rare
                            // and it's best to continue processing the other
                            // changes. The alternative would be to stop here
                            // and fix the bug, but in the meantime revisions
                            // will no longer be generated.
                            // The drawback is that once a change has been
                            // skipped it will never be processed again because
                            // the error will be in the past (before
                            // revisionService.lastProcessedChangeId)
                            //
                            // https://github.com/laurent22/joplin/issues/5531
                            logger.error(`collectRevisions: Processing one of the changes for note ${noteId} failed. Other changes will still be processed. Error was: `, error);
                            logger.error('collectRevisions: Change was:', change);
                        }
                    }
                    Setting_1.default.setValue('revisionService.lastProcessedChangeId', change.id);
                }
            }
        }
        catch (error) {
            if (error.code === 'revision_encrypted') {
                // One or more revisions are encrypted - stop processing for now
                // and these revisions will be processed next time the revision
                // collector runs.
                logger.info('collectRevisions: One or more revision was encrypted. Processing was stopped but will resume later when the revision is decrypted.', error);
            }
            else {
                // This should not happen anymore because we handle the error in
                // the loop above.
                logger.error('collectRevisions:', error);
            }
        }
        await Setting_1.default.saveAll();
        await ItemChangeUtils_1.default.deleteProcessedChanges();
        this.isCollecting_ = false;
        logger.info(`collectRevisions: Created revisions for ${doneNoteIds.length} notes`);
    }
    async deleteOldRevisions(ttl) {
        return Revision_1.default.deleteOldRevisions(ttl);
    }
    async revisionNote(revisions, index) {
        if (index < 0 || index >= revisions.length)
            throw new Error(`Invalid revision index: ${index}`);
        const rev = revisions[index];
        const merged = await Revision_1.default.mergeDiffs(rev, revisions);
        const output = Object.assign({ title: merged.title, body: merged.body }, merged.metadata);
        output.updated_time = output.user_updated_time;
        output.created_time = output.user_created_time;
        // eslint-disable-next-line @typescript-eslint/no-explicit-any -- Old code before rule was applied
        output.type_ = BaseModel_1.default.TYPE_NOTE;
        if (!('markup_language' in output))
            output.markup_language = renderer_1.MarkupLanguage.Markdown;
        return output;
    }
    restoreFolderTitle() {
        return (0, locale_1._)('Restored Notes');
    }
    async restoreFolder() {
        let folder = await Folder_1.default.loadByTitle(this.restoreFolderTitle());
        if (!folder) {
            folder = await Folder_1.default.save({ title: this.restoreFolderTitle() });
        }
        return folder;
    }
    // reverseRevIndex = 0 means restoring the latest version. reverseRevIndex =
    // 1 means the version before that, etc.
    async restoreNoteById(noteId, reverseRevIndex) {
        const revisions = await Revision_1.default.allByType(BaseModel_1.default.TYPE_NOTE, noteId);
        if (!revisions.length)
            throw new Error(`No revision for note "${noteId}"`);
        const revIndex = revisions.length - 1 - reverseRevIndex;
        const note = await this.revisionNote(revisions, revIndex);
        return this.importRevisionNote(note);
    }
    restoreSuccessMessage(note) {
        return (0, locale_1._)('The note "%s" has been successfully restored to the notebook "%s".', substrWithEllipsis(note.title, 0, 32), this.restoreFolderTitle());
    }
    async importRevisionNote(note) {
        const toImport = Object.assign({}, note);
        delete toImport.id;
        delete toImport.updated_time;
        delete toImport.created_time;
        delete toImport.encryption_applied;
        delete toImport.encryption_cipher_text;
        const folder = await this.restoreFolder();
        toImport.parent_id = folder.id;
        return Note_1.default.save(toImport);
    }
    async maintenance() {
        this.maintenanceCalls_.push(true);
        try {
            const startTime = Date.now();
            logger.info('maintenance: Starting...');
            if (!Setting_1.default.value('revisionService.enabled')) {
                logger.info('maintenance: Service is disabled');
                // We do as if we had processed all the latest changes so that they can be cleaned up
                // later on by ItemChangeUtils.deleteProcessedChanges().
                Setting_1.default.setValue('revisionService.lastProcessedChangeId', await ItemChange_1.default.lastChangeId());
                await this.deleteOldRevisions(Setting_1.default.value('revisionService.ttlDays') * 24 * 60 * 60 * 1000);
            }
            else {
                logger.info('maintenance: Service is enabled');
                await this.collectRevisions();
                await this.deleteOldRevisions(Setting_1.default.value('revisionService.ttlDays') * 24 * 60 * 60 * 1000);
                logger.info(`maintenance: Done in ${Date.now() - startTime}ms`);
            }
        }
        catch (error) {
            logger.error('maintenance:', error);
        }
        finally {
            this.maintenanceCalls_.pop();
        }
    }
    runInBackground(collectRevisionInterval = null) {
        if (this.isRunningInBackground_)
            return;
        this.isRunningInBackground_ = true;
        if (collectRevisionInterval === null)
            collectRevisionInterval = 1000 * 60 * 10;
        logger.info(`runInBackground: Starting background service with revision collection interval ${collectRevisionInterval}`);
        this.maintenanceTimer1_ = shim_1.default.setTimeout(() => {
            void this.maintenance();
        }, 1000 * 4);
        this.maintenanceTimer2_ = shim_1.default.setInterval(() => {
            void this.maintenance();
        }, collectRevisionInterval);
    }
    async cancelTimers() {
        if (this.maintenanceTimer1_) {
            shim_1.default.clearTimeout(this.maintenanceTimer1_);
            this.maintenanceTimer1_ = null;
        }
        if (this.maintenanceTimer2_) {
            shim_1.default.clearInterval(this.maintenanceTimer2_);
            this.maintenanceTimer2_ = null;
        }
        return new Promise((resolve) => {
            const iid = shim_1.default.setInterval(() => {
                if (!this.maintenanceCalls_.length) {
                    shim_1.default.clearInterval(iid);
                    resolve(null);
                }
            }, 100);
        });
    }
}
exports.default = RevisionService;
//# sourceMappingURL=RevisionService.js.map