"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const defaultResourceModel_1 = require("../defaultResourceModel");
const linkReplacement_1 = require("./linkReplacement");
const globals_1 = require("@jest/globals");
(0, globals_1.describe)('linkReplacement', () => {
    (0, globals_1.test)('should handle non-resource links', () => {
        const r = (0, linkReplacement_1.default)('https://example.com/test').html;
        (0, globals_1.expect)(r).toBe('<a data-from-md href=\'https://example.com/test\' onclick=\'postMessage(&quot;https://example.com/test&quot;, { resourceId: &quot;&quot; }); return false;\'>');
    });
    (0, globals_1.test)('should handle non-resource links - simple rendering', () => {
        const r = (0, linkReplacement_1.default)('https://example.com/test', { linkRenderingType: 2 }).html;
        (0, globals_1.expect)(r).toBe('<a data-from-md href=\'https://example.com/test\'>');
    });
    (0, globals_1.test)('should handle non-resource links with single quotes in it', () => {
        // Handles a link such as:
        // [Google](https://www.goo'onclick=javascript:alert(/1/);f=')
        const r = (0, linkReplacement_1.default)('https://www.goo\'onclick=javascript:alert(/1/);f=\'', { linkRenderingType: 1 }).html;
        (0, globals_1.expect)(r).toBe('<a data-from-md href=\'https://www.goo&apos;onclick=javascript:alert(/1/);f=&apos;\' onclick=\'postMessage(&quot;https://www.goo%27onclick=javascript:alert(/1/);f=%27&quot;, { resourceId: &quot;&quot; }); return false;\'>');
    });
    (0, globals_1.test)('should handle resource links - downloaded status', () => {
        const resourceId = 'f6afba55bdf74568ac94f8d1e3578d2c';
        const r = (0, linkReplacement_1.default)(`:/${resourceId}`, {
            ResourceModel: defaultResourceModel_1.default,
            resources: {
                [resourceId]: {
                    item: { id: 'test' },
                    localState: {
                        fetch_status: 2, // FETCH_STATUS_DONE
                    },
                },
            },
        }).html;
        (0, globals_1.expect)(r).toBe(`<a data-from-md data-resource-id='${resourceId}' href='#' onclick='postMessage(&quot;joplin://${resourceId}&quot;, { resourceId: &quot;${resourceId}&quot; }); return false;'><span class="resource-icon fa-joplin"></span>`);
    });
    (0, globals_1.test)('should handle resource links - idle status', () => {
        const resourceId = 'f6afba55bdf74568ac94f8d1e3578d2c';
        const r = (0, linkReplacement_1.default)(`:/${resourceId}`, {
            ResourceModel: defaultResourceModel_1.default,
            resources: {
                [resourceId]: {
                    item: { id: 'test' },
                    localState: {
                        fetch_status: 0, // FETCH_STATUS_IDLE
                    },
                },
            },
        }).html;
        // Since the icon is embedded as SVG, we only check for the prefix
        const expectedPrefix = `<a class="not-loaded-resource resource-status-notDownloaded" data-resource-id="${resourceId}"><img src="data:image/svg+xml;utf8`;
        (0, globals_1.expect)(r.indexOf(expectedPrefix)).toBe(0);
    });
    (0, globals_1.test)('should create ontouch listeners to handle longpress', () => {
        const resourceId = 'e6afba55bdf74568ac94f8d1e3578d2c';
        const linkHtml = (0, linkReplacement_1.default)(`:/${resourceId}`, {
            ResourceModel: defaultResourceModel_1.default,
            resources: {
                [resourceId]: {
                    item: { id: 'test' },
                    localState: {
                        fetch_status: 2, // FETCH_STATUS_DONE
                    },
                },
            },
            enableLongPress: true,
        }).html;
        (0, globals_1.expect)(linkHtml).toContain('ontouchstart');
        (0, globals_1.expect)(linkHtml).toContain('ontouchend');
        (0, globals_1.expect)(linkHtml).toContain('ontouchcancel');
    });
});
//# sourceMappingURL=linkReplacement.test.js.map