import Test  # For @test_throws_wrapped
using Test: Returned, Threw, Fail, Pass, ExecutionResult

"""
    @test_throws_wrapped exception expr

Similar to `Test.@test_throws`, but this tests that the expression `expr` either throws
`exception`, OR that it throws an expression _wrapping_ `exception`.

Users can use this function if they aren't concerned about whether an exception is wrapped
or not, e.g. not caring whether the user is using concurrency.

# Examples
```jldoctest
julia> @test_throws_wrapped BoundsError [1, 2, 3][4]
Test Passed
      Thrown: BoundsError

julia> @test_throws_wrapped DimensionMismatch fetch(@async [1, 2, 3] + [1, 2])
Test Passed
      Thrown: DimensionMismatch
```
"""
macro test_throws_wrapped(extype, ex)
    orig_ex = Expr(:inert, ex)
    result = quote
        try
            Returned($(esc(ex)), nothing, $(QuoteNode(__source__)))
        catch _e
            if $(esc(extype)) != InterruptException && _e isa InterruptException
                rethrow()
            end
            Threw(_e, nothing, $(QuoteNode(__source__)))
        end
    end
    Base.remove_linenums!(result)
    :(do_test_throws_wrapped($result, $orig_ex, $(esc(extype))))
end

# An internal function, called by the code generated by @test_throws
# to evaluate and catch the thrown exception - if it exists
function do_test_throws_wrapped(result::ExecutionResult, @nospecialize(orig_expr), @nospecialize(extype))
    if isa(result, Threw)
        # Check that the right type of exception was thrown
        success = false
        exc = result.exception
        if isa(extype, Type)
            success = has_wrapped_exception(exc, extype)
        else
            if has_wrapped_exception(exc, typeof(extype))
                exc = unwrap_exception_until(exc, typeof(extype))
                success = true
                for fld in 1:nfields(extype)
                    if !isequal(getfield(extype, fld), getfield(exc, fld))
                        success = false
                        break
                    end
                end
            end
        end
        if success
            testres = Pass(:test_throws, nothing, nothing, exc)
        else
            testres = Fail(:test_throws_wrong, orig_expr, extype, exc, result.source)
        end
    else
        testres = Fail(:test_throws_nothing, orig_expr, extype, nothing, result.source)
    end
    Test.record(Test.get_testset(), testres)
end
