using MacroTools: flatten, striplines


@testset "flatten try" begin # see julia#50710 and MacroTools#194
    # These tests were prompted due to the following two issues:
    # 1. on all Julia versions `flatten(striplines(:(try catch; false finally false end)))`
    # was completely breaking the try block due to turning `begin false end` into `false`
    # which have drastically different interpretations as 3rd or 4th argument of a try block.
    # 2. only on Julia 1.10, due to the new parser generating slightly more annotations (line number nodes)
    # `flatten(:(try f() catch end))` was turning into `Expr(:try, call, false, linenumbernode)`
    # instead of `Expr(:try, call, false, empty_block)`. Downstream consumers of the AST
    # were not expecting the line number node and were breaking, which is how this issue was
    # discovered.
    # The two issues have the same underlying cause: `begin expr end` was being turned into `expr`
    # which is valid everywhere in julia except in try blocks.
    # Notice that issue 1 is triggered only if one uses `striplines`. As such it was not really
    # triggered in the wild. However, issue 2 was seen with the slight modification to
    # parser annotations in Julia 1.10 which led to the discovery of issue 1.
    exs = [
        quote try; f(); catch; end; end,
        quote try; f(); catch; else; finally; end; end,
        quote try; f(); catch E; else; finally; end; end,
        quote try; f(); catch; finally; end; end,
        quote try; f(); catch E; finally; end; end,
        quote try; f(); catch E; 3+3; finally; 4+4; end; end,
        quote try; f(); catch E; 3+3; else; 2+2; finally; 4+4; end; end,
        quote try; f(); finally; end; end,
        quote try; f(); catch; false; finally; end; end,
        quote try; f(); catch; else; finally; false; end; end,
        quote try; f(); catch; else; end; end,
        quote try; f(); catch; 3+3; else; 2+2; end; end,
        quote try; f(); catch E; else; end; end,
        quote try; f(); catch E; 3+3; else; 2+2; end; end
    ]
    for ex in exs
        #@show ex
        @test flatten(ex) |> striplines == ex |> striplines
        @test flatten(striplines(ex)) == striplines(ex).args[1]
    end
    @test 123 == eval(flatten(striplines(:(try error() catch; 123 finally end))))
    @test 123 == eval(flatten(:(try error() catch; 123 finally end)))
    @test 234 == eval(flatten(striplines(:(try 1+1 catch; false; else 234; finally end))))
    @test 234 == eval(flatten(:(try 1+1 catch; false; else 234; finally end)))
    for (exa, exb) in [
        (quote try; begin f(); g(); end; catch; end; end,                               quote try; f(); g(); catch; end; end),
        (quote try; catch; begin f(); g(); end;  end; end,                              quote try; catch; f(); g(); end; end),
        (quote try; begin f(); g(); end; catch; finally; begin m(); n(); end; end; end, quote try; f(); g(); catch; finally; m(); n(); end; end)
    ]
        @test exa |> flatten |> striplines == exb |> striplines
        @test exa |> striplines |> flatten == (exb |> striplines).args[1]
    end
    # unnatural expressions that can not be generated by the Julia parser, but still get accepted and we do not want to break
    for ex in [
        Expr(:try, 1, false, 2)
        Expr(:try, 1, false, false, false)
    ]
        @test flatten(ex)==ex
    end
end
