module ParallelMergeCSR

using Base.Threads
using SparseArrays
using LinearAlgebra: Adjoint,
                     adjoint,
                     Transpose,
                     transpose, 
                     fill!
using SparseArrays: AbstractSparseMatrixCSC,
                    getcolptr,
                    nonzeros,
                    rowvals, 
                    rmul!
using Polyester: @batch


struct Counter end

"""
    Base.getindex(range::Range,index::Int)

Index into a `Range` using `index` with an exception thrown if the index falls out of bounds
of the range.
"""
@inline function Base.getindex(::Counter,index::Int)
    return index 
end

"""
    mutable struct Coordinate

An implementation of the C++ `CoordinateT` struct from the Merrill and Garland paper (https://doi.org/10.1109/SC.2016.57)

Holds the "coordinate" (a combination of a row offset value and an index into the non-zero values of 
generated by `merge_path_search`.

See also: `merge_path_search`
"""
mutable struct Coordinate
    x::Int
    y::Int
end


"""
    merge_path_search(diagonal::Int, a_len::Int, b_len::Int, a::AbstractVector, b::AbstractVector)

An implementation of the C++ `MergePathSearch` function from the Merrill and Garland paper (https://doi.org/10.1109/SC.2016.57)

Given a `diagonal` the grid formed by the row offsets `a` and the indices of the non-zero values `b` 
as well as their respective lengths `a_len` and `b_len`, find the coordinate on the grid where a thread could 
begin/end at for its workload.
"""
@inline function merge_path_search(diagonal::Int, a_len::Int, b_len::Int, a::AbstractVector, b::Counter)

    # diagonal -> i + j = k 
    # a_len and b_len can stay the same from the paper
    # a are row-end offsets so really pass in a[2:end]
    # b are "natural" numbers (need to start at 1 instead of paper's 0 due to Julia 1-indexing)
    # Diagonal search range (in x coordinate space)
    x_min = max(diagonal - b_len, 0)
    x_max = min(diagonal, a_len)
    # 2D binary-search along diagonal search range
    @inbounds while (x_min < x_max)
        pivot = (x_min + x_max) >> 1
        if (a[pivot + 2] <= b[diagonal - pivot])
            x_min = pivot + 1
        else
            x_max = pivot
        end
    end

    return Coordinate(
        min(x_min, a_len) + 1,
        diagonal - x_min + 1
    )

end

for (T, t) in ((Adjoint, adjoint), (Transpose, transpose))
    @eval begin
    """
        mul!(C::AbstractArray, xA::$($T){<:Any,<:AbstractSparseMatrixCSC}, B::AbstractArray, α::Number, β::Number)
    Performs the operation C = xABα + Cβ where xA is an `AbstractSparseMatrixCSC` that has been wrapped by `$($T)`
    (accomplishable by calling `$($t)` on the CSC matrix).
    """
        function mul!(C::AbstractArray, xA::$T{<:Any,<:AbstractSparseMatrixCSC}, B::AbstractArray, α::Number, β::Number)
            # obtains the original matrix underneath the "lazy wrapper"
            A = xA.parent
            size(A, 2) == size(C, 1) || throw(DimensionMismatch())
            size(A, 1) == size(B, 1) || throw(DimensionMismatch())
            size(B, 2) == size(C, 2) || throw(DimensionMismatch())
            if β != 1
                # preemptively handle β so we just handle C = ABα + C
                β != 0 ? rmul!(C, β) : fill!(C, zero(eltype(C)))
            end

            nzv = nonzeros(A)
            rv = rowvals(A)
            cp = getcolptr(A)
            nnz = length(nzv) 
            
            nrows = A.n
        
            nz_indices = Counter()
            num_merge_items = nnz + nrows # preserve the dimensions of the original matrix
        
            num_threads = nthreads()
        
            items_per_thread = (num_merge_items + num_threads - 1) ÷ num_threads
        
            row_carry_out = zeros(eltype(cp), num_threads)
            value_carry_out = zeros(eltype(C), num_threads) # value must match output
        

            for k in 1:size(C,2)
                # merge_csr_mv!(α, A, B, C, col_idx, $t)
                # Julia threads start id by 1, so make sure to offset!
                @batch for tid in 1:num_threads

                    diagonal = min(items_per_thread * (tid - 1), num_merge_items)
                    diagonal_end = min(diagonal + items_per_thread, num_merge_items)
            
                    # Get starting and ending thread coordinates (row, nzv)
                    thread_coord = merge_path_search(diagonal, nrows, nnz, cp, nz_indices)
                    thread_coord_end = merge_path_search(diagonal_end, nrows, nnz, cp, nz_indices)
            
                    # Consume merge items, whole rows first
                    running_total = zero(eltype(C))
                    @inbounds while thread_coord.x < thread_coord_end.x
                        while thread_coord.y < cp[thread_coord.x+1] 
                            running_total += $t(nzv[thread_coord.y]) * B[rv[thread_coord.y],k]
                            thread_coord.y += 1
                        end
            
                        C[thread_coord.x,k] += α * running_total
                        running_total = zero(eltype(C))
                        thread_coord.x += 1 
                    end
                    
                    # May have thread end up partially consuming a row.
                    # Save result form partial consumption and do one pass at the end to add it back to y
                    @inbounds while thread_coord.y < thread_coord_end.y
                        running_total += $t(nzv[thread_coord.y]) * B[rv[thread_coord.y],k]
                        thread_coord.y += 1
                    end
            
                    # Save carry-outs
                    @inbounds row_carry_out[tid] = thread_coord_end.x
                    @inbounds value_carry_out[tid] = running_total
            
                end
                
                @inbounds for tid in 1:num_threads
                    if row_carry_out[tid] <= nrows
                        C[row_carry_out[tid],k] += α * value_carry_out[tid]
                    end
                end 
            end
            C
            # end of @eval macro
        end
    # end of for loop
    end
end


end # end of the module
