"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceWorkerPlugin = void 0;
const path_1 = require("path");
const schema_utils_1 = require("schema-utils");
const workbox_webpack_plugin_1 = require("workbox-webpack-plugin");
const copy_webpack_plugin_1 = __importDefault(require("copy-webpack-plugin"));
const webpack_1 = __importDefault(require("webpack"));
const webpack_inject_entry_plugin_1 = __importDefault(require("webpack-inject-entry-plugin"));
const schema = {
    type: "object",
    properties: {
        enableInDevelopment: {
            type: "boolean",
        },
        enableWorkboxLogging: {
            type: "boolean",
        },
        registration: {
            type: "object",
            properties: {
                autoRegister: {
                    type: "boolean",
                },
                entry: {
                    type: "string",
                },
                scope: {
                    type: "string",
                },
                path: {
                    type: "string",
                },
            },
            additionalProperties: false,
        },
        workbox: {
            type: "object",
        },
    },
    additionalProperties: false,
};
function isInjectManifest(workboxConfig) {
    return "swSrc" in workboxConfig;
}
class ServiceWorkerPlugin {
    constructor(options = {}) {
        (0, schema_utils_1.validate)(schema, options, {
            name: "Service Worker Webpack Plugin",
            baseDataPath: "options",
        });
        this.config = options;
    }
    apply(compiler) {
        var _a, _b;
        const DEFAULT_SW_NAME = "service-worker.js";
        const DEFAULT_SW_DEST = (0, path_1.join)((_a = compiler.options.output.path) !== null && _a !== void 0 ? _a : "", DEFAULT_SW_NAME);
        // https://webpack.js.org/concepts/entry-points/#single-entry-shorthand-syntax
        const DEFAULT_ENTRY = "main";
        const { enableInDevelopment = false, enableWorkboxLogging = undefined, registration: { entry: registrationEntry = DEFAULT_ENTRY, path = `/${DEFAULT_SW_NAME}`, scope = undefined, autoRegister = true, } = {}, } = this.config;
        const defaults = {
            cleanupOutdatedCaches: true,
            clientsClaim: true,
            exclude: [/\.map$/, /^manifest.*\.js$/, /\.txt$/, /\.d\.ts$/],
            skipWaiting: true,
            sourcemap: false,
            swDest: DEFAULT_SW_DEST,
        };
        const workbox = (_b = this.config.workbox) !== null && _b !== void 0 ? _b : defaults;
        if (autoRegister) {
            new webpack_1.default.DefinePlugin({
                __SERVICE_WORKER_SW_DEST__: JSON.stringify(path),
                __SERVICE_WORKER_SCOPE__: JSON.stringify(scope),
            }).apply(compiler);
            new webpack_inject_entry_plugin_1.default({
                entry: registrationEntry,
                filepath: (0, path_1.join)(__dirname, "autoRegister.js"),
            }).apply(compiler);
        }
        if (compiler.options.mode === "development" && !enableInDevelopment) {
            new copy_webpack_plugin_1.default({
                patterns: [
                    {
                        from: (0, path_1.join)(__dirname, "service-worker-development.js"),
                        to: workbox.swDest,
                    },
                ],
            }).apply(compiler);
        }
        else {
            if (isInjectManifest(workbox)) {
                new workbox_webpack_plugin_1.InjectManifest(workbox).apply(compiler);
            }
            else {
                let mode = undefined;
                if (enableWorkboxLogging === true) {
                    mode = "development";
                }
                else if (enableWorkboxLogging === false) {
                    mode = "production";
                }
                new workbox_webpack_plugin_1.GenerateSW(Object.assign(Object.assign({ mode }, defaults), workbox)).apply(compiler);
            }
        }
    }
}
exports.ServiceWorkerPlugin = ServiceWorkerPlugin;
