#.rst:
# LIEFConfig
# --------
#
# Populate the native LIEF targets.
#
# IMPORTED Targets
# ^^^^^^^^^^^^^^^^
#
# This module defines :prop_tgt:`IMPORTED` target ``LIEF::LIEF``
#
# Hints
# ^^^^^
#
# A user may set ``LIEF_DIR`` to the directory where ``LIEFConfig.cmake`` file
# resides to tell CMake where to find this file.
#
# To choose between STATIC and SHARED version of LIEF library, one can use
# ``COMPONENTS STATIC`` of ``COMPONENTS SHARED``. If no components are
# specified, then a user may set ``LIEF_SHARED_LIBS`` to a true value to
# request shared libraries. By default, if no variables or components are set,
# the static library is preferred if present, else shared library will be used.
#
# .. code-block:: cmake
#
#   find_package(LIEF 0.11.0 REQUIRED COMPONENTS STATIC)


# CMake component support for static and shared libraries based on
# https://alexreinking.com/blog/building-a-dual-shared-and-static-library-with-cmake.html

# Process known components
set(LIEF_known_comps STATIC SHARED)
set(LIEF_comp_STATIC NO)
set(LIEF_comp_SHARED NO)
foreach(_comp ${LIEF_FIND_COMPONENTS})
  if(";${LIEF_known_comps};" MATCHES ";${_comp};")
    set(LIEF_comp_${_comp} YES)
  else()
    set(LIEF_NOT_FOUND_MESSAGE
      "LIEF does not recognize component `${_comp}`.")
    set(LIEF_FOUND FALSE)
    return()
  endif()
endforeach()


# Validate component selection makes sense
if(LIEF_comp_STATIC AND LIEF_comp_SHARED)
  set(LIEF_NOT_FOUND_MESSAGE
    "LIEF `STATIC` and `SHARED` components are mutually exclusive.")
  set(LIEF_FOUND FALSE)
  return()
endif()


# These files are generated by CMake and hold the LIEF library target(s).
set(LIEF_static_export "${CMAKE_CURRENT_LIST_DIR}/LIEFExport-static.cmake")
set(LIEF_shared_export "${CMAKE_CURRENT_LIST_DIR}/LIEFExport-shared.cmake")

# Fix when cross-compiling for OSX with 'CMAKE_INSTALL_NAME_DIR'
if (APPLE AND (NOT CMAKE_OSX_DEPLOYMENT_TARGET OR
               CMAKE_OSX_DEPLOYMENT_TARGET VERSION_GREATER 10.4))
  if(NOT CMAKE_SHARED_LIBRARY_RUNTIME_C_FLAG)
    set(CMAKE_SHARED_LIBRARY_RUNTIME_C_FLAG "-Wl,-rpath,")
  endif()
endif()

# Helper macro to load the requested targets, where `lib_type` is `static` or
# `shared`
macro(LIEF_load_targets lib_type)
  if(NOT EXISTS "${LIEF_${lib_type}_export}")
    set(LIEF_NOT_FOUND_MESSAGE
      "LIEF `${lib_type}` libraries were requested but not found.")
    set(LIEF_FOUND FALSE)
    return()
  endif ()

  if("${lib_type}" STREQUAL "static")
    # Need to find all dependencies even if they're private when LIEF is
    # compiled statically
    include(CMakeFindDependencyMacro)

    if(0)
      find_dependency(MbedTLS)
    endif()

    if(0)
      find_dependency(utf8cpp)
    endif()

    if(OFF)
      find_dependency(spdlog)
    endif()

    if(ON AND 1)
      find_dependency(nlohmann_json)
    endif()

    if(NOT OFF AND OFF)
      find_dependency(frozen)
    endif()

    if(ON)
      find_dependency(tl-expected)
    endif()

  endif()

  # Include the respective targets file
  include("${LIEF_${lib_type}_export}")
endmacro()

# Run the logic to choose static or shared libraries
# 1. Check components
if(LIEF_comp_STATIC)
  LIEF_load_targets("static")
elseif(LIEF_comp_SHARED)
  LIEF_load_targets("shared")
# 2. Check LIEF-only library selection
elseif(DEFINED LIEF_SHARED_LIBS AND LIEF_SHARED_LIBS)
  LIEF_load_targets("shared")
elseif(DEFINED LIEF_SHARED_LIBS AND NOT LIEF_SHARED_LIBS)
  LIEF_load_targets("static")
# 3. Check CMake build type and choose what's available
elseif(BUILD_SHARED_LIBS)
  if(EXISTS "${LIEF_shared_export}")
    LIEF_load_targets("shared")
  else()
    LIEF_load_targets("static")
  endif()
else()
  if(EXISTS "${LIEF_static_export}")
    LIEF_load_targets("static")
  else()
    LIEF_load_targets("shared")
  endif()
endif()


# Attach location of public interface libraries that don't have
# their own CMake config that would otherwise be found using `find_dependency`
# Header-only library can be found relatively easily

if(1)
  find_path(TL_EXPECTED_INCLUDE_DIR tl/expected.hpp)
  if(NOT TL_EXPECTED_INCLUDE_DIR)
    set(LIEF_NOT_FOUND_MESSAGE "Could not find include directory with 'tl/expected.hpp'. Consider setting `TL_EXPECTED_INCLUDE_DIR` during CMake configuration")
    set(LIEF_FOUND FALSE)
    return()
  endif()
  set_property(TARGET LIEF::LIEF APPEND PROPERTY
    INTERFACE_SYSTEM_INCLUDE_DIRECTORIES "${TL_EXPECTED_INCLUDE_DIR}"
  )
endif()

if(0)
  find_path(TCB_SPAN_INCLUDE_DIR tcb/span.hpp)
  if(NOT TCB_SPAN_INCLUDE_DIR)
    set(LIEF_NOT_FOUND_MESSAGE "Could not find include directory with 'tcb/span.hpp'. Consider setting `TCB_SPAN_INCLUDE_DIR` during CMake configuration")
    set(LIEF_FOUND FALSE)
    return()
  endif()
  set_property(TARGET LIEF::LIEF APPEND PROPERTY
    INTERFACE_SYSTEM_INCLUDE_DIRECTORIES "${TCB_SPAN_INCLUDE_DIR}"
  )
endif()

if(0)
  set(LIEF_CORE_EXTENDED TRUE)
else()
  set(LIEF_CORE_EXTENDED FALSE)
endif()

# Set this for backwards compatibility. This isn't quite the path to the
# actual library file, but using the target is strictly better because it
# includes all other information needed to correctly link and use LIEF
set(LIEF_LIBRARIES LIEF::LIEF)

# Set this for backwards compatibility
get_target_property(LIEF_INCLUDE_DIRS LIEF::LIEF INTERFACE_INCLUDE_DIRECTORIES)

# Set the following for backwards compatibility
if(DEFINED LIEF_INCLUDE_DIR)
  if(EXISTS "${LIEF_INCLUDE_DIR}/LIEF/version.h")
    file(STRINGS "${LIEF_INCLUDE_DIR}/LIEF/version.h" LIEF_H REGEX "^#define LIEF_VERSION \"[^\"]*\"$")

    string(REGEX REPLACE "^.*LIEF_VERSION \"([0-9]+).*$" "\\1"                   LIEF_VERSION_MAJOR "${LIEF_H}")
    string(REGEX REPLACE "^.*LIEF_VERSION \"[0-9]+\\.([0-9]+).*$" "\\1"          LIEF_VERSION_MINOR "${LIEF_H}")
    string(REGEX REPLACE "^.*LIEF_VERSION \"[0-9]+\\.[0-9]+\\.([0-9]+).*$" "\\1" LIEF_VERSION_PATCH "${LIEF_H}")
    set(LIEF_VERSION_STRING "${LIEF_VERSION_MAJOR}.${LIEF_VERSION_MINOR}.${LIEF_VERSION_PATCH}")

    set(LIEF_MAJOR_VERSION "${LIEF_VERSION_MAJOR}")
    set(LIEF_MINOR_VERSION "${LIEF_VERSION_MINOR}")
    set(LIEF_PATCH_VERSION "${LIEF_VERSION_PATCH}")
  endif()
endif()
