<?php

namespace CirrusSearch\Query;

use CirrusSearch\Search\SearchContext;
use CirrusSearch\WarningCollector;
use CirrusSearch\Wikimedia\WeightedTagsHooks;
use Elastica\Query\DisMax;
use Elastica\Query\Term;
use MediaWiki\Message\Message;

/**
 * Finds pages based on how well they match a given keyword
 * (e.g.articletopic:term, articlecountry:term), based on scores provided by
 * (Wikimedia-specific) ML models.
 * @package CirrusSearch\Wikimedia
 * @see WeightedTagsHooks
 * @see https://www.mediawiki.org/wiki/Help:CirrusSearch#Articletopic
 */
class ArticlePredictionKeyword extends SimpleKeywordFeature {
	public const ARTICLE_TOPIC_TAG_PREFIX = 'classification.prediction.articletopic';
	/**
	 * @var string
	 * @deprecated TODO remove, once all indexes have been reindexed
	 */
	public const ARTICLE_TOPIC_TAG_PREFIX_LEGACY = 'classification.ores.articletopic';
	public const DRAFT_TOPIC_TAG_PREFIX = 'classification.prediction.drafttopic';
	/**
	 * @var string
	 * @deprecated TODO remove, once all indexes have been reindexed
	 */
	public const DRAFT_TOPIC_TAG_PREFIX_LEGACY = 'classification.ores.drafttopic';
	public const ARTICLE_COUNTRY_TAG_PREFIX = 'classification.prediction.articlecountry';

	private const PREFIX_PER_KEYWORD = [
		'articletopic' => [
			self::ARTICLE_TOPIC_TAG_PREFIX,
			self::ARTICLE_TOPIC_TAG_PREFIX_LEGACY
		],
		'drafttopic' => [
			self::DRAFT_TOPIC_TAG_PREFIX,
			self::DRAFT_TOPIC_TAG_PREFIX_LEGACY
		],
		'articlecountry' => self::ARTICLE_COUNTRY_TAG_PREFIX,
	];

	/**
	 * @var array<string, string|array<string>>
	 */
	private const TERMS_PER_KEYWORD = [
		'articletopic' => ArticleTopicFeature::TERMS_TO_LABELS,
		'drafttopic' => ArticleTopicFeature::TERMS_TO_LABELS,
		// Suppresses a warning when ArticleCountryFeature::AREA_CODES_TO_COUNTRY_CODES
		// is empty. Using + operator for compile-time array union since array_merge()
		// can't be used in constant definitions
		// @phan-suppress-next-line PhanUselessBinaryAddRight
		'articlecountry' => ArticleCountryFeature::COUNTRY_CODES_TO_LABELS +
			ArticleCountryFeature::AREA_CODES_TO_COUNTRY_CODES,
	];

	private const WARN_MESSAGE_PER_KEYWORD = "cirrussearch-articleprediction-invalid-keyword";

	/**
	 * @inheritDoc
	 * @phan-return array{keywords:string[],tag_prefix:string}
	 */
	public function parseValue(
		$key, $value, $quotedValue, $valueDelimiter, $suffix, WarningCollector $warningCollector
	) {
		$keywords = explode( '|', strtolower( $value ) );
		$invalidKeywords = array_diff( $keywords, array_keys( self::TERMS_PER_KEYWORD[$key] ) );

		$validKeywords = array_filter( array_merge( ...array_map( static function ( $keyword )
		use ( $key ) {
			$value = self::TERMS_PER_KEYWORD[$key][$keyword] ?? null;
			// if $value is generated by taking the union of two arrays
			// of type array<string, string> and array<string, array<string>>, and
			// the latter is empty, phan will consider the type check an
			// impossible condition.
			// @phan-suppress-next-line PhanImpossibleCondition
			if ( is_array( $value ) ) {
				return array_map( static function ( $elem ) use ( $key ) {
					return self::TERMS_PER_KEYWORD[$key][strtolower( $elem )] ?? null;
				}, $value );
			}
			return [ $value ];
		}, array_diff( $keywords, $invalidKeywords ) ) ) );

		if ( $invalidKeywords ) {

			$warningCollector->addWarning( self::WARN_MESSAGE_PER_KEYWORD,
				Message::listParam( $invalidKeywords, 'comma' ), count( $invalidKeywords ), $key );
		}
		return [ 'keywords' => $validKeywords, 'tag_prefix' => self::PREFIX_PER_KEYWORD[$key] ];
	}

	/** @inheritDoc */
	protected function getKeywords() {
		return array_keys( self::PREFIX_PER_KEYWORD );
	}

	/** @inheritDoc */
	protected function doApply( SearchContext $context, $key, $value, $quotedValue, $negated ) {
		$parsed = $this->parseValue( $key, $value, $quotedValue, '', '', $context );
		$keywords = $parsed['keywords'];
		$tagPrefixes = is_array( $parsed['tag_prefix'] ) ? $parsed['tag_prefix'] : [ $parsed['tag_prefix'] ];
		if ( $keywords === [] ) {
			$context->setResultsPossible( false );
			return [ null, true ];
		}

		$query = new DisMax();
		foreach ( $keywords as $keyword ) {
			foreach ( $tagPrefixes as $tagPrefix ) {
				$keywordQuery = new Term();
				$keywordQuery->setTerm( WeightedTagsHooks::FIELD_NAME, $tagPrefix . '/' . $keyword );
				$query->addQuery( $keywordQuery );
			}
		}

		if ( !$negated ) {
			$context->addNonTextQuery( $query );
			return [ null, false ];
		} else {
			return [ $query, false ];
		}
	}

}
