#ifndef MPIABI_H
#define MPIABI_H

#include <assert.h>
#include <stddef.h>
#include <stdint.h>

#if defined __cplusplus && __cplusplus >= 201103L
#include <type_traits>
#endif

// Work around a bug in GCC 8.1.0
#if defined __STDC_VERSION__ && __STDC_VERSION__ >= 201112L
#ifndef static_assert
#define static_assert _Static_assert
#endif
#endif

// MPI ABI version (we use SemVer)

#define MPIABI_VERSION_MAJOR 2
#define MPIABI_VERSION_MINOR 10
#define MPIABI_VERSION_PATCH 0

// Compile-time constants

#define MPIABI_MPI_VERSION 3
#define MPIABI_MPI_SUBVERSION 1

// These limits must be no smaller than any MPI implementations' limit
#define MPIABI_MAX_DATAREP_STRING 128          // MPICH: 128, OpenMPI: 128
#define MPIABI_MAX_ERROR_STRING 1024           // MPICH: 1024, OpenMPI: 256
#define MPIABI_MAX_INFO_KEY 256                // MPICH: 256, OpenMPI: 36
#define MPIABI_MAX_INFO_VAL 1024               // MPICH: 1024, OpenMPI: 256
#define MPIABI_MAX_LIBRARY_VERSION_STRING 8192 // MPICH: 8192, OpenMPI: 256
#define MPIABI_MAX_OBJECT_NAME 128             // MPICH: 128, OpenMPI: 64
#define MPIABI_MAX_PORT_NAME 1024              // MPICH: 256, OpenMPI: 1024
#define MPIABI_MAX_PROCESSOR_NAME 256          // MPICH: 128, OpenMPI: 256

// Simple types

typedef intptr_t MPIABI_Aint;
typedef int64_t MPIABI_Count;
typedef int MPIABI_Fint;
typedef int64_t MPIABI_Offset;

// Handles

typedef uintptr_t MPIABI_Comm;
typedef uintptr_t MPIABI_Datatype;
typedef uintptr_t MPIABI_Errhandler;
typedef uintptr_t MPIABI_File;
typedef uintptr_t MPIABI_Group;
typedef uintptr_t MPIABI_Info;
typedef uintptr_t MPIABI_Message;
typedef uintptr_t MPIABI_Op;
typedef uintptr_t MPIABI_Request;
typedef uintptr_t MPIABI_Win;

// TODO: Don't define these publicly
typedef MPIABI_Comm *MPIABI_CommPtr;
typedef MPIABI_Datatype *MPIABI_DatatypePtr;
typedef MPIABI_Errhandler *MPIABI_ErrhandlerPtr;
typedef MPIABI_File *MPIABI_FilePtr;
typedef MPIABI_Group *MPIABI_GroupPtr;
typedef MPIABI_Info *MPIABI_InfoPtr;
typedef MPIABI_Message *MPIABI_MessagePtr;
typedef MPIABI_Op *MPIABI_OpPtr;
typedef MPIABI_Request *MPIABI_RequestPtr;
typedef MPIABI_Win *MPIABI_WinPtr;

// MPI_Status
// This is a difficult type for an ABI since it has user-accessible fields.

#ifdef __LP64__
#define MPIABI_STATUS_SIZE 10
#else
#define MPIABI_STATUS_SIZE 8
#endif

// We put the MPI_Status struct in the beginning. This way, we can
// cast from MPI_Status* to MPIABI_Status* when the status is
// undefined, or when it is `MPI_STATUS_IGNORE`.
typedef struct {
  union {
    struct {
      int f0;
      int f1;
      int f2;
      int f3;
      size_t f4;
    } status_OpenMPI; // also IBM Spectrum MPI
    struct {
      int f0;
      int f1;
      int f2;
      int f3;
      int f4;
    } status_MPICH; // also Intel MPI
  } mpi_status;
  int MPI_SOURCE;
  int MPI_TAG;
  int MPI_ERROR;
} MPIABI_Status;

// TODO: Don't define these publicly
typedef MPIABI_Status *MPIABI_StatusPtr;
typedef const MPIABI_Status *MPIABI_const_StatusPtr;

#if defined __STDC_VERSION__ && __STDC_VERSION__ >= 201112L
static_assert(MPIABI_STATUS_SIZE * sizeof(MPIABI_Fint) == sizeof(MPIABI_Status),
              "");
#endif
#if defined __cplusplus && __cplusplus >= 201103L
static_assert(MPIABI_STATUS_SIZE * sizeof(MPIABI_Fint) == sizeof(MPIABI_Status),
              "");
static_assert(std::is_pod<MPIABI_Status>::value, "");
#endif

// Helper types
// TODO: Don't define this publicly
typedef int MPIABI_array_int_3[3];

// Callback function types

typedef void MPIABI_Comm_copy_attr_function();
typedef void MPIABI_Comm_delete_attr_function();
typedef void MPIABI_Comm_errhandler_function();
typedef MPIABI_Comm_errhandler_function MPIABI_Comm_errhandler_fn;
typedef MPIABI_Comm_copy_attr_function MPIABI_Copy_function;
typedef int MPIABI_Datarep_conversion_function(void *userbuf,
                                               MPIABI_Datatype datatype,
                                               int count, void *filebuf,
                                               MPIABI_Offset position,
                                               void *extra_state);
typedef int MPIABI_Datarep_extent_function(MPIABI_Datatype datatype,
                                           MPIABI_Aint *extent,
                                           void *extra_state);
typedef void MPIABI_Delete_function();
typedef void MPIABI_File_errhandler_function();
typedef void MPIABI_File_errhandler_fn();
typedef int MPIABI_Grequest_cancel_function(void *extra_state, int complete);
typedef int MPIABI_Grequest_free_function(void *extra_state);
typedef int MPIABI_Grequest_query_function(void *extra_state,
                                           MPIABI_Status *status);
typedef void MPIABI_Type_copy_attr_function();
typedef void MPIABI_Type_delete_attr_function();
typedef void MPIABI_User_function();
typedef void MPIABI_Win_copy_attr_function();
typedef void MPIABI_Win_delete_attr_function();
typedef void MPIABI_Win_errhandler_function();
typedef MPIABI_Win_errhandler_function MPIABI_Win_errhandler_fn;

// Constants

#ifdef __cplusplus
extern "C" {
#endif

extern const int mpiabi_version_major;
extern const int mpiabi_version_minor;
extern const int mpiabi_version_patch;

#ifdef __cplusplus
}
#endif

#endif // #ifndef MPIABI_H
