// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Contains the {@linkcode STATUS_CODE} object which contains standard HTTP
 * status codes and provides several type guards for handling status codes
 * with type safety.
 *
 * @example The status code and status text
 * ```ts
 * import {
 *   STATUS_CODE,
 *   STATUS_TEXT,
 * } from "@std/http/status";
 *
 * console.log(STATUS_CODE.NotFound); // Returns 404
 * console.log(STATUS_TEXT[STATUS_CODE.NotFound]); // Returns "Not Found"
 * ```
 *
 * @example Checking the status code type
 * ```ts
 * import { isErrorStatus } from "@std/http/status";
 *
 * const res = await fetch("https://example.com/");
 *
 * if (isErrorStatus(res.status)) {
 *   // error handling here...
 * }
 * ```
 *
 * @module
 */ export const STATUS_CODE = {
  /** RFC 7231, 6.2.1 */ Continue: 100,
  /** RFC 7231, 6.2.2 */ SwitchingProtocols: 101,
  /** RFC 2518, 10.1 */ Processing: 102,
  /** RFC 8297 **/ EarlyHints: 103,
  /** RFC 7231, 6.3.1 */ OK: 200,
  /** RFC 7231, 6.3.2 */ Created: 201,
  /** RFC 7231, 6.3.3 */ Accepted: 202,
  /** RFC 7231, 6.3.4 */ NonAuthoritativeInfo: 203,
  /** RFC 7231, 6.3.5 */ NoContent: 204,
  /** RFC 7231, 6.3.6 */ ResetContent: 205,
  /** RFC 7233, 4.1 */ PartialContent: 206,
  /** RFC 4918, 11.1 */ MultiStatus: 207,
  /** RFC 5842, 7.1 */ AlreadyReported: 208,
  /** RFC 3229, 10.4.1 */ IMUsed: 226,
  /** RFC 7231, 6.4.1 */ MultipleChoices: 300,
  /** RFC 7231, 6.4.2 */ MovedPermanently: 301,
  /** RFC 7231, 6.4.3 */ Found: 302,
  /** RFC 7231, 6.4.4 */ SeeOther: 303,
  /** RFC 7232, 4.1 */ NotModified: 304,
  /** RFC 7231, 6.4.5 */ UseProxy: 305,
  /** RFC 7231, 6.4.7 */ TemporaryRedirect: 307,
  /** RFC 7538, 3 */ PermanentRedirect: 308,
  /** RFC 7231, 6.5.1 */ BadRequest: 400,
  /** RFC 7235, 3.1 */ Unauthorized: 401,
  /** RFC 7231, 6.5.2 */ PaymentRequired: 402,
  /** RFC 7231, 6.5.3 */ Forbidden: 403,
  /** RFC 7231, 6.5.4 */ NotFound: 404,
  /** RFC 7231, 6.5.5 */ MethodNotAllowed: 405,
  /** RFC 7231, 6.5.6 */ NotAcceptable: 406,
  /** RFC 7235, 3.2 */ ProxyAuthRequired: 407,
  /** RFC 7231, 6.5.7 */ RequestTimeout: 408,
  /** RFC 7231, 6.5.8 */ Conflict: 409,
  /** RFC 7231, 6.5.9 */ Gone: 410,
  /** RFC 7231, 6.5.10 */ LengthRequired: 411,
  /** RFC 7232, 4.2 */ PreconditionFailed: 412,
  /** RFC 7231, 6.5.11 */ ContentTooLarge: 413,
  /** RFC 7231, 6.5.12 */ URITooLong: 414,
  /** RFC 7231, 6.5.13 */ UnsupportedMediaType: 415,
  /** RFC 7233, 4.4 */ RangeNotSatisfiable: 416,
  /** RFC 7231, 6.5.14 */ ExpectationFailed: 417,
  /** RFC 7168, 2.3.3 */ Teapot: 418,
  /** RFC 7540, 9.1.2 */ MisdirectedRequest: 421,
  /** RFC 4918, 11.2 */ UnprocessableEntity: 422,
  /** RFC 4918, 11.3 */ Locked: 423,
  /** RFC 4918, 11.4 */ FailedDependency: 424,
  /** RFC 8470, 5.2 */ TooEarly: 425,
  /** RFC 7231, 6.5.15 */ UpgradeRequired: 426,
  /** RFC 6585, 3 */ PreconditionRequired: 428,
  /** RFC 6585, 4 */ TooManyRequests: 429,
  /** RFC 6585, 5 */ RequestHeaderFieldsTooLarge: 431,
  /** RFC 7725, 3 */ UnavailableForLegalReasons: 451,
  /** RFC 7231, 6.6.1 */ InternalServerError: 500,
  /** RFC 7231, 6.6.2 */ NotImplemented: 501,
  /** RFC 7231, 6.6.3 */ BadGateway: 502,
  /** RFC 7231, 6.6.4 */ ServiceUnavailable: 503,
  /** RFC 7231, 6.6.5 */ GatewayTimeout: 504,
  /** RFC 7231, 6.6.6 */ HTTPVersionNotSupported: 505,
  /** RFC 2295, 8.1 */ VariantAlsoNegotiates: 506,
  /** RFC 4918, 11.5 */ InsufficientStorage: 507,
  /** RFC 5842, 7.2 */ LoopDetected: 508,
  /** RFC 2774, 7 */ NotExtended: 510,
  /** RFC 6585, 6 */ NetworkAuthenticationRequired: 511
};
/** A record of all the status codes text. */ export const STATUS_TEXT = {
  [STATUS_CODE.Accepted]: "Accepted",
  [STATUS_CODE.AlreadyReported]: "Already Reported",
  [STATUS_CODE.BadGateway]: "Bad Gateway",
  [STATUS_CODE.BadRequest]: "Bad Request",
  [STATUS_CODE.Conflict]: "Conflict",
  [STATUS_CODE.Continue]: "Continue",
  [STATUS_CODE.Created]: "Created",
  [STATUS_CODE.EarlyHints]: "Early Hints",
  [STATUS_CODE.ExpectationFailed]: "Expectation Failed",
  [STATUS_CODE.FailedDependency]: "Failed Dependency",
  [STATUS_CODE.Forbidden]: "Forbidden",
  [STATUS_CODE.Found]: "Found",
  [STATUS_CODE.GatewayTimeout]: "Gateway Timeout",
  [STATUS_CODE.Gone]: "Gone",
  [STATUS_CODE.HTTPVersionNotSupported]: "HTTP Version Not Supported",
  [STATUS_CODE.IMUsed]: "IM Used",
  [STATUS_CODE.InsufficientStorage]: "Insufficient Storage",
  [STATUS_CODE.InternalServerError]: "Internal Server Error",
  [STATUS_CODE.LengthRequired]: "Length Required",
  [STATUS_CODE.Locked]: "Locked",
  [STATUS_CODE.LoopDetected]: "Loop Detected",
  [STATUS_CODE.MethodNotAllowed]: "Method Not Allowed",
  [STATUS_CODE.MisdirectedRequest]: "Misdirected Request",
  [STATUS_CODE.MovedPermanently]: "Moved Permanently",
  [STATUS_CODE.MultiStatus]: "Multi Status",
  [STATUS_CODE.MultipleChoices]: "Multiple Choices",
  [STATUS_CODE.NetworkAuthenticationRequired]: "Network Authentication Required",
  [STATUS_CODE.NoContent]: "No Content",
  [STATUS_CODE.NonAuthoritativeInfo]: "Non Authoritative Info",
  [STATUS_CODE.NotAcceptable]: "Not Acceptable",
  [STATUS_CODE.NotExtended]: "Not Extended",
  [STATUS_CODE.NotFound]: "Not Found",
  [STATUS_CODE.NotImplemented]: "Not Implemented",
  [STATUS_CODE.NotModified]: "Not Modified",
  [STATUS_CODE.OK]: "OK",
  [STATUS_CODE.PartialContent]: "Partial Content",
  [STATUS_CODE.PaymentRequired]: "Payment Required",
  [STATUS_CODE.PermanentRedirect]: "Permanent Redirect",
  [STATUS_CODE.PreconditionFailed]: "Precondition Failed",
  [STATUS_CODE.PreconditionRequired]: "Precondition Required",
  [STATUS_CODE.Processing]: "Processing",
  [STATUS_CODE.ProxyAuthRequired]: "Proxy Auth Required",
  [STATUS_CODE.ContentTooLarge]: "Content Too Large",
  [STATUS_CODE.RequestHeaderFieldsTooLarge]: "Request Header Fields Too Large",
  [STATUS_CODE.RequestTimeout]: "Request Timeout",
  [STATUS_CODE.URITooLong]: "URI Too Long",
  [STATUS_CODE.RangeNotSatisfiable]: "Range Not Satisfiable",
  [STATUS_CODE.ResetContent]: "Reset Content",
  [STATUS_CODE.SeeOther]: "See Other",
  [STATUS_CODE.ServiceUnavailable]: "Service Unavailable",
  [STATUS_CODE.SwitchingProtocols]: "Switching Protocols",
  [STATUS_CODE.Teapot]: "I'm a teapot",
  [STATUS_CODE.TemporaryRedirect]: "Temporary Redirect",
  [STATUS_CODE.TooEarly]: "Too Early",
  [STATUS_CODE.TooManyRequests]: "Too Many Requests",
  [STATUS_CODE.Unauthorized]: "Unauthorized",
  [STATUS_CODE.UnavailableForLegalReasons]: "Unavailable For Legal Reasons",
  [STATUS_CODE.UnprocessableEntity]: "Unprocessable Entity",
  [STATUS_CODE.UnsupportedMediaType]: "Unsupported Media Type",
  [STATUS_CODE.UpgradeRequired]: "Upgrade Required",
  [STATUS_CODE.UseProxy]: "Use Proxy",
  [STATUS_CODE.VariantAlsoNegotiates]: "Variant Also Negotiates"
};
/**
 * Returns whether the provided number is a valid HTTP status code.
 *
 * @example Usage
 * ```ts
 * import { isStatus } from "@std/http/status";
 * import { assert } from "@std/assert/assert";
 *
 * assert(isStatus(404));
 * ```
 *
 * @param status The status to assert against.
 * @returns Whether or not the provided status is a valid status code.
 */ export function isStatus(status) {
  return Object.values(STATUS_CODE).includes(status);
}
/**
 * A type guard that determines if the status code is informational.
 *
 * @example Usage
 * ```ts
 * import { isInformationalStatus } from "@std/http/status";
 * import { assert } from "@std/assert/assert";
 *
 * assert(isInformationalStatus(100));
 * ```
 *
 * @param status The status to assert against.
 * @returns Whether or not the provided status is an informational status code.
 */ export function isInformationalStatus(status) {
  return isStatus(status) && status >= 100 && status < 200;
}
/**
 * A type guard that determines if the status code is successful.
 *
 * @example Usage
 * ```ts
 * import { isSuccessfulStatus } from "@std/http/status";
 * import { assert } from "@std/assert/assert";
 *
 * assert(isSuccessfulStatus(200));
 * ```
 *
 * @param status The status to assert against.
 * @returns Whether or not the provided status is a successful status code.
 */ export function isSuccessfulStatus(status) {
  return isStatus(status) && status >= 200 && status < 300;
}
/**
 * A type guard that determines if the status code is a redirection.
 *
 * @example Usage
 * ```ts
 * import { isRedirectStatus } from "@std/http/status";
 * import { assert } from "@std/assert/assert";
 *
 * assert(isRedirectStatus(302));
 * ```
 *
 * @param status The status to assert against.
 * @returns Whether or not the provided status is a redirect status code.
 */ export function isRedirectStatus(status) {
  return isStatus(status) && status >= 300 && status < 400;
}
/**
 * A type guard that determines if the status code is a client error.
 *
 * @example Usage
 * ```ts
 * import { isClientErrorStatus } from "@std/http/status";
 * import { assert } from "@std/assert/assert";
 *
 * assert(isClientErrorStatus(404));
 * ```
 *
 * @param status The status to assert against.
 * @returns Whether or not the provided status is a client error status code.
 */ export function isClientErrorStatus(status) {
  return isStatus(status) && status >= 400 && status < 500;
}
/**
 * A type guard that determines if the status code is a server error.
 *
 * @example Usage
 * ```ts
 * import { isServerErrorStatus } from "@std/http/status";
 * import { assert } from "@std/assert/assert";
 *
 * assert(isServerErrorStatus(502));
 * ```
 *
 * @param status The status to assert against.
 * @returns Whether or not the provided status is a server error status code.
 */ export function isServerErrorStatus(status) {
  return isStatus(status) && status >= 500 && status < 600;
}
/**
 * A type guard that determines if the status code is an error.
 *
 * @example Usage
 * ```ts
 * import { isErrorStatus } from "@std/http/status";
 * import { assert } from "@std/assert/assert";
 *
 * assert(isErrorStatus(502));
 * ```
 *
 * @param status The status to assert against.
 * @returns Whether or not the provided status is an error status code.
 */ export function isErrorStatus(status) {
  return isStatus(status) && status >= 400 && status < 600;
}
//# sourceMappingURL=data:application/json;base64,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
// denoCacheMetadata=3672688449562238793,1858999716640372750