/**
 * Do nothing, but only compile if `_val` is of type `T`.
 */
export declare function proveType<T>(_val: T): void;
/**
 * Only compile if `_never` has type `never`.  If run, trace the
 * `message` and return `result`.
 */
export declare function proveNever<T>(_never: never, message: string, result: T): never;
/**
 * Throw an `Error` and trace `message`.
 */
export declare function panic(message?: string): never;
/**
 * Assert that `fact` is `true`.  If the assertion fails, [[panic]]
 * with `message`.
 */
export declare function assert(fact: boolean, message?: string): asserts fact;
/**
 * Only compile if `_never` has the type `never`.  If run, [[panic]]
 * with `message`.
 */
export declare function assertNever(_never: never, message?: string): never;
/**
 * If `v` is not `undefined`, return `v`.  Otherwise, [[panic]] with
 * a message including `reason`.
 */
export declare function defined<T>(v: T | undefined, reason?: string): T;
/**
 * The only thing this function does is to trace if the promise
 * `p` rejects.  It is useful to explain to ESLint that `p` is
 * intentionally not `await`ed.
 *
 * @deprecated Instead of `dontAwait(x)` use `void x`.
 */
export declare function dontAwait<T>(p: Promise<T> | T): void;
/**
 * A promise that resolves after `ms` milliseconds.
 */
export declare function sleep(ms: number): Promise<void>;
/**
 * If both `a` and `b` are `undefined`, return `undefined`.
 * If exactly one of the two is `undefined`, return the other.
 * If both are not `undefined`, return `f(a, b)`.
 */
export declare function reduceTwo<T>(a: T | undefined, b: T | undefined, f: (aa: T, bb: T) => T): T | undefined;
/**
 * Remove `readonly` from all properties of `T`.
 */
export type Writable<T> = {
    -readonly [P in keyof T]: T[P];
};
export type DeepWritable<T> = {
    -readonly [P in keyof T]: DeepWritable<T[P]>;
};
export type DeepReadonly<T> = {
    readonly [P in keyof T]: DeepReadonly<T[P]>;
};
/**
 * If `x` is undefined, return `undefined`.  Otherwise, return `f(x)`.
 */
export declare function definedMap<T, U>(x: T | undefined, f: (xx: T) => U): U | undefined;
/**
 * If `x` belongs to the enum `e`, return `true`.  Otherwise, return `false`.
 */
export declare function isEnumValue<T extends object>(e: T, x: unknown): x is T[keyof T];
/**
 * Returns whether `x` is an array.
 *
 * We have this because `Array.isArray(x)` returns `x is any[]`, which means
 * that this code compiles without error, for example:
 *
 * ```typescript
 * function thisShouldNotCompile(x: string | readonly string[]) {
 *     if (Array.isArray(x)) {
 *         const s = x[0];
 *         s.thisIsAnAny("but it's actually a string");
 *     }
 * }
 * ```
 */
export declare function isArray<T, U>(x: readonly T[] | U): x is readonly T[];
export declare function isArray<T, U>(x: T[] | U): x is T[];
/**
 * Returns whether `obj` has `name` as its own property.
 */
export declare function hasOwnProperty<T extends string>(obj: unknown, name: T): obj is {
    [P in T]: unknown;
};
/**
 * Map `f` over `iterable`, and return an array of all results
 * that are not `undefined`.
 */
export declare function mapFilterUndefined<T, U>(iterable: Iterable<T>, f: (x: T, i: number) => U | undefined): U[];
/**
 * Filters out all elements in `arr` that are `undefined`.
 */
export declare function filterUndefined<T>(arr: Iterable<T | undefined>): T[];
export declare function mapRecord<T, U>(r: Record<string, T>, f: (v: T, n: string) => U): Record<string, U>;
export declare function mapRecordFilterUndefined<T, U>(r: Record<string, T>, f: (v: T, n: string) => U | undefined): Record<string, U>;
/**
 * Returns a string representation of `e`, which is supposed to be an
 * exception.
 */
export declare function exceptionToString(e: unknown): string;
/**
 * Returns the exception `e` as an instance of `Error`.  If `e` is already an
 * `Error`, it just returns `e`, otherwise it returns an error with
 * `exceptionToString(e)`.
 */
export declare function exceptionToError(e: unknown): Error;
export { DefaultMap, ReadonlyDefaultMap } from "./default-map";
export { BrandedString, brandString, makeBrandString } from "./branded-strings";
