'use strict';

var index_js = require('../../metadata/index.js');
var theme_js = require('./theme.js');
var token_js = require('./token.js');

function getSummaryActions({ actions, tag, type }) {
  if (!actions) return [];
  if (actions.length === 1) return actions;
  const list = ["transaction-multisig"];
  return list.includes(`${tag}-${type}`) ? actions : actions.filter((t) => t.tag === tag && t.type === type);
}
function formatPlain(activity) {
  const list = format(activity, theme_js.themePlain).filter((s) => s !== "");
  const clean = [];
  for (let i = 0; i < list.length; i++) {
    if (list[i] === " " && list[i + 1] === " ") continue;
    clean.push(list[i] ?? "");
  }
  return clean.join("");
}
function format(activity, theme) {
  const ts = tokenizeActivity(activity);
  return ts.map((t) => {
    if (!theme[t.type]) {
      return theme.unknown(t.content);
    }
    return theme[t.type](t.content);
  });
}
function tokenizeActivity(activity) {
  const actions = getSummaryActions(activity);
  if (activity.tag === "social" && actions.length > 1) {
    return tokenizeAction(activity, actions[0]);
  }
  if (activity.tag === "unknown" || activity.type === "unknown") {
    return [token_js.token("unknown", "Carried out an activity")];
  }
  const tokens = actions.reduce((acc, action) => {
    if (acc.length > 0) {
      acc.push(token_js.tokenSeparator);
    }
    acc.push(...tokenizeAction(activity, action));
    return acc;
  }, []);
  tokens.push(token_js.tokenSpace, token_js.tokenTime(activity.timestamp));
  return tokens;
}
function tokenizeToActions(activity) {
  const actions = activity.actions ?? [];
  if (activity.tag === "social" && actions.length > 1) {
    return [tokenizeAction(activity, actions[0])];
  }
  if (activity.tag === "unknown" || activity.type === "unknown") {
    return [[token_js.token("unknown", "Carried out an activity")]];
  }
  return actions.map((action) => tokenizeAction(activity, action));
}
function tokenizeToSummaryActions(activity) {
  const actions = getSummaryActions(activity);
  if (activity.tag === "social" && actions.length > 1) {
    return [tokenizeAction(activity, actions[0])];
  }
  if (activity.tag === "unknown" || activity.type === "unknown") {
    return [[token_js.token("unknown", "Carried out an activity")]];
  }
  return actions.map((action) => tokenizeAction(activity, action));
}
function tokenizeAction(activity, action) {
  const direction = activity.direction;
  let res = [token_js.tokenText("Carried out an activity")];
  index_js.handleMetadata(action, {
    "transaction-transfer": (m) => {
      if (direction === "in") {
        res = token_js.join([
          token_js.tokenAddr(action.to),
          token_js.tokenText("received"),
          ...token_js.tokenValue(m),
          token_js.tokenText("from"),
          token_js.tokenAddr(action.from)
        ]);
      } else {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("sent"),
          ...token_js.tokenValue(m),
          token_js.tokenText("to"),
          token_js.tokenAddr(action.to)
        ]);
      }
    },
    "transaction-approval": (m) => {
      if (m.action === "approve") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("approved"),
          ...token_js.tokenValue(m),
          token_js.tokenText("to"),
          token_js.tokenAddr(action.to)
        ]);
      } else {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("revoked the approval of"),
          ...token_js.tokenValue(m),
          token_js.tokenText("to"),
          token_js.tokenAddr(action.to)
        ]);
      }
    },
    "transaction-mint": (m) => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("minted"),
        ...token_js.tokenValue(m)
      ]);
    },
    "transaction-burn": (m) => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("burned"),
        ...token_js.tokenValue(m)
      ]);
    },
    "transaction-bridge": (m) => {
      let network = [];
      if (m.sourceNetwork) {
        network = [
          token_js.tokenText("from"),
          token_js.tokenNetwork(m.sourceNetwork),
          token_js.tokenText("to"),
          token_js.tokenNetwork(m.targetNetwork)
        ];
      }
      if (m.action === "deposit") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("deposited"),
          ...token_js.tokenValue(m.token),
          ...network,
          ...token_js.tokenPlatform(activity)
        ]);
      } else {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("withdrew"),
          ...token_js.tokenValue(m.token),
          ...network,
          ...token_js.tokenPlatform(activity)
        ]);
      }
    },
    // for collectible or nft related action, it will use image_url as the image link
    "collectible-transfer": (m) => {
      const meta = {
        address: m.address,
        id: m.id,
        network: activity.network,
        preview: m.parsedImageUrl
      };
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("transferred"),
        ...token_js.tokenAsset(m.name || "an asset", meta),
        token_js.tokenText("to"),
        token_js.tokenAddr(action.to)
      ]);
    },
    "collectible-approval": (m) => {
      const meta = {
        address: m.address,
        id: m.id,
        network: activity.network,
        preview: m.parsedImageUrl
      };
      if (m.action === "approve") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("approved"),
          token_js.tokenImage(m.parsedImageUrl),
          ...token_js.tokenAsset(m.name || "collection", meta),
          token_js.tokenText("to"),
          token_js.tokenAddr(action.to)
        ]);
      } else {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("revoked the approval of"),
          ...token_js.tokenAsset(m.name || "collection", meta),
          token_js.tokenText("to"),
          token_js.tokenAddr(action.to)
        ]);
      }
    },
    "collectible-mint": (m) => {
      const meta = {
        address: m.address,
        id: m.id,
        network: activity.network,
        preview: m.parsedImageUrl
      };
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("minted"),
        ...token_js.tokenAsset(m.name || "an asset", meta),
        token_js.tokenText("to"),
        token_js.tokenAddr(action.to)
      ]);
    },
    "collectible-burn": (m) => {
      const meta = {
        address: m.address,
        id: m.id,
        network: activity.network,
        preview: m.parsedImageUrl
      };
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("burned"),
        ...token_js.tokenAsset(m.name || "an asset", meta)
      ]);
    },
    "collectible-trade": (m) => {
      const meta = {
        address: m.address,
        id: m.id,
        network: activity.network,
        preview: m.parsedImageUrl
      };
      if (m.action === "buy") {
        res = token_js.join([
          token_js.tokenAddr(action.to),
          token_js.tokenText("bought"),
          ...token_js.tokenAsset(m.name || "an asset", meta),
          token_js.tokenText("from"),
          token_js.tokenAddr(action.from),
          ...token_js.tokenPlatform(action)
        ]);
      } else {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("sold"),
          ...token_js.tokenAsset(m.name || "an asset", meta),
          token_js.tokenText("to"),
          token_js.tokenAddr(action.to),
          ...token_js.tokenPlatform(action)
        ]);
      }
    },
    "exchange-swap": (m) => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("swapped"),
        ...token_js.tokenValue(m.from),
        token_js.tokenText("to"),
        ...token_js.tokenValue(m.to),
        ...token_js.tokenPlatform(action)
      ]);
    },
    // todo add the action invoker
    "exchange-liquidity": (m) => {
      const tokens = m.tokens.flatMap((t) => token_js.join([...token_js.tokenValue(t), token_js.tokenText(",")])).slice(0, -1);
      if (m.action === "add") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("added"),
          ...tokens,
          token_js.tokenText("to liquidity"),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "remove") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("removed"),
          ...tokens,
          token_js.tokenText("from liquidity"),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "collect") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("collected"),
          ...tokens,
          token_js.tokenText("from liquidity"),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "borrow") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("borrowed"),
          ...tokens,
          token_js.tokenText("from liquidity"),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "repay") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("repaid"),
          ...tokens,
          token_js.tokenText("to liquidity"),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "supply") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("supplied"),
          ...tokens,
          token_js.tokenText("to liquidity"),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "withdraw") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("withDrew"),
          ...tokens,
          token_js.tokenText("from liquidity"),
          ...token_js.tokenPlatform(action)
        ]);
      }
    },
    "social-post": (m) => {
      res = token_js.join([
        token_js.tokenHandle(m.handle || action.from, action.from, activity.network),
        token_js.tokenText("published a post"),
        token_js.tokenPost(action),
        ...token_js.tokenPlatform(action)
      ]);
    },
    "social-comment": (m) => {
      res = token_js.join([
        token_js.tokenHandle(m.handle || action.from, action.from, activity.network),
        token_js.tokenText("made a comment"),
        token_js.tokenPost(action),
        ...token_js.tokenPlatform(action)
      ]);
    },
    "social-share": () => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("shared a post"),
        token_js.tokenPost(action),
        ...token_js.tokenPlatform(action)
      ]);
    },
    "social-mint": () => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("minted a post"),
        token_js.tokenPost(action),
        ...token_js.tokenPlatform(action)
      ]);
    },
    "social-revise": (m) => {
      res = token_js.join([
        token_js.tokenAddr(m.handle || action.from),
        token_js.tokenText("revised a post"),
        token_js.tokenPost(action),
        ...token_js.tokenPlatform(action)
      ]);
    },
    "social-profile": (m) => {
      if (m.action === "create") {
        res = token_js.join([
          token_js.tokenAddr(m.handle || action.from),
          token_js.tokenText("created a profile"),
          token_js.tokenImage(m.imageUri),
          token_js.tokenName(m.name || m.handle || ""),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "update") {
        res = token_js.join([
          token_js.tokenAddr(m.handle || action.from),
          token_js.tokenText("updated a profile"),
          token_js.tokenImage(m.imageUri),
          token_js.tokenName(m.name || m.handle || ""),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "renew") {
        res = token_js.join([
          token_js.tokenAddr(m.handle || action.from),
          token_js.tokenText("renewed a profile"),
          token_js.tokenImage(m.imageUri),
          token_js.tokenName(m.name || m.handle || ""),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "wrap") {
        res = token_js.join([
          token_js.tokenAddr(m.handle || action.from),
          token_js.tokenText("wrapped a profile"),
          token_js.tokenImage(m.imageUri),
          token_js.tokenName(m.name || m.handle || ""),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "unwrap") {
        res = token_js.join([
          token_js.tokenAddr(m.handle || action.from),
          token_js.tokenText("unwrapped a profile"),
          token_js.tokenImage(m.imageUri),
          token_js.tokenName(m.name || m.handle || ""),
          ...token_js.tokenPlatform(action)
        ]);
      }
    },
    "social-delete": () => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("deleted a post"),
        token_js.tokenPost(action),
        ...token_js.tokenPlatform(action)
      ]);
    },
    "social-proxy": (m) => {
      if (m.action === "appoint") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("approved a proxy"),
          ...token_js.tokenPlatform(action)
        ]);
      } else {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("removed a proxy"),
          ...token_js.tokenPlatform(action)
        ]);
      }
    },
    "metaverse-transfer": (m) => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("transferred"),
        token_js.tokenImage(m.parsedImageUrl),
        token_js.tokenName(m.name || "an asset"),
        token_js.tokenText("to"),
        token_js.tokenAddr(action.to)
      ]);
    },
    "metaverse-mint": (m) => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("minted"),
        token_js.tokenImage(m.parsedImageUrl),
        token_js.tokenName(m.name || "an asset"),
        ...token_js.tokenPlatform(action)
      ]);
    },
    "metaverse-burn": (m) => {
      res = token_js.join([
        token_js.tokenAddr(action.from),
        token_js.tokenText("burned"),
        token_js.tokenImage(m.parsedImageUrl),
        token_js.tokenName(m.name || "an asset"),
        ...token_js.tokenPlatform(action)
      ]);
    },
    "metaverse-trade": (m) => {
      if (m.action === "buy") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("bought"),
          token_js.tokenImage(m.parsedImageUrl),
          token_js.tokenName(m.name || "an asset"),
          token_js.tokenText("from"),
          token_js.tokenAddr(action.from),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "sell") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("sold"),
          token_js.tokenImage(m.parsedImageUrl),
          token_js.tokenName(m.name || "an asset"),
          token_js.tokenText("from"),
          token_js.tokenAddr(action.from),
          ...token_js.tokenPlatform(action)
        ]);
      } else if (m.action === "list") {
        res = token_js.join([
          token_js.tokenAddr(action.from),
          token_js.tokenText("listed"),
          token_js.tokenImage(m.parsedImageUrl),
          token_js.tokenName(m.name || "an asset"),
          token_js.tokenText("from"),
          token_js.tokenAddr(action.from),
          ...token_js.tokenPlatform(action)
        ]);
      }
    }
  });
  return res;
}
function hasMultiPrimaryActions(activity) {
  const actions = getSummaryActions(activity);
  let count = 0;
  for (const action of actions) {
    if (action.type === activity.type && action.tag === activity.tag) {
      count++;
      if (count > 1) {
        return true;
      }
    }
  }
  return false;
}
function flatActivity(activity) {
  const hasMulti = hasMultiPrimaryActions(activity);
  if (hasMulti) {
    return getSummaryActions(activity).map(
      (action) => ({
        ...activity,
        actions: [action]
      })
    );
  }
  return [activity];
}

exports.flatActivity = flatActivity;
exports.format = format;
exports.formatPlain = formatPlain;
exports.getSummaryActions = getSummaryActions;
exports.hasMultiPrimaryActions = hasMultiPrimaryActions;
exports.tokenizeAction = tokenizeAction;
exports.tokenizeActivity = tokenizeActivity;
exports.tokenizeToActions = tokenizeToActions;
exports.tokenizeToSummaryActions = tokenizeToSummaryActions;
Object.keys(theme_js).forEach(function (k) {
  if (k !== 'default' && !Object.prototype.hasOwnProperty.call(exports, k)) Object.defineProperty(exports, k, {
    enumerable: true,
    get: function () { return theme_js[k]; }
  });
});
//# sourceMappingURL=index.cjs.map
//# sourceMappingURL=index.cjs.map