# exceptions.py: Store exception definitions.
#
# Copyright (C) 2023 Upscaler Contributors
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-only

from os import PathLike
from pathlib import Path
from typing import Any


class AlgorithmFailedError(Exception):
    """Raise when the algorithm has failed."""

    def __init__(self, result_code: int, output: str) -> None:
        """Initialize exception."""
        super().__init__()
        self.result_code = result_code
        self.output = output

    def __str__(self) -> str:
        """Return a detailed string representation of a failed upscale attempt."""
        return (
            f"Algorithm failed.\nResult code: {self.result_code}\nOutput: {self.output}"
        )


class AlgorithmWarning(Warning):
    """Raise when the output could be damaged."""


class InvalidApplicationError(TypeError):
    """Raise when the application is not a `Gio.Application`."""

    def __init__(self, variable: Any) -> None:
        """Initialize exception."""
        super().__init__(
            f"Application should be a `Gio.Application`, not a {type(variable)}.",
        )


class UnsupportedImageTypeError(TypeError):
    """Raise when the user opens an unsupported image type."""

    def __init__(self, file_path: Path | PathLike[str]) -> None:
        """Initialize exception."""
        path = str(file_path)
        super().__init__(f"Unsupported image type from path: {path}")
