// This is mul/vil3d/vil3d_image_view_base.h
#ifndef vil3d_image_view_base_h_
#define vil3d_image_view_base_h_
//:
// \file
// \brief A base class reference-counting view of some image data.
// \author Ian Scott - Manchester
//
// \verbatim
//  Modifications
//   10 Sep. 2004 Peter Vanroose  Inlined all 1-line methods in class decl
// \endverbatim

#include <iosfwd>
#include <string>
#include <iostream>
#include <cstddef>
#include <cassert>
#ifdef _MSC_VER
#  include <vcl_msvc_warnings.h>
#endif
#include <vil/vil_pixel_format.h>
#include <vil/vil_smart_ptr.h>
#include <vgl/vgl_box_3d.h>

//: An abstract base class of smart pointers to actual image data in memory.
// If you want an actual image, try instantiating vil3d_image_view<T>.

class vil3d_image_view_base
{
 protected:
  //: Number of columns.
   unsigned ni_{0};
   //: Number of rasters.
   unsigned nj_{0};
   //: Number of slices.
   unsigned nk_{0};
   //: Number of planes.
   unsigned nplanes_{1};

   vil3d_image_view_base(unsigned ni, unsigned nj, unsigned nk,
                         unsigned nplanes)
       : ni_(ni), nj_(nj), nk_(nk), nplanes_(nplanes) {}

   //: Default is an empty one-plane image
   //  Don't set nplanes_ to zero as it confuses set_size(nx,ny,nz) later
   vil3d_image_view_base() = default;

 public:
  // The destructor must be virtual so that the memory chunk is destroyed.
  virtual ~vil3d_image_view_base() = default;

  //: Width
  unsigned ni()  const {return ni_;}
  //: Height
  unsigned nj()  const {return nj_;}
  //: Depth
  unsigned nk()  const {return nk_;}
  //: Number of planes
  unsigned nplanes() const {return nplanes_;}

  //: Return a box describing the voxel region
  vgl_box_3d<int> bounds() const
  {
    if (size()==0) return {};  // Empty
    return {0,0,0,static_cast<int>(ni()-1),static_cast<int>(nj()-1),static_cast<int>(nk()-1)};
  }

  //: The number of pixels.
  std::size_t size() const { return static_cast<std::size_t>(ni_) * nj_ * nk_ * nplanes_; }

  //: resize current planes to ni x nj * nk
  // If already correct size, this function returns quickly
  virtual void set_size(unsigned ni, unsigned nj, unsigned nk) =0;

  //: resize to ni x nj * nk with nplanes planes.
  // If already correct size, this function returns quickly
  virtual void set_size(unsigned ni, unsigned nj, unsigned nk, unsigned nplanes) =0;

  //: Print a 1-line summary of contents
  virtual void print(std::ostream&) const =0;

  //: Return class name
  virtual std::string is_a() const =0;

  //: Return a description of the concrete data pixel type.
  // For example if the value is VIL_PIXEL_FORMAT_BYTE,
  // you can safely cast, or assign the base class reference to
  // a vil3d_image_view<vxl_byte>.
  virtual enum vil_pixel_format pixel_format() const=0;

  //: True if this is (or is derived from) class s
  virtual bool is_class(std::string const& s) const { return s=="vil3d_image_view_base"; }

 private:
  // You probably should not use a vil3d_image_view in a vbl_smart_ptr, so the
  // ref functions are private
  friend class vil_smart_ptr<vil3d_image_view_base>;
  void ref() { ++reference_count_; }
  void unref() {
    assert(reference_count_>0);
    if (--reference_count_<=0) delete this;}
  int reference_count_{0};
};

//: An interface between vil3d_image_views and vil3d_image_resources
// This object is used internally by vil to provide a type-independent
// transient storage for a view as it is being assigned to a
// vil3d_image_view<T> from a vil3d_image_resource::get_view(),
// vil3d_load() or vil3d_convert_..() function call.
// If you want a type independent image container, you are recommended to
// use a vil3d_image_resource_sptr
typedef vil_smart_ptr<vil3d_image_view_base> vil3d_image_view_base_sptr;

//: Print a 1-line summary of contents
inline
std::ostream& operator<<(std::ostream& s, vil3d_image_view_base const& im) {
  im.print(s); return s;
}

#endif // vil3d_image_view_base_h_
