/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.authentication.authenticators.x509;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.SignatureException;
import java.security.cert.CRLException;
import java.security.cert.CertPathBuilder;
import java.security.cert.CertPathValidatorException;
import java.security.cert.CertSelector;
import java.security.cert.CertStore;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXCertPathBuilderResult;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CRL;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.InitialDirContext;
import javax.security.auth.x500.X500Principal;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.jboss.logging.Logger;
import org.keycloak.common.crypto.CryptoIntegration;
import org.keycloak.common.util.PemUtils;
import org.keycloak.common.util.Time;
import org.keycloak.connections.httpclient.HttpClientProvider;
import org.keycloak.models.Constants;
import org.keycloak.models.KeycloakSession;
import org.keycloak.truststore.TruststoreProvider;
import org.keycloak.utils.CRLUtils;
import org.keycloak.utils.OCSPProvider;

public class CertificateValidator {
    private static final Logger logger = Logger.getLogger(CertificateValidator.class);
    KeycloakSession session;
    X509Certificate[] _certChain;
    int _keyUsageBits;
    List<String> _extendedKeyUsage;
    List<String> _certificatePolicy;
    String _certificatePolicyMode;
    boolean _crlCheckingEnabled;
    boolean _crlAbortIfNonUpdated;
    boolean _crldpEnabled;
    CRLLoaderImpl _crlLoader;
    boolean _ocspEnabled;
    boolean _ocspFailOpen;
    OCSPChecker ocspChecker;
    boolean _timestampValidationEnabled;
    boolean _trustValidationEnabled;

    public CertificateValidator() {
    }

    protected CertificateValidator(X509Certificate[] certChain, int keyUsageBits, List<String> extendedKeyUsage, List<String> certificatePolicy, String certificatePolicyMode, boolean cRLCheckingEnabled, boolean cRLAbortIfNonUpdated, boolean cRLDPCheckingEnabled, CRLLoaderImpl crlLoader, boolean oCSPCheckingEnabled, boolean ocspFailOpen, OCSPChecker ocspChecker, KeycloakSession session, boolean timestampValidationEnabled, boolean trustValidationEnabled) {
        this._certChain = certChain;
        this._keyUsageBits = keyUsageBits;
        this._extendedKeyUsage = extendedKeyUsage;
        this._certificatePolicy = certificatePolicy;
        this._certificatePolicyMode = certificatePolicyMode;
        this._crlCheckingEnabled = cRLCheckingEnabled;
        this._crlAbortIfNonUpdated = cRLAbortIfNonUpdated;
        this._crldpEnabled = cRLDPCheckingEnabled;
        this._crlLoader = crlLoader;
        this._ocspEnabled = oCSPCheckingEnabled;
        this._ocspFailOpen = ocspFailOpen;
        this.ocspChecker = ocspChecker;
        this.session = session;
        this._timestampValidationEnabled = timestampValidationEnabled;
        this._trustValidationEnabled = trustValidationEnabled;
        if (ocspChecker == null) {
            throw new IllegalArgumentException("ocspChecker");
        }
    }

    private static void validateKeyUsage(X509Certificate[] certs, int expected) throws GeneralSecurityException {
        boolean[] keyUsageBits = certs[0].getKeyUsage();
        if (keyUsageBits == null) {
            if (expected != 0) {
                String message = "Key usage extension is expected, but unavailable.";
                throw new GeneralSecurityException(message);
            }
            return;
        }
        boolean isCritical = false;
        Set<String> critSet = certs[0].getCriticalExtensionOIDs();
        if (critSet != null) {
            isCritical = critSet.contains("2.5.29.15");
        }
        int n = expected;
        StringBuilder sb = new StringBuilder();
        int i = 0;
        while (i < keyUsageBits.length) {
            boolean value = keyUsageBits[i];
            if ((n & 1) == 1 && !value) {
                String message = String.format("Key Usage bit '%s' is not set.", KeyUsageBits.fromValue(i).getName());
                if (sb.length() > 0) {
                    sb.append("\n");
                }
                sb.append(message);
                logger.warn((Object)message);
            }
            ++i;
            n >>= 1;
        }
        if (sb.length() > 0 && isCritical) {
            throw new GeneralSecurityException(sb.toString());
        }
    }

    private static void validateExtendedKeyUsage(X509Certificate[] certs, List<String> expectedEKU) throws GeneralSecurityException {
        if (expectedEKU == null || expectedEKU.size() == 0) {
            logger.debug((Object)"Extended Key Usage validation is not enabled.");
            return;
        }
        List<String> extendedKeyUsage = certs[0].getExtendedKeyUsage();
        if (extendedKeyUsage == null) {
            String message = "Extended key usage extension is expected, but unavailable";
            throw new GeneralSecurityException(message);
        }
        boolean isCritical = false;
        Set<String> critSet = certs[0].getCriticalExtensionOIDs();
        if (critSet != null) {
            isCritical = critSet.contains("2.5.29.37");
        }
        LinkedList ekuList = new LinkedList();
        extendedKeyUsage.forEach(s -> ekuList.add(s.toLowerCase()));
        for (String eku : expectedEKU) {
            if (ekuList.contains(eku.toLowerCase())) continue;
            String message = String.format("Extended Key Usage '%s' is missing.", eku);
            if (isCritical) {
                throw new GeneralSecurityException(message);
            }
            logger.warn((Object)message);
        }
    }

    private static void validatePolicy(X509Certificate[] certs, List<String> expectedPolicies, String policyCheckMode) throws GeneralSecurityException {
        if (expectedPolicies == null || expectedPolicies.isEmpty()) {
            logger.debug((Object)"Certificate Policy validation is not enabled.");
            return;
        }
        List policyList = CryptoIntegration.getProvider().getCertificateUtils().getCertificatePolicyList(certs[0]);
        logger.debugf("Certificate policies found: %s", (Object)String.join((CharSequence)",", policyList));
        if ("Any".equals(policyCheckMode)) {
            boolean hasMatch = expectedPolicies.stream().anyMatch(p -> policyList.contains(p.toLowerCase()));
            if (!hasMatch) {
                String message = String.format("Certificate Policy check failed: mode = ANY, found = '%s', expected = '%s'.", String.join((CharSequence)",", policyList), String.join((CharSequence)",", expectedPolicies));
                throw new GeneralSecurityException(message);
            }
        } else {
            for (String policy : expectedPolicies) {
                if (policyList.contains(policy.toLowerCase())) continue;
                String message = String.format("Certificate Policy check failed: mode = ALL, certificate policy '%s' is missing.", policy);
                throw new GeneralSecurityException(message);
            }
        }
    }

    public CertificateValidator validateKeyUsage() throws GeneralSecurityException {
        CertificateValidator.validateKeyUsage(this._certChain, this._keyUsageBits);
        return this;
    }

    public CertificateValidator validateExtendedKeyUsage() throws GeneralSecurityException {
        CertificateValidator.validateExtendedKeyUsage(this._certChain, this._extendedKeyUsage);
        return this;
    }

    public CertificateValidator validatePolicy() throws GeneralSecurityException {
        CertificateValidator.validatePolicy(this._certChain, this._certificatePolicy, this._certificatePolicyMode);
        return this;
    }

    public CertificateValidator validateTimestamps() throws GeneralSecurityException {
        if (!this._timestampValidationEnabled) {
            return this;
        }
        for (int i = 0; i < this._certChain.length; ++i) {
            X509Certificate x509Certificate = this._certChain[i];
            if (x509Certificate.getNotBefore().getTime() > Time.currentTimeMillis()) {
                String serialNumber = x509Certificate.getSerialNumber().toString(16).replaceAll("..(?!$)", "$0 ");
                String message = "certificate with serialnumber '" + serialNumber + "' is not valid yet: " + x509Certificate.getNotBefore().toString();
                throw new GeneralSecurityException(message);
            }
            if (x509Certificate.getNotAfter().getTime() >= Time.currentTimeMillis()) continue;
            String serialNumber = x509Certificate.getSerialNumber().toString(16).replaceAll("..(?!$)", "$0 ");
            String message = "certificate with serialnumber '" + serialNumber + "' has expired on: " + x509Certificate.getNotAfter().toString();
            throw new GeneralSecurityException(message);
        }
        return this;
    }

    public CertificateValidator validateTrust() throws GeneralSecurityException {
        if (!this._trustValidationEnabled) {
            return this;
        }
        TruststoreProvider truststoreProvider = (TruststoreProvider)this.session.getProvider(TruststoreProvider.class);
        if (truststoreProvider == null || truststoreProvider.getTruststore() == null) {
            throw new GeneralSecurityException("Cannot validate client certificate trust: Truststore not available. Please make sure to correctly configure truststore provider in order to be able to revalidate certificate trust");
        }
        Set<X509Certificate> trustedRootCerts = truststoreProvider.getRootCertificates().entrySet().stream().flatMap(t -> ((List)t.getValue()).stream()).collect(Collectors.toSet());
        Set<X509Certificate> trustedIntermediateCerts = truststoreProvider.getIntermediateCertificates().entrySet().stream().flatMap(t -> ((List)t.getValue()).stream()).collect(Collectors.toSet());
        logger.debugf("Found %d trusted root certs, %d trusted intermediate certs", trustedRootCerts.size(), trustedIntermediateCerts.size());
        CertificateValidator.verifyCertificateTrust(this._certChain, trustedRootCerts, trustedIntermediateCerts);
        return this;
    }

    private static PKIXCertPathBuilderResult verifyCertificateTrust(X509Certificate[] certChain, Set<X509Certificate> trustedRootCerts, Set<X509Certificate> trustedIntermediateCerts) throws GeneralSecurityException {
        X509CertSelector selector = new X509CertSelector();
        selector.setCertificate(certChain[0]);
        HashSet<TrustAnchor> trustAnchors = new HashSet<TrustAnchor>();
        for (X509Certificate trustedRootCert : trustedRootCerts) {
            trustAnchors.add(new TrustAnchor(trustedRootCert, null));
        }
        PKIXBuilderParameters pkixParams = new PKIXBuilderParameters(trustAnchors, (CertSelector)selector);
        pkixParams.setRevocationEnabled(false);
        HashSet<X509Certificate> intermediateCerts = new HashSet<X509Certificate>();
        for (X509Certificate intermediateCert : trustedIntermediateCerts) {
            intermediateCerts.add(intermediateCert);
        }
        for (X509Certificate clientCert : certChain) {
            intermediateCerts.add(clientCert);
        }
        CertStore intermediateCertStore = CryptoIntegration.getProvider().getCertStore(new CollectionCertStoreParameters(intermediateCerts));
        pkixParams.addCertStore(intermediateCertStore);
        CertPathBuilder builder = CryptoIntegration.getProvider().getCertPathBuilder();
        PKIXCertPathBuilderResult result = (PKIXCertPathBuilderResult)builder.build(pkixParams);
        return result;
    }

    private X509Certificate findCAInTruststore(X509Certificate cert) throws GeneralSecurityException {
        TruststoreProvider truststoreProvider = (TruststoreProvider)this.session.getProvider(TruststoreProvider.class);
        if (truststoreProvider == null || truststoreProvider.getTruststore() == null) {
            return null;
        }
        Map rootCerts = truststoreProvider.getRootCertificates();
        X500Principal issuer = cert.getIssuerX500Principal();
        List cas = (List)rootCerts.get(issuer);
        X509Certificate ca = null;
        if (cas == null) {
            cas = (List)truststoreProvider.getIntermediateCertificates().get(issuer);
        }
        if (cas != null) {
            for (X509Certificate cacert : cas) {
                try {
                    cert.verify(cacert.getPublicKey());
                }
                catch (InvalidKeyException | SignatureException e) {
                    continue;
                }
                ca = cacert;
                break;
            }
        }
        if (ca != null) {
            ca.checkValidity();
        }
        return ca;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void checkRevocationUsingOCSP(X509Certificate[] certs) throws GeneralSecurityException {
        if (logger.isDebugEnabled() && certs != null) {
            for (X509Certificate cert : certs) {
                logger.debugf("Certificate: %s", (Object)cert.getSubjectDN().getName());
            }
        }
        X509Certificate cert = null;
        X509Certificate issuer = null;
        if (certs == null || certs.length == 0) {
            throw new GeneralSecurityException("No certificates sent");
        }
        if (certs.length > 1) {
            cert = certs[0];
            issuer = certs[1];
        } else {
            cert = certs[0];
            issuer = this.findCAInTruststore(cert);
            if (issuer == null) {
                throw new GeneralSecurityException(String.format("No trusted CA in certificate found: %s. Add it to truststore SPI if valid.", cert.getIssuerX500Principal()));
            }
        }
        try {
            OCSPProvider.OCSPRevocationStatus rs = this.ocspChecker.check(cert, issuer);
            if (rs == null) {
                if (!this._ocspFailOpen) throw new GeneralSecurityException("Unable to check client revocation status using OCSP");
                logger.warnf("Unable to check client revocation status using OCSP - continuing certificate authentication because of fail-open OCSP configuration setting", new Object[0]);
            }
            if (rs.getRevocationStatus() == OCSPProvider.RevocationStatus.UNKNOWN) {
                if (!this._ocspFailOpen) throw new GeneralSecurityException("Unable to determine certificate's revocation status.");
                logger.warnf("Unable to determine certificate's revocation status - continuing certificate authentication because of fail-open OCSP configuration setting", new Object[0]);
                return;
            } else {
                if (rs.getRevocationStatus() != OCSPProvider.RevocationStatus.REVOKED) return;
                StringBuilder sb = new StringBuilder();
                sb.append("Certificate's been revoked.");
                sb.append("\n");
                sb.append(rs.getRevocationReason().toString());
                sb.append("\n");
                sb.append(String.format("Revoked on: %s", rs.getRevocationTime().toString()));
                throw new GeneralSecurityException(sb.toString());
            }
        }
        catch (CertPathValidatorException e) {
            if (!this._ocspFailOpen) throw e;
            logger.warnf("Unable to check client revocation status using OCSP - continuing certificate authentication because of fail-open OCSP configuration setting", new Object[0]);
            return;
        }
    }

    private static void checkRevocationStatusUsingCRL(X509Certificate[] certs, CRLLoaderImpl crLoader, KeycloakSession session) throws GeneralSecurityException {
        Collection<X509CRL> crlColl = crLoader.getX509CRLs();
        if (crlColl != null && !crlColl.isEmpty()) {
            for (X509CRL it : crlColl) {
                CRLUtils.check(certs, it, session);
            }
        }
    }

    private static List<String> getCRLDistributionPoints(X509Certificate cert) {
        try {
            return CryptoIntegration.getProvider().getCertificateUtils().getCRLDistributionPoints(cert);
        }
        catch (IOException e) {
            logger.error((Object)e.getMessage());
            return new ArrayList<String>();
        }
    }

    private static void checkRevocationStatusUsingCRLDistributionPoints(X509Certificate[] certs, KeycloakSession session, boolean abortIfNonUpdated) throws GeneralSecurityException {
        List<String> distributionPoints = CertificateValidator.getCRLDistributionPoints(certs[0]);
        if (distributionPoints == null || distributionPoints.isEmpty()) {
            throw new GeneralSecurityException("Could not find any CRL distribution points in the certificate, unable to check the certificate revocation status using CRL/DP.");
        }
        for (String dp : distributionPoints) {
            logger.tracef("CRL Distribution point: \"%s\"", (Object)dp);
            CertificateValidator.checkRevocationStatusUsingCRL(certs, new CRLFileLoader(session, dp, abortIfNonUpdated), session);
        }
    }

    public CertificateValidator checkRevocationStatus() throws GeneralSecurityException {
        if (!this._crlCheckingEnabled && !this._ocspEnabled) {
            return this;
        }
        if (this._crlCheckingEnabled) {
            if (!this._crldpEnabled) {
                CertificateValidator.checkRevocationStatusUsingCRL(this._certChain, this._crlLoader, this.session);
            } else {
                CertificateValidator.checkRevocationStatusUsingCRLDistributionPoints(this._certChain, this.session, this._crlAbortIfNonUpdated);
            }
        }
        if (this._ocspEnabled) {
            this.checkRevocationUsingOCSP(this._certChain);
        }
        return this;
    }

    public static abstract class CRLLoaderImpl {
        public abstract Collection<X509CRL> getX509CRLs() throws GeneralSecurityException;
    }

    public static abstract class OCSPChecker {
        public abstract OCSPProvider.OCSPRevocationStatus check(X509Certificate var1, X509Certificate var2) throws CertPathValidatorException;
    }

    static enum KeyUsageBits {
        DIGITAL_SIGNATURE(0, "digitalSignature"),
        NON_REPUDIATION(1, "nonRepudiation"),
        KEY_ENCIPHERMENT(2, "keyEncipherment"),
        DATA_ENCIPHERMENT(3, "dataEncipherment"),
        KEY_AGREEMENT(4, "keyAgreement"),
        KEYCERTSIGN(5, "keyCertSign"),
        CRLSIGN(6, "cRLSign"),
        ENCIPHERMENT_ONLY(7, "encipherOnly"),
        DECIPHER_ONLY(8, "decipherOnly");

        private int value;
        private String name;

        private KeyUsageBits(int value, String name) {
            if (value < 0 || value > 8) {
                throw new IllegalArgumentException("value");
            }
            if (name == null || name.trim().length() == 0) {
                throw new IllegalArgumentException("name");
            }
            this.value = value;
            this.name = name.trim();
        }

        public int getInt() {
            return this.value;
        }

        public String getName() {
            return this.name;
        }

        public static KeyUsageBits parse(String name) throws IllegalArgumentException, IndexOutOfBoundsException {
            if (name == null || name.trim().length() == 0) {
                throw new IllegalArgumentException("name");
            }
            for (KeyUsageBits bit : KeyUsageBits.values()) {
                if (!bit.getName().equalsIgnoreCase(name)) continue;
                return bit;
            }
            throw new IndexOutOfBoundsException("name");
        }

        public static KeyUsageBits fromValue(int value) throws IndexOutOfBoundsException {
            if (value < 0 || value > 8) {
                throw new IndexOutOfBoundsException("value");
            }
            for (KeyUsageBits bit : KeyUsageBits.values()) {
                if (bit.getInt() != value) continue;
                return bit;
            }
            throw new IndexOutOfBoundsException("value");
        }
    }

    public static class CRLFileLoader
    extends CRLLoaderImpl {
        private final KeycloakSession session;
        private final String cRLPath;
        private final LdapContext ldapContext;
        private final boolean abortIfNonUpdated;

        public CRLFileLoader(KeycloakSession session, String cRLPath, boolean abortIfNonUpdated) {
            this(session, cRLPath, abortIfNonUpdated, new LdapContext());
        }

        public CRLFileLoader(KeycloakSession session, String cRLPath, boolean abortIfNonUpdated, LdapContext ldapContext) {
            this.session = session;
            this.cRLPath = cRLPath;
            this.abortIfNonUpdated = abortIfNonUpdated;
            this.ldapContext = ldapContext;
            if (ldapContext == null) {
                throw new NullPointerException("Context cannot be null");
            }
        }

        @Override
        public Collection<X509CRL> getX509CRLs() throws GeneralSecurityException {
            X509CRL crl = this.loadCRL();
            if (crl == null) {
                throw new GeneralSecurityException(String.format("Unable to load CRL from \"%s\"", this.cRLPath));
            }
            if (crl.getNextUpdate() != null && crl.getNextUpdate().compareTo(new Date(Time.currentTimeMillis())) < 0) {
                String message = String.format("CRL from '%s' is not refreshed. Next update is %s.", this.cRLPath, crl.getNextUpdate());
                logger.warn((Object)message);
                if (this.abortIfNonUpdated) {
                    throw new GeneralSecurityException(message);
                }
            }
            return Collections.singletonList(crl);
        }

        private X509CRL loadCRL() throws GeneralSecurityException {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            X509CRL crl = null;
            if (this.cRLPath != null) {
                if (this.cRLPath.startsWith("http") || this.cRLPath.startsWith("https")) {
                    try {
                        crl = this.loadFromURI(cf, new URI(this.cRLPath));
                    }
                    catch (URISyntaxException e) {
                        logger.error((Object)e.getMessage());
                    }
                } else if (this.cRLPath.startsWith("ldap")) {
                    try {
                        crl = this.loadCRLFromLDAP(cf, new URI(this.cRLPath));
                    }
                    catch (URISyntaxException e) {
                        logger.error((Object)e.getMessage());
                    }
                } else {
                    crl = this.loadCRLFromFile(cf, this.cRLPath);
                }
            }
            return crl;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive exception aggregation
         */
        private X509CRL loadFromURI(CertificateFactory cf, URI remoteURI) throws GeneralSecurityException {
            try {
                logger.debugf("Loading CRL from %s", (Object)remoteURI.toString());
                CloseableHttpClient httpClient = ((HttpClientProvider)this.session.getProvider(HttpClientProvider.class)).getHttpClient();
                HttpGet get = new HttpGet(remoteURI);
                get.setHeader("Pragma", "no-cache");
                get.setHeader("Cache-Control", "no-cache, no-store");
                try (CloseableHttpResponse response = httpClient.execute((HttpUriRequest)get);){
                    X509CRL x509CRL;
                    try {
                        X509CRL crl;
                        InputStream content = response.getEntity().getContent();
                        x509CRL = crl = this.loadFromStream(cf, content);
                    }
                    catch (Throwable throwable) {
                        EntityUtils.consumeQuietly((HttpEntity)response.getEntity());
                        throw throwable;
                    }
                    EntityUtils.consumeQuietly((HttpEntity)response.getEntity());
                    return x509CRL;
                }
            }
            catch (IOException ex) {
                logger.errorf(ex.getMessage(), new Object[0]);
                return null;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private X509CRL loadCRLFromLDAP(CertificateFactory cf, URI remoteURI) throws GeneralSecurityException {
            block7: {
                X509CRL x509CRL;
                Hashtable<String, String> env = new Hashtable<String, String>(2);
                env.put("java.naming.factory.initial", this.ldapContext.getLdapFactoryClassName());
                env.put("java.naming.provider.url", remoteURI.toString());
                InitialDirContext ctx = new InitialDirContext(env);
                try {
                    X509CRL crl;
                    Attributes attrs = ctx.getAttributes("");
                    Attribute cRLAttribute = attrs.get("certificateRevocationList;binary");
                    byte[] data = (byte[])cRLAttribute.get();
                    if (data == null || data.length == 0) {
                        throw new CertificateException(String.format("Failed to download CRL from \"%s\"", remoteURI.toString()));
                    }
                    x509CRL = crl = this.loadFromStream(cf, new ByteArrayInputStream(data));
                }
                catch (Throwable throwable) {
                    try {
                        ctx.close();
                        throw throwable;
                    }
                    catch (NamingException e) {
                        logger.error((Object)e.getMessage());
                        break block7;
                    }
                    catch (IOException e) {
                        logger.error((Object)e.getMessage());
                    }
                }
                ctx.close();
                return x509CRL;
            }
            return null;
        }

        private X509CRL loadCRLFromFile(CertificateFactory cf, String relativePath) throws GeneralSecurityException {
            block8: {
                X509CRL x509CRL;
                File f;
                String configDir = System.getProperty("jboss.server.config.dir");
                if (configDir == null || !(f = new File(configDir + File.separator + relativePath)).isFile()) break block8;
                logger.debugf("Loading CRL from %s", (Object)f.getAbsolutePath());
                if (!f.canRead()) {
                    throw new IOException(String.format("Unable to read CRL from \"%s\"", f.getAbsolutePath()));
                }
                FileInputStream is = new FileInputStream(f.getAbsolutePath());
                try {
                    X509CRL crl;
                    x509CRL = crl = this.loadFromStream(cf, is);
                }
                catch (Throwable throwable) {
                    try {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                        throw throwable;
                    }
                    catch (IOException ex) {
                        logger.errorf(ex.getMessage(), new Object[0]);
                    }
                }
                is.close();
                return x509CRL;
            }
            return null;
        }

        private X509CRL loadFromStream(CertificateFactory cf, InputStream is) throws IOException, CRLException {
            DataInputStream dis = new DataInputStream(is);
            X509CRL crl = (X509CRL)cf.generateCRL(dis);
            dis.close();
            return crl;
        }
    }

    public static class CertificateValidatorBuilder {
        KeycloakSession session;
        int _keyUsageBits = 0;
        List<String> _extendedKeyUsage = new LinkedList<String>();
        List<String> _certificatePolicy = new LinkedList<String>();
        String _certificatePolicyMode;
        boolean _crlCheckingEnabled;
        boolean _crlAbortIfNonUpdated;
        boolean _crldpEnabled;
        CRLLoaderImpl _crlLoader;
        boolean _ocspEnabled;
        boolean _ocspFailOpen;
        String _responderUri;
        X509Certificate _responderCert;
        boolean _timestampValidationEnabled;
        boolean _trustValidationEnabled;

        public CertificateValidatorBuilder session(KeycloakSession session) {
            this.session = session;
            return this;
        }

        public KeyUsageValidationBuilder keyUsage() {
            return new KeyUsageValidationBuilder(this);
        }

        public ExtendedKeyUsageValidationBuilder extendedKeyUsage() {
            return new ExtendedKeyUsageValidationBuilder(this);
        }

        public CertificatePolicyValidationBuilder certificatePolicy() {
            return new CertificatePolicyValidationBuilder(this);
        }

        public RevocationStatusCheckBuilder revocation() {
            return new RevocationStatusCheckBuilder(this);
        }

        public TimestampValidationBuilder timestampValidation() {
            return new TimestampValidationBuilder(this);
        }

        public TrustValidationBuilder trustValidation() {
            return new TrustValidationBuilder(this);
        }

        public CertificateValidator build(X509Certificate[] certs) {
            if (this._crlLoader == null) {
                this._crlLoader = new CRLFileLoader(this.session, "", this._crlAbortIfNonUpdated);
            }
            return new CertificateValidator(certs, this._keyUsageBits, this._extendedKeyUsage, this._certificatePolicy, this._certificatePolicyMode, this._crlCheckingEnabled, this._crlAbortIfNonUpdated, this._crldpEnabled, this._crlLoader, this._ocspEnabled, this._ocspFailOpen, new BouncyCastleOCSPChecker(this.session, this._responderUri, this._responderCert), this.session, this._timestampValidationEnabled, this._trustValidationEnabled);
        }

        public class KeyUsageValidationBuilder {
            CertificateValidatorBuilder _parent;

            KeyUsageValidationBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public KeyUsageValidationBuilder enableDigitalSignatureBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.DIGITAL_SIGNATURE.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enablecRLSignBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.CRLSIGN.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableDataEncriphermentBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.DATA_ENCIPHERMENT.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableDecipherOnlyBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.DECIPHER_ONLY.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableEnciphermentOnlyBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.ENCIPHERMENT_ONLY.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableKeyAgreementBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.KEY_AGREEMENT.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableKeyEnciphermentBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.KEY_ENCIPHERMENT.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableKeyCertSign() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.KEYCERTSIGN.getInt();
                return this;
            }

            public KeyUsageValidationBuilder enableNonRepudiationBit() {
                CertificateValidatorBuilder.this._keyUsageBits |= 1 << KeyUsageBits.NON_REPUDIATION.getInt();
                return this;
            }

            public CertificateValidatorBuilder back() {
                return this._parent;
            }

            CertificateValidatorBuilder parse(String keyUsage) {
                String[] strs;
                if (keyUsage == null || keyUsage.trim().length() == 0) {
                    return this._parent;
                }
                for (String s : strs = keyUsage.split("[,]")) {
                    try {
                        KeyUsageBits bit = KeyUsageBits.parse(s.trim());
                        switch (bit.ordinal()) {
                            case 6: {
                                this.enablecRLSignBit();
                                break;
                            }
                            case 3: {
                                this.enableDataEncriphermentBit();
                                break;
                            }
                            case 8: {
                                this.enableDecipherOnlyBit();
                                break;
                            }
                            case 0: {
                                this.enableDigitalSignatureBit();
                                break;
                            }
                            case 7: {
                                this.enableEnciphermentOnlyBit();
                                break;
                            }
                            case 4: {
                                this.enableKeyAgreementBit();
                                break;
                            }
                            case 2: {
                                this.enableKeyEnciphermentBit();
                                break;
                            }
                            case 5: {
                                this.enableKeyCertSign();
                                break;
                            }
                            case 1: {
                                this.enableNonRepudiationBit();
                            }
                        }
                    }
                    catch (IllegalArgumentException e) {
                        logger.warnf("Unable to parse key usage bit: \"%s\"", (Object)s);
                    }
                    catch (IndexOutOfBoundsException e) {
                        logger.warnf("Invalid key usage bit: \"%s\"", (Object)s);
                    }
                }
                return this._parent;
            }
        }

        public class ExtendedKeyUsageValidationBuilder {
            CertificateValidatorBuilder _parent;

            protected ExtendedKeyUsageValidationBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public CertificateValidatorBuilder parse(String extendedKeyUsage) {
                String[] strs;
                if (extendedKeyUsage == null || extendedKeyUsage.trim().length() == 0) {
                    return this._parent;
                }
                for (String str : strs = extendedKeyUsage.split("[,;:]")) {
                    CertificateValidatorBuilder.this._extendedKeyUsage.add(str.trim());
                }
                return this._parent;
            }
        }

        public class CertificatePolicyValidationBuilder {
            CertificateValidatorBuilder _parent;

            protected CertificatePolicyValidationBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public CertificatePolicyValidationBuilder mode(String mode) {
                CertificateValidatorBuilder.this._certificatePolicyMode = mode;
                return this;
            }

            public CertificateValidatorBuilder parse(String certificatePolicy) {
                String[] strs;
                if (certificatePolicy == null || certificatePolicy.trim().length() == 0) {
                    return this._parent;
                }
                for (String str : strs = certificatePolicy.split("[,;:]")) {
                    CertificateValidatorBuilder.this._certificatePolicy.add(str.trim());
                }
                return this._parent;
            }
        }

        public class RevocationStatusCheckBuilder {
            CertificateValidatorBuilder _parent;

            protected RevocationStatusCheckBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public RevocationStatusCheckBuilder crlAbortIfNonUpdated(boolean abortIfNonUpdated) {
                CertificateValidatorBuilder.this._crlAbortIfNonUpdated = abortIfNonUpdated;
                return this;
            }

            public GotCRL cRLEnabled(boolean value) {
                CertificateValidatorBuilder.this._crlCheckingEnabled = value;
                return new GotCRL();
            }

            public class GotCRL {
                public GotCRLDP cRLDPEnabled(boolean value) {
                    CertificateValidatorBuilder.this._crldpEnabled = value;
                    return new GotCRLDP();
                }
            }

            public class GotOCSPFailOpen {
                public GotOCSPFailOpen oCSPResponseCertificate(String responderCert) {
                    if (responderCert != null && !responderCert.isEmpty()) {
                        try {
                            CertificateValidatorBuilder.this._responderCert = PemUtils.decodeCertificate((String)responderCert);
                            CertificateValidatorBuilder.this._responderCert.checkValidity();
                        }
                        catch (CertificateException e) {
                            logger.warnf("Ignoring invalid certificate: %s", (Object)CertificateValidatorBuilder.this._responderCert);
                            CertificateValidatorBuilder.this._responderCert = null;
                        }
                    }
                    return new GotOCSPFailOpen();
                }

                public CertificateValidatorBuilder oCSPResponderURI(String responderURI) {
                    CertificateValidatorBuilder.this._responderUri = responderURI;
                    return RevocationStatusCheckBuilder.this._parent;
                }
            }

            public class GotOCSP {
                public GotOCSPFailOpen oCSPFailOpen(boolean ocspFailOpen) {
                    CertificateValidatorBuilder.this._ocspFailOpen = ocspFailOpen;
                    return new GotOCSPFailOpen();
                }
            }

            public class GotCRLDP {
                public GotCRLRelativePath cRLrelativePath(String value) {
                    if (value != null) {
                        CertificateValidatorBuilder.this._crlLoader = new CRLListLoader(CertificateValidatorBuilder.this.session, value, CertificateValidatorBuilder.this._crlAbortIfNonUpdated);
                    }
                    return new GotCRLRelativePath();
                }

                public GotCRLRelativePath cRLLoader(CRLLoaderImpl cRLLoader) {
                    if (cRLLoader != null) {
                        CertificateValidatorBuilder.this._crlLoader = cRLLoader;
                    }
                    return new GotCRLRelativePath();
                }
            }

            public class GotCRLRelativePath {
                public GotOCSP oCSPEnabled(boolean value) {
                    CertificateValidatorBuilder.this._ocspEnabled = value;
                    return new GotOCSP();
                }
            }
        }

        public class TimestampValidationBuilder {
            CertificateValidatorBuilder _parent;

            protected TimestampValidationBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public CertificateValidatorBuilder enabled(boolean timestampValidationEnabled) {
                CertificateValidatorBuilder.this._timestampValidationEnabled = timestampValidationEnabled;
                return this._parent;
            }
        }

        public class TrustValidationBuilder {
            CertificateValidatorBuilder _parent;

            protected TrustValidationBuilder(CertificateValidatorBuilder parent) {
                this._parent = parent;
            }

            public CertificateValidatorBuilder enabled(boolean value) {
                CertificateValidatorBuilder.this._trustValidationEnabled = value;
                return this._parent;
            }
        }
    }

    public static class CRLListLoader
    extends CRLLoaderImpl {
        private final List<CRLLoaderImpl> delegates;

        public CRLListLoader(KeycloakSession session, String cRLConfigValue, boolean abortIfNonUpdated) {
            String[] delegatePaths = Constants.CFG_DELIMITER_PATTERN.split(cRLConfigValue);
            this.delegates = Arrays.stream(delegatePaths).map(cRLPath -> new CRLFileLoader(session, (String)cRLPath, abortIfNonUpdated)).collect(Collectors.toList());
        }

        @Override
        public Collection<X509CRL> getX509CRLs() throws GeneralSecurityException {
            LinkedList<X509CRL> result = new LinkedList<X509CRL>();
            for (CRLLoaderImpl delegate : this.delegates) {
                result.addAll(delegate.getX509CRLs());
            }
            return result;
        }
    }

    public static class CRLLoaderProxy
    extends CRLLoaderImpl {
        private final X509CRL _crl;

        public CRLLoaderProxy(X509CRL crl) {
            this._crl = crl;
        }

        @Override
        public Collection<X509CRL> getX509CRLs() throws GeneralSecurityException {
            return Collections.singleton(this._crl);
        }
    }

    public static class BouncyCastleOCSPChecker
    extends OCSPChecker {
        private final KeycloakSession session;
        private final String responderUri;
        private final X509Certificate responderCert;

        BouncyCastleOCSPChecker(KeycloakSession session, String responderUri, X509Certificate responderCert) {
            this.session = session;
            this.responderUri = responderUri;
            this.responderCert = responderCert;
        }

        @Override
        public OCSPProvider.OCSPRevocationStatus check(X509Certificate cert, X509Certificate issuerCertificate) throws CertPathValidatorException {
            OCSPProvider.OCSPRevocationStatus ocspRevocationStatus = null;
            if (this.responderUri == null || this.responderUri.trim().length() == 0) {
                OCSPProvider ocspProvider = (OCSPProvider)CryptoIntegration.getProvider().getOCSPProver(OCSPProvider.class);
                ocspRevocationStatus = ocspProvider.check(this.session, cert, issuerCertificate);
            } else {
                URI uri;
                try {
                    uri = new URI(this.responderUri);
                }
                catch (URISyntaxException e) {
                    String message = String.format("Unable to check certificate revocation status using OCSP.\n%s", e.getMessage());
                    throw new CertPathValidatorException(message, e);
                }
                logger.tracef("Responder URI \"%s\" will be used to verify revocation status of the certificate using OCSP with responderCert=%s", (Object)uri.toString(), (Object)this.responderCert);
                OCSPProvider ocspProvider = (OCSPProvider)CryptoIntegration.getProvider().getOCSPProver(OCSPProvider.class);
                ocspRevocationStatus = ocspProvider.check(this.session, cert, issuerCertificate, uri, this.responderCert, null);
            }
            return ocspRevocationStatus;
        }
    }

    public static class LdapContext {
        private final String ldapFactoryClassName;

        public LdapContext() {
            this.ldapFactoryClassName = "com.sun.jndi.ldap.LdapCtxFactory";
        }

        public LdapContext(String ldapFactoryClassName) {
            this.ldapFactoryClassName = ldapFactoryClassName;
        }

        public String getLdapFactoryClassName() {
            return this.ldapFactoryClassName;
        }
    }
}

