// Copyright 2017 The PDFium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// Original code copyright 2014 Foxit Software Inc. http://www.foxitsoftware.com

#include "core/fxcrt/widetext_buffer.h"

#include "core/fxcrt/fx_safe_types.h"
#include "core/fxcrt/fx_system.h"

namespace fxcrt {

size_t WideTextBuffer::GetLength() const {
  return GetSize() / sizeof(wchar_t);
}

pdfium::span<wchar_t> WideTextBuffer::GetWideSpan() {
  return pdfium::make_span(reinterpret_cast<wchar_t*>(m_buffer.data()),
                           GetLength());
}

pdfium::span<const wchar_t> WideTextBuffer::GetWideSpan() const {
  return pdfium::make_span(reinterpret_cast<const wchar_t*>(m_buffer.data()),
                           GetLength());
}

WideStringView WideTextBuffer::AsStringView() const {
  return WideStringView(GetWideSpan());
}

WideString WideTextBuffer::MakeString() const {
  return WideString(AsStringView());
}

void WideTextBuffer::AppendChar(wchar_t ch) {
  pdfium::span<wchar_t> new_span = ExpandWideBuf(1);
  new_span[0] = ch;
}

void WideTextBuffer::Delete(size_t start_index, size_t count) {
  DeleteBuf(start_index * sizeof(wchar_t), count * sizeof(wchar_t));
}

WideTextBuffer& WideTextBuffer::operator<<(ByteStringView ascii) {
  pdfium::span<wchar_t> new_span = ExpandWideBuf(ascii.GetLength());
  for (size_t i = 0; i < ascii.GetLength(); ++i)
    new_span[i] = ascii[i];
  return *this;
}

WideTextBuffer& WideTextBuffer::operator<<(WideStringView str) {
  AppendBlock(str.unterminated_c_str(), str.GetLength() * sizeof(wchar_t));
  return *this;
}

WideTextBuffer& WideTextBuffer::operator<<(const WideString& str) {
  AppendBlock(str.c_str(), str.GetLength() * sizeof(wchar_t));
  return *this;
}

WideTextBuffer& WideTextBuffer::operator<<(const wchar_t* lpsz) {
  AppendBlock(lpsz, wcslen(lpsz) * sizeof(wchar_t));
  return *this;
}

WideTextBuffer& WideTextBuffer::operator<<(const WideTextBuffer& buf) {
  AppendBlock(buf.m_buffer.data(), buf.GetSize());
  return *this;
}

pdfium::span<wchar_t> WideTextBuffer::ExpandWideBuf(size_t char_count) {
  size_t original_count = GetLength();
  FX_SAFE_SIZE_T safe_bytes = char_count;
  safe_bytes *= sizeof(wchar_t);
  size_t bytes = safe_bytes.ValueOrDie();
  ExpandBuf(bytes);
  m_DataSize += bytes;
  return GetWideSpan().subspan(original_count);
}

}  // namespace fxcrt
