// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/quick_pair/repository/fast_pair_repository_impl.h"

#include "ash/quick_pair/common/fast_pair/fast_pair_metrics.h"
#include "ash/quick_pair/common/logging.h"
#include "ash/quick_pair/proto/fastpair.pb.h"
#include "ash/quick_pair/proto/fastpair_data.pb.h"
#include "ash/quick_pair/repository/fast_pair/device_id_map.h"
#include "ash/quick_pair/repository/fast_pair/device_image_store.h"
#include "ash/quick_pair/repository/fast_pair/device_metadata_fetcher.h"
#include "ash/quick_pair/repository/fast_pair/fast_pair_image_decoder_impl.h"
#include "ash/quick_pair/repository/fast_pair/footprints_fetcher.h"
#include "ash/quick_pair/repository/fast_pair/footprints_fetcher_impl.h"
#include "ash/quick_pair/repository/fast_pair/pending_write_store.h"
#include "ash/quick_pair/repository/fast_pair/proto_conversions.h"
#include "ash/quick_pair/repository/fast_pair/saved_device_registry.h"
#include "base/callback_helpers.h"
#include "base/memory/scoped_refptr.h"
#include "base/strings/string_number_conversions.h"
#include "chromeos/ash/components/network/network_handler.h"
#include "chromeos/ash/components/network/network_state_handler.h"
#include "chromeos/ash/services/bluetooth_config/public/cpp/device_image_info.h"
#include "chromeos/ash/services/quick_pair/public/cpp/account_key_filter.h"
#include "crypto/sha2.h"
#include "device/bluetooth/bluetooth_adapter.h"
#include "device/bluetooth/bluetooth_adapter_factory.h"
#include "device/bluetooth/bluetooth_device.h"
#include "device/bluetooth/public/cpp/bluetooth_address.h"

namespace {

constexpr base::TimeDelta kOfflineRetryTimeout = base::Minutes(1);
constexpr base::TimeDelta kCacheInvalidationTime = base::Minutes(30);

// Checks if the mac address of a FastPairDevice is the same as the given
// |mac_address| by checking if the SHA256 from the given |device| equals to
// SHA256(concat(account_key of |device|, |mac_address|)).
bool IsDeviceSha256Matched(const nearby::fastpair::FastPairDevice& device,
                           const std::string& mac_address) {
  if (!device.has_account_key() ||
      !device.has_sha256_account_key_public_address()) {
    return false;
  }

  return device.sha256_account_key_public_address() ==
         ash::quick_pair::FastPairRepository::
             GenerateSha256OfAccountKeyAndMacAddress(device.account_key(),
                                                     mac_address);
}

}  // namespace

namespace ash {
namespace quick_pair {

FastPairRepositoryImpl::FastPairRepositoryImpl()
    : device_metadata_fetcher_(std::make_unique<DeviceMetadataFetcher>()),
      footprints_fetcher_(std::make_unique<FootprintsFetcherImpl>()),
      image_decoder_(std::make_unique<FastPairImageDecoderImpl>()),
      device_image_store_(
          std::make_unique<DeviceImageStore>(image_decoder_.get())),
      footprints_last_updated_(base::Time::UnixEpoch()) {
  device::BluetoothAdapterFactory::Get()->GetAdapter(base::BindOnce(
      &FastPairRepositoryImpl::OnGetAdapter, weak_ptr_factory_.GetWeakPtr()));
  chromeos::NetworkHandler::Get()->network_state_handler()->AddObserver(
      this, FROM_HERE);
}

void FastPairRepositoryImpl::OnGetAdapter(
    scoped_refptr<device::BluetoothAdapter> adapter) {
  adapter_ = adapter;
  device_id_map_ = std::make_unique<DeviceIdMap>(adapter_);
  saved_device_registry_ = std::make_unique<SavedDeviceRegistry>(adapter_);
}

FastPairRepositoryImpl::FastPairRepositoryImpl(
    scoped_refptr<device::BluetoothAdapter> adapter,
    std::unique_ptr<DeviceMetadataFetcher> device_metadata_fetcher,
    std::unique_ptr<FootprintsFetcher> footprints_fetcher,
    std::unique_ptr<FastPairImageDecoder> image_decoder,
    std::unique_ptr<DeviceIdMap> device_id_map,
    std::unique_ptr<DeviceImageStore> device_image_store,
    std::unique_ptr<SavedDeviceRegistry> saved_device_registry,
    std::unique_ptr<PendingWriteStore> pending_write_store)
    : adapter_(adapter),
      device_metadata_fetcher_(std::move(device_metadata_fetcher)),
      footprints_fetcher_(std::move(footprints_fetcher)),
      image_decoder_(std::move(image_decoder)),
      device_id_map_(std::move(device_id_map)),
      device_image_store_(std::move(device_image_store)),
      saved_device_registry_(std::move(saved_device_registry)),
      pending_write_store_(std::move(pending_write_store)),
      footprints_last_updated_(base::Time::UnixEpoch()),
      retry_write_last_attempted_(base::Time::UnixEpoch()) {}

FastPairRepositoryImpl::~FastPairRepositoryImpl() {
  chromeos::NetworkHandler::Get()->network_state_handler()->RemoveObserver(
      this, FROM_HERE);
}

void FastPairRepositoryImpl::GetDeviceMetadata(
    const std::string& hex_model_id,
    DeviceMetadataCallback callback) {
  std::string normalized_id = base::ToUpperASCII(hex_model_id);
  if (metadata_cache_.contains(normalized_id)) {
    QP_LOG(VERBOSE) << __func__ << ": Data already in cache.";
    RecordFastPairRepositoryCacheResult(/*success=*/true);
    std::move(callback).Run(metadata_cache_[normalized_id].get(),
                            /*has_retryable_error=*/false);
    return;
  }
  QP_LOG(VERBOSE) << __func__ << ": Not cached, fetching from web service.";
  RecordFastPairRepositoryCacheResult(/*success=*/false);
  device_metadata_fetcher_->LookupHexDeviceId(
      normalized_id, base::BindOnce(&FastPairRepositoryImpl::OnMetadataFetched,
                                    weak_ptr_factory_.GetWeakPtr(),
                                    normalized_id, std::move(callback)));
}

void FastPairRepositoryImpl::OnMetadataFetched(
    const std::string& normalized_model_id,
    DeviceMetadataCallback callback,
    absl::optional<nearby::fastpair::GetObservedDeviceResponse> response,
    bool has_retryable_error) {
  if (!response) {
    std::move(callback).Run(nullptr, has_retryable_error);
    return;
  }
  if (response->image().empty()) {
    metadata_cache_[normalized_model_id] =
        std::make_unique<DeviceMetadata>(std::move(*response), gfx::Image());
    std::move(callback).Run(metadata_cache_[normalized_model_id].get(),
                            /*has_retryable_error=*/false);
    return;
  }

  const std::string& string_data = response->image();
  std::vector<uint8_t> binary_data(string_data.begin(), string_data.end());

  image_decoder_->DecodeImage(
      binary_data,
      /*resize_to_notification_size=*/true,
      base::BindOnce(&FastPairRepositoryImpl::OnImageDecoded,
                     weak_ptr_factory_.GetWeakPtr(), normalized_model_id,
                     std::move(callback), *response));
}

void FastPairRepositoryImpl::OnImageDecoded(
    const std::string& normalized_model_id,
    DeviceMetadataCallback callback,
    nearby::fastpair::GetObservedDeviceResponse response,
    gfx::Image image) {
  metadata_cache_[normalized_model_id] =
      std::make_unique<DeviceMetadata>(response, std::move(image));
  std::move(callback).Run(metadata_cache_[normalized_model_id].get(),
                          /*has_retryable_error=*/false);
}

bool FastPairRepositoryImpl::IsAccountKeyPairedLocally(
    const std::vector<uint8_t>& account_key) {
  // Before we check if the |account_key| matches any of the devices saved in
  // the Saved Device Registry, we fetch all the devices saved to the user's
  // account and cross check the SHA256(concat(account_key, mac_address)) of
  // the saved devices with the SHA256(concat(account_key, mac_address)) of any
  // devices paired to the adapter using the paired device's mac address and
  // the given |account_key|. If there are any matches, we should add the
  // (mac_address, account_key) pair to the Saved Device Registry before
  // completing our check. This handles the edge case where a user pairs a
  // device already saved to their account from another platform via classic
  // BT pairing, and the device still emits a not discoverable advertisement,
  // and we want to prevent showing a Subsequent pairing notification in this
  // case.
  for (device::BluetoothDevice* device : adapter_->GetDevices()) {
    if (!device->IsPaired())
      continue;

    // Use the paired device's |mac_address| and the given |account_key| to
    // generate a SHA256(concat(account_key, mac_address)), and use this
    // SHA256 hash to check if there are any matches with any saved devices in
    // Footprints.
    const std::string& mac_address = device->GetAddress();
    std::string paired_device_hash = GenerateSha256OfAccountKeyAndMacAddress(
        std::string(account_key.begin(), account_key.end()), mac_address);

    for (const auto& info : user_devices_cache_.fast_pair_info()) {
      if (info.has_device() &&
          info.device().has_sha256_account_key_public_address() &&
          info.device().sha256_account_key_public_address() ==
              paired_device_hash) {
        QP_LOG(VERBOSE)
            << __func__
            << ": paired device already saved to account at address = "
            << mac_address << "; adding to registry";
        saved_device_registry_->SaveAccountKey(mac_address, account_key);

        // We only expect there to be at most one match with |account_key| in
        // the devices saved to Footprints. An account key is uniquely written
        // to one device in the pairing flows. Since we found a match and
        // saved it locally, we can return "true" since the account key matches
        // a paired device.
        return true;
      }
    }
  }

  return saved_device_registry_->IsAccountKeySavedToRegistry(account_key);
}

void FastPairRepositoryImpl::CheckAccountKeys(
    const AccountKeyFilter& account_key_filter,
    CheckAccountKeysCallback callback) {
  CheckAccountKeysImpl(account_key_filter, std::move(callback),
                       /*allow_cache_refresh=*/true);
}

void FastPairRepositoryImpl::CheckAccountKeysImpl(
    const AccountKeyFilter& account_key_filter,
    CheckAccountKeysCallback callback,
    bool allow_cache_refresh) {
  QP_LOG(INFO) << __func__;
  if (allow_cache_refresh &&
      (base::Time::Now() - footprints_last_updated_) > kCacheInvalidationTime) {
    // If it has been >30 minutes since the cache was updated, try to get
    // user devices from the server before proceeding.
    footprints_fetcher_->GetUserDevices(base::BindOnce(
        &FastPairRepositoryImpl::UpdateCacheAndRetryCheckAccountKeys,
        weak_ptr_factory_.GetWeakPtr(), account_key_filter,
        std::move(callback)));
    return;
  }

  for (const auto& info : user_devices_cache_.fast_pair_info()) {
    if (info.has_device()) {
      const std::string& string_key = info.device().account_key();
      const std::vector<uint8_t> binary_key(string_key.begin(),
                                            string_key.end());
      if (account_key_filter.IsAccountKeyInFilter(binary_key)) {
        nearby::fastpair::StoredDiscoveryItem device;
        if (device.ParseFromString(info.device().discovery_item_bytes())) {
          QP_LOG(INFO) << "Account key matched with a paired device: "
                       << device.title();
          GetDeviceMetadata(
              device.id(),
              base::BindOnce(&FastPairRepositoryImpl::CompleteAccountKeyLookup,
                             weak_ptr_factory_.GetWeakPtr(),
                             std::move(callback), std::move(binary_key)));
          return;
        }
      }
    }
  }

  // On cache miss, query the server to make sure the device isn't saved to the
  // account unless we've already queried the server in the past minute.
  if (allow_cache_refresh &&
      (base::Time::Now() - footprints_last_updated_) > base::Minutes(1)) {
    footprints_fetcher_->GetUserDevices(
        base::BindOnce(&FastPairRepositoryImpl::RetryCheckAccountKeys,
                       weak_ptr_factory_.GetWeakPtr(), account_key_filter,
                       std::move(callback)));
    return;
  }

  std::move(callback).Run(absl::nullopt);
}

void FastPairRepositoryImpl::RetryCheckAccountKeys(
    const AccountKeyFilter& account_key_filter,
    CheckAccountKeysCallback callback,
    absl::optional<nearby::fastpair::UserReadDevicesResponse> user_devices) {
  QP_LOG(INFO) << __func__;
  if (!user_devices) {
    std::move(callback).Run(absl::nullopt);
    return;
  }

  UpdateUserDevicesCache(user_devices);
  CheckAccountKeysImpl(account_key_filter, std::move(callback),
                       /*allow_cache_refresh=*/false);
}

void FastPairRepositoryImpl::UpdateCacheAndRetryCheckAccountKeys(
    const AccountKeyFilter& account_key_filter,
    CheckAccountKeysCallback callback,
    absl::optional<nearby::fastpair::UserReadDevicesResponse> user_devices) {
  QP_LOG(INFO) << __func__;
  if (!user_devices) {
    QP_LOG(INFO) << __func__
                 << "Failed to update user devices cache. Using stale cache";
  } else {
    UpdateUserDevicesCache(user_devices);
  }
  CheckAccountKeysImpl(account_key_filter, std::move(callback),
                       /*allow_cache_refresh=*/false);
}

void FastPairRepositoryImpl::CompleteAccountKeyLookup(
    CheckAccountKeysCallback callback,
    const std::vector<uint8_t> account_key,
    DeviceMetadata* device_metadata,
    bool has_retryable_error) {
  if (!device_metadata) {
    std::move(callback).Run(absl::nullopt);
    return;
  }
  std::move(callback).Run(
      PairingMetadata(device_metadata, std::move(account_key)));
}

void FastPairRepositoryImpl::UpdateUserDevicesCache(
    absl::optional<nearby::fastpair::UserReadDevicesResponse> user_devices) {
  if (user_devices) {
    QP_LOG(VERBOSE) << "Updated user devices cache with "
                    << user_devices->fast_pair_info_size() << " devices.";
    user_devices_cache_ = std::move(*user_devices);
    footprints_last_updated_ = base::Time::Now();
  }
}

void FastPairRepositoryImpl::AssociateAccountKey(
    scoped_refptr<Device> device,
    const std::vector<uint8_t>& account_key) {
  QP_LOG(INFO) << __func__;
  DCHECK(device->classic_address());
  GetDeviceMetadata(
      device->metadata_id,
      base::BindOnce(&FastPairRepositoryImpl::WriteDeviceToFootprints,
                     weak_ptr_factory_.GetWeakPtr(), device->metadata_id,
                     device->classic_address().value(), account_key));
}

bool FastPairRepositoryImpl::AssociateAccountKeyLocally(
    scoped_refptr<Device> device) {
  QP_LOG(VERBOSE) << __func__;

  absl::optional<std::vector<uint8_t>> account_key = device->account_key();
  if (!account_key) {
    QP_LOG(WARNING) << __func__ << ": Account key not found for device.";
    return false;
  }

  DCHECK(device->classic_address());
  saved_device_registry_->SaveAccountKey(device->classic_address().value(),
                                         account_key.value());
  QP_LOG(INFO) << __func__ << ": Saved account key locally.";
  return true;
}

void FastPairRepositoryImpl::WriteDeviceToFootprints(
    const std::string& hex_model_id,
    const std::string& mac_address,
    const std::vector<uint8_t>& account_key,
    DeviceMetadata* metadata,
    bool has_retryable_error) {
  if (!metadata) {
    QP_LOG(WARNING) << __func__ << ": Unable to retrieve metadata.";
    return;
  }

  pending_write_store_->WritePairedDevice(mac_address, hex_model_id);
  footprints_fetcher_->AddUserFastPairInfo(
      BuildFastPairInfo(hex_model_id, account_key, mac_address, metadata),
      base::BindOnce(&FastPairRepositoryImpl::OnWriteDeviceToFootprintsComplete,
                     weak_ptr_factory_.GetWeakPtr(), mac_address, account_key));
}

void FastPairRepositoryImpl::OnWriteDeviceToFootprintsComplete(
    const std::string& mac_address,
    const std::vector<uint8_t>& account_key,
    bool success) {
  if (!success) {
    QP_LOG(WARNING)
        << __func__
        << ": Failed to write device to Footprints--"
           "deferring addition to SavedDeviceRegistry until we succeed.";
    return;
  }
  QP_LOG(INFO) << __func__ << ": Successfully added device to Footprints.";

  // Remove pending write on successful Footprints write
  pending_write_store_->OnPairedDeviceSaved(mac_address);

  // save/update account key in saved device registry
  saved_device_registry_->SaveAccountKey(mac_address, account_key);

  if (saved_device_registry_->IsAccountKeySavedToRegistry(account_key)) {
    QP_LOG(INFO) << __func__
                 << ": Successfully wrote device to Saved Device Registry.";
    return;
  }

  QP_LOG(WARNING) << __func__
                  << ": Failed to write device to Saved Device Registry.";
}

void FastPairRepositoryImpl::CheckOptInStatus(
    CheckOptInStatusCallback callback) {
  footprints_fetcher_->GetUserDevices(
      base::BindOnce(&FastPairRepositoryImpl::OnCheckOptInStatus,
                     weak_ptr_factory_.GetWeakPtr(), std::move(callback)));
}

void FastPairRepositoryImpl::OnCheckOptInStatus(
    CheckOptInStatusCallback callback,
    absl::optional<nearby::fastpair::UserReadDevicesResponse> user_devices) {
  QP_LOG(INFO) << __func__;
  if (!user_devices) {
    QP_LOG(WARNING)
        << __func__
        << ": Missing UserReadDevicesResponse from call to Footprints";
    std::move(callback).Run(nearby::fastpair::OptInStatus::STATUS_UNKNOWN);
    return;
  }

  for (const auto& info : user_devices->fast_pair_info()) {
    if (info.has_opt_in_status()) {
      std::move(callback).Run(info.opt_in_status());
      return;
    }
  }

  std::move(callback).Run(nearby::fastpair::OptInStatus::STATUS_UNKNOWN);
}

void FastPairRepositoryImpl::UpdateOptInStatus(
    nearby::fastpair::OptInStatus opt_in_status,
    UpdateOptInStatusCallback callback) {
  footprints_fetcher_->AddUserFastPairInfo(
      BuildFastPairInfoForOptIn(opt_in_status),
      base::BindOnce(&FastPairRepositoryImpl::OnUpdateOptInStatusComplete,
                     weak_ptr_factory_.GetWeakPtr(), std::move(callback)));
}

void FastPairRepositoryImpl::OnUpdateOptInStatusComplete(
    UpdateOptInStatusCallback callback,
    bool success) {
  QP_LOG(INFO) << __func__ << ": success=" << success;
  std::move(callback).Run(success);
}

void FastPairRepositoryImpl::GetSavedDevices(GetSavedDevicesCallback callback) {
  footprints_fetcher_->GetUserDevices(
      base::BindOnce(&FastPairRepositoryImpl::OnGetSavedDevices,
                     weak_ptr_factory_.GetWeakPtr(), std::move(callback)));
}

void FastPairRepositoryImpl::OnGetSavedDevices(
    GetSavedDevicesCallback callback,
    absl::optional<nearby::fastpair::UserReadDevicesResponse> user_devices) {
  RecordGetSavedDevicesResult(/*success=*/user_devices.has_value());

  // |user_devices| will be null if we either didn't get a response from the
  // Footprints server or if we could not parse the response. Therefore we
  // should bubble up an error status and empty device list to the UI.
  if (!user_devices.has_value()) {
    QP_LOG(WARNING)
        << __func__
        << ": Missing UserReadDevicesResponse from call to Footprints";
    std::move(callback).Run(nearby::fastpair::OptInStatus::
                                STATUS_ERROR_RETRIEVING_FROM_FOOTPRINTS_SERVER,
                            /*saved_device_list=*/{});
    return;
  }

  nearby::fastpair::OptInStatus opt_in_status =
      nearby::fastpair::OptInStatus::STATUS_UNKNOWN;
  std::vector<nearby::fastpair::FastPairDevice> saved_devices;
  for (const auto& info : user_devices->fast_pair_info()) {
    if (info.has_opt_in_status()) {
      opt_in_status = info.opt_in_status();
    }

    if (info.has_device()) {
      saved_devices.push_back(info.device());
    }
  }

  // If the opt in status is `STATUS_OPTED_OUT`, then we can expect the list of
  // saved devices to be empty, since an opted out status removes all saved
  // devices from the list, although there still might be saved devices, if
  // an Android or Chromebook writes to the user's account against their wishes.
  std::move(callback).Run(opt_in_status, std::move(saved_devices));
}

void FastPairRepositoryImpl::DeleteAssociatedDevice(
    const std::string& mac_address,
    DeleteAssociatedDeviceCallback callback) {
  absl::optional<const std::vector<uint8_t>> account_key =
      saved_device_registry_->GetAccountKey(mac_address);
  if (!account_key) {
    QP_LOG(WARNING) << __func__ << ": No saved account key.";
    std::move(callback).Run(/*success=*/false);
    return;
  }
  std::string hex_account_key = base::HexEncode(*account_key);

  QP_LOG(VERBOSE) << __func__
                  << ": Removing device from Footprints with address: "
                  << mac_address;
  pending_write_store_->DeletePairedDevice(mac_address, hex_account_key);
  footprints_fetcher_->DeleteUserDevice(
      hex_account_key,
      base::BindOnce(&FastPairRepositoryImpl::OnDeleteAssociatedDevice,
                     weak_ptr_factory_.GetWeakPtr(), mac_address,
                     std::move(callback)));
}

void FastPairRepositoryImpl::OnDeleteAssociatedDevice(
    const std::string& mac_address,
    DeleteAssociatedDeviceCallback callback,
    bool success) {
  if (!success) {
    QP_LOG(WARNING)
        << __func__
        << ": Failed to remove device from Footprints--"
           "deferring removal from SavedDeviceRegistry until we succeed.";
    std::move(callback).Run(/*success=*/false);
    return;
  }
  QP_LOG(INFO) << __func__ << ": Successfully removed device from Footprints.";

  // Remove pending delete on successful Footprints delete.
  pending_write_store_->OnPairedDeviceDeleted(mac_address);

  // Query the server and update the cache so the removal is reflected.
  footprints_fetcher_->GetUserDevices(
      base::BindOnce(&FastPairRepositoryImpl::UpdateUserDevicesCache,
                     weak_ptr_factory_.GetWeakPtr()));

  if (!saved_device_registry_->GetAccountKey(mac_address).has_value()) {
    QP_LOG(INFO) << __func__
                 << ": Device was already removed from Saved Device Registry.";
    std::move(callback).Run(/*success=*/true);
    return;
  }

  if (saved_device_registry_->DeleteAccountKey(mac_address)) {
    QP_LOG(INFO) << __func__
                 << ": Successfully removed device from Saved Device Registry.";
    std::move(callback).Run(/*success=*/true);
    return;
  }

  QP_LOG(WARNING) << __func__
                  << ": Failed to remove device from Saved Device Registry.";
  std::move(callback).Run(/*success=*/false);
}

void FastPairRepositoryImpl::DefaultNetworkChanged(
    const NetworkState* network) {
  // Only retry when we have an active connected network.
  if (!network || !network->IsConnectedState()) {
    return;
  }

  if (pending_write_store_->GetPendingDeletes().empty())
    return;

  // To prevent API call spam, only try to retry once per timeout.
  if ((base::Time::Now() - retry_write_last_attempted_) <
      kOfflineRetryTimeout) {
    return;
  }

  retry_write_last_attempted_ = base::Time::Now();

  GetSavedDevices(base::BindOnce(&FastPairRepositoryImpl::RetryPendingDeletes,
                                 weak_ptr_factory_.GetWeakPtr()));
}

// Parameter |status| is passed but not used.
void FastPairRepositoryImpl::RetryPendingDeletes(
    nearby::fastpair::OptInStatus status,
    std::vector<nearby::fastpair::FastPairDevice> devices) {
  // For each pending delete, check to see if the account key is stored in
  // Footprints. While the device failed to delete on this Chromebook, it could
  // have been successfully deleted on a different device.
  for (const PendingWriteStore::PendingDelete& pending_delete :
       pending_write_store_->GetPendingDeletes()) {
    QP_LOG(VERBOSE) << __func__
                    << ": Checking if failed delete should be retried "
                       "for account key: "
                    << pending_delete.hex_account_key;

    // Check if this pending delete is for a device that is in Footprints.
    bool found_in_saved_devices = false;
    for (const auto& device : devices) {
      // Account key may be null for a device removed from Android Saved
      // Devices.
      if (!device.has_account_key())
        continue;

      const std::string saved_account_key =
          base::HexEncode(std::vector<uint8_t>(device.account_key().begin(),
                                               device.account_key().end()));
      found_in_saved_devices =
          saved_account_key == pending_delete.hex_account_key;
      if (found_in_saved_devices)
        break;
    }

    // If our failed-to-delete account key is still found in Footprints, then
    // proceed with retrying the delete.
    if (found_in_saved_devices) {
      QP_LOG(VERBOSE) << __func__
                      << ": Retrying delete for device with account key "
                      << pending_delete.hex_account_key;
      footprints_fetcher_->DeleteUserDevice(
          pending_delete.hex_account_key,
          base::BindOnce(&FastPairRepositoryImpl::OnDeleteAssociatedDevice,
                         weak_ptr_factory_.GetWeakPtr(),
                         pending_delete.mac_address, base::DoNothing()));
    } else if (saved_device_registry_->GetAccountKey(pending_delete.mac_address)
                   .has_value()) {
      // If the device was already removed from Footprints, but hasn't been
      // removed from the registry, ensure that we remove it from the registry.
      bool result =
          saved_device_registry_->DeleteAccountKey(pending_delete.mac_address);
      QP_LOG(INFO) << __func__
                   << ": Device removed from Footprints, removing from "
                      "SavedDeviceRegistry was "
                   << (result ? "sucessful." : "unsuccessful.");

      // Remove from our list of pending deletes since the device isn't in
      // Footprints.
      pending_write_store_->OnPairedDeviceDeleted(pending_delete.mac_address);
    }
  }
}

void FastPairRepositoryImpl::DeleteAssociatedDeviceByAccountKey(
    const std::vector<uint8_t>& account_key,
    DeleteAssociatedDeviceByAccountKeyCallback callback) {
  QP_LOG(INFO) << __func__ << ": Removing device from Footprints.";
  footprints_fetcher_->DeleteUserDevice(
      base::HexEncode(account_key),
      base::BindOnce(
          &FastPairRepositoryImpl::OnDeleteAssociatedDeviceByAccountKey,
          weak_ptr_factory_.GetWeakPtr(), account_key, std::move(callback)));
}

void FastPairRepositoryImpl::OnDeleteAssociatedDeviceByAccountKey(
    const std::vector<uint8_t>& account_key,
    DeleteAssociatedDeviceByAccountKeyCallback callback,
    bool footprints_removal_success) {
  // Remove pending delete on successful Footprints delete.
  pending_write_store_->OnPairedDeviceDeleted(account_key);

  bool saved_device_registry_removal_success =
      saved_device_registry_->DeleteAccountKey(account_key);

  QP_LOG(INFO) << __func__ << ": Device removal: from Footprints: "
               << footprints_removal_success << "; from SavedDeviceRegistry: "
               << saved_device_registry_removal_success;
  if (footprints_removal_success) {
    // If removing from footprints was successful, Query the server and update
    // the cache so the removal is reflected.
    footprints_fetcher_->GetUserDevices(
        base::BindOnce(&FastPairRepositoryImpl::UpdateUserDevicesCache,
                       weak_ptr_factory_.GetWeakPtr()));
  }

  std::move(callback).Run(/*success=*/footprints_removal_success &&
                          saved_device_registry_removal_success);
}

void FastPairRepositoryImpl::FetchDeviceImages(scoped_refptr<Device> device) {
  QP_LOG(INFO) << __func__ << ": Fetching device images for model ID "
               << device->metadata_id;
  // Save a record of the device ID -> model ID for this device so that we can
  // display images for device objects that lack a model ID, such as
  // device::BluetoothDevice.
  if (!device_id_map_->SaveModelIdForDevice(device)) {
    QP_LOG(WARNING) << __func__
                    << ": Unable to save device ID -> model ID"
                       " mapping for model ID "
                    << device->metadata_id;
  }

  GetDeviceMetadata(
      device->metadata_id,
      base::BindOnce(&FastPairRepositoryImpl::CompleteFetchDeviceImages,
                     weak_ptr_factory_.GetWeakPtr(), device->metadata_id));
}

absl::optional<std::string>
FastPairRepositoryImpl::GetDeviceDisplayNameFromCache(
    std::vector<uint8_t> account_key) {
  std::string account_key_str =
      std::string(account_key.begin(), account_key.end());

  QP_LOG(INFO) << __func__ << ": Scanning cache for device name.";
  for (const auto& info : user_devices_cache_.fast_pair_info()) {
    if (!info.has_device()) {
      continue;
    }
    const std::string& device_account_key_str = info.device().account_key();

    if (account_key_str == device_account_key_str) {
      nearby::fastpair::StoredDiscoveryItem item;
      item.ParseFromString(info.device().discovery_item_bytes());
      QP_LOG(VERBOSE) << __func__ << ": Found display name: " << item.title();
      return item.title();
    }
  }
  return nullptr;
}

void FastPairRepositoryImpl::CompleteFetchDeviceImages(
    const std::string& hex_model_id,
    DeviceMetadata* device_metadata,
    bool has_retryable_error) {
  if (!device_metadata) {
    QP_LOG(WARNING) << __func__ << ": No metadata available for "
                    << hex_model_id;
    return;
  }

  QP_LOG(INFO) << __func__
               << ": Completing fetching device images for model ID "
               << hex_model_id;
  device_image_store_->FetchDeviceImages(hex_model_id, device_metadata,
                                         base::DoNothing());
}

bool FastPairRepositoryImpl::PersistDeviceImages(scoped_refptr<Device> device) {
  QP_LOG(INFO) << __func__ << ": Persisting device images for model ID "
               << device->metadata_id;
  if (!device_id_map_->PersistRecordsForDevice(device)) {
    QP_LOG(WARNING) << __func__
                    << ": Unable to persist address -> model ID"
                       " mapping for model ID "
                    << device->metadata_id;
  }
  return device_image_store_->PersistDeviceImages(device->metadata_id);
}

bool FastPairRepositoryImpl::EvictDeviceImages(
    const device::BluetoothDevice* device) {
  const std::string device_id = device->GetIdentifier();
  absl::optional<const std::string> hex_model_id =
      device_id_map_->GetModelIdForDeviceId(device_id);
  if (!hex_model_id)
    return false;
  device_id_map_->EvictDeviceIdRecord(device_id);

  // Before evicting images, check if other device IDs map to this model ID.
  if (device_id_map_->HasPersistedRecordsForModelId(hex_model_id.value()))
    return false;

  return device_image_store_->EvictDeviceImages(hex_model_id.value());
}

absl::optional<bluetooth_config::DeviceImageInfo>
FastPairRepositoryImpl::GetImagesForDevice(const std::string& device_id) {
  absl::optional<const std::string> hex_model_id =
      device_id_map_->GetModelIdForDeviceId(device_id);
  if (!hex_model_id) {
    return absl::nullopt;
  }

  return device_image_store_->GetImagesForDeviceModel(hex_model_id.value());
}

void FastPairRepositoryImpl::IsDeviceSavedToAccount(
    const std::string& mac_address,
    IsDeviceSavedToAccountCallback callback) {
  footprints_fetcher_->GetUserDevices(base::BindOnce(
      &FastPairRepositoryImpl::CompleteIsDeviceSavedToAccount,
      weak_ptr_factory_.GetWeakPtr(), mac_address, std::move(callback)));
}

void FastPairRepositoryImpl::CompleteIsDeviceSavedToAccount(
    const std::string& mac_address,
    IsDeviceSavedToAccountCallback callback,
    absl::optional<nearby::fastpair::UserReadDevicesResponse> user_devices) {
  QP_LOG(INFO) << __func__;

  if (!user_devices) {
    QP_LOG(WARNING)
        << __func__
        << ": Missing UserReadDevicesResponse from call to Footprints";
    std::move(callback).Run(false);
    return;
  }

  for (const auto& info : user_devices->fast_pair_info()) {
    if (info.has_device() &&
        IsDeviceSha256Matched(info.device(), mac_address)) {
      QP_LOG(VERBOSE) << __func__
                      << ": found a SHA256 match for device at address = "
                      << mac_address;
      std::move(callback).Run(true);
      return;
    }
  }

  std::move(callback).Run(false);
}

}  // namespace quick_pair
}  // namespace ash
